/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.component.hdfs;

import java.io.IOException;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.camel.Endpoint;
import org.apache.camel.Exchange;
import org.apache.camel.component.hdfs.HdfsConfiguration;
import org.apache.camel.component.hdfs.HdfsEndpoint;
import org.apache.camel.component.hdfs.HdfsHeader;
import org.apache.camel.component.hdfs.HdfsOutputStream;
import org.apache.camel.impl.DefaultProducer;

public class HdfsProducer
extends DefaultProducer {
    private final HdfsConfiguration config;
    private final StringBuilder hdfsPath;
    private final AtomicBoolean idle = new AtomicBoolean(false);
    private ScheduledExecutorService scheduler;
    private HdfsOutputStream ostream;
    private long splitNum;

    public HdfsProducer(HdfsEndpoint endpoint, HdfsConfiguration config) {
        super((Endpoint)endpoint);
        this.config = config;
        this.hdfsPath = config.getFileSystemType().getHdfsPath(config);
    }

    protected void doStart() throws Exception {
        super.doStart();
        StringBuilder actualPath = new StringBuilder(this.hdfsPath);
        if (this.config.getSplitStrategies().size() > 0) {
            actualPath = this.newFileName();
        }
        this.ostream = HdfsOutputStream.createOutputStream(actualPath.toString(), this.config);
        SplitStrategy idleStrategy = null;
        for (SplitStrategy strategy : this.config.getSplitStrategies()) {
            if (strategy.type != SplitStrategyType.IDLE) continue;
            idleStrategy = strategy;
            break;
        }
        if (idleStrategy != null) {
            this.scheduler = this.getEndpoint().getCamelContext().getExecutorServiceStrategy().newScheduledThreadPool((Object)this, "IdleCheck", 1);
            this.log.debug("Creating IdleCheck task scheduled to run every {} millis", (Object)this.config.getCheckIdleInterval());
            this.scheduler.scheduleAtFixedRate(new IdleCheck(idleStrategy), 1000L, this.config.getCheckIdleInterval(), TimeUnit.MILLISECONDS);
        }
    }

    protected void doStop() throws Exception {
        super.doStop();
        this.ostream.close();
        if (this.scheduler != null) {
            this.getEndpoint().getCamelContext().getExecutorServiceStrategy().shutdown((ExecutorService)this.scheduler);
            this.scheduler = null;
        }
    }

    public void process(Exchange exchange) throws Exception {
        Object body = exchange.getIn().getBody();
        Object key = exchange.getIn().getHeader(HdfsHeader.KEY.name());
        boolean split = false;
        List<SplitStrategy> strategies = this.config.getSplitStrategies();
        for (SplitStrategy splitStrategy : strategies) {
            split |= splitStrategy.getType().split(this.ostream, splitStrategy.value, this);
        }
        if (split) {
            this.ostream.close();
            StringBuilder actualPath = this.newFileName();
            this.ostream = HdfsOutputStream.createOutputStream(actualPath.toString(), this.config);
        }
        this.ostream.append(key, body, exchange.getContext().getTypeConverter());
        this.idle.set(false);
    }

    public HdfsOutputStream getOstream() {
        return this.ostream;
    }

    private StringBuilder newFileName() {
        StringBuilder actualPath = new StringBuilder(this.hdfsPath);
        actualPath.append(this.splitNum);
        ++this.splitNum;
        return actualPath;
    }

    public final AtomicBoolean isIdle() {
        return this.idle;
    }

    private final class IdleCheck
    implements Runnable {
        private final SplitStrategy strategy;

        private IdleCheck(SplitStrategy strategy) {
            this.strategy = strategy;
        }

        @Override
        public void run() {
            HdfsProducer.this.log.trace("IdleCheck running");
            if (System.currentTimeMillis() - HdfsProducer.this.ostream.getLastAccess() > this.strategy.value && !HdfsProducer.this.idle.get() && !HdfsProducer.this.ostream.isBusy().get()) {
                HdfsProducer.this.idle.set(true);
                try {
                    HdfsProducer.this.ostream.close();
                }
                catch (IOException iOException) {
                    // empty catch block
                }
            }
        }

        public String toString() {
            return "IdleCheck";
        }
    }

    public static enum SplitStrategyType {
        BYTES{

            @Override
            public boolean split(HdfsOutputStream oldOstream, long value, HdfsProducer producer) {
                return oldOstream.getNumOfWrittenBytes() >= value;
            }
        }
        ,
        MESSAGES{

            @Override
            public boolean split(HdfsOutputStream oldOstream, long value, HdfsProducer producer) {
                return oldOstream.getNumOfWrittenMessages() >= value;
            }
        }
        ,
        IDLE{

            @Override
            public boolean split(HdfsOutputStream oldOstream, long value, HdfsProducer producer) {
                return producer.idle.get();
            }
        };


        public abstract boolean split(HdfsOutputStream var1, long var2, HdfsProducer var4);
    }

    public static final class SplitStrategy {
        private SplitStrategyType type;
        private long value;

        public SplitStrategy(SplitStrategyType type, long value) {
            this.type = type;
            this.value = value;
        }

        public SplitStrategyType getType() {
            return this.type;
        }

        public long getValue() {
            return this.value;
        }
    }
}

