/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.component.hdfs;

import java.io.Closeable;
import java.io.IOException;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.camel.Endpoint;
import org.apache.camel.Exchange;
import org.apache.camel.component.hdfs.HdfsConfiguration;
import org.apache.camel.component.hdfs.HdfsEndpoint;
import org.apache.camel.component.hdfs.HdfsHeader;
import org.apache.camel.component.hdfs.HdfsOutputStream;
import org.apache.camel.impl.DefaultProducer;
import org.apache.camel.util.IOHelper;
import org.slf4j.Logger;

public class HdfsProducer
extends DefaultProducer {
    private final HdfsConfiguration config;
    private final StringBuilder hdfsPath;
    private final AtomicBoolean idle = new AtomicBoolean(false);
    private volatile ScheduledExecutorService scheduler;
    private volatile HdfsOutputStream ostream;
    private long splitNum;

    public HdfsProducer(HdfsEndpoint endpoint, HdfsConfiguration config) {
        super((Endpoint)endpoint);
        this.config = config;
        this.hdfsPath = config.getFileSystemType().getHdfsPath(config);
    }

    public HdfsEndpoint getEndpoint() {
        return (HdfsEndpoint)super.getEndpoint();
    }

    protected void doStart() throws Exception {
        super.doStart();
        if (this.getEndpoint().getConfig().isConnectOnStartup()) {
            this.ostream = this.setupHdfs(true);
        }
        SplitStrategy idleStrategy = null;
        for (SplitStrategy strategy : this.config.getSplitStrategies()) {
            if (strategy.type != SplitStrategyType.IDLE) continue;
            idleStrategy = strategy;
            break;
        }
        if (idleStrategy != null) {
            this.scheduler = this.getEndpoint().getCamelContext().getExecutorServiceManager().newSingleThreadScheduledExecutor((Object)this, "HdfsIdleCheck");
            this.log.debug("Creating IdleCheck task scheduled to run every {} millis", (Object)this.config.getCheckIdleInterval());
            this.scheduler.scheduleAtFixedRate(new IdleCheck(idleStrategy), 1000L, this.config.getCheckIdleInterval(), TimeUnit.MILLISECONDS);
        }
    }

    private synchronized HdfsOutputStream setupHdfs(boolean onStartup) throws Exception {
        if (this.ostream != null) {
            return this.ostream;
        }
        StringBuilder actualPath = new StringBuilder(this.hdfsPath);
        if (this.config.getSplitStrategies().size() > 0) {
            actualPath = this.newFileName();
        }
        if (onStartup) {
            this.log.info("Connecting to hdfs file-system {}:{}/{} (may take a while if connection is not available)", new Object[]{this.config.getHostName(), this.config.getPort(), actualPath.toString()});
        } else if (this.log.isDebugEnabled()) {
            this.log.debug("Connecting to hdfs file-system {}:{}/{} (may take a while if connection is not available)", new Object[]{this.config.getHostName(), this.config.getPort(), actualPath.toString()});
        }
        HdfsOutputStream answer = HdfsOutputStream.createOutputStream(actualPath.toString(), this.config);
        if (onStartup) {
            this.log.info("Connected to hdfs file-system {}:{}/{}", new Object[]{this.config.getHostName(), this.config.getPort(), actualPath.toString()});
        } else if (this.log.isDebugEnabled()) {
            this.log.debug("Connected to hdfs file-system {}:{}/{}", new Object[]{this.config.getHostName(), this.config.getPort(), actualPath.toString()});
        }
        return answer;
    }

    protected void doStop() throws Exception {
        super.doStop();
        if (this.scheduler != null) {
            this.getEndpoint().getCamelContext().getExecutorServiceManager().shutdownNow((ExecutorService)this.scheduler);
            this.scheduler = null;
        }
        if (this.ostream != null) {
            IOHelper.close((Closeable)this.ostream, (String)"output stream", (Logger)this.log);
            this.ostream = null;
        }
    }

    public void process(Exchange exchange) throws Exception {
        Object body = exchange.getIn().getBody();
        Object key = exchange.getIn().getHeader(HdfsHeader.KEY.name());
        if (this.ostream == null) {
            this.ostream = this.setupHdfs(false);
        }
        boolean split = false;
        List<SplitStrategy> strategies = this.config.getSplitStrategies();
        for (SplitStrategy splitStrategy : strategies) {
            split |= splitStrategy.getType().split(this.ostream, splitStrategy.value, this);
        }
        if (split) {
            if (this.ostream != null) {
                IOHelper.close((Closeable)this.ostream, (String)"output stream", (Logger)this.log);
            }
            StringBuilder actualPath = this.newFileName();
            this.ostream = HdfsOutputStream.createOutputStream(actualPath.toString(), this.config);
        }
        this.ostream.append(key, body, exchange.getContext().getTypeConverter());
        this.idle.set(false);
    }

    private StringBuilder newFileName() {
        StringBuilder actualPath = new StringBuilder(this.hdfsPath);
        actualPath.append(this.splitNum);
        ++this.splitNum;
        return actualPath;
    }

    private final class IdleCheck
    implements Runnable {
        private final SplitStrategy strategy;

        private IdleCheck(SplitStrategy strategy) {
            this.strategy = strategy;
        }

        @Override
        public void run() {
            if (HdfsProducer.this.ostream == null) {
                return;
            }
            HdfsProducer.this.log.trace("IdleCheck running");
            if (System.currentTimeMillis() - HdfsProducer.this.ostream.getLastAccess() > this.strategy.value && !HdfsProducer.this.idle.get() && !HdfsProducer.this.ostream.isBusy().get()) {
                HdfsProducer.this.idle.set(true);
                try {
                    HdfsProducer.this.ostream.close();
                }
                catch (IOException iOException) {
                    // empty catch block
                }
            }
        }

        public String toString() {
            return "IdleCheck";
        }
    }

    public static enum SplitStrategyType {
        BYTES{

            @Override
            public boolean split(HdfsOutputStream oldOstream, long value, HdfsProducer producer) {
                return oldOstream.getNumOfWrittenBytes() >= value;
            }
        }
        ,
        MESSAGES{

            @Override
            public boolean split(HdfsOutputStream oldOstream, long value, HdfsProducer producer) {
                return oldOstream.getNumOfWrittenMessages() >= value;
            }
        }
        ,
        IDLE{

            @Override
            public boolean split(HdfsOutputStream oldOstream, long value, HdfsProducer producer) {
                return producer.idle.get();
            }
        };


        public abstract boolean split(HdfsOutputStream var1, long var2, HdfsProducer var4);
    }

    public static final class SplitStrategy {
        private SplitStrategyType type;
        private long value;

        public SplitStrategy(SplitStrategyType type, long value) {
            this.type = type;
            this.value = value;
        }

        public SplitStrategyType getType() {
            return this.type;
        }

        public long getValue() {
            return this.value;
        }
    }
}

