/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.component.hdfs;

import java.io.Closeable;
import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.stream.Collectors;
import javax.security.auth.login.Configuration;
import org.apache.camel.Endpoint;
import org.apache.camel.Exchange;
import org.apache.camel.ExtendedExchange;
import org.apache.camel.Message;
import org.apache.camel.Processor;
import org.apache.camel.RuntimeCamelException;
import org.apache.camel.component.hdfs.HdfsComponent;
import org.apache.camel.component.hdfs.HdfsConfiguration;
import org.apache.camel.component.hdfs.HdfsEndpoint;
import org.apache.camel.component.hdfs.HdfsFileType;
import org.apache.camel.component.hdfs.HdfsHeader;
import org.apache.camel.component.hdfs.HdfsInfo;
import org.apache.camel.component.hdfs.HdfsInfoFactory;
import org.apache.camel.component.hdfs.HdfsInputStream;
import org.apache.camel.component.hdfs.Holder;
import org.apache.camel.support.ScheduledPollConsumer;
import org.apache.camel.util.IOHelper;
import org.apache.commons.lang.StringUtils;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class HdfsConsumer
extends ScheduledPollConsumer {
    private static final Logger LOG = LoggerFactory.getLogger(HdfsConsumer.class);
    private final HdfsConfiguration endpointConfig;
    private final StringBuilder hdfsPath;
    private final Processor processor;
    private final HdfsInfoFactory hdfsInfoFactory;
    private final ReadWriteLock rwLock = new ReentrantReadWriteLock();

    public HdfsConsumer(HdfsEndpoint endpoint, Processor processor, HdfsConfiguration endpointConfig) {
        this(endpoint, processor, endpointConfig, new HdfsInfoFactory(endpointConfig), endpointConfig.getFileSystemType().getHdfsPath(endpointConfig));
    }

    HdfsConsumer(HdfsEndpoint endpoint, Processor processor, HdfsConfiguration endpointConfig, HdfsInfoFactory hdfsInfoFactory, StringBuilder hdfsPath) {
        super((Endpoint)endpoint, processor);
        this.processor = processor;
        this.endpointConfig = endpointConfig;
        this.hdfsPath = hdfsPath;
        this.hdfsInfoFactory = hdfsInfoFactory;
        this.setUseFixedDelay(true);
    }

    public HdfsEndpoint getEndpoint() {
        return (HdfsEndpoint)super.getEndpoint();
    }

    protected void doStart() throws Exception {
        super.doStart();
        if (this.endpointConfig.isConnectOnStartup()) {
            this.setupHdfs(true);
        }
    }

    private HdfsInfo setupHdfs(boolean onStartup) throws IOException {
        String hdfsFsDescription = this.endpointConfig.getFileSystemLabel(this.hdfsPath.toString());
        if (onStartup) {
            LOG.info("Connecting to hdfs file-system {} (may take a while if connection is not available)", (Object)hdfsFsDescription);
        } else {
            LOG.debug("Connecting to hdfs file-system {} (may take a while if connection is not available)", (Object)hdfsFsDescription);
        }
        HdfsInfo answer = this.hdfsInfoFactory.newHdfsInfo(this.hdfsPath.toString());
        if (onStartup) {
            LOG.info("Connected to hdfs file-system {}", (Object)hdfsFsDescription);
        } else {
            LOG.debug("Connected to hdfs file-system {}", (Object)hdfsFsDescription);
        }
        return answer;
    }

    protected int poll() throws Exception {
        Configuration auth = HdfsComponent.getJAASConfiguration();
        try {
            int n = this.doPoll();
            return n;
        }
        finally {
            HdfsComponent.setJAASConfiguration(auth);
        }
    }

    protected int doPoll() throws IOException {
        FileStatus[] fileStatuses;
        HdfsInfo info = this.setupHdfs(false);
        if (info.getFileSystem().isFile(info.getPath())) {
            fileStatuses = info.getFileSystem().globStatus(info.getPath());
        } else {
            Path pattern = info.getPath().suffix("/" + this.endpointConfig.getPattern());
            class ExcludePathFilter
            implements PathFilter {
                ExcludePathFilter() {
                }

                public boolean accept(Path path) {
                    return !path.toString().endsWith(HdfsConsumer.this.endpointConfig.getOpenedSuffix()) && !path.toString().endsWith(HdfsConsumer.this.endpointConfig.getReadSuffix());
                }
            }
            fileStatuses = info.getFileSystem().globStatus(pattern, (PathFilter)new ExcludePathFilter());
        }
        fileStatuses = Optional.ofNullable(fileStatuses).orElse(new FileStatus[0]);
        return this.processFileStatuses(info, fileStatuses);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private int processFileStatuses(HdfsInfo info, FileStatus[] fileStatuses) {
        AtomicInteger totalMessageCount = new AtomicInteger(0);
        List hdfsFiles = Arrays.stream(fileStatuses).filter(status -> this.normalFileIsDirectoryHasSuccessFile((FileStatus)status, info)).filter(this::hasMatchingOwner).limit(this.endpointConfig.getMaxMessagesPerPoll()).map(this::asHdfsFile).filter(Objects::nonNull).collect(Collectors.toList());
        LOG.info("Processing [{}] valid files out of [{}] available.", (Object)hdfsFiles.size(), (Object)fileStatuses.length);
        for (int i = 0; i < hdfsFiles.size(); ++i) {
            HdfsInputStream hdfsFile = (HdfsInputStream)hdfsFiles.get(i);
            try {
                int messageCount = this.processHdfsInputStream(hdfsFile, totalMessageCount);
                LOG.debug("Processed [{}] files out of [{}].", (Object)i, (Object)hdfsFiles.size());
                LOG.debug("File [{}] was split to [{}] messages.", (Object)i, (Object)messageCount);
                continue;
            }
            finally {
                IOHelper.close((Closeable)hdfsFile, (String)"hdfs file", (Logger)LOG);
            }
        }
        return totalMessageCount.get();
    }

    private int processHdfsInputStream(HdfsInputStream hdfsFile, AtomicInteger totalMessageCount) {
        AtomicInteger messageCount = new AtomicInteger(0);
        Holder<Object> currentKey = new Holder<Object>();
        Holder<Object> currentValue = new Holder<Object>();
        while (hdfsFile.next(currentKey, currentValue) >= 0L) {
            this.processHdfsInputStream(hdfsFile, currentKey, currentValue, messageCount, totalMessageCount);
            messageCount.incrementAndGet();
        }
        return messageCount.get();
    }

    private void processHdfsInputStream(HdfsInputStream hdfsFile, Holder<Object> key, Holder<Object> value, AtomicInteger messageCount, AtomicInteger totalMessageCount) {
        Exchange exchange = this.getEndpoint().createExchange();
        Message message = exchange.getIn();
        String fileName = StringUtils.substringAfterLast((String)hdfsFile.getActualPath(), (String)"/");
        message.setHeader("CamelFileName", (Object)fileName);
        message.setHeader("CamelFileNameConsumed", (Object)fileName);
        message.setHeader("CamelFileAbsolutePath", (Object)hdfsFile.getActualPath());
        if (key.getValue() != null) {
            message.setHeader(HdfsHeader.KEY.name(), key.getValue());
        }
        if (hdfsFile.getNumOfReadBytes() >= 0L) {
            message.setHeader("CamelFileLength", (Object)hdfsFile.getNumOfReadBytes());
        }
        message.setBody(value.getValue());
        this.updateNewExchange(exchange, messageCount.get(), hdfsFile);
        LOG.debug("Processing file [{}]", (Object)fileName);
        try {
            this.processor.process(exchange);
            totalMessageCount.incrementAndGet();
        }
        catch (Exception e) {
            exchange.setException((Throwable)e);
        }
        if (exchange.getException() != null) {
            this.getExceptionHandler().handleException((Throwable)exchange.getException());
        }
    }

    private boolean normalFileIsDirectoryHasSuccessFile(FileStatus fileStatus, HdfsInfo info) {
        if (this.endpointConfig.getFileType().equals((Object)HdfsFileType.NORMAL_FILE) && fileStatus.isDirectory()) {
            try {
                Path successPath = new Path(fileStatus.getPath().toString() + "/_SUCCESS");
                if (!info.getFileSystem().exists(successPath)) {
                    return false;
                }
            }
            catch (IOException e) {
                throw new RuntimeCamelException((Throwable)e);
            }
        }
        return true;
    }

    private boolean hasMatchingOwner(FileStatus fileStatus) {
        if (this.endpointConfig.getOwner() != null && !this.endpointConfig.getOwner().equals(fileStatus.getOwner())) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Skipping file: {} as not matching owner: {}", (Object)fileStatus.getPath(), (Object)this.endpointConfig.getOwner());
            }
            return false;
        }
        return true;
    }

    private HdfsInputStream asHdfsFile(FileStatus fileStatus) {
        try {
            this.rwLock.writeLock().lock();
            HdfsInputStream hdfsInputStream = HdfsInputStream.createInputStream(fileStatus.getPath().toString(), this.hdfsInfoFactory);
            return hdfsInputStream;
        }
        finally {
            this.rwLock.writeLock().unlock();
        }
    }

    protected void updateNewExchange(Exchange exchange, int index, HdfsInputStream hdfsFile) {
        ((ExtendedExchange)exchange.adapt(ExtendedExchange.class)).setUnitOfWork(null);
        exchange.setProperty("CamelSplitIndex", (Object)index);
        if (hdfsFile.hasNext()) {
            exchange.setProperty("CamelSplitComplete", (Object)Boolean.FALSE);
        } else {
            exchange.setProperty("CamelSplitComplete", (Object)Boolean.TRUE);
            exchange.setProperty("CamelSplitSize", (Object)(index + 1));
        }
    }
}

