/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.component.kafka;

import java.io.Closeable;
import java.lang.reflect.Field;
import java.time.Duration;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;
import java.util.regex.Pattern;
import org.apache.camel.component.kafka.KafkaConsumer;
import org.apache.camel.component.kafka.KafkaConsumerFatalException;
import org.apache.camel.component.kafka.PollExceptionStrategy;
import org.apache.camel.component.kafka.PollOnError;
import org.apache.camel.component.kafka.consumer.CommitManager;
import org.apache.camel.component.kafka.consumer.CommitManagers;
import org.apache.camel.component.kafka.consumer.support.KafkaConsumerResumeStrategy;
import org.apache.camel.component.kafka.consumer.support.KafkaRecordProcessorFacade;
import org.apache.camel.component.kafka.consumer.support.PartitionAssignmentListener;
import org.apache.camel.component.kafka.consumer.support.ProcessingResult;
import org.apache.camel.component.kafka.consumer.support.ResumeStrategyFactory;
import org.apache.camel.support.BridgeExceptionHandlerToErrorHandler;
import org.apache.camel.support.task.ForegroundTask;
import org.apache.camel.support.task.Tasks;
import org.apache.camel.support.task.budget.Budgets;
import org.apache.camel.support.task.budget.IterationBudget;
import org.apache.camel.util.IOHelper;
import org.apache.camel.util.ReflectionHelper;
import org.apache.camel.util.TimeUtils;
import org.apache.kafka.clients.consumer.Consumer;
import org.apache.kafka.clients.consumer.ConsumerRebalanceListener;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.internals.ConsumerNetworkClient;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.InterruptException;
import org.apache.kafka.common.errors.WakeupException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class KafkaFetchRecords
implements Runnable {
    private static final Logger LOG = LoggerFactory.getLogger(KafkaFetchRecords.class);
    private final KafkaConsumer kafkaConsumer;
    private Consumer consumer;
    private String clientId;
    private final String topicName;
    private final Pattern topicPattern;
    private final String threadId;
    private final Properties kafkaProps;
    private final Map<String, Long> lastProcessedOffset = new HashMap<String, Long>();
    private final PollExceptionStrategy pollExceptionStrategy;
    private final BridgeExceptionHandlerToErrorHandler bridge;
    private final ReentrantLock lock = new ReentrantLock();
    private CommitManager commitManager;
    private Exception lastError;
    private boolean terminated;
    private long currentBackoffInterval;
    private boolean retry = true;
    private boolean reconnect;
    private boolean connected;

    KafkaFetchRecords(KafkaConsumer kafkaConsumer, PollExceptionStrategy pollExceptionStrategy, BridgeExceptionHandlerToErrorHandler bridge, String topicName, Pattern topicPattern, String id, Properties kafkaProps) {
        this.kafkaConsumer = kafkaConsumer;
        this.pollExceptionStrategy = pollExceptionStrategy;
        this.bridge = bridge;
        this.topicName = topicName;
        this.topicPattern = topicPattern;
        this.threadId = topicName + "-Thread " + id;
        this.kafkaProps = kafkaProps;
    }

    @Override
    public void run() {
        if (!this.isKafkaConsumerRunnable()) {
            return;
        }
        do {
            this.terminated = false;
            if (!this.isConnected()) {
                this.currentBackoffInterval = this.kafkaConsumer.getEndpoint().getComponent().getCreateConsumerBackoffInterval();
                ForegroundTask task = Tasks.foregroundTask().withName("Create KafkaConsumer").withBudget((IterationBudget)Budgets.iterationBudget().withMaxIterations(this.kafkaConsumer.getEndpoint().getComponent().getCreateConsumerBackoffMaxAttempts()).withInitialDelay(Duration.ZERO).withInterval(Duration.ofMillis(this.currentBackoffInterval)).build()).build();
                boolean success = task.run(() -> {
                    try {
                        this.createConsumer();
                        this.commitManager = CommitManagers.createCommitManager(this.consumer, this.kafkaConsumer, this.threadId, this.getPrintableTopic());
                    }
                    catch (Exception e) {
                        this.setConnected(false);
                        LOG.warn("Error creating org.apache.kafka.clients.consumer.KafkaConsumer due to: {}", (Object)e.getMessage(), (Object)e);
                        this.lastError = e;
                        return false;
                    }
                    return true;
                });
                if (!success) {
                    int max = this.kafkaConsumer.getEndpoint().getComponent().getCreateConsumerBackoffMaxAttempts();
                    String time = TimeUtils.printDuration((Duration)task.elapsed());
                    String topic = this.getPrintableTopic();
                    String msg = "Gave up creating org.apache.kafka.clients.consumer.KafkaConsumer " + this.threadId + " to " + topic + " after " + max + " attempts (elapsed: " + time + ").";
                    LOG.warn(msg);
                    this.lastError = new KafkaConsumerFatalException(msg, this.lastError);
                    this.terminated = true;
                    break;
                }
                this.currentBackoffInterval = this.kafkaConsumer.getEndpoint().getComponent().getSubscribeConsumerBackoffInterval();
                task = Tasks.foregroundTask().withName("Subscribe KafkaConsumer").withBudget((IterationBudget)Budgets.iterationBudget().withMaxIterations(this.kafkaConsumer.getEndpoint().getComponent().getSubscribeConsumerBackoffMaxAttempts()).withInitialDelay(Duration.ZERO).withInterval(Duration.ofMillis(this.currentBackoffInterval)).build()).build();
                success = task.run(() -> {
                    try {
                        this.initializeConsumer();
                    }
                    catch (Exception e) {
                        this.setConnected(false);
                        LOG.warn("Error subscribing org.apache.kafka.clients.consumer.KafkaConsumer due to: {}", (Object)e.getMessage(), (Object)e);
                        this.lastError = e;
                        return false;
                    }
                    return true;
                });
                if (!success) {
                    int max = this.kafkaConsumer.getEndpoint().getComponent().getCreateConsumerBackoffMaxAttempts();
                    String time = TimeUtils.printDuration((Duration)task.elapsed());
                    String topic = this.getPrintableTopic();
                    String msg = "Gave up subscribing org.apache.kafka.clients.consumer.KafkaConsumer " + this.threadId + " to " + topic + " after " + max + " attempts (elapsed: " + time + ").";
                    LOG.warn(msg);
                    this.lastError = new KafkaConsumerFatalException(msg, this.lastError);
                    this.terminated = true;
                    break;
                }
                this.setConnected(true);
            }
            this.lastError = null;
            this.startPolling();
        } while ((this.isRetrying() || this.isReconnect()) && this.isKafkaConsumerRunnable());
        if (LOG.isInfoEnabled()) {
            LOG.info("Terminating KafkaConsumer thread {} receiving from {}", (Object)this.threadId, (Object)this.getPrintableTopic());
        }
        this.safeUnsubscribe();
        IOHelper.close((Closeable)this.consumer);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void createConsumer() {
        ClassLoader threadClassLoader = Thread.currentThread().getContextClassLoader();
        try {
            Thread.currentThread().setContextClassLoader(org.apache.kafka.clients.consumer.KafkaConsumer.class.getClassLoader());
            long delay = this.kafkaConsumer.getEndpoint().getConfiguration().getPollTimeoutMs();
            String prefix = this.consumer == null ? "Connecting" : "Reconnecting";
            LOG.info("{} Kafka consumer thread ID {} with poll timeout of {} ms", new Object[]{prefix, this.threadId, delay});
            this.consumer = this.kafkaConsumer.getEndpoint().getKafkaClientFactory().getConsumer(this.kafkaProps);
            if (this.clientId == null) {
                this.clientId = this.getKafkaProps().getProperty("client.id");
                if (this.clientId == null) {
                    try {
                        this.clientId = (String)ReflectionHelper.getField((Field)this.consumer.getClass().getDeclaredField("clientId"), (Object)this.consumer);
                    }
                    catch (Exception e) {
                        this.clientId = "";
                    }
                }
            }
        }
        finally {
            Thread.currentThread().setContextClassLoader(threadClassLoader);
        }
    }

    private void initializeConsumer() {
        this.subscribe();
        this.setConnected(false);
        this.setRetry(true);
    }

    private void subscribe() {
        KafkaConsumerResumeStrategy resumeStrategy = ResumeStrategyFactory.newResumeStrategy(this.kafkaConsumer);
        resumeStrategy.setConsumer(this.consumer);
        PartitionAssignmentListener listener = new PartitionAssignmentListener(this.threadId, this.kafkaConsumer.getEndpoint().getConfiguration(), this.lastProcessedOffset, this::isRunnable, this.commitManager, resumeStrategy);
        if (LOG.isInfoEnabled()) {
            LOG.info("Subscribing {} to {}", (Object)this.threadId, (Object)this.getPrintableTopic());
        }
        if (this.topicPattern != null) {
            this.consumer.subscribe(this.topicPattern, (ConsumerRebalanceListener)listener);
        } else {
            this.consumer.subscribe(Arrays.asList(this.topicName.split(",")), (ConsumerRebalanceListener)listener);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void startPolling() {
        long partitionLastOffset = -1L;
        try {
            this.lock.lock();
            long pollTimeoutMs = this.kafkaConsumer.getEndpoint().getConfiguration().getPollTimeoutMs();
            if (LOG.isTraceEnabled()) {
                LOG.trace("Polling {} from {} with timeout: {}", new Object[]{this.threadId, this.getPrintableTopic(), pollTimeoutMs});
            }
            KafkaRecordProcessorFacade recordProcessorFacade = new KafkaRecordProcessorFacade(this.kafkaConsumer, this.lastProcessedOffset, this.threadId, this.commitManager);
            Duration pollDuration = Duration.ofMillis(pollTimeoutMs);
            while (this.isKafkaConsumerRunnable() && this.isRetrying() && this.isConnected()) {
                ConsumerRecords allRecords = this.consumer.poll(pollDuration);
                this.commitManager.processAsyncCommits();
                ProcessingResult result = recordProcessorFacade.processPolledRecords((ConsumerRecords<Object, Object>)allRecords);
                if (!result.isBreakOnErrorHit()) continue;
                LOG.debug("We hit an error ... setting flags to force reconnect");
                this.setReconnect(true);
                this.setConnected(false);
                this.setRetry(false);
            }
            if (!this.isConnected()) {
                LOG.debug("Not reconnecting, check whether to auto-commit or not ...");
                this.commitManager.commit();
            }
            this.safeUnsubscribe();
        }
        catch (InterruptException e) {
            this.kafkaConsumer.getExceptionHandler().handleException("Interrupted while consuming " + this.threadId + " from kafka topic", (Throwable)e);
            this.commitManager.commit();
            LOG.info("Unsubscribing {} from {}", (Object)this.threadId, (Object)this.getPrintableTopic());
            this.safeUnsubscribe();
            Thread.currentThread().interrupt();
        }
        catch (WakeupException e) {
            if (LOG.isTraceEnabled()) {
                LOG.trace("The kafka consumer was woken up while polling on thread {} for {}", (Object)this.threadId, (Object)this.getPrintableTopic());
            }
            this.safeUnsubscribe();
        }
        catch (Exception e) {
            if (LOG.isDebugEnabled()) {
                LOG.warn("Exception {} caught while polling {} from kafka {} at offset {}: {}", new Object[]{e.getClass().getName(), this.threadId, this.getPrintableTopic(), this.lastProcessedOffset, e.getMessage(), e});
            } else {
                LOG.warn("Exception {} caught while polling {} from kafka {} at offset {}: {}", new Object[]{e.getClass().getName(), this.threadId, this.getPrintableTopic(), this.lastProcessedOffset, e.getMessage()});
            }
            this.handleAccordingToStrategy(partitionLastOffset, e);
        }
        finally {
            this.lock.unlock();
            if (!this.isRetrying()) {
                LOG.debug("Closing consumer {}", (Object)this.threadId);
                this.safeUnsubscribe();
                IOHelper.close((Closeable)this.consumer);
            }
        }
    }

    private void handleAccordingToStrategy(long partitionLastOffset, Exception e) {
        PollOnError onError = this.pollExceptionStrategy.handleException(e);
        if (PollOnError.RETRY == onError) {
            this.handlePollRetry();
        } else if (PollOnError.RECONNECT == onError) {
            this.handlePollReconnect();
        } else if (PollOnError.ERROR_HANDLER == onError) {
            this.handlePollErrorHandler(partitionLastOffset, e);
        } else if (PollOnError.DISCARD == onError) {
            this.handlePollDiscard(partitionLastOffset);
        } else if (PollOnError.STOP == onError) {
            this.handlePollStop();
        }
    }

    private void safeUnsubscribe() {
        if (this.consumer == null) {
            return;
        }
        String printableTopic = this.getPrintableTopic();
        try {
            this.consumer.unsubscribe();
        }
        catch (IllegalStateException e) {
            LOG.warn("The consumer is likely already closed. Skipping unsubscribing thread {} from kafka {}", (Object)this.threadId, (Object)printableTopic);
        }
        catch (Exception e) {
            this.kafkaConsumer.getExceptionHandler().handleException("Error unsubscribing thread " + this.threadId + " from kafka " + printableTopic, (Throwable)e);
        }
    }

    private String getPrintableTopic() {
        if (this.topicPattern != null) {
            return "topic pattern " + this.topicPattern;
        }
        return "topic " + this.topicName;
    }

    private void handlePollStop() {
        LOG.warn("Requesting the consumer to stop based on polling exception strategy");
        this.setRetry(false);
        this.setConnected(false);
    }

    private void handlePollDiscard(long partitionLastOffset) {
        LOG.warn("Requesting the consumer to discard the message and continue to the next based on polling exception strategy");
        this.seekToNextOffset(partitionLastOffset);
    }

    private void handlePollErrorHandler(long partitionLastOffset, Exception e) {
        LOG.warn("Deferring processing to the exception handler based on polling exception strategy");
        this.bridge.handleException((Throwable)e);
        this.seekToNextOffset(partitionLastOffset);
    }

    private void handlePollReconnect() {
        LOG.warn("Requesting the consumer to re-connect on the next run based on polling exception strategy");
        this.setReconnect(true);
        this.setConnected(false);
        this.setRetry(false);
    }

    private void handlePollRetry() {
        LOG.warn("Requesting the consumer to retry polling the same message based on polling exception strategy");
        this.setRetry(true);
    }

    private boolean isKafkaConsumerRunnable() {
        return this.kafkaConsumer.isRunAllowed() && !this.kafkaConsumer.isStoppingOrStopped() && !this.kafkaConsumer.isSuspendingOrSuspended();
    }

    private boolean isRunnable() {
        return this.kafkaConsumer.getEndpoint().getCamelContext().isStopping() && !this.kafkaConsumer.isRunAllowed();
    }

    private void seekToNextOffset(long partitionLastOffset) {
        block5: {
            Set tps;
            boolean logged;
            block4: {
                logged = false;
                tps = this.consumer.assignment();
                if (tps == null || partitionLastOffset == -1L) break block4;
                long next = partitionLastOffset + 1L;
                if (LOG.isInfoEnabled()) {
                    LOG.info("Consumer seeking to next offset {} to continue polling next message from {}", (Object)next, (Object)this.getPrintableTopic());
                }
                for (TopicPartition tp : tps) {
                    this.consumer.seek(tp, next);
                }
                break block5;
            }
            if (tps == null) break block5;
            for (TopicPartition tp : tps) {
                long next = this.consumer.position(tp) + 1L;
                if (!logged) {
                    LOG.info("Consumer seeking to next offset {} to continue polling next message from {}", (Object)next, (Object)this.getPrintableTopic());
                    logged = true;
                }
                this.consumer.seek(tp, next);
            }
        }
    }

    private boolean isRetrying() {
        return this.retry;
    }

    private void setRetry(boolean value) {
        this.retry = value;
    }

    private boolean isReconnect() {
        return this.reconnect;
    }

    private void setReconnect(boolean value) {
        this.reconnect = value;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void safeStop() {
        if (this.consumer == null) {
            return;
        }
        long timeout = this.kafkaConsumer.getEndpoint().getConfiguration().getShutdownTimeout();
        try {
            LOG.info("Waiting up to {} milliseconds for the processing to finish", (Object)timeout);
            if (!this.lock.tryLock(timeout, TimeUnit.MILLISECONDS)) {
                LOG.warn("The processing of the current record did not finish within {} seconds", (Object)timeout);
            }
            this.consumer.wakeup();
        }
        catch (InterruptedException e) {
            this.consumer.wakeup();
            Thread.currentThread().interrupt();
        }
        finally {
            this.lock.unlock();
        }
    }

    void stop() {
        this.safeStop();
    }

    public boolean isConnected() {
        return this.connected;
    }

    public void setConnected(boolean connected) {
        this.connected = connected;
    }

    public boolean isReady() {
        if (!this.connected) {
            return false;
        }
        boolean ready = true;
        try {
            if (this.consumer instanceof org.apache.kafka.clients.consumer.KafkaConsumer) {
                org.apache.kafka.clients.consumer.KafkaConsumer kc = (org.apache.kafka.clients.consumer.KafkaConsumer)this.consumer;
                ConsumerNetworkClient nc = (ConsumerNetworkClient)ReflectionHelper.getField((Field)kc.getClass().getDeclaredField("client"), (Object)kc);
                LOG.trace("Health-Check calling org.apache.kafka.clients.consumer.internals.ConsumerNetworkClient.hasReadyNode");
                ready = nc.hasReadyNodes(System.currentTimeMillis());
            }
        }
        catch (Exception e) {
            LOG.debug("Cannot check hasReadyNodes on KafkaConsumer client (ConsumerNetworkClient) due to: " + e.getMessage() + ". This exception is ignored.", (Throwable)e);
        }
        return ready;
    }

    Properties getKafkaProps() {
        return this.kafkaProps;
    }

    String getClientId() {
        return this.clientId;
    }

    Exception getLastError() {
        return this.lastError;
    }

    boolean isTerminated() {
        return this.terminated;
    }

    boolean isRecoverable() {
        return (this.isRetrying() || this.isReconnect()) && this.isKafkaConsumerRunnable();
    }

    long getCurrentRecoveryInterval() {
        return this.currentBackoffInterval;
    }
}

