/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.component.lumberjack.io;

import com.fasterxml.jackson.databind.ObjectMapper;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufOutputStream;
import io.netty.channel.ChannelHandlerContext;
import io.netty.handler.codec.ByteToMessageDecoder;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.zip.Inflater;
import java.util.zip.InflaterOutputStream;
import org.apache.camel.component.lumberjack.io.LumberjackMessage;
import org.apache.camel.component.lumberjack.io.LumberjackWindow;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class LumberjackFrameDecoder
extends ByteToMessageDecoder {
    private static final Logger LOG = LoggerFactory.getLogger(LumberjackFrameDecoder.class);
    private final ObjectMapper jackson = new ObjectMapper();
    private LumberjackWindow window;

    LumberjackFrameDecoder() {
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    protected void decode(ChannelHandlerContext ctx, ByteBuf in, List<Object> out) throws Exception {
        in.markReaderIndex();
        boolean frameDecoded = false;
        try {
            if (!in.isReadable(2)) {
                return;
            }
            byte frameVersion = in.readByte();
            this.verifyVersion(frameVersion);
            short frameType = in.readUnsignedByte();
            LOG.debug("Received a lumberjack frame of type {}", (Object)Character.valueOf((char)frameType));
            switch (frameType) {
                case 74: {
                    frameDecoded = this.handleJsonFrame(in, out);
                    return;
                }
                case 68: {
                    frameDecoded = this.handleDataFrame(in, out);
                    return;
                }
                case 87: {
                    frameDecoded = this.handleWindowFrame(in, frameVersion, out);
                    return;
                }
                case 67: {
                    frameDecoded = this.handleCompressedFrame(ctx, in, out);
                    return;
                }
                default: {
                    throw new RuntimeException("Unsupported frame type=" + frameType);
                }
            }
        }
        finally {
            if (!frameDecoded) {
                LOG.debug("Not enough data to decode a complete frame, retry when more data is available. Reader index was {}", (Object)in.readerIndex());
                in.resetReaderIndex();
            }
        }
    }

    private boolean handleJsonFrame(ByteBuf in, List<Object> out) throws IOException {
        if (!in.isReadable(8)) {
            return false;
        }
        int sequenceNumber = in.readInt();
        String jsonStr = this.readLengthPrefixedString(in);
        if (jsonStr == null) {
            return false;
        }
        Object jsonMessage = this.jackson.readValue(jsonStr, Object.class);
        this.window.addMessage(new LumberjackMessage(sequenceNumber, jsonMessage));
        this.sendWindowIfComplete(out);
        return true;
    }

    private boolean handleDataFrame(ByteBuf in, List<Object> out) {
        if (!in.isReadable(8)) {
            return false;
        }
        int sequenceNumber = in.readInt();
        int entriesCount = in.readInt();
        LinkedHashMap<String, String> dataMessage = new LinkedHashMap<String, String>();
        while (entriesCount-- > 0) {
            String key = this.readLengthPrefixedString(in);
            if (key == null) {
                return false;
            }
            String value = this.readLengthPrefixedString(in);
            if (value == null) {
                return false;
            }
            dataMessage.put(key, value);
        }
        this.window.addMessage(new LumberjackMessage(sequenceNumber, dataMessage));
        this.sendWindowIfComplete(out);
        return true;
    }

    private boolean handleWindowFrame(ByteBuf in, byte frameVersion, List<Object> out) {
        if (!in.isReadable(4)) {
            return false;
        }
        if (this.window != null) {
            LOG.warn("New window size received but the current window was not complete, sending the current window");
            out.add(this.window);
            this.window = null;
        }
        int windowSize = in.readInt();
        this.window = new LumberjackWindow(frameVersion, windowSize);
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean handleCompressedFrame(ChannelHandlerContext ctx, ByteBuf in, List<Object> out) throws Exception {
        if (!in.isReadable(4)) {
            return false;
        }
        int compressedPayloadLength = in.readInt();
        if (!in.isReadable(compressedPayloadLength)) {
            return false;
        }
        Inflater inflater = new Inflater();
        ByteBuf decompressed = ctx.alloc().buffer(compressedPayloadLength);
        try (ByteBufOutputStream buffOutput = new ByteBufOutputStream(decompressed);
             InflaterOutputStream inflaterStream = new InflaterOutputStream((OutputStream)buffOutput, inflater);){
            in.readBytes((OutputStream)inflaterStream, compressedPayloadLength);
        }
        finally {
            inflater.end();
        }
        try {
            while (decompressed.readableBytes() > 0) {
                this.decode(ctx, decompressed, out);
            }
        }
        finally {
            decompressed.release();
        }
        return true;
    }

    private String readLengthPrefixedString(ByteBuf in) {
        if (!in.isReadable(4)) {
            return null;
        }
        int length = in.readInt();
        if (!in.isReadable(length)) {
            return null;
        }
        String str = in.toString(in.readerIndex(), length, StandardCharsets.UTF_8);
        in.skipBytes(length);
        return str;
    }

    private void verifyVersion(byte version) {
        if (this.window == null || this.window.getVersion() == -1) {
            if (version != 49 && version != 50) {
                throw new RuntimeException("Unsupported frame version=" + version);
            }
            LOG.debug("Lumberjack protocol version is {}", (Object)Character.valueOf((char)version));
        } else if (this.window.getVersion() != version) {
            throw new IllegalStateException("Protocol version changed during session from " + this.window.getVersion() + " to " + version);
        }
    }

    public void sendWindowIfComplete(List<Object> out) {
        if (this.window.isComplete()) {
            out.add(this.window);
            this.window = null;
        }
    }
}

