/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.compress.utils;

import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.WritableByteChannel;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.commons.compress.utils.FixedLengthBlockOutputStream;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.hamcrest.Matchers;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;

public class FixedLengthBlockOutputStreamTest {
    private static void assertContainsAtOffset(String msg, byte[] expected, int offset, byte[] actual) {
        MatcherAssert.assertThat((Object)actual.length, (Matcher)Matchers.greaterThanOrEqualTo((Comparable)Integer.valueOf(offset + expected.length)));
        for (int i = 0; i < expected.length; ++i) {
            Assertions.assertEquals((byte)expected[i], (byte)actual[i + offset], (String)String.format("%s ([%d])", msg, i));
        }
    }

    private ByteBuffer getByteBuffer(byte[] msg) {
        int len = msg.length;
        ByteBuffer buf = ByteBuffer.allocate(len);
        buf.put(msg);
        buf.flip();
        return buf;
    }

    private FixedLengthBlockOutputStream newClosedFLBOS() throws IOException {
        int blockSize = 512;
        FixedLengthBlockOutputStream out = new FixedLengthBlockOutputStream((OutputStream)new MockOutputStream(512, false), 512);
        out.write(1);
        Assertions.assertTrue((boolean)out.isOpen());
        out.close();
        Assertions.assertFalse((boolean)out.isOpen());
        return out;
    }

    private void testBuf(int blockSize, String text) throws IOException {
        try (MockWritableByteChannel mock = new MockWritableByteChannel(blockSize, false);){
            ByteArrayOutputStream bos = mock.bos;
            byte[] msg = text.getBytes();
            ByteBuffer buf = this.getByteBuffer(msg);
            try (FixedLengthBlockOutputStream out = new FixedLengthBlockOutputStream((WritableByteChannel)mock, blockSize);){
                out.write(buf);
            }
            double v = Math.ceil((double)msg.length / (double)blockSize) * (double)blockSize;
            Assertions.assertEquals((long)((long)v), (long)bos.size(), (String)"wrong size");
            byte[] output = bos.toByteArray();
            String l = new String(output, 0, msg.length);
            Assertions.assertEquals((Object)text, (Object)l);
            for (int i = msg.length; i < bos.size(); ++i) {
                Assertions.assertEquals((int)0, (int)output[i], (String)String.format("output[%d]", i));
            }
        }
    }

    @Test
    public void testMultiWriteBuf() throws IOException {
        int blockSize = 13;
        try (MockWritableByteChannel mock = new MockWritableByteChannel(13, false);){
            int i;
            String testString = "hello world";
            byte[] msg = "hello world".getBytes();
            int reps = 17;
            try (FixedLengthBlockOutputStream out = new FixedLengthBlockOutputStream((WritableByteChannel)mock, 13);){
                for (int i2 = 0; i2 < 17; ++i2) {
                    ByteBuffer buf = this.getByteBuffer(msg);
                    out.write(buf);
                }
            }
            ByteArrayOutputStream bos = mock.bos;
            double v = Math.ceil((double)(17 * msg.length) / 13.0) * 13.0;
            Assertions.assertEquals((long)((long)v), (long)bos.size(), (String)"wrong size");
            int strLen = msg.length * 17;
            byte[] output = bos.toByteArray();
            String l = new String(output, 0, strLen);
            StringBuilder buf = new StringBuilder(strLen);
            for (i = 0; i < 17; ++i) {
                buf.append("hello world");
            }
            Assertions.assertEquals((Object)buf.toString(), (Object)l);
            for (i = strLen; i < output.length; ++i) {
                Assertions.assertEquals((int)0, (int)output[i]);
            }
        }
    }

    @Test
    public void testPartialWritingThrowsException() {
        IOException e = (IOException)Assertions.assertThrows(IOException.class, () -> this.testWriteAndPad(512, "hello world!\n", true), (String)"Exception for partial write not thrown");
        String msg = e.getMessage();
        Assertions.assertEquals((Object)"Failed to write 512 bytes atomically. Only wrote  511", (Object)msg, (String)"exception message");
    }

    @Test
    public void testSmallWrite() throws IOException {
        this.testWriteAndPad(10240, "hello world!\n", false);
        this.testWriteAndPad(512, "hello world!\n", false);
        this.testWriteAndPad(11, "hello world!\n", false);
        this.testWriteAndPad(3, "hello world!\n", false);
    }

    @Test
    public void testSmallWriteToStream() throws IOException {
        this.testWriteAndPadToStream(10240, "hello world!\n", false);
        this.testWriteAndPadToStream(512, "hello world!\n", false);
        this.testWriteAndPadToStream(11, "hello world!\n", false);
        this.testWriteAndPadToStream(3, "hello     world!\n", false);
    }

    @Test
    public void testWithFileOutputStream() throws IOException {
        int i;
        Path tempFile = Files.createTempFile("xxx", "yyy", new FileAttribute[0]);
        Runtime.getRuntime().addShutdownHook(new Thread(() -> {
            try {
                Files.deleteIfExists(tempFile);
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }));
        int blockSize = 512;
        int reps = 1000;
        try (FixedLengthBlockOutputStream out = new FixedLengthBlockOutputStream(Files.newOutputStream(tempFile.toFile().toPath(), new OpenOption[0]), 512);){
            DataOutputStream dos = new DataOutputStream((OutputStream)out);
            for (int i2 = 0; i2 < 1000; ++i2) {
                dos.writeInt(i2);
            }
        }
        long expectedDataSize = 4000L;
        long expectedFileSize = (long)Math.ceil(7.8125) * 512L;
        Assertions.assertEquals((long)expectedFileSize, (long)Files.size(tempFile), (String)"file size");
        DataInputStream din = new DataInputStream(Files.newInputStream(tempFile, new OpenOption[0]));
        for (i = 0; i < 1000; ++i) {
            Assertions.assertEquals((int)i, (int)din.readInt(), (String)"file int");
        }
        i = 0;
        while ((long)i < expectedFileSize - 4000L) {
            Assertions.assertEquals((int)0, (int)din.read());
            ++i;
        }
        Assertions.assertEquals((int)-1, (int)din.read());
    }

    private void testWriteAndPad(int blockSize, String text, boolean doPartialWrite) throws IOException {
        try (MockWritableByteChannel mock = new MockWritableByteChannel(blockSize, doPartialWrite);){
            byte[] msg = text.getBytes(StandardCharsets.US_ASCII);
            ByteArrayOutputStream bos = mock.bos;
            try (FixedLengthBlockOutputStream out = new FixedLengthBlockOutputStream((WritableByteChannel)mock, blockSize);){
                out.write(msg);
                Assertions.assertEquals((int)(msg.length / blockSize * blockSize), (int)bos.size(), (String)"no partial write");
            }
            this.validate(blockSize, msg, bos.toByteArray());
        }
    }

    private void testWriteAndPadToStream(int blockSize, String text, boolean doPartialWrite) throws IOException {
        try (MockOutputStream mock = new MockOutputStream(blockSize, doPartialWrite);){
            byte[] msg = text.getBytes(StandardCharsets.US_ASCII);
            ByteArrayOutputStream bos = mock.bos;
            try (FixedLengthBlockOutputStream out = new FixedLengthBlockOutputStream((OutputStream)mock, blockSize);){
                out.write(msg);
                Assertions.assertEquals((int)(msg.length / blockSize * blockSize), (int)bos.size(), (String)"no partial write");
            }
            this.validate(blockSize, msg, bos.toByteArray());
        }
    }

    @Test
    public void testWriteBuf() throws IOException {
        String hwa = "hello world avengers";
        this.testBuf(4, "hello world avengers");
        this.testBuf(512, "hello world avengers");
        this.testBuf(10240, "hello world avengers");
        this.testBuf(11, "hello world avengershello world avengershello world avengers");
    }

    @Test
    public void testWriteFailsAfterDestClosedThrowsException() throws IOException {
        int blockSize = 2;
        try (MockOutputStream mock = new MockOutputStream(2, false);
             FixedLengthBlockOutputStream out = new FixedLengthBlockOutputStream((OutputStream)mock, 2);){
            Assertions.assertThrows(IOException.class, () -> {
                out.write(1);
                Assertions.assertTrue((boolean)out.isOpen());
                mock.close();
                out.write(1);
            }, (String)"expected IO Exception");
            Assertions.assertFalse((boolean)out.isOpen());
        }
    }

    @Test
    public void testWriteFailsAfterFLClosedThrowsException() {
        Assertions.assertThrowsExactly(ClosedChannelException.class, () -> {
            try (FixedLengthBlockOutputStream out = this.newClosedFLBOS();){
                out.write(1);
            }
        }, (String)"expected Closed Channel Exception");
        Assertions.assertThrowsExactly(ClosedChannelException.class, () -> {
            try (FixedLengthBlockOutputStream out = this.newClosedFLBOS();){
                out.write(new byte[]{0, 1, 2, 3});
            }
        }, (String)"expected Closed Channel Exception");
        Assertions.assertThrowsExactly(ClosedChannelException.class, () -> {
            try (FixedLengthBlockOutputStream out = this.newClosedFLBOS();){
                out.write(ByteBuffer.wrap(new byte[]{0, 1, 2, 3}));
            }
        }, (String)"expected Closed Channel Exception");
    }

    @Test
    public void testWriteSingleBytes() throws IOException {
        int blockSize = 4;
        try (MockWritableByteChannel mock = new MockWritableByteChannel(4, false);){
            ByteArrayOutputStream bos = mock.bos;
            String text = "hello world avengers";
            byte[] msg = "hello world avengers".getBytes();
            int len = msg.length;
            try (FixedLengthBlockOutputStream out = new FixedLengthBlockOutputStream((WritableByteChannel)mock, 4);){
                for (int i = 0; i < len; ++i) {
                    out.write((int)msg[i]);
                }
            }
            byte[] output = bos.toByteArray();
            this.validate(4, msg, output);
        }
    }

    private void validate(int blockSize, byte[] expectedBytes, byte[] actualBytes) {
        double v = Math.ceil((double)expectedBytes.length / (double)blockSize) * (double)blockSize;
        Assertions.assertEquals((long)((long)v), (long)actualBytes.length, (String)"wrong size");
        FixedLengthBlockOutputStreamTest.assertContainsAtOffset("output", expectedBytes, 0, actualBytes);
        for (int i = expectedBytes.length; i < actualBytes.length; ++i) {
            Assertions.assertEquals((int)0, (int)actualBytes[i], (String)String.format("output[%d]", i));
        }
    }

    private static final class MockOutputStream
    extends OutputStream {
        final ByteArrayOutputStream bos = new ByteArrayOutputStream();
        private final int requiredWriteSize;
        private final boolean doPartialWrite;
        private final AtomicBoolean closed = new AtomicBoolean();

        private MockOutputStream(int requiredWriteSize, boolean doPartialWrite) {
            this.requiredWriteSize = requiredWriteSize;
            this.doPartialWrite = doPartialWrite;
        }

        private void checkIsOpen() throws IOException {
            if (this.closed.get()) {
                throw new IOException("Closed");
            }
        }

        @Override
        public void close() throws IOException {
            if (this.closed.compareAndSet(false, true)) {
                this.bos.close();
            }
        }

        @Override
        public void write(byte[] b, int off, int len) throws IOException {
            this.checkIsOpen();
            Assertions.assertEquals((int)this.requiredWriteSize, (int)len, (String)"write size");
            if (this.doPartialWrite) {
                --len;
            }
            this.bos.write(b, off, len);
        }

        @Override
        public void write(int b) throws IOException {
            this.checkIsOpen();
            Assertions.assertEquals((int)this.requiredWriteSize, (int)1, (String)"write size");
            this.bos.write(b);
        }
    }

    private static final class MockWritableByteChannel
    implements WritableByteChannel {
        final ByteArrayOutputStream bos = new ByteArrayOutputStream();
        private final int requiredWriteSize;
        private final boolean doPartialWrite;
        final AtomicBoolean closed = new AtomicBoolean();

        private MockWritableByteChannel(int requiredWriteSize, boolean doPartialWrite) {
            this.requiredWriteSize = requiredWriteSize;
            this.doPartialWrite = doPartialWrite;
        }

        @Override
        public void close() throws IOException {
            this.closed.compareAndSet(false, true);
        }

        @Override
        public boolean isOpen() {
            return !this.closed.get();
        }

        @Override
        public int write(ByteBuffer src) throws IOException {
            Assertions.assertEquals((int)this.requiredWriteSize, (int)src.remaining(), (String)"write size");
            if (this.doPartialWrite) {
                src.limit(src.limit() - 1);
            }
            int bytesOut = src.remaining();
            while (src.hasRemaining()) {
                this.bos.write(src.get());
            }
            return bytesOut;
        }
    }
}

