/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.commons.configuration2.builder;

import org.apache.commons.configuration2.ImmutableConfiguration;
import org.apache.commons.configuration2.event.EventType;

/**
 * <p>
 * A specialized event class which is generated by a
 * {@link ConfigurationBuilder} when a result configuration has been created.
 * </p>
 * <p>
 * Events of this type are fired in the {@code getConfiguration()} method of a
 * configuration builder each time a new result object is created. At the time
 * the event is fired, no lock is held. The newly created {@code ImmutableConfiguration}
 * object is available as a property of this event.
 * </p>
 * <p>
 * A use case for this event is to perform special initializations on newly
 * created configuration objects. It is also an indication that a builder is now
 * fully initialized; i.e. the managed configuration is available.
 * </p>
 *
 * @version $Id: ConfigurationBuilderResultCreatedEvent.java 1706911 2015-10-05 20:01:32Z oheger $
 * @since 2.0
 */
public class ConfigurationBuilderResultCreatedEvent extends
        ConfigurationBuilderEvent
{
    /**
     * The specialized event type for a newly created result configuration.
     * Events of this type are generated by a configuration builder when a
     * result configuration has been created.
     */
    public static final EventType<ConfigurationBuilderResultCreatedEvent> RESULT_CREATED =
            new EventType<ConfigurationBuilderResultCreatedEvent>(ANY,
                    "RESULT_CREATED");

    /** The newly created configuration object. */
    private final ImmutableConfiguration configuration;

    /**
     * Creates a new instance of {@code ConfigurationBuilderResultCreatedEvent}
     * and initializes its properties.
     *
     * @param source the {@code ConfigurationBuilder} object which triggered
     *        this event (must not be <b>null</b>)
     * @param evType the type of this event (must not be <b>null</b>)
     * @param createdConfiguration the newly created {@code ImmutableConfiguration}
     *        object (must not be <b>null</b>)
     * @throws IllegalArgumentException if a required parameter is null
     */
    public ConfigurationBuilderResultCreatedEvent(
            ConfigurationBuilder<?> source,
            EventType<? extends ConfigurationBuilderResultCreatedEvent> evType,
            ImmutableConfiguration createdConfiguration)
    {
        super(source, evType);
        if (createdConfiguration == null)
        {
            throw new IllegalArgumentException(
                    "Configuration must not be null!");
        }
        configuration = createdConfiguration;
    }

    /**
     * Returns the newly created {@code ImmutableConfiguration} object.
     *
     * @return the newly created {@code ImmutableConfiguration}
     */
    public ImmutableConfiguration getConfiguration()
    {
        return configuration;
    }
}
