/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.cxf.jaxws.handler.soap;

import java.net.URI;
import java.util.HashSet;
import java.util.Set;

import javax.xml.namespace.QName;
import javax.xml.soap.SOAPBody;
import javax.xml.soap.SOAPException;
import javax.xml.soap.SOAPMessage;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import javax.xml.transform.dom.DOMSource;
import javax.xml.ws.Binding;
import javax.xml.ws.handler.Handler;
import javax.xml.ws.handler.MessageContext;
import javax.xml.ws.handler.soap.SOAPHandler;
import javax.xml.ws.handler.soap.SOAPMessageContext;

import org.apache.cxf.binding.soap.SoapMessage;
import org.apache.cxf.binding.soap.interceptor.AbstractSoapInterceptor;
import org.apache.cxf.binding.soap.interceptor.MustUnderstandInterceptor;
import org.apache.cxf.binding.soap.interceptor.SoapInterceptor;
import org.apache.cxf.binding.soap.interceptor.SoapPreProtocolOutInterceptor;
import org.apache.cxf.binding.soap.saaj.SAAJInInterceptor;
import org.apache.cxf.binding.soap.saaj.SAAJOutInterceptor;
import org.apache.cxf.endpoint.Endpoint;
import org.apache.cxf.helpers.CastUtils;
import org.apache.cxf.interceptor.Fault;
import org.apache.cxf.interceptor.InterceptorChain;
import org.apache.cxf.interceptor.OutgoingChainInterceptor;
import org.apache.cxf.jaxws.handler.AbstractProtocolHandlerInterceptor;
import org.apache.cxf.jaxws.handler.HandlerChainInvoker;
import org.apache.cxf.jaxws.support.ContextPropertiesMapping;
import org.apache.cxf.message.Exchange;
import org.apache.cxf.message.Message;
import org.apache.cxf.phase.Phase;
import org.apache.cxf.phase.PhaseInterceptorChain;
import org.apache.cxf.staxutils.StaxUtils;
import org.apache.cxf.transport.MessageObserver;

public class SOAPHandlerInterceptor extends
        AbstractProtocolHandlerInterceptor<SoapMessage> implements
        SoapInterceptor {
    private static final SAAJOutInterceptor SAAJ_OUT = new SAAJOutInterceptor();

    AbstractSoapInterceptor ending = new AbstractSoapInterceptor(
            SOAPHandlerInterceptor.class.getName() + ".ENDING",
            Phase.USER_PROTOCOL) {

        public void handleMessage(SoapMessage message) throws Fault {
            handleMessageInternal(message);
        }
    };

    public SOAPHandlerInterceptor(Binding binding) {
        super(binding, Phase.PRE_PROTOCOL);
        addAfter(MustUnderstandInterceptor.class.getName());
        addAfter(SAAJOutInterceptor.class.getName());
    }

    public Set<URI> getRoles() {
        Set<URI> roles = new HashSet<URI>();
        //TODO
        return roles;
    }

    public Set<QName> getUnderstoodHeaders() {
        Set<QName> understood = new HashSet<QName>();
        for (Handler h : getBinding().getHandlerChain()) {
            if (h instanceof SOAPHandler) {
                Set<QName> headers = CastUtils.cast(((SOAPHandler) h).getHeaders());
                if (headers != null) {
                    understood.addAll(headers);
                }
            }
        }
        return understood;
    }

    public void handleMessage(SoapMessage message) {
        if (getInvoker(message).getProtocolHandlers().isEmpty()) {
            return;
        }

        if (getInvoker(message).isOutbound()) {

            SAAJ_OUT.handleMessage(message);

            message.getInterceptorChain().add(ending);
        } else {
            boolean isFault = handleMessageInternal(message);
            SOAPMessage msg = message.getContent(SOAPMessage.class);
            if (msg != null) {
                XMLStreamReader xmlReader = createXMLStreamReaderFromSOAPMessage(msg);
                message.setContent(XMLStreamReader.class, xmlReader);
                // replace headers
                try {
                    SAAJInInterceptor.replaceHeaders(msg, message);
                } catch (SOAPException e) {
                    e.printStackTrace();
                }
            }
            if (isFault) {
                Endpoint ep = message.getExchange().get(Endpoint.class);
                message.getInterceptorChain().abort();
                if (ep.getInFaultObserver() != null) {
                    ep.getInFaultObserver().onMessage(message);
                    
                }
            }
        }
    }

    private boolean handleMessageInternal(SoapMessage message) {
        
        MessageContext context = createProtocolMessageContext(message);
        if (context == null) {
            return true;
        }
                
        HandlerChainInvoker invoker = getInvoker(message);
        invoker.setProtocolMessageContext(context);

        if (!invoker.invokeProtocolHandlers(isRequestor(message), context)) {
            handleAbort(message, context);
        }

        // If this is the outbound and end of MEP, call MEP completion
        if (isRequestor(message) && invoker.getLogicalHandlers().isEmpty()
            && !isOutbound(message) && isMEPComlete(message)) {
            onCompletion(message);
        } else if (isOutbound(message) && isMEPComlete(message)) {
            onCompletion(message);
        }
        return false;
    }

    private void handleAbort(SoapMessage message, MessageContext context) {
        if (isRequestor(message)) {
            // client side outbound
            if (getInvoker(message).isOutbound()) {
                message.getInterceptorChain().abort();
                
                MessageObserver observer = (MessageObserver)message.getExchange().get(MessageObserver.class);
                if (!message.getExchange().isOneWay()
                    && observer != null) {
                    Endpoint e = message.getExchange().get(Endpoint.class);
                    Message responseMsg = e.getBinding().createMessage();
    
                    // the request message becomes the response message
                    message.getExchange().setInMessage(responseMsg);
                    SOAPMessage soapMessage = ((SOAPMessageContext)context).getMessage();

                    if (soapMessage != null) {
                        responseMsg.setContent(SOAPMessage.class, soapMessage);
                        XMLStreamReader xmlReader = createXMLStreamReaderFromSOAPMessage(soapMessage);
                        responseMsg.setContent(XMLStreamReader.class, xmlReader);
                    }
                    responseMsg.put(PhaseInterceptorChain.STARTING_AT_INTERCEPTOR_ID,
                                    SOAPHandlerInterceptor.class.getName());
                    observer.onMessage(responseMsg);
                }
                //We dont call onCompletion here, as onCompletion will be called by inbound
                //LogicalHandlerInterceptor
            } else {
                // client side inbound - Normal handler message processing
                // stops, but the inbound interceptor chain still continues, dispatch the message
                //By onCompletion here, we can skip following Logical handlers
                onCompletion(message);
            }
        } else {
            if (!getInvoker(message).isOutbound()) {
                // server side inbound
                message.getInterceptorChain().abort();
                Endpoint e = message.getExchange().get(Endpoint.class);
                Message responseMsg = e.getBinding().createMessage();
                if (!message.getExchange().isOneWay()) {
                    message.getExchange().setOutMessage(responseMsg);
                    SOAPMessage soapMessage = ((SOAPMessageContext)context).getMessage();

                    responseMsg.setContent(SOAPMessage.class, soapMessage);

                    InterceptorChain chain = OutgoingChainInterceptor.getOutInterceptorChain(message
                        .getExchange());
                    responseMsg.setInterceptorChain(chain);
                    // so the idea of starting interceptor chain from any
                    // specified point does not work
                    // well for outbound case, as many outbound interceptors
                    // have their ending interceptors.
                    // For example, we can not skip MessageSenderInterceptor.
                    chain.doInterceptStartingAfter(responseMsg, 
                                                   SoapPreProtocolOutInterceptor.class.getName());
                }

            } else {
                // server side outbound - Normal handler message processing
                // stops, but still continue the outbound interceptor chain, dispatch the message
            }
        }
    }

    @Override
    protected MessageContext createProtocolMessageContext(SoapMessage message) {
        SOAPMessageContextImpl sm = new SOAPMessageContextImpl(message);
        
        boolean requestor = isRequestor(message);
        ContextPropertiesMapping.mapCxf2Jaxws(message.getExchange(), sm, requestor);
        Exchange exch = message.getExchange();
        setupBindingOperationInfo(exch, sm);
        try {            
            SOAPMessage msg = sm.getMessage();
            if (msg.getSOAPPart().getEnvelope().getBody() != null 
                && msg.getSOAPPart().getEnvelope().getBody().hasFault()) {
                return null;
            }            
        } catch (SOAPException e) {
            throw new Fault(e);
        }
        
        return sm;
    }

    private XMLStreamReader createXMLStreamReaderFromSOAPMessage(SOAPMessage soapMessage) {
        // responseMsg.setContent(SOAPMessage.class, soapMessage);
        XMLStreamReader xmlReader = null;
        try {
            DOMSource bodySource = new DOMSource(soapMessage.getSOAPPart().getEnvelope().getBody());
            xmlReader = StaxUtils.createXMLStreamReader(bodySource);
            xmlReader.nextTag();
            xmlReader.nextTag(); // move past body tag
        } catch (SOAPException e) {
            e.printStackTrace();
        } catch (XMLStreamException e) {
            e.printStackTrace();
        }
        return xmlReader;
    }

    public void handleFault(SoapMessage message) {
    }
    
    protected QName getOpQName(Exchange ex, Object data) {
        SOAPMessageContextImpl sm = (SOAPMessageContextImpl)data;
        try {
            SOAPMessage msg = sm.getMessage();
            if (msg == null) {
                return null;
            }
            SOAPBody body = msg.getSOAPBody();
            if (body == null) {
                return null;
            }
            org.w3c.dom.Node nd = body.getFirstChild();
            while (nd != null && !(nd instanceof org.w3c.dom.Element)) {
                nd = nd.getNextSibling();
            }
            if (nd != null) {
                return new QName(nd.getNamespaceURI(), nd.getLocalName());
            }
        } catch (SOAPException e) {
            //ignore, nothing we can do
        }
        return null;
    }

}
