/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.cxf.jaxrs.model.wadl;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.io.StringWriter;
import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.lang.reflect.Type;
import java.net.URI;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Logger;

import javax.ws.rs.DefaultValue;
import javax.ws.rs.Encoded;
import javax.ws.rs.FormParam;
import javax.ws.rs.HeaderParam;
import javax.ws.rs.MatrixParam;
import javax.ws.rs.PathParam;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.MultivaluedMap;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.UriBuilder;
import javax.ws.rs.core.UriInfo;
import javax.ws.rs.ext.MessageBodyWriter;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlSeeAlso;
import javax.xml.bind.annotation.XmlType;
import javax.xml.namespace.QName;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamWriter;
import javax.xml.transform.dom.DOMResult;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamSource;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

import org.apache.cxf.Bus;
import org.apache.cxf.BusFactory;
import org.apache.cxf.common.WSDLConstants;
import org.apache.cxf.common.jaxb.JAXBBeanInfo;
import org.apache.cxf.common.jaxb.JAXBContextProxy;
import org.apache.cxf.common.jaxb.JAXBUtils;
import org.apache.cxf.common.logging.LogUtils;
import org.apache.cxf.common.util.PackageUtils;
import org.apache.cxf.common.util.StringUtils;
import org.apache.cxf.common.util.XmlSchemaPrimitiveUtils;
import org.apache.cxf.common.xmlschema.SchemaCollection;
import org.apache.cxf.common.xmlschema.XmlSchemaConstants;
import org.apache.cxf.endpoint.Endpoint;
import org.apache.cxf.helpers.CastUtils;
import org.apache.cxf.helpers.DOMUtils;
import org.apache.cxf.helpers.IOUtils;
import org.apache.cxf.helpers.XMLUtils;
import org.apache.cxf.jaxrs.JAXRSServiceImpl;
import org.apache.cxf.jaxrs.ext.Oneway;
import org.apache.cxf.jaxrs.ext.RequestHandler;
import org.apache.cxf.jaxrs.ext.multipart.Multipart;
import org.apache.cxf.jaxrs.ext.xml.XMLSource;
import org.apache.cxf.jaxrs.impl.HttpHeadersImpl;
import org.apache.cxf.jaxrs.impl.UriInfoImpl;
import org.apache.cxf.jaxrs.model.ClassResourceInfo;
import org.apache.cxf.jaxrs.model.OperationResourceInfo;
import org.apache.cxf.jaxrs.model.Parameter;
import org.apache.cxf.jaxrs.model.ParameterType;
import org.apache.cxf.jaxrs.model.ResourceTypes;
import org.apache.cxf.jaxrs.model.URITemplate;
import org.apache.cxf.jaxrs.provider.ProviderFactory;
import org.apache.cxf.jaxrs.utils.AnnotationUtils;
import org.apache.cxf.jaxrs.utils.ExceptionUtils;
import org.apache.cxf.jaxrs.utils.InjectionUtils;
import org.apache.cxf.jaxrs.utils.JAXRSUtils;
import org.apache.cxf.jaxrs.utils.ResourceUtils;
import org.apache.cxf.jaxrs.utils.schemas.SchemaHandler;
import org.apache.cxf.message.Message;
import org.apache.cxf.service.Service;
import org.apache.cxf.service.model.EndpointInfo;
import org.apache.cxf.staxutils.DelegatingXMLStreamWriter;
import org.apache.cxf.staxutils.StaxUtils;
import org.apache.ws.commons.schema.XmlSchema;

public class WadlGenerator implements RequestHandler {

    public static final String WADL_QUERY = "_wadl";
    public static final MediaType WADL_TYPE = JAXRSUtils.toMediaType("application/vnd.sun.wadl+xml");
    public static final String WADL_NS = "http://wadl.dev.java.net/2009/02";

    private static final MediaType DEFAULT_MEDIA_TYPE = MediaType.APPLICATION_XML_TYPE;
    private static final Logger LOG = LogUtils.getL7dLogger(WadlGenerator.class);
    private static final String JAXB_DEFAULT_NAMESPACE = "##default";
    private static final String JAXB_DEFAULT_NAME = "##default";
    private static final String CLASSPATH_PREFIX = "classpath:";
    private static final String DEFAULT_NS_PREFIX = "prefix";
    private static final Map<ParameterType, Class<? extends Annotation>> PARAMETER_TYPE_MAP;
    static {
        PARAMETER_TYPE_MAP = new HashMap<ParameterType, Class<? extends Annotation>>();
        PARAMETER_TYPE_MAP.put(ParameterType.FORM, FormParam.class);
        PARAMETER_TYPE_MAP.put(ParameterType.QUERY, QueryParam.class);
        PARAMETER_TYPE_MAP.put(ParameterType.HEADER, HeaderParam.class);
        PARAMETER_TYPE_MAP.put(ParameterType.PATH, PathParam.class);
        PARAMETER_TYPE_MAP.put(ParameterType.MATRIX, MatrixParam.class);
    }
    
    private String wadlNamespace;
    private boolean ignoreMessageWriters = true;
    private boolean singleResourceMultipleMethods = true;
    private boolean useSingleSlashResource;
    private boolean ignoreForwardSlash;
    private boolean addResourceAndMethodIds;
    private boolean ignoreRequests;
    private boolean linkAnyMediaTypeToXmlSchema;
    private boolean useJaxbContextForQnames = true;
    private boolean supportCollections = true;
    private boolean supportJaxbXmlType = true;
    private boolean supportJaxbSubstitutions = true;

    private List<String> externalSchemasCache;
    private List<URI> externalSchemaLinks;
    private Map<String, List<String>> externalQnamesMap;

    private ConcurrentHashMap<String, String> docLocationMap = new ConcurrentHashMap<String, String>();

    private ElementQNameResolver resolver;
    private List<String> privateAddresses;
    private String applicationTitle;
    private String nsPrefix = DEFAULT_NS_PREFIX;
    private MediaType defaultMediaType = DEFAULT_MEDIA_TYPE;

    private Bus bus; 

    public WadlGenerator() {

    }

    public WadlGenerator(Bus bus) {
        this.bus = bus;
    }

    public WadlGenerator(WadlGenerator other) {
        this.wadlNamespace = other.wadlNamespace;
        this.externalQnamesMap = other.externalQnamesMap;
        this.externalSchemaLinks = other.externalSchemaLinks;
        this.externalSchemasCache = other.externalSchemasCache;
        this.ignoreMessageWriters = other.ignoreMessageWriters;
        this.ignoreForwardSlash = other.ignoreForwardSlash; 
        this.ignoreRequests = other.ignoreRequests;
        this.linkAnyMediaTypeToXmlSchema = other.linkAnyMediaTypeToXmlSchema;  
        this.privateAddresses = other.privateAddresses;
        this.resolver = other.resolver;
        this.addResourceAndMethodIds = other.addResourceAndMethodIds;
        this.singleResourceMultipleMethods = other.singleResourceMultipleMethods;
        this.useJaxbContextForQnames = other.useJaxbContextForQnames;
        this.useSingleSlashResource = other.useSingleSlashResource;
        this.supportCollections = other.supportCollections;
        this.supportJaxbXmlType = other.supportJaxbXmlType;
        this.nsPrefix = other.nsPrefix;
        this.applicationTitle = other.applicationTitle;
        this.defaultMediaType = other.defaultMediaType;
        this.bus = other.bus;     
    }

    public Response handleRequest(Message m, ClassResourceInfo resource) {

        if (!"GET".equals(m.get(Message.HTTP_REQUEST_METHOD))) {
            return null;
        }

        UriInfo ui = new UriInfoImpl(m);
        if (!ui.getQueryParameters().containsKey(WADL_QUERY)) {
            if (!docLocationMap.isEmpty()) {
                String path = ui.getPath(false);
                if (path.startsWith("/") && path.length() > 0) {
                    path = path.substring(1);
                }
                if (docLocationMap.containsKey(path)) {
                    return getExistingResource(m, ui, path);
                }
            }
            return null;
        }

        if (ignoreRequests) {
            return Response.status(404).build();
        }

        HttpHeaders headers = new HttpHeadersImpl(m);
        List<MediaType> accepts = headers.getAcceptableMediaTypes();
        MediaType type = accepts.contains(WADL_TYPE) ? WADL_TYPE : accepts
            .contains(MediaType.APPLICATION_JSON_TYPE) ? MediaType.APPLICATION_JSON_TYPE : defaultMediaType;

        Response response = getExistingWadl(m, ui, type);
        if (response != null) {
            return response;
        }

        boolean isJson = type == MediaType.APPLICATION_JSON_TYPE;

        StringBuilder sbMain = generateWADL(getBaseURI(m, ui), getResourcesList(m, resource), isJson, m, ui);

        m.getExchange().put(JAXRSUtils.IGNORE_MESSAGE_WRITERS, !isJson && ignoreMessageWriters);
        return Response.ok().type(type).entity(createResponseEntity(sbMain.toString(), isJson)).build();
    }

    public StringBuilder generateWADL(String baseURI, 
                                       List<ClassResourceInfo> cris, 
                                       boolean isJson,
                                       Message m,
                                       UriInfo ui) {
        StringBuilder sbMain = new StringBuilder();
        sbMain.append("<application");
        if (!isJson) {
            sbMain.append(" xmlns=\"").append(getNamespace()).append("\" xmlns:xs=\"")
                .append(XmlSchemaConstants.XSD_NAMESPACE_URI).append("\"");
        }
        StringBuilder sbGrammars = new StringBuilder();
        sbGrammars.append("<grammars>");

        StringBuilder sbResources = new StringBuilder();
        sbResources.append("<resources base=\"").append(baseURI).append("\">");

        MessageBodyWriter<?> jaxbWriter = (m != null && useJaxbContextForQnames)
             ? ProviderFactory.getInstance(m).getRegisteredJaxbWriter() : null;
        ResourceTypes resourceTypes = ResourceUtils.getAllRequestResponseTypes(cris, 
                                                                               useJaxbContextForQnames,
                                                                               jaxbWriter);
        checkXmlSeeAlso(resourceTypes);
        Set<Class<?>> allTypes = resourceTypes.getAllTypes().keySet();

        JAXBContext jaxbContext = useJaxbContextForQnames ? ResourceUtils
            .createJaxbContext(new HashSet<Class<?>>(allTypes), null, null) : null;

        SchemaWriter schemaWriter = createSchemaWriter(resourceTypes, jaxbContext, ui);
        ElementQNameResolver qnameResolver = schemaWriter == null
            ? null : createElementQNameResolver(jaxbContext);

        Map<Class<?>, QName> clsMap = new IdentityHashMap<Class<?>, QName>();
        Set<ClassResourceInfo> visitedResources = new LinkedHashSet<ClassResourceInfo>();
        for (ClassResourceInfo cri : cris) {
            startResourceTag(sbResources, cri.getServiceClass(), cri.getURITemplate().getValue());
            Annotation description = AnnotationUtils.getClassAnnotation(cri.getServiceClass(), Description.class);
            if (description == null) {
                description = AnnotationUtils.getClassAnnotation(cri.getServiceClass(), Descriptions.class);
            }
            if (description != null) {
                handleDocs(new Annotation[] {description}, sbResources, DocTarget.RESOURCE, true, isJson);
            }
            handleResource(sbResources, allTypes, qnameResolver, clsMap, cri, visitedResources, isJson);
            sbResources.append("</resource>");
        }
        sbResources.append("</resources>");

        handleGrammars(sbMain, sbGrammars, schemaWriter, clsMap);

        sbGrammars.append("</grammars>");
        sbMain.append(">");
        handleApplicationDocs(sbMain);
        sbMain.append(sbGrammars.toString());
        sbMain.append(sbResources.toString());
        sbMain.append("</application>");
        return sbMain;
    }

    private Object createResponseEntity(String entity, boolean isJson) {
        if (!isJson) {
            return entity;
        }
        try {
            return StaxUtils.read(new StringReader(entity));
        } catch (Exception ex) {
            throw ExceptionUtils.toInternalServerErrorException(ex, null);
        }
    }

    private String getBaseURI(Message m, UriInfo ui) {
        EndpointInfo ei = m.getExchange().get(Endpoint.class).getEndpointInfo();
        String publishedEndpointUrl = (String)ei.getProperty("publishedEndpointUrl");
        if (publishedEndpointUrl == null) {
            return ui.getBaseUri().toString();
        } else {
            return publishedEndpointUrl;
        }
    }

    protected void handleGrammars(StringBuilder sbApp, StringBuilder sbGrammars, SchemaWriter writer,
                                  Map<Class<?>, QName> clsMap) {
        if (writer == null) {
            return;
        }

        Map<String, String> map = new HashMap<String, String>();
        for (QName qname : clsMap.values()) {
            map.put(qname.getPrefix(), qname.getNamespaceURI());
        }
        for (Map.Entry<String, String> entry : map.entrySet()) {
            sbApp.append(" xmlns:").append(entry.getKey()).append("=\"").append(entry.getValue())
                .append("\"");
        }

        writer.write(sbGrammars);
    }

    protected void handleResource(StringBuilder sb, Set<Class<?>> jaxbTypes,
                                  ElementQNameResolver qnameResolver, Map<Class<?>, QName> clsMap,
                                  ClassResourceInfo cri, Set<ClassResourceInfo> visitedResources,
                                  boolean isJson) {
        visitedResources.add(cri);
        Map<Parameter, Object> classParams = getClassParameters(cri);

        List<OperationResourceInfo> sortedOps = sortOperationsByPath(cri.getMethodDispatcher()
            .getOperationResourceInfos());

        boolean resourceTagOpened = false;
        for (int i = 0; i < sortedOps.size(); i++) {
            OperationResourceInfo ori = sortedOps.get(i);

            if (ori.getHttpMethod() == null) {
                Class<?> cls = getMethod(ori).getReturnType();
                ClassResourceInfo subcri = cri.findResource(cls, cls);
                if (subcri != null && !visitedResources.contains(subcri)) {
                    startResourceTag(sb, subcri.getServiceClass(), ori.getURITemplate().getValue());
                    handleDocs(subcri.getServiceClass().getAnnotations(), sb, DocTarget.RESOURCE, true,
                               isJson);
                    handlePathAndMatrixParams(sb, ori, isJson);
                    handleResource(sb, jaxbTypes, qnameResolver, clsMap, subcri, visitedResources, isJson);
                    sb.append("</resource>");
                } else {
                    handleDynamicSubresource(sb, jaxbTypes, qnameResolver, clsMap, ori, subcri, isJson);
                }
                continue;
            }
            OperationResourceInfo nextOp = i + 1 < sortedOps.size() ? sortedOps.get(i + 1) : null;
            resourceTagOpened = handleOperation(sb, jaxbTypes, qnameResolver, clsMap, ori, classParams,
                                                nextOp, resourceTagOpened, isJson, i);
        }
    }

    private Map<Parameter, Object> getClassParameters(ClassResourceInfo cri) {
        Map<Parameter, Object> classParams = new LinkedHashMap<Parameter, Object>();
        List<Method> paramMethods = cri.getParameterMethods();
        for (Method m : paramMethods) {
            classParams.put(ResourceUtils.getParameter(0, m.getAnnotations(), m.getParameterTypes()[0]), m);
        }
        List<Field> fieldParams = cri.getParameterFields();
        for (Field f : fieldParams) {
            classParams.put(ResourceUtils.getParameter(0, f.getAnnotations(), f.getType()), f);
        }
        return classParams;
    }

    private void startResourceTag(StringBuilder sb, Class<?> serviceClass, String path) {
        sb.append("<resource path=\"").append(getPath(path)).append("\"");
        if (addResourceAndMethodIds) {
            QName jaxbQname = null;
            if (useJaxbContextForQnames) {
                jaxbQname = getJaxbQName(null, serviceClass, new HashMap<Class<?>, QName>(0));
            }
            String pName = jaxbQname == null ? PackageUtils.getPackageName(serviceClass) : jaxbQname
                .getNamespaceURI();
            String localName = jaxbQname == null ? serviceClass.getSimpleName() : jaxbQname.getLocalPart();
            String finalName = jaxbQname == null ? pName + "." : "{" + pName + "}";
            sb.append(" id=\"").append(finalName + localName).append("\"");
        }
        sb.append(">");
    }

    protected String getPath(String path) {
        String thePath = null;
        if (ignoreForwardSlash && path.startsWith("/") && path.length() > 0) {
            thePath = path.substring(1);
        } else {
            thePath = path;
        }
        
        return xmlEncodeIfNeeded(thePath);
    }

    private void checkXmlSeeAlso(ResourceTypes resourceTypes) {
        if (!this.useJaxbContextForQnames) {
            return;
        }
        List<Class<?>> extraClasses = new LinkedList<Class<?>>();
        for (Class<?> cls : resourceTypes.getAllTypes().keySet()) {
            if (!isXmlRoot(cls) || Modifier.isAbstract(cls.getModifiers())) {
                XmlSeeAlso seeAlsoAnn = cls.getAnnotation(XmlSeeAlso.class);
                if (seeAlsoAnn != null) {
                    List<Class<?>> seeAlsoList = CastUtils.cast(Arrays.asList(seeAlsoAnn.value()));
                    if (this.supportJaxbSubstitutions) {
                        for (Class<?> seeAlsoCls : seeAlsoList) {
                            resourceTypes.getSubstitutions().put(seeAlsoCls, cls);
                        }
                    } 
                    extraClasses.addAll(seeAlsoList);
                }
            }
        }
        for (Class<?> cls : extraClasses) {
            resourceTypes.getAllTypes().put(cls, cls);
        }
    }
    
    private String xmlEncodeIfNeeded(String value) {
        return XMLUtils.xmlEncode(value);
    }
    
    private void startMethodTag(StringBuilder sb, OperationResourceInfo ori) {
        sb.append("<method name=\"").append(ori.getHttpMethod()).append("\"");
        if (addResourceAndMethodIds) {
            sb.append(" id=\"").append(getMethod(ori).getName()).append("\"");
        }
        sb.append(">");
    }
    protected void endMethodTag(StringBuilder sb, OperationResourceInfo ori) {
        sb.append("</method>");
    }
    protected void startMethodRequestTag(StringBuilder sb, OperationResourceInfo ori) {
        sb.append("<request>");
    }
    protected void startMethodResponseTag(StringBuilder sb, OperationResourceInfo ori) {
        sb.append("<response");
    }
    protected void endMethodRequestTag(StringBuilder sb, OperationResourceInfo ori) {
        sb.append("</request>");
    }
    protected void endMethodResponseTag(StringBuilder sb, OperationResourceInfo ori) {
        sb.append("</response>");
    }
    protected void startResourceTag(StringBuilder sb, OperationResourceInfo ori, String path) {
        sb.append("<resource path=\"").append(path).append("\">");
    }
    protected void endResourceTag(StringBuilder sb, OperationResourceInfo ori) {
        sb.append("</resource>");
    }

    // CHECKSTYLE:OFF
    protected boolean handleOperation(StringBuilder sb, Set<Class<?>> jaxbTypes,
                                      ElementQNameResolver qnameResolver, Map<Class<?>, QName> clsMap,
                                      OperationResourceInfo ori, Map<Parameter, Object> classParams,
                                      OperationResourceInfo nextOp, boolean resourceTagOpened,
                                      boolean isJson, int index) {
        Annotation[] anns = getMethod(ori).getAnnotations();
        // CHECKSTYLE:ON
        boolean samePathOperationFollows = singleResourceMultipleMethods && compareOperations(ori, nextOp);

        String path = ori.getURITemplate().getValue();
        if (!resourceTagOpened && openResource(path)) {
            resourceTagOpened = true;
            URITemplate template = ori.getClassResourceInfo().getURITemplate();
            if (template != null) {
                String parentPath = template.getValue();
                if (parentPath.endsWith("/") && path.startsWith("/") && path.length() > 1) {
                    path = path.substring(1);
                }
            }
            startResourceTag(sb, ori, getPath(path));
            handleDocs(anns, sb, DocTarget.RESOURCE, false, isJson);
            handlePathAndMatrixClassParams(sb, classParams, isJson);
            handlePathAndMatrixParams(sb, ori, isJson);
        } else if (index == 0) {
            handlePathAndMatrixClassParams(sb, classParams, isJson);
            handlePathAndMatrixParams(sb, ori, isJson);
        }

        startMethodTag(sb, ori);
        handleDocs(anns, sb, DocTarget.METHOD, true, isJson);
        if (getMethod(ori).getParameterTypes().length != 0 || classParams.size() != 0) {
            startMethodRequestTag(sb, ori);
            handleDocs(anns, sb, DocTarget.REQUEST, false, isJson);

            boolean isForm = isFormRequest(ori);

            doHandleClassParams(sb, classParams, isJson, ParameterType.QUERY, ParameterType.HEADER);
            doHandleJaxrsBeanParamClassParams(sb, classParams, isJson, 
                                              ParameterType.QUERY, ParameterType.HEADER);
            for (Parameter p : ori.getParameters()) {
                if (isForm && p.getType() == ParameterType.REQUEST_BODY) {
                    continue;
                }
                handleParameter(sb, jaxbTypes, qnameResolver, clsMap, ori, p, isJson);
            }
            if (isForm) {
                handleFormRepresentation(sb, jaxbTypes, qnameResolver, clsMap, ori, getFormClass(ori), isJson);
            }
            endMethodRequestTag(sb, ori);
        }
        startMethodResponseTag(sb, ori);
        Class<?> returnType = getMethod(ori).getReturnType();
        boolean isVoid = void.class == returnType;
        if (isVoid) {
            boolean oneway = getMethod(ori).getAnnotation(Oneway.class) != null;
            sb.append(" status=\"" + (oneway ? 202 : 204) + "\"");
        }
        sb.append(">");
        handleDocs(anns, sb, DocTarget.RESPONSE, false, isJson);
        if (!isVoid) {
            handleRepresentation(sb, jaxbTypes, qnameResolver, clsMap, ori, returnType, isJson, false);
        }
        endMethodResponseTag(sb, ori);

        endMethodTag(sb, ori);

        if (resourceTagOpened && !samePathOperationFollows) {
            endResourceTag(sb, ori);
            resourceTagOpened = false;
        }
        return resourceTagOpened;
    }

    protected boolean compareOperations(OperationResourceInfo ori1, OperationResourceInfo ori2) {
        if (ori1 == null || ori2 == null
            || !ori1.getURITemplate().getValue().equals(ori2.getURITemplate().getValue())
            || ori1.getHttpMethod() != null && ori2.getHttpMethod() == null || ori2.getHttpMethod() != null
            && ori1.getHttpMethod() == null) {
            return false;
        }
        int ori1PathParams = 0;
        int ori1MatrixParams = 0;
        for (Parameter p : ori1.getParameters()) {
            if (p.getType() == ParameterType.PATH) {
                ori1PathParams++;
            } else if (p.getType() == ParameterType.MATRIX) {
                ori1MatrixParams++;
            }
        }

        int ori2PathParams = 0;
        int ori2MatrixParams = 0;
        for (Parameter p : ori2.getParameters()) {
            if (p.getType() == ParameterType.PATH) {
                ori2PathParams++;
            } else if (p.getType() == ParameterType.MATRIX) {
                ori2MatrixParams++;
            }
        }

        return ori1PathParams == ori2PathParams && ori1MatrixParams == ori2MatrixParams;
    }

    private boolean openResource(String path) {
        if ("/".equals(path)) {
            return useSingleSlashResource;
        }
        return true;
    }

    protected void handleDynamicSubresource(StringBuilder sb, Set<Class<?>> jaxbTypes,
                                            ElementQNameResolver qnameResolver, Map<Class<?>, QName> clsMap,
                                            OperationResourceInfo ori, ClassResourceInfo subcri,
                                            boolean isJson) {
        if (!isJson) {
            if (subcri != null) {
                sb.append("<!-- Recursive subresource -->");
            } else {
                sb.append("<!-- Dynamic subresource -->");
            }
        }
        startResourceTag(sb, subcri != null ? subcri.getServiceClass() : Object.class, ori.getURITemplate()
            .getValue());
        handlePathAndMatrixParams(sb, ori, isJson);
        sb.append("</resource>");
    }

    protected void handlePathAndMatrixClassParams(StringBuilder sb, Map<Parameter, Object> params,
                                                  boolean isJson) {
        doHandleClassParams(sb, params, isJson, ParameterType.PATH);
        doHandleClassParams(sb, params, isJson, ParameterType.MATRIX);
        doHandleJaxrsBeanParamClassParams(sb, params, isJson, ParameterType.PATH, ParameterType.MATRIX);
    }

    protected void doHandleClassParams(StringBuilder sb, Map<Parameter, Object> params, boolean isJson,
                                       ParameterType... pType) {
        Set<ParameterType> pTypes = new LinkedHashSet<ParameterType>(Arrays.asList(pType));
        for (Map.Entry<Parameter, Object> entry : params.entrySet()) {
            Parameter pm = entry.getKey();
            Object obj = entry.getValue();
            if (pTypes.contains(pm.getType())) {
                Class<?> cls = obj instanceof Method ? ((Method)obj).getParameterTypes()[0] : ((Field)obj)
                    .getType();
                Type type = obj instanceof Method
                    ? ((Method)obj).getGenericParameterTypes()[0] : ((Field)obj).getGenericType();
                Annotation[] ann = obj instanceof Method
                    ? ((Method)obj).getParameterAnnotations()[0] : ((Field)obj).getAnnotations();
                doWriteParam(sb, pm, cls, type, pm.getName(), ann, isJson);
            }
        }
    }
    
    protected void doHandleJaxrsBeanParamClassParams(StringBuilder sb, 
                                       Map<Parameter, Object> params, 
                                       boolean isJson,
                                       ParameterType... pType) {
        for (Map.Entry<Parameter, Object> entry : params.entrySet()) {
            Parameter pm = entry.getKey();
            Object obj = entry.getValue();
            if (pm.getType() == ParameterType.BEAN) {
                Class<?> cls = obj instanceof Method ? ((Method)obj).getParameterTypes()[0] : ((Field)obj)
                    .getType();
                doWriteJaxrsBeanParam(sb, cls, isJson, pType);
            }
        }
    } 

    protected void handlePathAndMatrixParams(StringBuilder sb, OperationResourceInfo ori, boolean isJson) {
        handleParams(sb, ori, ParameterType.PATH, isJson);
        handleParams(sb, ori, ParameterType.MATRIX, isJson);
        doWriteJaxrsBeanParams(sb, ori, isJson, ParameterType.PATH, ParameterType.MATRIX);
    }

    protected void handleParameter(StringBuilder sb, Set<Class<?>> jaxbTypes,
                                   ElementQNameResolver qnameResolver, Map<Class<?>, QName> clsMap,
                                   OperationResourceInfo ori, Parameter pm, boolean isJson) {
        Class<?> cls = getMethod(ori).getParameterTypes()[pm.getIndex()];
        if (pm.getType() == ParameterType.REQUEST_BODY) {
            handleRepresentation(sb, jaxbTypes, qnameResolver, clsMap, ori, cls, isJson, true);
            return;
        }
        if (pm.getType() == ParameterType.PATH || pm.getType() == ParameterType.MATRIX) {
            return;
        }
        if (pm.getType() == ParameterType.HEADER || pm.getType() == ParameterType.QUERY) {
            writeParam(sb, pm, ori, isJson);
        }
        if (pm.getType() == ParameterType.BEAN) {
            doWriteJaxrsBeanParams(sb, ori, isJson, ParameterType.HEADER, ParameterType.QUERY);
        }
    }

    protected void handleParams(StringBuilder sb, OperationResourceInfo ori, ParameterType type,
                                boolean isJson) {
        for (Parameter pm : ori.getParameters()) {
            if (pm.getType() == type) {
                writeParam(sb, pm, ori, isJson);
            }
        }
    }

    private Annotation[] getBodyAnnotations(OperationResourceInfo ori, boolean inbound) {
        Method opMethod = getMethod(ori);
        if (inbound) {
            for (Parameter pm : ori.getParameters()) {
                if (pm.getType() == ParameterType.REQUEST_BODY) {
                    return opMethod.getParameterAnnotations()[pm.getIndex()];
                }
            }
            return new Annotation[] {};
        } else {
            return opMethod.getDeclaredAnnotations();
        }
    }

    private void writeParam(StringBuilder sb, Parameter pm, OperationResourceInfo ori, boolean isJson) {
        Method method = getMethod(ori);
        Class<?> type = method.getParameterTypes()[pm.getIndex()];
        if (!"".equals(pm.getName())) {
            doWriteParam(sb, pm, type, method.getGenericParameterTypes()[pm.getIndex()], pm.getName(),
                         method.getParameterAnnotations()[pm.getIndex()], isJson);
        } else {
            List<Class<?>> parentBeanClasses = new LinkedList<Class<?>>();
            parentBeanClasses.add(type);
            doWriteBeanParam(sb, type, pm, null, parentBeanClasses, isJson);
        }
    }

    private void doWriteJaxrsBeanParams(StringBuilder sb, 
                                       OperationResourceInfo ori, 
                                       boolean isJson,
                                       ParameterType ...parameterTypes) {
        for (Parameter p : ori.getParameters()) {
            if (p.getType() == ParameterType.BEAN) {
                Method method = getMethod(ori);
                Class<?> type = method.getParameterTypes()[p.getIndex()];
                doWriteJaxrsBeanParam(sb, type, isJson, parameterTypes);
            }
        }
    }
    
    private void doWriteJaxrsBeanParam(StringBuilder sb, 
                                       Class<?> beanType, 
                                       boolean isJson,
                                       ParameterType ...parameterTypes) {
        for (Method m : beanType.getMethods()) {
            if (m.getName().startsWith("set")) {
                for (ParameterType parameterType : parameterTypes) {
                    Class<? extends Annotation> annClass = getAnnotationFromParamType(parameterType);
                    Annotation annotation = m.getAnnotation(annClass);
                    if (annotation != null) {
                        String propertyName = StringUtils.uncapitalize(m.getName().substring(3));
                        Parameter pm = new Parameter(parameterType, propertyName);
                        pm.setEncoded(m.getAnnotation(Encoded.class) != null);
                        DefaultValue dv = m.getAnnotation(DefaultValue.class);
                        if (dv != null) {
                            pm.setDefaultValue(dv.value());
                        }
                        doWriteParam(sb,
                                     pm, 
                                     m.getParameterTypes()[0], 
                                     m.getGenericParameterTypes()[0], 
                                     propertyName,
                                     new Annotation[]{}, 
                                     isJson);
                        
                    }
                }
            }
        }
    }

    private Class<? extends Annotation> getAnnotationFromParamType(ParameterType pt) {
        return PARAMETER_TYPE_MAP.get(pt);
    }

    private void doWriteBeanParam(StringBuilder sb, Class<?> type, Parameter pm, String parentName,
                                  List<Class<?>> parentBeanClasses, boolean isJson) {
        Map<Parameter, Class<?>> pms = InjectionUtils.getParametersFromBeanClass(type, pm.getType(), true);
        for (Map.Entry<Parameter, Class<?>> entry : pms.entrySet()) {
            String name = entry.getKey().getName();
            if (parentName != null) {
                name = parentName + "." + name;
            }
            Class<?> paramCls = entry.getValue();
            boolean isPrimitive = InjectionUtils.isPrimitive(paramCls) || paramCls.isEnum();
            if (isPrimitive || Date.class.isAssignableFrom(paramCls)
                || InjectionUtils.isSupportedCollectionOrArray(paramCls)) {
                doWriteParam(sb, entry.getKey(), paramCls, paramCls, name, new Annotation[] {}, isJson);
            } else if (!parentBeanClasses.contains(paramCls)) {
                parentBeanClasses.add(paramCls);
                doWriteBeanParam(sb, paramCls, entry.getKey(), name, parentBeanClasses, isJson);
            }
        }
    }

    protected void doWriteParam(StringBuilder sb, Parameter pm, Class<?> type, Type genericType,
                                String paramName, Annotation[] anns, boolean isJson) {
        ParameterType pType = pm.getType();
        boolean isForm = isFormParameter(pm, type, anns);
        if (paramName == null && isForm) {
            Multipart m = AnnotationUtils.getAnnotation(anns, Multipart.class);
            if (m != null) {
                paramName = m.value();
            }
        }
        sb.append("<param name=\"").append(paramName).append("\" ");
        String style = ParameterType.PATH == pType ? "template" : isForm
            ? "query" : ParameterType.REQUEST_BODY == pType ? "plain" : pType.toString().toLowerCase();
        sb.append("style=\"").append(style).append("\"");
        if (pm.getDefaultValue() != null) {
            sb.append(" default=\"").append(xmlEncodeIfNeeded(pm.getDefaultValue()))
                .append("\"");
        }
        if (InjectionUtils.isSupportedCollectionOrArray(type)) {
            type = InjectionUtils.getActualType(genericType);
            sb.append(" repeating=\"true\"");
        }

        String value = XmlSchemaPrimitiveUtils.getSchemaRepresentation(type);
        if (value == null) {
            if (type.isEnum()) {
                value = "xs:string";
            } else if (type == InputStream.class) {
                value = "xs:anyType";
            }
        }
        if (value != null) {
            if (isJson) {
                value = value.substring(3);
            }
            sb.append(" type=\"").append(value).append("\"");
        }
        if (type.isEnum()) {
            sb.append(">");
            handleDocs(anns, sb, DocTarget.PARAM, true, isJson);
            setEnumOptions(sb, type);
            sb.append("</param>");
        } else {
            addDocsAndCloseElement(sb, anns, "param", DocTarget.PARAM, true, isJson);
        }
    }

    private void setEnumOptions(StringBuilder sb, Class<?> enumClass) {
        try {
            Method m = enumClass.getMethod("values", new Class[] {});
            Object[] values = (Object[])m.invoke(null, new Object[] {});
            m = enumClass.getMethod("toString", new Class[] {});
            for (Object o : values) {
                String str = (String)m.invoke(o, new Object[] {});
                sb.append("<option value=\"" + str + "\"/>");
            }

        } catch (Throwable ex) {
            // ignore
        }
    }

    private void addDocsAndCloseElement(StringBuilder sb, Annotation[] anns, String elementName,
                                        String category, boolean allowDefault, boolean isJson) {
        if (isDocAvailable(anns)) {
            sb.append(">");
            handleDocs(anns, sb, category, allowDefault, isJson);
            sb.append("</" + elementName + ">");
        } else {
            sb.append("/>");
        }
    }

    private boolean isDocAvailable(Annotation[] anns) {
        return AnnotationUtils.getAnnotation(anns, Description.class) != null
               || AnnotationUtils.getAnnotation(anns, Descriptions.class) != null;
    }

    // TODO: Collapse multiple parameters into a holder
    // CHECKSTYLE:OFF
    protected void handleRepresentation(StringBuilder sb, Set<Class<?>> jaxbTypes,
                                        ElementQNameResolver qnameResolver, Map<Class<?>, QName> clsMap,
                                        OperationResourceInfo ori, Class<?> type, boolean isJson,
                                        boolean inbound) {
        // CHECKSTYLE:ON
        List<MediaType> types = inbound ? ori.getConsumeTypes() : ori.getProduceTypes();
        if (MultivaluedMap.class.isAssignableFrom(type)) {
            types = Collections.singletonList(MediaType.APPLICATION_FORM_URLENCODED_TYPE);
        } else if (isWildcard(types)) {
            types = Collections.singletonList(MediaType.APPLICATION_OCTET_STREAM_TYPE);
        }

        Method opMethod = getMethod(ori);
        boolean isPrimitive = InjectionUtils.isPrimitive(type);
        for (MediaType mt : types) {

            sb.append("<representation");
            sb.append(" mediaType=\"").append(JAXRSUtils.mediaTypeToString(mt)).append("\"");
            if (isJson && !mt.getSubtype().contains("json")) {
                sb.append("/>");
                continue;
            }

            boolean allowDefault = true;
            String docCategory;
            Annotation[] anns;
            if (inbound) {
                int index = getRequestBodyParam(ori).getIndex();
                anns = opMethod.getParameterAnnotations()[index];
                if (!isDocAvailable(anns)) {
                    anns = opMethod.getAnnotations();
                }
                docCategory = DocTarget.PARAM;
            } else {
                anns = opMethod.getAnnotations();
                docCategory = DocTarget.RETURN;
                allowDefault = false;
            }
            if (isPrimitive) {
                sb.append(">");
                Parameter p = inbound ? getRequestBodyParam(ori) : new Parameter(ParameterType.REQUEST_BODY,
                                                                                 0, "result");
                doWriteParam(sb, p, type, type, p.getName() == null ? "request" : p.getName(), anns, isJson);
                sb.append("</representation>");
            } else {
                boolean isCollection = InjectionUtils.isSupportedCollectionOrArray(type);
                Class<?> theActualType = null;
                if (isCollection) {
                    theActualType = InjectionUtils.getActualType(!inbound ? opMethod.getGenericReturnType() : opMethod
                        .getGenericParameterTypes()[getRequestBodyParam(ori).getIndex()]);
                } else {
                    theActualType = ResourceUtils.getActualJaxbType(type, opMethod, inbound);
                }
                if (isJson) {
                    sb.append(" element=\"").append(theActualType.getSimpleName()).append("\"");
                } else if (qnameResolver != null
                           && (linkAnyMediaTypeToXmlSchema || mt.getSubtype().contains("xml"))
                           && jaxbTypes.contains(theActualType)) {
                    generateQName(sb, qnameResolver, clsMap, theActualType, isCollection,
                                  getBodyAnnotations(ori, inbound));
                }
                addDocsAndCloseElement(sb, anns, "representation", docCategory, allowDefault, isJson);
            }
        }

    }

    private Parameter getRequestBodyParam(OperationResourceInfo ori) {
        for (Parameter p : ori.getParameters()) {
            if (p.getType() == ParameterType.REQUEST_BODY) {
                return p;
            }
        }
        throw new IllegalStateException();
    }

    private boolean isWildcard(List<MediaType> types) {
        return types.size() == 1 && types.get(0).equals(MediaType.WILDCARD_TYPE);
    }

    private void handleFormRepresentation(StringBuilder sb, Set<Class<?>> jaxbTypes,
                                          ElementQNameResolver qnameResolver, Map<Class<?>, QName> clsMap,
                                          OperationResourceInfo ori, Class<?> type, boolean isJson) {
        if (type != null) {
            handleRepresentation(sb, jaxbTypes, qnameResolver, clsMap, ori, type, false, true);
        } else {
            List<MediaType> types = ori.getConsumeTypes();
            MediaType formType = isWildcard(types) ? MediaType.APPLICATION_FORM_URLENCODED_TYPE : types
                .get(0);
            sb.append("<representation");
            sb.append(" mediaType=\"").append(formType).append("\"");
            if (isJson) {
                sb.append("/>");
            } else {
                sb.append(">");
                List<Parameter> params = ori.getParameters();
                for (int i = 0; i < params.size(); i++) {
                    if (isFormParameter(params.get(i), getMethod(ori).getParameterTypes()[i], getMethod(ori)
                        .getParameterAnnotations()[i])) {
                        writeParam(sb, params.get(i), ori, isJson);
                    }
                }
                sb.append("</representation>");
            }
        }
    }

    protected List<OperationResourceInfo> sortOperationsByPath(Set<OperationResourceInfo> ops) {
        List<OperationResourceInfo> opsWithSamePath = new LinkedList<OperationResourceInfo>(ops);
        Collections.sort(opsWithSamePath, new Comparator<OperationResourceInfo>() {

            public int compare(OperationResourceInfo op1, OperationResourceInfo op2) {
                boolean sub1 = op1.getHttpMethod() == null;
                boolean sub2 = op2.getHttpMethod() == null;
                if (sub1 && !sub2) {
                    return 1;
                } else if (!sub1 && sub2) {
                    return -1;
                }
                URITemplate ut1 = op1.getURITemplate();
                URITemplate ut2 = op2.getURITemplate();
                int result = ut1.getValue().compareTo(ut2.getValue());
                if (result == 0 && !(sub1 && sub2)) {
                    result = op1.getHttpMethod().compareTo(op2.getHttpMethod());
                }
                return result;
            }

        });
        return opsWithSamePath;
    }

    public List<ClassResourceInfo> getResourcesList(Message m, ClassResourceInfo cri) {
        return cri != null ? Collections.singletonList(cri)
               : ((JAXRSServiceImpl)m.getExchange().get(Service.class)).getClassResourceInfos();
    }

    // TODO: deal with caching later on
    public Response getExistingWadl(Message m, UriInfo ui, MediaType mt) {
        Endpoint ep = m.getExchange().get(Endpoint.class);
        if (ep != null) {
            String loc = (String)ep.get(JAXRSUtils.DOC_LOCATION);
            if (loc != null) {
                try {
                    InputStream is = ResourceUtils.getResourceStream(loc, (Bus)ep.get(Bus.class.getName()));
                    if (is != null) {
                        Element appEl = StaxUtils.read(is).getDocumentElement();

                        List<Element> grammarEls = DOMUtils.getChildrenWithName(appEl, WadlGenerator.WADL_NS,
                                                                                "grammars");
                        if (grammarEls.size() == 1) {
                            handleExistingDocRefs(DOMUtils.getChildrenWithName(grammarEls.get(0),
                                                                               WadlGenerator.WADL_NS,
                                                                               "include"), "href", loc, "",
                                                  m, ui);
                        }

                        List<Element> resourcesEls = DOMUtils.getChildrenWithName(appEl,
                                                                                  WadlGenerator.WADL_NS,
                                                                                  "resources");
                        if (resourcesEls.size() == 1) {
                            DOMUtils.setAttribute(resourcesEls.get(0), "base", getBaseURI(m, ui));

                            List<Element> resourceEls = DOMUtils.getChildrenWithName(resourcesEls.get(0),
                                                                                     WadlGenerator.WADL_NS,
                                                                                     "resource");
                            handleExistingDocRefs(resourceEls, "type", loc, "", m, ui);

                            return Response.ok().type(mt).entity(new DOMSource(appEl)).build();
                        }

                    }
                } catch (Exception ex) {
                    throw ExceptionUtils.toInternalServerErrorException(ex, null);
                }
            }
        }
        return null;
    }

    // TODO: deal with caching later on
    public Response getExistingResource(Message m, UriInfo ui, String href) {
        String loc = docLocationMap.get(href);
        Endpoint ep = m.getExchange().get(Endpoint.class);
        if (ep != null && loc != null) {
            try {
                int fragmentIndex = loc.lastIndexOf("#");
                if (fragmentIndex != -1) {
                    loc = loc.substring(0, fragmentIndex);
                }
                InputStream is = ResourceUtils.getResourceStream(loc, (Bus)ep.get(Bus.class.getName()));
                if (is != null) {
                    Element docEl = StaxUtils.read(is).getDocumentElement();
                    if (fragmentIndex != -1) {
                        List<Element> grammarEls = DOMUtils.getChildrenWithName(docEl, WadlGenerator.WADL_NS,
                                                                                "grammars");
                        if (grammarEls.size() == 1) {
                            handleExistingDocRefs(DOMUtils.getChildrenWithName(grammarEls.get(0),
                                                                               WadlGenerator.WADL_NS,
                                                                               "include"), "href", loc, href,
                                                  m, ui);
                        }
                    } else {
                        handleExistingDocRefs(DOMUtils.getChildrenWithName(docEl,
                                                                           XmlSchemaConstants.XSD_NAMESPACE_URI,
                                                                           "import"), "schemaLocation", loc,
                                              href, m, ui);
                        handleExistingDocRefs(DOMUtils.getChildrenWithName(docEl,
                                                                           XmlSchemaConstants.XSD_NAMESPACE_URI,
                                                                           "include"), "schemaLocation", loc,
                                              href, m, ui);
                    }

                    return Response.ok().type(MediaType.APPLICATION_XML_TYPE).entity(new DOMSource(docEl))
                        .build();
                }
            } catch (Exception ex) {
                throw ExceptionUtils.toBadRequestException(ex, null);
            }

        }
        return null;
    }

    private void handleExistingDocRefs(List<Element> elements, String attrName, String parentDocLoc,
                                       String parentRef, Message m, UriInfo ui) {
        int index = parentDocLoc.lastIndexOf('/');
        parentDocLoc = index == -1 ? parentDocLoc : parentDocLoc.substring(0, index + 1);

        index = parentRef.lastIndexOf('/');
        parentRef = index == -1 ? "" : parentRef.substring(0, index + 1);

        for (Element element : elements) {
            String href = element.getAttribute(attrName);
            String originalRef = href;
            if (!StringUtils.isEmpty(href) && !href.startsWith("#")) {
                int fragmentIndex = href.lastIndexOf("#");
                String fragment = null;
                if (fragmentIndex != -1) {
                    fragment = href.substring(fragmentIndex + 1);
                    href = href.substring(0, fragmentIndex);
                }

                String actualRef = parentRef + href;
                docLocationMap.put(actualRef, parentDocLoc + originalRef);
                UriBuilder ub = UriBuilder.fromUri(getBaseURI(m, ui)).path(actualRef).fragment(fragment);
                URI schemaURI = ub.build();
                DOMUtils.setAttribute(element, attrName, schemaURI.toString());
            }
        }
    }

    private void generateQName(StringBuilder sb, ElementQNameResolver qnameResolver,
                               Map<Class<?>, QName> clsMap, Class<?> type, boolean isCollection,
                               Annotation[] annotations) {
        if (!isCollection) {
            QName typeQName = clsMap.get(type);
            if (typeQName != null) {
                writeQName(sb, typeQName);
                return;
            }
        }

        QName qname = qnameResolver.resolve(type, annotations, Collections.unmodifiableMap(clsMap));

        if (qname != null) {
            if (!isCollection) {
                writeQName(sb, qname);
                clsMap.put(type, qname);
            } else {
                XMLName name = AnnotationUtils.getAnnotation(annotations, XMLName.class);
                String localPart = null;
                if (name != null) {
                    localPart = JAXRSUtils.convertStringToQName(name.value()).getLocalPart();
                } else {
                    localPart = qname.getLocalPart() + "s";    
                }
                QName collectionName = new QName(qname.getNamespaceURI(), localPart, qname.getPrefix());
                writeQName(sb, collectionName);
            }
        }
    }

    private void writeQName(StringBuilder sb, QName qname) {
        sb.append(" element=\"").append(qname.getPrefix()).append(':').append(qname.getLocalPart())
            .append("\"");
    }

    private boolean isXmlRoot(Class<?> cls) {
        return cls.getAnnotation(XmlRootElement.class) != null;
    }
    
    private SchemaCollection getSchemaCollection(ResourceTypes resourceTypes, JAXBContext context) {
        if (context == null) {
            return null;
        }
        SchemaCollection xmlSchemaCollection = new SchemaCollection();
        Collection<DOMSource> schemas = new HashSet<DOMSource>();
        List<String> targetNamespaces = new ArrayList<String>();
        try {
            for (DOMResult r : JAXBUtils.generateJaxbSchemas(context, CastUtils.cast(Collections.emptyMap(),
                                                                                     String.class,
                                                                                     DOMResult.class))) {
                Document doc = (Document)r.getNode();
                ElementQNameResolver theResolver = createElementQNameResolver(context);
                String tns = doc.getDocumentElement().getAttribute("targetNamespace");
                
                String tnsPrefix = doc.getDocumentElement().lookupPrefix(tns);
                if (tnsPrefix == null) {
                    String tnsDecl = doc.getDocumentElement().getAttribute("xmlns:tns");
                    tnsPrefix = tnsDecl != null && tnsDecl.equals(tns) ? "tns:" : "";
                } else {
                    tnsPrefix += ":";
                }

                if (supportJaxbXmlType) {
                    for (Class<?> cls : resourceTypes.getAllTypes().keySet()) {
                        if (isXmlRoot(cls)) {
                            continue;
                        }
                        XmlType root = cls.getAnnotation(XmlType.class);
                        if (root != null) {
                            QName typeName = theResolver.resolve(cls, new Annotation[] {},
                                                           Collections.<Class<?>, QName> emptyMap());
                            if (typeName != null && tns.equals(typeName.getNamespaceURI())) {
                                QName elementName = resourceTypes.getXmlNameMap().get(cls);
                                if (elementName == null) {
                                    elementName = typeName;
                                }
                                Element newElement = doc
                                    .createElementNS(XmlSchemaConstants.XSD_NAMESPACE_URI, "xs:element");
                                newElement.setAttribute("name", elementName.getLocalPart());
                                newElement.setAttribute("type", tnsPrefix + typeName.getLocalPart());
                                
                                if (Modifier.isAbstract(cls.getModifiers()) 
                                    && resourceTypes.getSubstitutions().values().contains(cls)) {
                                    newElement.setAttribute("abstract", "true");
                                }
                                
                                doc.getDocumentElement().appendChild(newElement);
                            }
                        }
                    }
                    if (supportJaxbSubstitutions) {
                        for (Map.Entry<Class<?>, Class<?>> entry : resourceTypes.getSubstitutions().entrySet()) {
                            QName typeName = theResolver.resolve(entry.getKey(), new Annotation[] {},
                                                                 Collections.<Class<?>, QName> emptyMap());
                            List<Element> elements = DOMUtils.findAllElementsByTagNameNS(doc.getDocumentElement(),
                                                                XmlSchemaConstants.XSD_NAMESPACE_URI, "element");
                            for (Element element : elements) {
                                if (element.getAttribute("name").equals(typeName.getLocalPart())) {
                                    QName groupName = theResolver.resolve(entry.getValue(), new Annotation[] {},
                                                                         Collections.<Class<?>, QName> emptyMap());
                                    if (groupName != null) {
                                        element.setAttribute("substitutionGroup", tnsPrefix + groupName.getLocalPart());
                                    }
                                }
                            }
                        }
                    }
                }
                if (supportCollections && !resourceTypes.getCollectionMap().isEmpty()) {
                    for (Map.Entry<Class<?>, QName> entry : resourceTypes.getCollectionMap().entrySet()) {
                        QName colQName = entry.getValue();
                        if (colQName == null) {
                            colQName = theResolver.resolve(entry.getKey(), new Annotation[] {},
                                                Collections.<Class<?>, QName> emptyMap());
                            if (colQName != null) {
                                colQName = new QName(colQName.getNamespaceURI(), 
                                                     colQName.getLocalPart() + "s",
                                                     colQName.getPrefix());
                            }
                        }
                        if (colQName == null) {
                            continue;
                        }
                        if (tns.equals(colQName.getNamespaceURI())) {
                            QName typeName = theResolver.resolve(entry.getKey(), new Annotation[] {},
                                                                  Collections.<Class<?>, QName> emptyMap());
                            if (typeName != null) {
                                Element newElement = doc
                                    .createElementNS(XmlSchemaConstants.XSD_NAMESPACE_URI, "xs:element");
                                newElement.setAttribute("name", colQName.getLocalPart());
                                Element ctElement = doc.createElementNS(XmlSchemaConstants.XSD_NAMESPACE_URI,
                                                                        "xs:complexType");
                                newElement.appendChild(ctElement);
                                Element seqElement = doc
                                    .createElementNS(XmlSchemaConstants.XSD_NAMESPACE_URI, "xs:sequence");
                                ctElement.appendChild(seqElement);
                                Element xsElement = doc.createElementNS(XmlSchemaConstants.XSD_NAMESPACE_URI,
                                                                        "xs:element");
                                seqElement.appendChild(xsElement);
                                xsElement.setAttribute("ref", tnsPrefix + typeName.getLocalPart());
                                xsElement.setAttribute("minOccurs", "0");
                                xsElement.setAttribute("maxOccurs", "unbounded");

                                doc.getDocumentElement().appendChild(newElement);
                            }
                        }
                    }
                }
                DOMSource source = new DOMSource(doc, r.getSystemId());
                schemas.add(source);
                if (!StringUtils.isEmpty(tns)) {
                    targetNamespaces.add(tns);
                }
            }
        } catch (IOException e) {
            LOG.fine("No schema can be generated");
            return null;
        }

        boolean hackAroundEmptyNamespaceIssue = false;
        for (DOMSource r : schemas) {
            hackAroundEmptyNamespaceIssue = addSchemaDocument(xmlSchemaCollection, targetNamespaces,
                                                              (Document)r.getNode(), r.getSystemId(),
                                                              hackAroundEmptyNamespaceIssue);
        }
        return xmlSchemaCollection;
    }

    private QName getJaxbQName(String name, String namespace, Class<?> type, Map<Class<?>, QName> clsMap) {

        QName qname = getQNameFromParts(name, namespace, type, clsMap);
        if (qname != null) {
            return qname;
        }
        String ns = JAXBUtils.getPackageNamespace(type);
        if (ns != null) {
            return getQNameFromParts(name, ns, type, clsMap);
        } else {
            return null;
        }
        
    }

    private QName getJaxbQName(JAXBContextProxy jaxbProxy, Class<?> type, Map<Class<?>, QName> clsMap) {

        XmlRootElement root = type.getAnnotation(XmlRootElement.class);
        if (root != null) {
            return getJaxbQName(root.name(), root.namespace(), type, clsMap);
        }

        try {
            JAXBBeanInfo jaxbInfo = jaxbProxy == null ? null : JAXBUtils.getBeanInfo(jaxbProxy, type);
            if (jaxbInfo == null) {
                return null;
            }
            Object instance = type.newInstance();
            return getQNameFromParts(jaxbInfo.getElementLocalName(instance),
                                     jaxbInfo.getElementNamespaceURI(instance), type, clsMap);
        } catch (Exception ex) {
            // ignore
        }
        return null;
    }

    private String getPrefix(String ns, Map<Class<?>, QName> clsMap) {
        String prefix = null;
        for (QName name : clsMap.values()) {
            if (name.getNamespaceURI().equals(ns)) {
                prefix = name.getPrefix();
                break;
            }
        }
        if (prefix == null) {
            int size = new HashSet<QName>(clsMap.values()).size();
            prefix = nsPrefix + (size + 1);
        }
        return prefix;
    }

    private boolean isFormRequest(OperationResourceInfo ori) {
        for (Parameter p : ori.getParameters()) {
            if (p.getType() == ParameterType.FORM
                || p.getType() == ParameterType.REQUEST_BODY
                && (getMethod(ori).getParameterTypes()[p.getIndex()] == MultivaluedMap.class || AnnotationUtils
                    .getAnnotation(getMethod(ori).getParameterAnnotations()[p.getIndex()], Multipart.class) != null)) {
                return true;
            }
        }
        return false;
    }

    private Class<?> getFormClass(OperationResourceInfo ori) {
        List<Parameter> params = ori.getParameters();
        for (int i = 0; i < params.size(); i++) {
            if (isFormParameter(params.get(i), getMethod(ori).getParameterTypes()[i], getMethod(ori)
                .getParameterAnnotations()[i])) {
                return null;
            }
        }
        return MultivaluedMap.class;
    }

    private boolean isFormParameter(Parameter pm, Class<?> type, Annotation[] anns) {
        return ParameterType.FORM == pm.getType() || ParameterType.REQUEST_BODY == pm.getType()
               && AnnotationUtils.getAnnotation(anns, Multipart.class) != null
               && (InjectionUtils.isPrimitive(type) || type == InputStream.class);
    }

    // TODO : can we reuse this block with JAXBBinding somehow ?
    public boolean addSchemaDocument(SchemaCollection col, List<String> tnsList, Document d, String systemId,
                                     boolean hackAroundEmptyNamespaceIssue) {
        String ns = d.getDocumentElement().getAttribute("targetNamespace");

        if (StringUtils.isEmpty(ns)) {
            if (DOMUtils.getFirstElement(d.getDocumentElement()) == null) {
                hackAroundEmptyNamespaceIssue = true;
                return hackAroundEmptyNamespaceIssue;
            }
            // create a copy of the dom so we
            // can modify it.
            d = copy(d);
            ns = tnsList.isEmpty() ? "" : tnsList.get(0);
            d.getDocumentElement().setAttribute("targetNamespace", ns);
        }

        if (hackAroundEmptyNamespaceIssue) {
            d = doEmptyNamespaceHack(d);
        }

        Node n = d.getDocumentElement().getFirstChild();
        while (n != null) {
            if (n instanceof Element) {
                Element e = (Element)n;
                if (e.getLocalName().equals("import")) {
                    e.removeAttribute("schemaLocation");
                }
            }
            n = n.getNextSibling();
        }

        synchronized (d) {
            col.read(d, systemId);
        }
        return hackAroundEmptyNamespaceIssue;
    }

    private Document doEmptyNamespaceHack(Document d) {
        boolean hasStuffToRemove = false;
        Element el = DOMUtils.getFirstElement(d.getDocumentElement());
        while (el != null) {
            if ("import".equals(el.getLocalName()) && StringUtils.isEmpty(el.getAttribute("targetNamespace"))) {
                hasStuffToRemove = true;
                break;
            }
            el = DOMUtils.getNextElement(el);
        }
        if (hasStuffToRemove) {
            // create a copy of the dom so we
            // can modify it.
            d = copy(d);
            el = DOMUtils.getFirstElement(d.getDocumentElement());
            while (el != null) {
                if ("import".equals(el.getLocalName())
                    && StringUtils.isEmpty(el.getAttribute("targetNamespace"))) {
                    d.getDocumentElement().removeChild(el);
                    el = DOMUtils.getFirstElement(d.getDocumentElement());
                } else {
                    el = DOMUtils.getNextElement(el);
                }
            }
        }

        return d;
    }

    private Document copy(Document doc) {
        try {
            return StaxUtils.copy(doc);
        } catch (XMLStreamException e) {
            // ignore
        } catch (ParserConfigurationException e) {
            // ignore
        }
        return doc;
    }

    private QName getQNameFromParts(String name, String namespace, Class<?> type, Map<Class<?>, QName> clsMap) {
        if (namespace == null || JAXB_DEFAULT_NAMESPACE.equals(namespace) || namespace.length() == 0) {
            return null;
        }
        if (name == null || name.length() == 0) {
            return null;
        }
        if (JAXB_DEFAULT_NAME.equals(name)) {
            name = StringUtils.uncapitalize(type.getSimpleName());
        }
        String prefix = getPrefix(namespace, clsMap);
        return new QName(namespace, name, prefix);
    }

    public void setIgnoreMessageWriters(boolean ignoreMessageWriters) {
        this.ignoreMessageWriters = ignoreMessageWriters;
    }

    private void handleApplicationDocs(StringBuilder sbApp) {
        if (applicationTitle != null) {
            sbApp.append("<doc title=\"" + xmlEncodeIfNeeded(applicationTitle) + "\"/>");
        }
    }

    protected void handleDocs(Annotation[] anns, StringBuilder sb, String category, boolean allowDefault,
                            boolean isJson) {
        for (Annotation a : anns) {
            if (a.annotationType() == Descriptions.class) {
                Descriptions ds = (Descriptions)a;
                handleDocs(ds.value(), sb, category, allowDefault, isJson);
                return;
            }
            if (a.annotationType() == Description.class) {
                Description d = (Description)a;
                if (d.target().length() == 0 && !allowDefault || d.target().length() > 0
                    && !d.target().equals(category)) {
                    continue;
                }

                sb.append("<doc");
                if (!isJson && d.lang().length() > 0) {
                    sb.append(" xml:lang=\"" + d.lang() + "\"");
                }
                if (d.title().length() > 0) {
                    sb.append(" title=\"" + xmlEncodeIfNeeded(d.title()) + "\"");
                }
                sb.append(">");
                if (d.value().length() > 0) {
                    sb.append(xmlEncodeIfNeeded(d.value()));
                } else if (d.docuri().length() > 0) {
                    InputStream is = null;
                    if (d.docuri().startsWith(CLASSPATH_PREFIX)) {
                        String path = d.docuri().substring(CLASSPATH_PREFIX.length());
                        is = ResourceUtils.getClasspathResourceStream(path, SchemaHandler.class,
                            bus == null ? BusFactory.getDefaultBus() : bus);
                        if (is != null) {
                            try {
                                sb.append(IOUtils.toString(is));
                            } catch (IOException ex) {
                                // ignore
                            }
                        }
                    }
                }
                sb.append("</doc>");
            }
        }
    }

    private String getNamespace() {
        return wadlNamespace != null ? wadlNamespace : WADL_NS;
    }

    public void setWadlNamespace(String namespace) {
        this.wadlNamespace = namespace;
    }

    public void setSingleResourceMultipleMethods(boolean singleResourceMultipleMethods) {
        this.singleResourceMultipleMethods = singleResourceMultipleMethods;
    }

    public void setUseSingleSlashResource(boolean useSingleSlashResource) {
        this.useSingleSlashResource = useSingleSlashResource;
    }

    public void setLinkJsonToXmlSchema(boolean link) {
        setLinkAnyMediaTypeToXmlSchema(link);
    }

    public void setLinkAnyMediaTypeToXmlSchema(boolean link) {
        linkAnyMediaTypeToXmlSchema = link;
    }

    public void setSchemaLocations(List<String> locations) {

        externalQnamesMap = new HashMap<String, List<String>>();
        externalSchemasCache = new ArrayList<String>(locations.size());
        for (int i = 0; i < locations.size(); i++) {
            String loc = locations.get(i);
            try {
                loadSchemasIntoCache(loc);
            } catch (Exception ex) {
                LOG.warning("No schema resource " + loc + " can be loaded : " + ex.getMessage());
                externalSchemasCache = null;
                externalQnamesMap = null;
                return;
            }
        }
    }

    private void loadSchemasIntoCache(String loc) throws Exception {
        InputStream is = ResourceUtils.getResourceStream(loc, 
            bus == null ? BusFactory.getDefaultBus() : bus);
        if (is == null) {
            return;
        }
        ByteArrayInputStream bis = IOUtils.loadIntoBAIS(is);
        XMLSource source = new XMLSource(bis);
        source.setBuffering();
        String targetNs = source.getValue("/*/@targetNamespace");

        Map<String, String> nsMap = Collections.singletonMap("xs", XmlSchemaConstants.XSD_NAMESPACE_URI);
        String[] elementNames = source.getValues("/*/xs:element/@name", nsMap);
        externalQnamesMap.put(targetNs, Arrays.asList(elementNames));
        String schemaValue = source.getNode("/xs:schema", nsMap, String.class);
        externalSchemasCache.add(schemaValue);
    }

    public void setUseJaxbContextForQnames(boolean checkJaxbOnly) {
        this.useJaxbContextForQnames = checkJaxbOnly;
    }

    protected ElementQNameResolver createElementQNameResolver(JAXBContext context) {
        if (resolver != null) {
            return resolver;
        }
        if (useJaxbContextForQnames) {
            if (context != null) {
                JAXBContextProxy proxy = JAXBUtils.createJAXBContextProxy(context);
                return new JaxbContextQNameResolver(proxy);
            } else {
                return null;
            }
        } else if (externalQnamesMap != null) {
            return new SchemaQNameResolver(externalQnamesMap);
        } else {
            return new XMLNameQNameResolver();
        }
    }

    protected SchemaWriter createSchemaWriter(ResourceTypes resourceTypes, JAXBContext context, UriInfo ui) {
        // if neither externalSchemaLinks nor externalSchemasCache is set
        // then JAXBContext will be used to generate the schema
        if (externalSchemaLinks != null && externalSchemasCache == null) {
            return new ExternalSchemaWriter(externalSchemaLinks, ui);
        } else if (externalSchemasCache != null) {
            return new StringSchemaWriter(externalSchemasCache, externalSchemaLinks, ui);
        } else if (context != null) {
            SchemaCollection coll = getSchemaCollection(resourceTypes, context);
            if (coll != null) {
                return new SchemaCollectionWriter(coll);
            }
        }
        return null;
    }

    public void setExternalLinks(List<String> externalLinks) {
        externalSchemaLinks = new LinkedList<URI>();
        for (String s : externalLinks) {
            try {
                String href = s;
                if (href.startsWith("classpath:")) {
                    int index = href.lastIndexOf('/');
                    href = index == -1 ? href.substring(9) : href.substring(index + 1);
                    docLocationMap.put(href, s);
                }
                externalSchemaLinks.add(URI.create(href));
            } catch (Exception ex) {
                LOG.warning("Not a valid URI : " + s);
                externalSchemaLinks = null;
                break;
            }
        }
    }

    protected interface SchemaWriter {
        void write(StringBuilder sb);
    }

    private class StringSchemaWriter implements SchemaWriter {

        private List<String> theSchemas;

        public StringSchemaWriter(List<String> schemas, List<URI> links, UriInfo ui) {

            this.theSchemas = new LinkedList<String>();
            // we'll need to do the proper schema caching eventually
            for (String s : schemas) {
                XMLSource source = new XMLSource(new ByteArrayInputStream(s.getBytes()));
                source.setBuffering();
                Map<String, String> locs = getLocationsMap(source, "import", links, ui);
                locs.putAll(getLocationsMap(source, "include", links, ui));
                String actualSchema = !locs.isEmpty() ? transformSchema(s, locs) : s;
                theSchemas.add(actualSchema);
            }
        }

        private Map<String, String> getLocationsMap(XMLSource source, String elementName, List<URI> links,
                                                    UriInfo ui) {
            Map<String, String> nsMap = Collections.singletonMap("xs", XmlSchemaConstants.XSD_NAMESPACE_URI);
            String[] locations = source.getValues("/*/xs:" + elementName + "/@schemaLocation", nsMap);

            Map<String, String> locs = new HashMap<String, String>();
            if (locations == null) {
                return locs;
            }

            for (String loc : locations) {
                try {
                    URI uri = URI.create(loc);
                    if (!uri.isAbsolute()) {
                        if (links != null) {
                            for (URI overwriteURI : links) {
                                if (overwriteURI.toString().endsWith(loc)) {
                                    if (overwriteURI.isAbsolute()) {
                                        locs.put(loc, overwriteURI.toString());
                                    } else {
                                        locs.put(loc, ui.getBaseUriBuilder().path(overwriteURI.toString())
                                            .build().toString());
                                    }
                                    break;
                                }
                            }
                        }
                        if (!locs.containsKey(loc)) {
                            locs.put(loc, ui.getBaseUriBuilder().path(loc).build().toString());
                        }
                    }
                } catch (Exception ex) {
                    // continue
                }
            }
            return locs;
        }

        private String transformSchema(String schema, Map<String, String> locs) {
            ByteArrayOutputStream bos = new ByteArrayOutputStream();
            SchemaConverter sc = new SchemaConverter(StaxUtils.createXMLStreamWriter(bos), locs);
            try {
                StaxUtils.copy(new StreamSource(new StringReader(schema)), sc);
                sc.flush();
                sc.close();
                return bos.toString();
            } catch (Exception ex) {
                return schema;
            }

        }

        public void write(StringBuilder sb) {
            for (String s : theSchemas) {
                sb.append(s);
            }
        }
    }

    private class SchemaCollectionWriter implements SchemaWriter {

        private SchemaCollection coll;

        public SchemaCollectionWriter(SchemaCollection coll) {
            this.coll = coll;
        }

        public void write(StringBuilder sb) {
            for (XmlSchema xs : coll.getXmlSchemas()) {
                if (xs.getItems().isEmpty() || WSDLConstants.NS_SCHEMA_XSD.equals(xs.getTargetNamespace())) {
                    continue;
                }
                StringWriter writer = new StringWriter();
                xs.write(writer);
                sb.append(writer.toString());
            }
        }
    }

    private class ExternalSchemaWriter implements SchemaWriter {

        private List<URI> links;
        private UriInfo uriInfo;

        public ExternalSchemaWriter(List<URI> links, UriInfo ui) {
            this.links = links;
            this.uriInfo = ui;
        }

        public void write(StringBuilder sb) {
            for (URI link : links) {
                try {
                    URI value = link.isAbsolute() ? link : uriInfo.getBaseUriBuilder().path(link.toString())
                        .build();
                    sb.append("<include href=\"").append(value.toString()).append("\"/>");
                } catch (Exception ex) {
                    LOG.warning("WADL grammar section will be incomplete, this link is not a valid URI : "
                                + link.toString());
                }
            }
        }
    }

    private class JaxbContextQNameResolver implements ElementQNameResolver {

        private JAXBContextProxy proxy;

        public JaxbContextQNameResolver(JAXBContextProxy proxy) {
            this.proxy = proxy;
        }

        public QName resolve(Class<?> type, Annotation[] annotations, Map<Class<?>, QName> clsMap) {
            QName qname = WadlGenerator.this.getJaxbQName(proxy, type, clsMap);
            if (qname == null && supportJaxbXmlType) {
                XmlType root = type.getAnnotation(XmlType.class);
                if (root != null) {
                    XMLName name = AnnotationUtils.getAnnotation(annotations, XMLName.class);
                    if (name == null) {
                        qname = getJaxbQName(root.name(), root.namespace(), type, clsMap);
                    } else {
                        QName tempQName = JAXRSUtils.convertStringToQName(name.value());
                        qname = new QName(tempQName.getNamespaceURI(),
                                          tempQName.getLocalPart(),
                                          getPrefix(tempQName.getNamespaceURI(), clsMap));
                    }
                }
            }
            return qname;
        }

    }

    private class XMLNameQNameResolver implements ElementQNameResolver {

        public QName resolve(Class<?> type, Annotation[] annotations, Map<Class<?>, QName> clsMap) {
            XMLName name = AnnotationUtils.getAnnotation(annotations, XMLName.class);
            if (name == null) {
                name = type.getAnnotation(XMLName.class);
            }
            if (name != null) {
                QName qname = XMLUtils.convertStringToQName(name.value(), name.prefix());
                if (qname.getPrefix().length() > 0) {
                    return qname;
                } else {
                    return getQNameFromParts(qname.getLocalPart(), qname.getNamespaceURI(), type, clsMap);
                }
            }
            return null;
        }

    }

    private class SchemaQNameResolver implements ElementQNameResolver {

        private Map<String, List<String>> map;

        public SchemaQNameResolver(Map<String, List<String>> map) {
            this.map = map;
        }

        public QName resolve(Class<?> type, Annotation[] annotations, Map<Class<?>, QName> clsMap) {
            String name = type.getSimpleName();
            for (Map.Entry<String, List<String>> entry : map.entrySet()) {
                String elementName = null;
                if (entry.getValue().contains(name)) {
                    elementName = name;
                } else if (entry.getValue().contains(name.toLowerCase())) {
                    elementName = name.toLowerCase();
                }
                if (elementName != null) {
                    return getQNameFromParts(elementName, entry.getKey(), type, clsMap);
                }
            }
            return null;
        }

    }

    public void setResolver(ElementQNameResolver resolver) {
        this.resolver = resolver;
    }

    public void setPrivateAddresses(List<String> privateAddresses) {
        this.privateAddresses = privateAddresses;
    }

    public List<String> getPrivateAddresses() {
        return privateAddresses;
    }

    public void setAddResourceAndMethodIds(boolean addResourceAndMethodIds) {
        this.addResourceAndMethodIds = addResourceAndMethodIds;
    }

    private Method getMethod(OperationResourceInfo ori) {
        Method annMethod = ori.getAnnotatedMethod();
        return annMethod != null ? annMethod : ori.getMethodToInvoke();
    }

    public void setApplicationTitle(String applicationTitle) {
        this.applicationTitle = applicationTitle;
    }

    public void setNamespacePrefix(String prefix) {
        this.nsPrefix = prefix;
    }

    public void setIgnoreForwardSlash(boolean ignoreForwardSlash) {
        this.ignoreForwardSlash = ignoreForwardSlash;
    }

    public void setIgnoreRequests(boolean ignoreRequests) {
        this.ignoreRequests = ignoreRequests;
    }

    public void setSupportCollections(boolean support) {
        this.supportCollections = support;
    }

    public void setDefaultMediaType(String mt) {
        this.defaultMediaType = JAXRSUtils.toMediaType(mt);
    }

    public void setSupportJaxbXmlType(boolean supportJaxbXmlType) {
        this.supportJaxbXmlType = supportJaxbXmlType;
    }

    public void setSupportJaxbSubstitutions(boolean supportJaxbSubstitutions) {
        this.supportJaxbSubstitutions = supportJaxbSubstitutions;
    }

    private static class SchemaConverter extends DelegatingXMLStreamWriter {
        private static final String SCHEMA_LOCATION = "schemaLocation";
        private Map<String, String> locsMap;

        public SchemaConverter(XMLStreamWriter writer, Map<String, String> locsMap) {
            super(writer);
            this.locsMap = locsMap;
        }

        public void writeAttribute(String local, String value) throws XMLStreamException {
            if (SCHEMA_LOCATION.equals(local) && locsMap.containsKey(value)) {
                value = locsMap.get(value);
            }
            super.writeAttribute(local, value);
        }
    }

}
