/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cxf.xkms.x509.repo.file;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.math.BigInteger;
import java.net.URISyntaxException;
import java.security.cert.CRLException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509CRL;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.regex.Pattern;
import org.apache.cxf.xkms.exception.XKMSConfigurationException;
import org.apache.cxf.xkms.model.xkms.ResultMajorEnum;
import org.apache.cxf.xkms.model.xkms.ResultMinorEnum;
import org.apache.cxf.xkms.model.xkms.UseKeyWithType;
import org.apache.cxf.xkms.x509.repo.CertificateRepo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileCertificateRepo
implements CertificateRepo {
    private static final Logger LOG = LoggerFactory.getLogger(FileCertificateRepo.class);
    private static final String CN_PREFIX = "cn=";
    private static final String TRUSTED_CAS_PATH = "trusted_cas";
    private static final String CRLS_PATH = "crls";
    private static final String CAS_PATH = "cas";
    private final File storageDir;
    private final CertificateFactory certFactory;

    public FileCertificateRepo(String path) {
        this.storageDir = new File(path);
        try {
            this.certFactory = CertificateFactory.getInstance("X.509");
        }
        catch (Exception e) {
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    @Override
    public void saveCertificate(X509Certificate cert, UseKeyWithType id) {
        this.saveCategorizedCertificate(cert, id, false, false);
    }

    public void saveTrustedCACertificate(X509Certificate cert, UseKeyWithType id) {
        this.saveCategorizedCertificate(cert, id, true, false);
    }

    public void saveCACertificate(X509Certificate cert, UseKeyWithType id) {
        this.saveCategorizedCertificate(cert, id, false, true);
    }

    public void saveCRL(X509CRL crl, UseKeyWithType id) {
        String name = crl.getIssuerX500Principal().getName();
        try {
            String path = this.convertDnForFileSystem(name) + ".cer";
            Pattern p = Pattern.compile("[a-zA-Z_0-9-_]");
            if (!p.matcher(path).find()) {
                throw new URISyntaxException(path, "Input did not match [a-zA-Z_0-9-_].");
            }
            File certFile = new File(this.storageDir + "/" + CRLS_PATH, path);
            certFile.getParentFile().mkdirs();
            FileOutputStream fos = new FileOutputStream(certFile);
            BufferedOutputStream bos = new BufferedOutputStream(fos);
            bos.write(crl.getEncoded());
            bos.close();
            fos.close();
        }
        catch (Exception e) {
            throw new RuntimeException("Error saving CRL " + name + ": " + e.getMessage(), e);
        }
    }

    private boolean saveCategorizedCertificate(X509Certificate cert, UseKeyWithType id, boolean isTrustedCA, boolean isCA) {
        String name = cert.getSubjectX500Principal().getName();
        String category = "";
        if (isTrustedCA) {
            category = TRUSTED_CAS_PATH;
        }
        if (isCA) {
            category = CAS_PATH;
        }
        try {
            File certFile = new File(this.storageDir + "/" + category, this.getRelativePathForSubjectDn(cert));
            certFile.getParentFile().mkdirs();
            FileOutputStream fos = new FileOutputStream(certFile);
            BufferedOutputStream bos = new BufferedOutputStream(fos);
            bos.write(cert.getEncoded());
            bos.close();
            fos.close();
        }
        catch (Exception e) {
            throw new RuntimeException("Error saving certificate " + name + ": " + e.getMessage(), e);
        }
        return true;
    }

    public String convertDnForFileSystem(String dn) {
        String result = dn.replace("=", "-");
        result = result.replace(", ", "_");
        result = result.replace(",", "_");
        result = result.replace("/", "_");
        result = result.replace("\\", "_");
        result = result.replace("{", "_");
        result = result.replace("}", "_");
        result = result.replace(":", "_");
        return result;
    }

    public String getRelativePathForSubjectDn(X509Certificate cert) throws URISyntaxException {
        BigInteger serialNumber = cert.getSerialNumber();
        String issuer = cert.getIssuerX500Principal().getName();
        String path = this.convertDnForFileSystem(issuer) + "-" + serialNumber.toString() + ".cer";
        Pattern p = Pattern.compile("[a-zA-Z_0-9-_]");
        if (p.matcher(path).find()) {
            return path;
        }
        throw new URISyntaxException(path, "Input did not match [a-zA-Z_0-9-_].");
    }

    private File[] getX509Files() {
        ArrayList<File> certificateFiles = new ArrayList<File>();
        try {
            certificateFiles.addAll(Arrays.asList(this.storageDir.listFiles()));
            certificateFiles.addAll(Arrays.asList(new File(this.storageDir + "/" + TRUSTED_CAS_PATH).listFiles()));
            certificateFiles.addAll(Arrays.asList(new File(this.storageDir + "/" + CAS_PATH).listFiles()));
            certificateFiles.addAll(Arrays.asList(new File(this.storageDir + "/" + CRLS_PATH).listFiles()));
        }
        catch (NullPointerException nullPointerException) {
            // empty catch block
        }
        if (certificateFiles.isEmpty()) {
            throw new XKMSConfigurationException(ResultMajorEnum.HTTP_WWW_W_3_ORG_2002_03_XKMS_RECEIVER, ResultMinorEnum.HTTP_WWW_W_3_ORG_2002_03_XKMS_FAILURE, "File base persistence storage is not found: " + this.storageDir.getPath());
        }
        return certificateFiles.toArray(new File[certificateFiles.size()]);
    }

    public X509Certificate readCertificate(File certFile) throws CertificateException, FileNotFoundException {
        FileInputStream fis = new FileInputStream(certFile);
        return (X509Certificate)this.certFactory.generateCertificate(fis);
    }

    public X509CRL readCRL(File crlFile) throws FileNotFoundException, CRLException {
        FileInputStream fis = new FileInputStream(crlFile);
        return (X509CRL)this.certFactory.generateCRL(fis);
    }

    @Override
    public List<X509Certificate> getTrustedCaCerts() {
        File[] list;
        ArrayList<X509Certificate> results = new ArrayList<X509Certificate>();
        for (File certFile : list = this.getX509Files()) {
            try {
                if (certFile.isDirectory() || !certFile.getParent().endsWith(TRUSTED_CAS_PATH)) continue;
                X509Certificate cert = this.readCertificate(certFile);
                results.add(cert);
            }
            catch (Exception e) {
                LOG.warn(String.format("Cannot load certificate from file: %s. Error: %s", certFile, e.getMessage()));
            }
        }
        return results;
    }

    @Override
    public List<X509Certificate> getCaCerts() {
        File[] list;
        ArrayList<X509Certificate> results = new ArrayList<X509Certificate>();
        for (File certFile : list = this.getX509Files()) {
            try {
                if (certFile.isDirectory() || !certFile.getParent().endsWith(CAS_PATH)) continue;
                X509Certificate cert = this.readCertificate(certFile);
                results.add(cert);
            }
            catch (Exception e) {
                LOG.warn(String.format("Cannot load certificate from file: %s. Error: %s", certFile, e.getMessage()));
            }
        }
        return results;
    }

    @Override
    public List<X509CRL> getCRLs() {
        File[] list;
        ArrayList<X509CRL> results = new ArrayList<X509CRL>();
        for (File crlFile : list = this.getX509Files()) {
            try {
                if (crlFile.isDirectory() || !crlFile.getParent().endsWith(CRLS_PATH)) continue;
                X509CRL crl = this.readCRL(crlFile);
                results.add(crl);
            }
            catch (Exception e) {
                LOG.warn(String.format("Cannot load CRL from file: %s. Error: %s", crlFile, e.getMessage()));
            }
        }
        return results;
    }

    @Override
    public X509Certificate findByServiceName(String serviceName) {
        return this.findBySubjectDn(CN_PREFIX + serviceName);
    }

    @Override
    public X509Certificate findBySubjectDn(String subjectDn) {
        File[] list;
        ArrayList<X509Certificate> result = new ArrayList<X509Certificate>();
        for (File certFile : list = this.getX509Files()) {
            try {
                if (certFile.isDirectory()) continue;
                X509Certificate cert = this.readCertificate(certFile);
                LOG.debug("Searching for " + subjectDn + ". Checking cert " + cert.getSubjectDN().getName() + ", " + cert.getSubjectX500Principal().getName());
                if (!subjectDn.equalsIgnoreCase(cert.getSubjectDN().getName()) && !subjectDn.equalsIgnoreCase(cert.getSubjectX500Principal().getName())) continue;
                result.add(cert);
            }
            catch (Exception e) {
                LOG.warn(String.format("Cannot load certificate from file: %s. Error: %s", certFile, e.getMessage()));
            }
        }
        if (!result.isEmpty()) {
            return (X509Certificate)result.get(0);
        }
        return null;
    }

    @Override
    public X509Certificate findByIssuerSerial(String issuer, String serial) {
        File[] list;
        ArrayList<X509Certificate> result = new ArrayList<X509Certificate>();
        for (File certFile : list = this.getX509Files()) {
            try {
                if (certFile.isDirectory()) continue;
                X509Certificate cert = this.readCertificate(certFile);
                BigInteger cs = cert.getSerialNumber();
                BigInteger ss = new BigInteger(serial, 16);
                if (!issuer.equalsIgnoreCase(cert.getIssuerX500Principal().getName()) || !cs.equals(ss)) continue;
                result.add(cert);
            }
            catch (Exception e) {
                LOG.warn(String.format("Cannot load certificate from file: %s. Error: %s", certFile, e.getMessage()));
            }
        }
        if (!result.isEmpty()) {
            return (X509Certificate)result.get(0);
        }
        return null;
    }
}

