/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.daffodil.japi.debugger

/**
 * Abstract debugger runner, which can be implemented to create a custom debugger runner
 */
abstract class DebuggerRunner {
  /**
   * Called once at the beginning of a parse, allowing one to perform any
   * initialization steps that may be necessary.
   */
  def init(): Unit

  /**
   * Called by Daffodil when there is a pause in parsing to determine what
   * debugger actions should be taken.

   * @see <a target="_blank" href='https://daffodil.apache.org/debugger/'>Daffodil Interactive Debugger</a> - debugger commands
   *
   * @return a debugger command that tells the Daffodil debugger what step to
   *         take next.
   */
  def getCommand(): String

  /**
   * Called by Daffodil when a debugger command has produce output. This method
   * is called once for every line produced by the Daffodil debugger.
   *
   * @param line a single line of output generated by the Daffodil debugger
   */
  def lineOutput(line: String): Unit

  /**
   * Called once upon completion of a parse, allowing one to perform any
   * steps necessary to clean up the debugger runner.
   */
  def fini(): Unit
}

/* 
 * The below class is empty and is not ever actually used. It is just a place
 * holder. Whenever the Java API uses it, it is translated to the appropriate
 * Scala debugger runner. It is marked final so that it cannot be extended,
 * since the Scala pattern matcher would still match and use the equivalent
 * Scala trace runner and lose any added functionality. One must extend the
 * DebuggerRunner to create their own debugger runner.
 */

/**
 * [[DebuggerRunner]] that writes verbose information about parsing to stdout
 */
final class TraceDebuggerRunner extends DebuggerRunner {
  def init(): Unit = {}
  def getCommand(): String = ""
  def lineOutput(line: String) = {}
  def fini(): Unit = {}
}

