/*
 * Decompiled with CFR 0.152.
 */
package org.apache.directory.server.kerberos.kdc.authentication;

import java.io.IOException;
import java.net.InetAddress;
import java.util.Date;
import java.util.Set;
import javax.security.auth.kerberos.KerberosKey;
import javax.security.auth.kerberos.KerberosPrincipal;
import org.apache.directory.server.kerberos.kdc.KdcContext;
import org.apache.directory.server.kerberos.kdc.KdcServer;
import org.apache.directory.server.kerberos.kdc.authentication.AuthenticationContext;
import org.apache.directory.server.kerberos.sam.SamException;
import org.apache.directory.server.kerberos.sam.SamSubsystem;
import org.apache.directory.server.kerberos.shared.KerberosUtils;
import org.apache.directory.server.kerberos.shared.crypto.encryption.CipherTextHandler;
import org.apache.directory.server.kerberos.shared.crypto.encryption.EncryptionType;
import org.apache.directory.server.kerberos.shared.crypto.encryption.KeyUsage;
import org.apache.directory.server.kerberos.shared.crypto.encryption.RandomKeyFactory;
import org.apache.directory.server.kerberos.shared.exceptions.ErrorType;
import org.apache.directory.server.kerberos.shared.exceptions.KerberosException;
import org.apache.directory.server.kerberos.shared.io.decoder.EncryptedDataDecoder;
import org.apache.directory.server.kerberos.shared.io.encoder.EncryptionTypeInfoEncoder;
import org.apache.directory.server.kerberos.shared.io.encoder.PreAuthenticationDataEncoder;
import org.apache.directory.server.kerberos.shared.messages.AuthenticationReply;
import org.apache.directory.server.kerberos.shared.messages.Encodable;
import org.apache.directory.server.kerberos.shared.messages.KdcReply;
import org.apache.directory.server.kerberos.shared.messages.KdcRequest;
import org.apache.directory.server.kerberos.shared.messages.KerberosMessage;
import org.apache.directory.server.kerberos.shared.messages.components.EncTicketPart;
import org.apache.directory.server.kerberos.shared.messages.components.EncTicketPartModifier;
import org.apache.directory.server.kerberos.shared.messages.components.InvalidTicketException;
import org.apache.directory.server.kerberos.shared.messages.components.Ticket;
import org.apache.directory.server.kerberos.shared.messages.value.EncryptedData;
import org.apache.directory.server.kerberos.shared.messages.value.EncryptedTimeStamp;
import org.apache.directory.server.kerberos.shared.messages.value.EncryptionKey;
import org.apache.directory.server.kerberos.shared.messages.value.EncryptionTypeInfoEntry;
import org.apache.directory.server.kerberos.shared.messages.value.KerberosTime;
import org.apache.directory.server.kerberos.shared.messages.value.LastRequest;
import org.apache.directory.server.kerberos.shared.messages.value.PaData;
import org.apache.directory.server.kerberos.shared.messages.value.TransitedEncoding;
import org.apache.directory.server.kerberos.shared.messages.value.flags.KerberosFlag;
import org.apache.directory.server.kerberos.shared.messages.value.flags.TicketFlag;
import org.apache.directory.server.kerberos.shared.messages.value.types.PaDataType;
import org.apache.directory.server.kerberos.shared.replay.InMemoryReplayCache;
import org.apache.directory.server.kerberos.shared.replay.ReplayCache;
import org.apache.directory.server.kerberos.shared.store.PrincipalStore;
import org.apache.directory.server.kerberos.shared.store.PrincipalStoreEntry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class AuthenticationService {
    private static final Logger LOG = LoggerFactory.getLogger(AuthenticationService.class);
    private static final ReplayCache replayCache = new InMemoryReplayCache();
    private static final CipherTextHandler cipherTextHandler = new CipherTextHandler();
    private static final String SERVICE_NAME = "Authentication Service (AS)";

    public static void execute(AuthenticationContext authContext) throws Exception {
        if (LOG.isDebugEnabled()) {
            AuthenticationService.monitorRequest(authContext);
        }
        authContext.setReplayCache(replayCache);
        authContext.setCipherTextHandler(cipherTextHandler);
        if (authContext.getRequest().getProtocolVersionNumber() != 5) {
            throw new KerberosException(ErrorType.KDC_ERR_BAD_PVNO);
        }
        AuthenticationService.selectEncryptionType(authContext);
        AuthenticationService.getClientEntry(authContext);
        AuthenticationService.verifyPolicy(authContext);
        AuthenticationService.verifySam(authContext);
        AuthenticationService.verifyEncryptedTimestamp(authContext);
        if (authContext.getClientKey() == null) {
            AuthenticationService.verifyEncryptedTimestamp(authContext);
        }
        AuthenticationService.getServerEntry(authContext);
        AuthenticationService.generateTicket(authContext);
        AuthenticationService.buildReply(authContext);
        if (LOG.isDebugEnabled()) {
            AuthenticationService.monitorContext(authContext);
            AuthenticationService.monitorReply(authContext);
        }
        AuthenticationService.sealReply(authContext);
    }

    private static void selectEncryptionType(AuthenticationContext authContext) throws KerberosException, InvalidTicketException {
        AuthenticationContext kdcContext = authContext;
        KdcServer config = kdcContext.getConfig();
        Set requestedTypes = kdcContext.getRequest().getEType();
        EncryptionType bestType = KerberosUtils.getBestEncryptionType((Set)requestedTypes, config.getEncryptionTypes());
        LOG.debug("Session will use encryption type {}.", (Object)bestType);
        if (bestType == null) {
            throw new KerberosException(ErrorType.KDC_ERR_ETYPE_NOSUPP);
        }
        kdcContext.setEncryptionType(bestType);
    }

    private static void getClientEntry(AuthenticationContext authContext) throws KerberosException, InvalidTicketException {
        KerberosPrincipal principal = authContext.getRequest().getClientPrincipal();
        PrincipalStore store = authContext.getStore();
        PrincipalStoreEntry storeEntry = AuthenticationService.getEntry(principal, store, ErrorType.KDC_ERR_C_PRINCIPAL_UNKNOWN);
        authContext.setClientEntry(storeEntry);
    }

    private static void verifyPolicy(AuthenticationContext authContext) throws KerberosException, InvalidTicketException {
        PrincipalStoreEntry entry = authContext.getClientEntry();
        if (entry.isDisabled()) {
            throw new KerberosException(ErrorType.KDC_ERR_CLIENT_REVOKED);
        }
        if (entry.isLockedOut()) {
            throw new KerberosException(ErrorType.KDC_ERR_CLIENT_REVOKED);
        }
        if (entry.getExpiration().getTime() < new Date().getTime()) {
            throw new KerberosException(ErrorType.KDC_ERR_CLIENT_REVOKED);
        }
    }

    private static void verifySam(AuthenticationContext authContext) throws KerberosException, InvalidTicketException {
        LOG.debug("Verifying using SAM subsystem.");
        KdcRequest request = authContext.getRequest();
        KdcServer config = authContext.getConfig();
        PrincipalStoreEntry clientEntry = authContext.getClientEntry();
        String clientName = clientEntry.getPrincipal().getName();
        EncryptionKey clientKey = null;
        if (clientEntry.getSamType() != null) {
            PaData[] preAuthData;
            if (LOG.isDebugEnabled()) {
                LOG.debug("Entry for client principal {} has a valid SAM type.  Invoking SAM subsystem for pre-authentication.", (Object)clientName);
            }
            if ((preAuthData = request.getPreAuthData()) == null || preAuthData.length == 0) {
                throw new KerberosException(ErrorType.KDC_ERR_PREAUTH_REQUIRED, AuthenticationService.preparePreAuthenticationError(config.getEncryptionTypes()));
            }
            try {
                for (int ii = 0; ii < preAuthData.length; ++ii) {
                    if (!preAuthData[ii].getPaDataType().equals((Object)PaDataType.PA_ENC_TIMESTAMP)) continue;
                    KerberosKey samKey = SamSubsystem.getInstance().verify(clientEntry, preAuthData[ii].getPaDataValue());
                    clientKey = new EncryptionKey(EncryptionType.getTypeByOrdinal((int)samKey.getKeyType()), samKey.getEncoded());
                }
            }
            catch (SamException se) {
                throw new KerberosException(ErrorType.KRB_ERR_GENERIC, (Throwable)se);
            }
            authContext.setClientKey(clientKey);
            authContext.setPreAuthenticated(true);
            if (LOG.isDebugEnabled()) {
                LOG.debug("Pre-authentication using SAM subsystem successful for {}.", (Object)clientName);
            }
        }
    }

    private static void verifyEncryptedTimestamp(AuthenticationContext authContext) throws KerberosException, InvalidTicketException {
        LOG.debug("Verifying using encrypted timestamp.");
        KdcServer config = authContext.getConfig();
        KdcRequest request = authContext.getRequest();
        CipherTextHandler cipherTextHandler = authContext.getCipherTextHandler();
        PrincipalStoreEntry clientEntry = authContext.getClientEntry();
        String clientName = clientEntry.getPrincipal().getName();
        EncryptionKey clientKey = null;
        if (clientEntry.getSamType() == null) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Entry for client principal {} has no SAM type.  Proceeding with standard pre-authentication.", (Object)clientName);
            }
            EncryptionType encryptionType = authContext.getEncryptionType();
            clientKey = (EncryptionKey)clientEntry.getKeyMap().get(encryptionType);
            if (clientKey == null) {
                throw new KerberosException(ErrorType.KDC_ERR_NULL_KEY);
            }
            if (config.isPaEncTimestampRequired()) {
                PaData[] preAuthData = request.getPreAuthData();
                if (preAuthData == null) {
                    throw new KerberosException(ErrorType.KDC_ERR_PREAUTH_REQUIRED, AuthenticationService.preparePreAuthenticationError(config.getEncryptionTypes()));
                }
                EncryptedTimeStamp timestamp = null;
                for (int ii = 0; ii < preAuthData.length; ++ii) {
                    EncryptedData dataValue;
                    if (!preAuthData[ii].getPaDataType().equals((Object)PaDataType.PA_ENC_TIMESTAMP)) continue;
                    try {
                        dataValue = EncryptedDataDecoder.decode((byte[])preAuthData[ii].getPaDataValue());
                    }
                    catch (IOException ioe) {
                        throw new KerberosException(ErrorType.KRB_AP_ERR_BAD_INTEGRITY, (Throwable)ioe);
                    }
                    catch (ClassCastException cce) {
                        throw new KerberosException(ErrorType.KRB_AP_ERR_BAD_INTEGRITY, (Throwable)cce);
                    }
                    timestamp = (EncryptedTimeStamp)cipherTextHandler.unseal(EncryptedTimeStamp.class, clientKey, dataValue, KeyUsage.NUMBER1);
                }
                if (preAuthData.length > 0 && timestamp == null) {
                    throw new KerberosException(ErrorType.KDC_ERR_PADATA_TYPE_NOSUPP);
                }
                if (timestamp == null) {
                    throw new KerberosException(ErrorType.KDC_ERR_PREAUTH_REQUIRED, AuthenticationService.preparePreAuthenticationError(config.getEncryptionTypes()));
                }
                if (!timestamp.getTimeStamp().isInClockSkew(config.getAllowableClockSkew())) {
                    throw new KerberosException(ErrorType.KDC_ERR_PREAUTH_FAILED);
                }
            }
        }
        authContext.setClientKey(clientKey);
        authContext.setPreAuthenticated(true);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Pre-authentication by encrypted timestamp successful for {}.", (Object)clientName);
        }
    }

    private static void getServerEntry(AuthenticationContext authContext) throws KerberosException, InvalidTicketException {
        KerberosPrincipal principal = authContext.getRequest().getServerPrincipal();
        PrincipalStore store = authContext.getStore();
        authContext.setServerEntry(AuthenticationService.getEntry(principal, store, ErrorType.KDC_ERR_S_PRINCIPAL_UNKNOWN));
    }

    private static void generateTicket(AuthenticationContext authContext) throws KerberosException, InvalidTicketException {
        KdcRequest request = authContext.getRequest();
        CipherTextHandler cipherTextHandler = authContext.getCipherTextHandler();
        KerberosPrincipal serverPrincipal = request.getServerPrincipal();
        EncryptionType encryptionType = authContext.getEncryptionType();
        EncryptionKey serverKey = (EncryptionKey)authContext.getServerEntry().getKeyMap().get(encryptionType);
        KerberosPrincipal ticketPrincipal = request.getServerPrincipal();
        EncTicketPartModifier newTicketBody = new EncTicketPartModifier();
        KdcServer config = authContext.getConfig();
        newTicketBody.setFlag((KerberosFlag)TicketFlag.INITIAL);
        if (authContext.isPreAuthenticated()) {
            newTicketBody.setFlag((KerberosFlag)TicketFlag.PRE_AUTHENT);
        }
        if (request.getOption(1)) {
            if (!config.isForwardableAllowed()) {
                throw new KerberosException(ErrorType.KDC_ERR_POLICY);
            }
            newTicketBody.setFlag((KerberosFlag)TicketFlag.FORWARDABLE);
        }
        if (request.getOption(3)) {
            if (!config.isProxiableAllowed()) {
                throw new KerberosException(ErrorType.KDC_ERR_POLICY);
            }
            newTicketBody.setFlag((KerberosFlag)TicketFlag.PROXIABLE);
        }
        if (request.getOption(5)) {
            if (!config.isPostdatedAllowed()) {
                throw new KerberosException(ErrorType.KDC_ERR_POLICY);
            }
            newTicketBody.setFlag((KerberosFlag)TicketFlag.MAY_POSTDATE);
        }
        if (request.getOption(30) || request.getOption(31) || request.getOption(4) || request.getOption(2) || request.getOption(28)) {
            throw new KerberosException(ErrorType.KDC_ERR_BADOPTION);
        }
        EncryptionKey sessionKey = RandomKeyFactory.getRandomKey((EncryptionType)authContext.getEncryptionType());
        newTicketBody.setSessionKey(sessionKey);
        newTicketBody.setClientPrincipal(request.getClientPrincipal());
        newTicketBody.setTransitedEncoding(new TransitedEncoding());
        KerberosTime now = new KerberosTime();
        newTicketBody.setAuthTime(now);
        KerberosTime startTime = request.getFrom();
        if (startTime == null || startTime.lessThan(now) || startTime.isInClockSkew(config.getAllowableClockSkew()) && !request.getOption(6)) {
            startTime = now;
        }
        if (startTime != null && startTime.greaterThan(now) && !startTime.isInClockSkew(config.getAllowableClockSkew()) && !request.getOption(6)) {
            throw new KerberosException(ErrorType.KDC_ERR_CANNOT_POSTDATE);
        }
        if (request.getOption(6)) {
            if (!config.isPostdatedAllowed()) {
                throw new KerberosException(ErrorType.KDC_ERR_POLICY);
            }
            newTicketBody.setFlag((KerberosFlag)TicketFlag.POSTDATED);
            newTicketBody.setFlag((KerberosFlag)TicketFlag.INVALID);
            newTicketBody.setStartTime(startTime);
        }
        long till = 0L;
        till = request.getTill().getTime() == 0L ? Long.MAX_VALUE : request.getTill().getTime();
        long endTime = Math.min(till, startTime.getTime() + config.getMaximumTicketLifetime());
        KerberosTime kerberosEndTime = new KerberosTime(endTime);
        newTicketBody.setEndTime(kerberosEndTime);
        if (kerberosEndTime.lessThan(startTime)) {
            throw new KerberosException(ErrorType.KDC_ERR_NEVER_VALID);
        }
        long ticketLifeTime = Math.abs(startTime.getTime() - kerberosEndTime.getTime());
        if (ticketLifeTime < config.getAllowableClockSkew()) {
            throw new KerberosException(ErrorType.KDC_ERR_NEVER_VALID);
        }
        KerberosTime tempRtime = request.getRtime();
        if (request.getOption(27) && request.getTill().greaterThan(kerberosEndTime)) {
            if (!config.isRenewableAllowed()) {
                throw new KerberosException(ErrorType.KDC_ERR_POLICY);
            }
            request.setOption(8);
            tempRtime = request.getTill();
        }
        if (request.getOption(8)) {
            if (!config.isRenewableAllowed()) {
                throw new KerberosException(ErrorType.KDC_ERR_POLICY);
            }
            newTicketBody.setFlag((KerberosFlag)TicketFlag.RENEWABLE);
            if (tempRtime == null || tempRtime.isZero()) {
                tempRtime = KerberosTime.INFINITY;
            }
            long renewTill = Math.min(tempRtime.getTime(), startTime.getTime() + config.getMaximumRenewableLifetime());
            newTicketBody.setRenewTill(new KerberosTime(renewTill));
        }
        if (request.getAddresses() != null && request.getAddresses().getAddresses() != null && request.getAddresses().getAddresses().length > 0) {
            newTicketBody.setClientAddresses(request.getAddresses());
        } else if (!config.isEmptyAddressesAllowed()) {
            throw new KerberosException(ErrorType.KDC_ERR_POLICY);
        }
        EncTicketPart ticketPart = newTicketBody.getEncTicketPart();
        EncryptedData encryptedData = cipherTextHandler.seal(serverKey, (Encodable)ticketPart, KeyUsage.NUMBER2);
        Ticket newTicket = new Ticket(ticketPrincipal, encryptedData);
        newTicket.setEncTicketPart(ticketPart);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Ticket will be issued for access to {}.", (Object)serverPrincipal.toString());
        }
        authContext.setTicket(newTicket);
    }

    private static void buildReply(AuthenticationContext authContext) throws KerberosException, InvalidTicketException {
        KdcRequest request = authContext.getRequest();
        Ticket ticket = authContext.getTicket();
        AuthenticationReply reply = new AuthenticationReply();
        reply.setClientPrincipal(request.getClientPrincipal());
        reply.setTicket(ticket);
        reply.setKey(ticket.getEncTicketPart().getSessionKey());
        reply.setLastRequest(new LastRequest());
        reply.setNonce(request.getNonce());
        reply.setFlags(ticket.getEncTicketPart().getFlags());
        reply.setAuthTime(ticket.getEncTicketPart().getAuthTime());
        reply.setStartTime(ticket.getEncTicketPart().getStartTime());
        reply.setEndTime(ticket.getEncTicketPart().getEndTime());
        if (ticket.getEncTicketPart().getFlags().isRenewable()) {
            reply.setRenewTill(ticket.getEncTicketPart().getRenewTill());
        }
        reply.setServerPrincipal(ticket.getServerPrincipal());
        reply.setClientAddresses(ticket.getEncTicketPart().getClientAddresses());
        authContext.setReply((KerberosMessage)reply);
    }

    private static void sealReply(AuthenticationContext authContext) throws KerberosException, InvalidTicketException {
        AuthenticationReply reply = (AuthenticationReply)authContext.getReply();
        EncryptionKey clientKey = authContext.getClientKey();
        CipherTextHandler cipherTextHandler = authContext.getCipherTextHandler();
        EncryptedData encryptedData = cipherTextHandler.seal(clientKey, (Encodable)reply, KeyUsage.NUMBER3);
        reply.setEncPart(encryptedData);
    }

    private static void monitorRequest(KdcContext kdcContext) {
        KdcRequest request = kdcContext.getRequest();
        if (LOG.isDebugEnabled()) {
            try {
                String clientAddress = kdcContext.getClientAddress().getHostAddress();
                StringBuffer sb = new StringBuffer();
                sb.append("Received Authentication Service (AS) request:");
                sb.append("\n\tmessageType:           " + request.getMessageType());
                sb.append("\n\tprotocolVersionNumber: " + request.getProtocolVersionNumber());
                sb.append("\n\tclientAddress:         " + clientAddress);
                sb.append("\n\tnonce:                 " + request.getNonce());
                sb.append("\n\tkdcOptions:            " + request.getKdcOptions());
                sb.append("\n\tclientPrincipal:       " + request.getClientPrincipal());
                sb.append("\n\tserverPrincipal:       " + request.getServerPrincipal());
                sb.append("\n\tencryptionType:        " + KerberosUtils.getEncryptionTypesString((Set)request.getEType()));
                sb.append("\n\trealm:                 " + request.getRealm());
                sb.append("\n\tfrom time:             " + request.getFrom());
                sb.append("\n\ttill time:             " + request.getTill());
                sb.append("\n\trenew-till time:       " + request.getRtime());
                sb.append("\n\thostAddresses:         " + request.getAddresses());
                LOG.debug(sb.toString());
            }
            catch (Exception e) {
                LOG.error("Error in request monitor", (Throwable)e);
            }
        }
    }

    private static void monitorContext(AuthenticationContext authContext) {
        try {
            long clockSkew = authContext.getConfig().getAllowableClockSkew();
            InetAddress clientAddress = authContext.getClientAddress();
            StringBuilder sb = new StringBuilder();
            sb.append("Monitoring Authentication Service (AS) context:");
            sb.append("\n\tclockSkew              " + clockSkew);
            sb.append("\n\tclientAddress          " + clientAddress);
            KerberosPrincipal clientPrincipal = authContext.getClientEntry().getPrincipal();
            PrincipalStoreEntry clientEntry = authContext.getClientEntry();
            sb.append("\n\tprincipal              " + clientPrincipal);
            sb.append("\n\tcn                     " + clientEntry.getCommonName());
            sb.append("\n\trealm                  " + clientEntry.getRealmName());
            sb.append("\n\tprincipal              " + clientEntry.getPrincipal());
            sb.append("\n\tSAM type               " + clientEntry.getSamType());
            KerberosPrincipal serverPrincipal = authContext.getRequest().getServerPrincipal();
            PrincipalStoreEntry serverEntry = authContext.getServerEntry();
            sb.append("\n\tprincipal              " + serverPrincipal);
            sb.append("\n\tcn                     " + serverEntry.getCommonName());
            sb.append("\n\trealm                  " + serverEntry.getRealmName());
            sb.append("\n\tprincipal              " + serverEntry.getPrincipal());
            sb.append("\n\tSAM type               " + serverEntry.getSamType());
            EncryptionType encryptionType = authContext.getEncryptionType();
            int clientKeyVersion = ((EncryptionKey)clientEntry.getKeyMap().get(encryptionType)).getKeyVersion();
            int serverKeyVersion = ((EncryptionKey)serverEntry.getKeyMap().get(encryptionType)).getKeyVersion();
            sb.append("\n\tRequest key type       " + encryptionType);
            sb.append("\n\tClient key version     " + clientKeyVersion);
            sb.append("\n\tServer key version     " + serverKeyVersion);
            LOG.debug(sb.toString());
        }
        catch (Exception e) {
            LOG.error("Error in context monitor", (Throwable)e);
        }
    }

    private static void monitorReply(KdcContext kdcContext) {
        KerberosMessage reply = kdcContext.getReply();
        if (LOG.isDebugEnabled() && reply instanceof KdcReply) {
            KdcReply success = (KdcReply)reply;
            try {
                StringBuffer sb = new StringBuffer();
                sb.append("Responding with Authentication Service (AS) reply:");
                sb.append("\n\tmessageType:           " + success.getMessageType());
                sb.append("\n\tprotocolVersionNumber: " + success.getProtocolVersionNumber());
                sb.append("\n\tnonce:                 " + success.getNonce());
                sb.append("\n\tclientPrincipal:       " + success.getClientPrincipal());
                sb.append("\n\tclient realm:          " + success.getClientRealm());
                sb.append("\n\tserverPrincipal:       " + success.getServerPrincipal());
                sb.append("\n\tserver realm:          " + success.getServerRealm());
                sb.append("\n\tauth time:             " + success.getAuthTime());
                sb.append("\n\tstart time:            " + success.getStartTime());
                sb.append("\n\tend time:              " + success.getEndTime());
                sb.append("\n\trenew-till time:       " + success.getRenewTill());
                sb.append("\n\thostAddresses:         " + success.getClientAddresses());
                LOG.debug(sb.toString());
            }
            catch (Exception e) {
                LOG.error("Error in reply monitor", (Throwable)e);
            }
        }
    }

    private static PrincipalStoreEntry getEntry(KerberosPrincipal principal, PrincipalStore store, ErrorType errorType) throws KerberosException {
        PrincipalStoreEntry entry = null;
        try {
            entry = store.getPrincipal(principal);
        }
        catch (Exception e) {
            throw new KerberosException(errorType, (Throwable)e);
        }
        if (entry == null) {
            throw new KerberosException(errorType);
        }
        if (entry.getKeyMap() == null || entry.getKeyMap().isEmpty()) {
            throw new KerberosException(ErrorType.KDC_ERR_NULL_KEY);
        }
        return entry;
    }

    private static byte[] preparePreAuthenticationError(Set<EncryptionType> encryptionTypes) {
        PaData[] paDataSequence = new PaData[2];
        PaData paData = new PaData();
        paData.setPaDataType(PaDataType.PA_ENC_TIMESTAMP);
        paData.setPaDataValue(new byte[0]);
        paDataSequence[0] = paData;
        EncryptionTypeInfoEntry[] entries = new EncryptionTypeInfoEntry[encryptionTypes.size()];
        int i = 0;
        for (EncryptionType encryptionType : encryptionTypes) {
            entries[i++] = new EncryptionTypeInfoEntry(encryptionType, null);
        }
        byte[] encTypeInfo = null;
        try {
            encTypeInfo = EncryptionTypeInfoEncoder.encode((EncryptionTypeInfoEntry[])entries);
        }
        catch (IOException ioe) {
            return null;
        }
        PaData encType = new PaData();
        encType.setPaDataType(PaDataType.PA_ENCTYPE_INFO);
        encType.setPaDataValue(encTypeInfo);
        paDataSequence[1] = encType;
        try {
            return PreAuthenticationDataEncoder.encode((PaData[])paDataSequence);
        }
        catch (IOException ioe) {
            return null;
        }
    }
}

