    /*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *  
 *    http://www.apache.org/licenses/LICENSE-2.0
 *  
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License. 
 *  
 */
package org.apache.directory.server.ldap.handlers;


import org.apache.directory.server.ldap.ExtendedOperationHandler;
import org.apache.directory.server.ldap.handlers.extended.StartTlsHandler;
import org.apache.directory.shared.ldap.message.ExtendedRequest;
import org.apache.directory.shared.ldap.message.ExtendedResponse;
import org.apache.directory.shared.ldap.message.LdapResult;
import org.apache.directory.shared.ldap.message.ResultCodeEnum;
import org.apache.directory.shared.ldap.util.ExceptionUtils;
import org.apache.mina.common.IoSession;


/**
* A single reply handler for {@link ExtendedRequest}s.
 *
 * @author <a href="mailto:dev@directory.apache.org">Apache Directory Project</a>
 * @version $Rev: 669253 $
 */
public class DefaultExtendedHandler extends ExtendedHandler
{
    public void extendedMessageReceived( IoSession session, ExtendedRequest req ) throws Exception
    {
        ExtendedOperationHandler handler = getHandler( req.getOid() );

        if ( handler == null )
        {
            // As long as no extended operations are implemented, send appropriate
            // error back to the client.
            String msg = "Unrecognized extended operation EXTENSION_OID: " + req.getOid();
            LdapResult result = req.getResultResponse().getLdapResult();
            result.setResultCode( ResultCodeEnum.PROTOCOL_ERROR );
            result.setErrorMessage( msg );
            session.write( req.getResultResponse() );
        }
        else
        {
            try
            {
            	if ( ! req.getOid().equals( StartTlsHandler.EXTENSION_OID ) )
            	{
	                // protect against insecure conns when confidentiality is required 
	                if ( ! isConfidentialityRequirementSatisfied( session ) )
	                {
	                	LdapResult result = req.getResultResponse().getLdapResult();
	                	result.setResultCode( ResultCodeEnum.CONFIDENTIALITY_REQUIRED );
	                	result.setErrorMessage( "Confidentiality (TLS secured connection) is required." );
	                	session.write( req.getResultResponse() );
	                	return;
	                }
            	}
                
                handler.handleExtendedOperation( session, getSessionRegistry(), req );
            }
            catch ( Exception e )
            {
                LdapResult result = req.getResultResponse().getLdapResult();
                result.setResultCode( ResultCodeEnum.OTHER );
                result.setErrorMessage( "Extended operation handler for the specified EXTENSION_OID (" + req.getOid()
                    + ") has failed to process your request:\n" + ExceptionUtils.getStackTrace( e ) );
                ExtendedResponse resp = ( ExtendedResponse ) req.getResultResponse();
                resp.setResponse( new byte[0] );
                session.write( req.getResultResponse() );
            }
        }
    }
}