/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 *
 */
package org.apache.directory.shared.ldap.schema.normalizers;


import java.io.IOException;

import javax.naming.NamingException;

import org.apache.directory.shared.i18n.I18n;
import org.apache.directory.shared.ldap.constants.SchemaConstants;
import org.apache.directory.shared.ldap.entry.Value;
import org.apache.directory.shared.ldap.entry.client.ClientStringValue;
import org.apache.directory.shared.ldap.schema.Normalizer;
import org.apache.directory.shared.ldap.schema.PrepareString;


/**
 * Normalize Numeric Strings
 *
 * @author <a href="mailto:dev@directory.apache.org">Apache Directory Project</a>
 */
public class NumericNormalizer extends Normalizer
{
    /** The serial UID */
    public static final long serialVersionUID = 1L;

    /**
     * Creates a new instance of NumericNormalizer.
     */
    public NumericNormalizer()
    {
        super( SchemaConstants.NUMERIC_STRING_MATCH_MR_OID );
    }

   
   /**
    * {@inheritDoc}
    */
   public Value<?> normalize( Value<?> value ) throws NamingException
   {
       try
       {
           String normalized = PrepareString.normalize( value.getString(),
               PrepareString.StringType.NUMERIC_STRING );
           
           return new ClientStringValue( normalized );
       }
       catch ( IOException ioe )
       {
           throw new NamingException( I18n.err( I18n.ERR_04224, value ) );
       }
   }


   /**
    * {@inheritDoc}
    */
   public String normalize( String value ) throws NamingException
   {
       try
       {
           return PrepareString.normalize( value,
               PrepareString.StringType.NUMERIC_STRING );
       }
       catch ( IOException ioe )
       {
           throw new NamingException( I18n.err( I18n.ERR_04224, value ) );
       }
   }
}