/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.druid.segment.data;

import com.google.common.base.Supplier;
import org.apache.druid.java.util.common.StringUtils;

public class EntireLayoutColumnarLongsSupplier implements Supplier<ColumnarLongs>
{

  private final int totalSize;
  private final CompressionFactory.LongEncodingReader reader;

  public EntireLayoutColumnarLongsSupplier(int totalSize, CompressionFactory.LongEncodingReader reader)
  {
    this.totalSize = totalSize;
    this.reader = reader;
  }

  @Override
  public ColumnarLongs get()
  {
    return new EntireLayoutColumnarLongs();
  }

  private class EntireLayoutColumnarLongs implements ColumnarLongs
  {

    @Override
    public int size()
    {
      return totalSize;
    }

    @Override
    public long get(int index)
    {
      return reader.read(index);
    }

    @Override
    public void fill(int index, long[] toFill)
    {
      if (totalSize - index < toFill.length) {
        throw new IndexOutOfBoundsException(
            StringUtils.format(
                "Cannot fill array of size[%,d] at index[%,d].  Max size[%,d]", toFill.length, index, totalSize
            )
        );
      }
      for (int i = 0; i < toFill.length; i++) {
        toFill[i] = get(index + i);
      }
    }

    @Override
    public String toString()
    {
      return "EntireCompressedColumnarLongs_Anonymous{" +
             ", totalSize=" + totalSize +
             '}';
    }

    @Override
    public void close()
    {
    }
  }
}
