/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.math.expr;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.nio.ByteBuffer;
import java.util.Map;
import javax.annotation.Nullable;
import org.apache.druid.common.config.NullHandling;
import org.apache.druid.guice.NestedDataModule;
import org.apache.druid.java.util.common.IAE;
import org.apache.druid.java.util.common.Pair;
import org.apache.druid.java.util.common.StringUtils;
import org.apache.druid.math.expr.Expr;
import org.apache.druid.math.expr.ExprMacroTable;
import org.apache.druid.math.expr.ExpressionType;
import org.apache.druid.math.expr.ExpressionValidationException;
import org.apache.druid.math.expr.InputBindings;
import org.apache.druid.math.expr.Parser;
import org.apache.druid.segment.column.TypeSignature;
import org.apache.druid.segment.column.TypeStrategies;
import org.apache.druid.segment.column.TypeStrategiesTest;
import org.apache.druid.segment.column.TypeStrategy;
import org.apache.druid.segment.nested.StructuredData;
import org.apache.druid.testing.InitializedNullHandlingTest;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

public class FunctionTest
extends InitializedNullHandlingTest {
    private Expr.ObjectBinding bestEffortBindings;
    private Expr.ObjectBinding typedBindings;
    private Expr.ObjectBinding[] allBindings;

    @BeforeClass
    public static void setupClass() {
        TypeStrategies.registerComplex((String)TypeStrategiesTest.NULLABLE_TEST_PAIR_TYPE.getComplexTypeName(), (TypeStrategy)new TypeStrategiesTest.NullableLongPairTypeStrategy());
        NestedDataModule.registerHandlersAndSerde();
    }

    @Before
    public void setup() {
        ImmutableMap.Builder inputTypesBuilder = ImmutableMap.builder();
        inputTypesBuilder.put((Object)"x", (Object)ExpressionType.STRING).put((Object)"y", (Object)ExpressionType.LONG).put((Object)"z", (Object)ExpressionType.DOUBLE).put((Object)"d", (Object)ExpressionType.DOUBLE).put((Object)"maxLong", (Object)ExpressionType.LONG).put((Object)"minLong", (Object)ExpressionType.LONG).put((Object)"f", (Object)ExpressionType.DOUBLE).put((Object)"nan", (Object)ExpressionType.DOUBLE).put((Object)"inf", (Object)ExpressionType.DOUBLE).put((Object)"-inf", (Object)ExpressionType.DOUBLE).put((Object)"o", (Object)ExpressionType.LONG).put((Object)"od", (Object)ExpressionType.DOUBLE).put((Object)"of", (Object)ExpressionType.DOUBLE).put((Object)"a", (Object)ExpressionType.STRING_ARRAY).put((Object)"b", (Object)ExpressionType.LONG_ARRAY).put((Object)"c", (Object)ExpressionType.DOUBLE_ARRAY).put((Object)"someComplex", (Object)ExpressionType.fromColumnType((TypeSignature)TypeStrategiesTest.NULLABLE_TEST_PAIR_TYPE)).put((Object)"str1", (Object)ExpressionType.STRING).put((Object)"str2", (Object)ExpressionType.STRING).put((Object)"nestedArray", (Object)ExpressionType.NESTED_DATA);
        StructuredData nestedArray = StructuredData.wrap((Object)ImmutableList.of((Object)ImmutableMap.of((Object)"x", (Object)2L, (Object)"y", (Object)3.3), (Object)ImmutableMap.of((Object)"x", (Object)4L, (Object)"y", (Object)6.6)));
        ImmutableMap.Builder builder = ImmutableMap.builder();
        builder.put((Object)"x", (Object)"foo").put((Object)"y", (Object)2).put((Object)"z", (Object)3.1).put((Object)"d", (Object)34.56).put((Object)"maxLong", (Object)Long.MAX_VALUE).put((Object)"minLong", (Object)Long.MIN_VALUE).put((Object)"f", (Object)Float.valueOf(12.34f)).put((Object)"nan", (Object)Double.NaN).put((Object)"inf", (Object)Double.POSITIVE_INFINITY).put((Object)"-inf", (Object)Double.NEGATIVE_INFINITY).put((Object)"o", (Object)0).put((Object)"od", (Object)0.0).put((Object)"of", (Object)Float.valueOf(0.0f)).put((Object)"a", (Object)new String[]{"foo", "bar", "baz", "foobar"}).put((Object)"b", (Object)new Long[]{1L, 2L, 3L, 4L, 5L}).put((Object)"c", (Object)new Double[]{3.1, 4.2, 5.3}).put((Object)"someComplex", (Object)new TypeStrategiesTest.NullableLongPair(1L, 2L)).put((Object)"str1", (Object)"v1").put((Object)"str2", (Object)"v2").put((Object)"nestedArray", (Object)nestedArray);
        this.bestEffortBindings = InputBindings.forMap((Map)builder.build());
        this.typedBindings = InputBindings.forMap((Map)builder.build(), (Expr.InputBindingInspector)InputBindings.inspectorFromTypeMap((Map)inputTypesBuilder.build()));
        this.allBindings = new Expr.ObjectBinding[]{this.bestEffortBindings, this.typedBindings};
    }

    @Test
    public void testCaseSimple() {
        this.assertExpr("case_simple(x,'baz','is baz','foo','is foo','is other')", "is foo");
        this.assertExpr("case_simple(x,'baz','is baz','bar','is bar','is other')", "is other");
        this.assertExpr("case_simple(y,2,'is 2',3,'is 3','is other')", "is 2");
        this.assertExpr("case_simple(z,2,'is 2',3,'is 3','is other')", "is other");
    }

    @Test
    public void testCaseSearched() {
        this.assertExpr("case_searched(x=='baz','is baz',x=='foo','is foo','is other')", "is foo");
        this.assertExpr("case_searched(x=='baz','is baz',x=='bar','is bar','is other')", "is other");
        this.assertExpr("case_searched(y==2,'is 2',y==3,'is 3','is other')", "is 2");
        this.assertExpr("case_searched(z==2,'is 2',z==3,'is 3','is other')", "is other");
    }

    @Test
    public void testConcat() {
        this.assertExpr("concat(x,' ',y)", "foo 2");
        if (NullHandling.replaceWithDefault()) {
            this.assertExpr("concat(x,' ',nonexistent,' ',y)", "foo  2");
        } else {
            this.assertArrayExpr("concat(x,' ',nonexistent,' ',y)", null);
        }
        this.assertExpr("concat(z)", "3.1");
        this.assertArrayExpr("concat()", null);
    }

    @Test
    public void testReplace() {
        this.assertExpr("replace(x,'oo','ab')", "fab");
        this.assertExpr("replace(x,x,'ab')", "ab");
        this.assertExpr("replace(x,'oo',y)", "f2");
    }

    @Test
    public void testSubstring() {
        this.assertExpr("substring(x,0,2)", "fo");
        this.assertExpr("substring(x,1,2)", "oo");
        this.assertExpr("substring(x,y,1)", "o");
        this.assertExpr("substring(x,0,-1)", "foo");
        this.assertExpr("substring(x,0,100)", "foo");
    }

    @Test
    public void testStrlen() {
        this.assertExpr("strlen(x)", 3L);
        this.assertExpr("strlen(nonexistent)", null);
    }

    @Test
    public void testStrpos() {
        this.assertExpr("strpos(x, 'o')", 1L);
        this.assertExpr("strpos(x, 'o', 0)", 1L);
        this.assertExpr("strpos(x, 'o', 1)", 1L);
        this.assertExpr("strpos(x, 'o', 2)", 2L);
        this.assertExpr("strpos(x, 'o', 3)", -1L);
        this.assertExpr("strpos(x, '')", 0L);
        this.assertExpr("strpos(x, 'x')", -1L);
    }

    @Test
    public void testLower() {
        this.assertExpr("lower('FOO')", "foo");
    }

    @Test
    public void testUpper() {
        this.assertExpr("upper(x)", "FOO");
    }

    @Test
    public void testIsNull() {
        this.assertExpr("isnull(null)", 1L);
        this.assertExpr("isnull('abc')", 0L);
    }

    @Test
    public void testIsNotNull() {
        this.assertExpr("notnull(null)", 0L);
        this.assertExpr("notnull('abc')", 1L);
    }

    @Test
    public void testLpad() {
        this.assertExpr("lpad(x, 5, 'ab')", "abfoo");
        this.assertExpr("lpad(x, 4, 'ab')", "afoo");
        this.assertExpr("lpad(x, 2, 'ab')", "fo");
        this.assertExpr("lpad(x, -1, 'ab')", NullHandling.replaceWithDefault() ? null : "");
        this.assertExpr("lpad(null, 5, 'ab')", null);
        this.assertExpr("lpad(x, 2, '')", NullHandling.replaceWithDefault() ? null : "fo");
        this.assertExpr("lpad(x, 6, '')", NullHandling.replaceWithDefault() ? null : "foo");
        this.assertExpr("lpad('', 3, '*')", NullHandling.replaceWithDefault() ? null : "***");
        this.assertExpr("lpad(x, 2, null)", null);
        this.assertExpr("lpad(a, 4, '*')", "[foo");
        this.assertExpr("lpad(a, 2, '*')", "[f");
        this.assertExpr("lpad(a, 2, '')", NullHandling.replaceWithDefault() ? null : "[f");
        this.assertExpr("lpad(b, 4, '*')", "[1, ");
        this.assertExpr("lpad(b, 2, '')", NullHandling.replaceWithDefault() ? null : "[1");
        this.assertExpr("lpad(b, 2, null)", null);
        this.assertExpr("lpad(x, 5, x)", "fofoo");
        this.assertExpr("lpad(x, 5, y)", "22foo");
        this.assertExpr("lpad(x, 5, z)", "3.foo");
        this.assertExpr("lpad(y, 5, x)", "foof2");
        this.assertExpr("lpad(z, 5, y)", "223.1");
    }

    @Test
    public void testRpad() {
        this.assertExpr("rpad(x, 5, 'ab')", "fooab");
        this.assertExpr("rpad(x, 4, 'ab')", "fooa");
        this.assertExpr("rpad(x, 2, 'ab')", "fo");
        this.assertExpr("rpad(x, -1, 'ab')", NullHandling.replaceWithDefault() ? null : "");
        this.assertExpr("rpad(null, 5, 'ab')", null);
        this.assertExpr("rpad(x, 2, '')", NullHandling.replaceWithDefault() ? null : "fo");
        this.assertExpr("rpad(x, 6, '')", NullHandling.replaceWithDefault() ? null : "foo");
        this.assertExpr("rpad('', 3, '*')", NullHandling.replaceWithDefault() ? null : "***");
        this.assertExpr("rpad(x, 2, null)", null);
        this.assertExpr("rpad(a, 2, '*')", "[f");
        this.assertExpr("rpad(a, 2, '')", NullHandling.replaceWithDefault() ? null : "[f");
        this.assertExpr("rpad(b, 4, '*')", "[1, ");
        this.assertExpr("rpad(b, 2, '')", NullHandling.replaceWithDefault() ? null : "[1");
        this.assertExpr("rpad(b, 2, null)", null);
        this.assertExpr("rpad(x, 5, x)", "foofo");
        this.assertExpr("rpad(x, 5, y)", "foo22");
        this.assertExpr("rpad(x, 5, z)", "foo3.");
        this.assertExpr("rpad(y, 5, x)", "2foof");
        this.assertExpr("rpad(z, 5, y)", "3.122");
    }

    @Test
    public void testArrayConstructor() {
        this.assertArrayExpr("array(1, 2, 3, 4)", new Long[]{1L, 2L, 3L, 4L});
        this.assertArrayExpr("array(1, 2, 3, 'bar')", new Long[]{1L, 2L, 3L, null});
        this.assertArrayExpr("array(1.0)", new Double[]{1.0});
        this.assertArrayExpr("array('foo', 'bar')", new String[]{"foo", "bar"});
    }

    @Test
    public void testArrayLength() {
        this.assertExpr("array_length([1,2,3])", 3L);
        this.assertExpr("array_length(a)", 4L);
        this.assertExpr("array_length(nestedArray)", 2L, this.typedBindings);
    }

    @Test
    public void testArrayOffset() {
        this.assertExpr("array_offset([1, 2, 3], 2)", 3L);
        this.assertArrayExpr("array_offset([1, 2, 3], 3)", null);
        this.assertExpr("array_offset(a, 2)", "baz");
        this.assertExpr("array_offset(nestedArray, 1)", ImmutableMap.of((Object)"x", (Object)4L, (Object)"y", (Object)6.6), this.typedBindings);
    }

    @Test
    public void testArrayOrdinal() {
        this.assertExpr("array_ordinal([1, 2, 3], 3)", 3L);
        this.assertArrayExpr("array_ordinal([1, 2, 3], 4)", null);
        this.assertExpr("array_ordinal(a, 3)", "baz");
        this.assertExpr("array_ordinal(nestedArray, 2)", ImmutableMap.of((Object)"x", (Object)4L, (Object)"y", (Object)6.6), this.typedBindings);
    }

    @Test
    public void testArrayOffsetOf() {
        this.assertExpr("array_offset_of([1, 2, 3], 3)", 2L);
        this.assertExpr("array_offset_of([1, 2, 3], 4)", NullHandling.replaceWithDefault() ? Long.valueOf(-1L) : null);
        this.assertExpr("array_offset_of(a, 'baz')", 2L);
    }

    @Test
    public void testArrayOrdinalOf() {
        this.assertExpr("array_ordinal_of([1, 2, 3], 3)", 3L);
        this.assertExpr("array_ordinal_of([1, 2, 3], 4)", NullHandling.replaceWithDefault() ? Long.valueOf(-1L) : null);
        this.assertExpr("array_ordinal_of(a, 'baz')", 3L);
    }

    @Test
    public void testArrayContains() {
        this.assertExpr("array_contains([1, 2, 3], 2)", 1L);
        this.assertExpr("array_contains([1, 2, 3], 4)", 0L);
        this.assertExpr("array_contains([1, 2, 3], [2, 3])", 1L);
        this.assertExpr("array_contains([1, 2, 3], [3, 4])", 0L);
        this.assertExpr("array_contains(b, [3, 4])", 1L);
    }

    @Test
    public void testArrayOverlap() {
        this.assertExpr("array_overlap([1, 2, 3], [2, 4, 6])", 1L);
        this.assertExpr("array_overlap([1, 2, 3], [4, 5, 6])", 0L);
    }

    @Test
    public void testArrayAppend() {
        this.assertArrayExpr("array_append([1, 2, 3], 4)", new Long[]{1L, 2L, 3L, 4L});
        this.assertArrayExpr("array_append([1, 2, 3], 'bar')", new Long[]{1L, 2L, 3L, null});
        this.assertArrayExpr("array_append([], 1)", new String[]{"1"});
        this.assertArrayExpr("array_append(<LONG>[], 1)", new Long[]{1L});
    }

    @Test
    public void testArrayConcat() {
        this.assertArrayExpr("array_concat([1, 2, 3], [2, 4, 6])", new Long[]{1L, 2L, 3L, 2L, 4L, 6L});
        this.assertArrayExpr("array_concat([1, 2, 3], 4)", new Long[]{1L, 2L, 3L, 4L});
        this.assertArrayExpr("array_concat(0, [1, 2, 3])", new Long[]{0L, 1L, 2L, 3L});
        this.assertArrayExpr("array_concat(map(y -> y * 3, b), [1, 2, 3])", new Long[]{3L, 6L, 9L, 12L, 15L, 1L, 2L, 3L});
        this.assertArrayExpr("array_concat(0, 1)", new Long[]{0L, 1L});
    }

    @Test
    public void testArraySetAdd() {
        this.assertArrayExpr("array_set_add([1, 2, 3], 4)", new Long[]{1L, 2L, 3L, 4L});
        this.assertArrayExpr("array_set_add([1, 2, 3], 'bar')", new Long[]{null, 1L, 2L, 3L});
        this.assertArrayExpr("array_set_add([1, 2, 2], 1)", new Long[]{1L, 2L});
        this.assertArrayExpr("array_set_add([], 1)", new String[]{"1"});
        this.assertArrayExpr("array_set_add(<LONG>[], 1)", new Long[]{1L});
        this.assertArrayExpr("array_set_add(<LONG>[], null)", new Long[]{null});
    }

    @Test
    public void testArraySetAddAll() {
        this.assertArrayExpr("array_set_add_all([1, 2, 3], [2, 4, 6])", new Long[]{1L, 2L, 3L, 4L, 6L});
        this.assertArrayExpr("array_set_add_all([1, 2, 3], 4)", new Long[]{1L, 2L, 3L, 4L});
        this.assertArrayExpr("array_set_add_all(0, [1, 2, 3])", new Long[]{0L, 1L, 2L, 3L});
        this.assertArrayExpr("array_set_add_all(map(y -> y * 3, b), [1, 2, 3])", new Long[]{1L, 2L, 3L, 6L, 9L, 12L, 15L});
        this.assertArrayExpr("array_set_add_all(0, 1)", new Long[]{0L, 1L});
    }

    @Test
    public void testArrayToString() {
        this.assertExpr("array_to_string([1, 2, 3], ',')", "1,2,3");
        this.assertExpr("array_to_string([1], '|')", "1");
        this.assertExpr("array_to_string(a, '|')", "foo|bar|baz|foobar");
    }

    @Test
    public void testStringToArray() {
        this.assertArrayExpr("string_to_array('1,2,3', ',')", new String[]{"1", "2", "3"});
        this.assertArrayExpr("string_to_array(null, ',')", null);
        this.assertArrayExpr("string_to_array('1', ',')", new String[]{"1"});
        this.assertArrayExpr("string_to_array(array_to_string(a, ','), ',')", new String[]{"foo", "bar", "baz", "foobar"});
    }

    @Test
    public void testArrayCastLegacy() {
        this.assertArrayExpr("cast([1, 2, 3], 'STRING_ARRAY')", new String[]{"1", "2", "3"});
        this.assertArrayExpr("cast([1, 2, 3], 'DOUBLE_ARRAY')", new Double[]{1.0, 2.0, 3.0});
        this.assertArrayExpr("cast(c, 'LONG_ARRAY')", new Long[]{3L, 4L, 5L});
        this.assertArrayExpr("cast(string_to_array(array_to_string(b, ','), ','), 'LONG_ARRAY')", new Long[]{1L, 2L, 3L, 4L, 5L});
        this.assertArrayExpr("cast(['1.0', '2.0', '3.0'], 'LONG_ARRAY')", new Long[]{1L, 2L, 3L});
    }

    @Test
    public void testArrayCast() {
        this.assertArrayExpr("cast([1, 2, 3], 'ARRAY<STRING>')", new String[]{"1", "2", "3"});
        this.assertArrayExpr("cast([1, 2, 3], 'ARRAY<DOUBLE>')", new Double[]{1.0, 2.0, 3.0});
        this.assertArrayExpr("cast(c, 'ARRAY<LONG>')", new Long[]{3L, 4L, 5L});
        this.assertArrayExpr("cast(string_to_array(array_to_string(b, ','), ','), 'ARRAY<LONG>')", new Long[]{1L, 2L, 3L, 4L, 5L});
        this.assertArrayExpr("cast(['1.0', '2.0', '3.0'], 'ARRAY<LONG>')", new Long[]{1L, 2L, 3L});
    }

    @Test
    public void testArraySlice() {
        this.assertArrayExpr("array_slice([1, 2, 3, 4], 1, 3)", new Long[]{2L, 3L});
        this.assertArrayExpr("array_slice([1.0, 2.1, 3.2, 4.3], 2)", new Double[]{3.2, 4.3});
        this.assertArrayExpr("array_slice(['a', 'b', 'c', 'd'], 4, 6)", new String[]{null, null});
        this.assertArrayExpr("array_slice([1, 2, 3, 4], 2, 2)", new Long[0]);
        this.assertArrayExpr("array_slice([1, 2, 3, 4], 5, 7)", null);
        this.assertArrayExpr("array_slice([1, 2, 3, 4], 2, 1)", null);
    }

    @Test
    public void testArrayPrepend() {
        this.assertArrayExpr("array_prepend(4, [1, 2, 3])", new Long[]{4L, 1L, 2L, 3L});
        this.assertArrayExpr("array_prepend('bar', [1, 2, 3])", new Long[]{null, 1L, 2L, 3L});
        this.assertArrayExpr("array_prepend(1, [])", new String[]{"1"});
        this.assertArrayExpr("array_prepend(1, <LONG>[])", new Long[]{1L});
        this.assertArrayExpr("array_prepend(1, <DOUBLE>[])", new Double[]{1.0});
    }

    @Test
    public void testRoundWithNonNumericValuesShouldReturn0() {
        this.assertExpr("round(nan)", 0.0);
        this.assertExpr("round(nan, 5)", 0.0);
        this.assertExpr("round(inf)", Double.MAX_VALUE);
        this.assertExpr("round(inf, 4)", Double.MAX_VALUE);
        this.assertExpr("round(-inf)", -1.7976931348623157E308);
        this.assertExpr("round(-inf, 3)", -1.7976931348623157E308);
        this.assertExpr("round(-inf, -5)", -1.7976931348623157E308);
        this.assertExpr("round(0/od)", 0.0);
        this.assertExpr("round(od/od)", 0.0);
        this.assertExpr("round(1/od)", Double.MAX_VALUE);
        this.assertExpr("round(-1/od)", -1.7976931348623157E308);
        this.assertExpr("round(0/of)", 0.0);
        this.assertExpr("round(of/of)", 0.0);
        this.assertExpr("round(1/of)", Double.MAX_VALUE);
        this.assertExpr("round(-1/of)", -1.7976931348623157E308);
    }

    @Test
    public void testRoundWithLong() {
        this.assertExpr("round(y)", 2L);
        this.assertExpr("round(y, 2)", 2L);
        this.assertExpr("round(y, -1)", 0L);
    }

    @Test
    public void testRoundWithDouble() {
        this.assertExpr("round(d)", 35.0);
        this.assertExpr("round(d, 2)", 34.56);
        this.assertExpr("round(d, y)", 34.56);
        this.assertExpr("round(d, 1)", 34.6);
        this.assertExpr("round(d, -1)", 30.0);
    }

    @Test
    public void testRoundWithFloat() {
        this.assertExpr("round(f)", 12.0);
        this.assertExpr("round(f, 2)", 12.34);
        this.assertExpr("round(f, y)", 12.34);
        this.assertExpr("round(f, 1)", 12.3);
        this.assertExpr("round(f, -1)", 10.0);
    }

    @Test
    public void testRoundWithExtremeNumbers() {
        this.assertExpr("round(maxLong)", BigDecimal.valueOf(Long.MAX_VALUE).setScale(0, RoundingMode.HALF_UP).longValue());
        this.assertExpr("round(minLong)", BigDecimal.valueOf(Long.MIN_VALUE).setScale(0, RoundingMode.HALF_UP).longValue());
        this.assertExpr("round(maxLong + 1, 1)", BigDecimal.valueOf(Long.MIN_VALUE).setScale(1, RoundingMode.HALF_UP).longValue());
        this.assertExpr("round(minLong - 1, -2)", BigDecimal.valueOf(Long.MAX_VALUE).setScale(-2, RoundingMode.HALF_UP).longValue());
        this.assertExpr("round(CAST(maxLong, 'DOUBLE') + 1, 1)", BigDecimal.valueOf(9.223372036854776E18).setScale(1, RoundingMode.HALF_UP).doubleValue());
        this.assertExpr("round(CAST(minLong, 'DOUBLE') - 1, -2)", BigDecimal.valueOf(-9.223372036854776E18).setScale(-2, RoundingMode.HALF_UP).doubleValue());
    }

    @Test
    public void testRoundWithNullValueOrInvalid() {
        ImmutableSet invalidArguments = ImmutableSet.of((Object)Pair.of((Object)"null", (Object)"STRING"), (Object)Pair.of((Object)"x", (Object)"STRING"), (Object)Pair.of((Object)"b", (Object)"ARRAY<LONG>"), (Object)Pair.of((Object)"c", (Object)"ARRAY<DOUBLE>"), (Object)Pair.of((Object)"a", (Object)"ARRAY<STRING>"));
        for (Pair argAndType : invalidArguments) {
            if (NullHandling.sqlCompatible()) {
                this.assertExpr(StringUtils.format((String)"round(%s)", (Object[])new Object[]{argAndType.lhs}), null);
                continue;
            }
            try {
                this.assertExpr(StringUtils.format((String)"round(%s)", (Object[])new Object[]{argAndType.lhs}), null);
                Assert.fail((String)"Did not throw IllegalArgumentException");
            }
            catch (ExpressionValidationException e) {
                Assert.assertEquals((Object)StringUtils.format((String)"Function[round] first argument should be a LONG or DOUBLE but got %s instead", (Object[])new Object[]{argAndType.rhs}), (Object)e.getMessage());
            }
        }
    }

    @Test
    public void testRoundWithInvalidSecondArgument() {
        ImmutableSet invalidArguments = ImmutableSet.of((Object)Pair.of((Object)"1.2", (Object)"DOUBLE"), (Object)Pair.of((Object)"x", (Object)"STRING"), (Object)Pair.of((Object)"a", (Object)"ARRAY<STRING>"), (Object)Pair.of((Object)"c", (Object)"ARRAY<DOUBLE>"));
        for (Pair argAndType : invalidArguments) {
            try {
                this.assertExpr(StringUtils.format((String)"round(d, %s)", (Object[])new Object[]{argAndType.lhs}), null);
                Assert.fail((String)"Did not throw IllegalArgumentException");
            }
            catch (ExpressionValidationException e) {
                Assert.assertEquals((Object)StringUtils.format((String)"Function[round] second argument should be a LONG but got %s instead", (Object[])new Object[]{argAndType.rhs}), (Object)e.getMessage());
            }
        }
    }

    @Test
    public void testGreatest() {
        this.assertExpr("greatest(y, 0)", 2L);
        this.assertExpr("greatest(34.0, z, 5.0, 767.0)", 767.0);
        this.assertExpr("greatest('B', x, 'A')", "foo");
        this.assertExpr("greatest(-1, z, 'A')", "A");
        this.assertExpr("greatest(-1, z)", 3.1);
        this.assertExpr("greatest(1, 'A')", "A");
        try {
            this.assertExpr("greatest(1, ['A'])", null);
            Assert.fail((String)"Did not throw IllegalArgumentException");
        }
        catch (ExpressionValidationException e) {
            Assert.assertEquals((Object)"Function[greatest] does not accept ARRAY<STRING> types", (Object)e.getMessage());
        }
        this.assertExpr("greatest()", null);
        this.assertExpr("greatest(null, null)", null);
        this.assertExpr("greatest(1, null, 'A')", "A");
    }

    @Test
    public void testLeast() {
        this.assertExpr("least(y, 0)", 0L);
        this.assertExpr("least(34.0, z, 5.0, 767.0)", 3.1);
        this.assertExpr("least('B', x, 'A')", "A");
        this.assertExpr("least(-1, z, 'A')", "-1");
        this.assertExpr("least(-1, z)", -1.0);
        this.assertExpr("least(1, 'A')", "1");
        try {
            this.assertExpr("least(1, [2, 3])", null);
            Assert.fail((String)"Did not throw IllegalArgumentException");
        }
        catch (ExpressionValidationException e) {
            Assert.assertEquals((Object)"Function[least] does not accept ARRAY<LONG> types", (Object)e.getMessage());
        }
        this.assertExpr("least()", null);
        this.assertExpr("least(null, null)", null);
        this.assertExpr("least(1, null, 'A')", "1");
    }

    @Test
    public void testSizeFormat() {
        this.assertExpr("human_readable_binary_byte_format(-1024)", "-1.00 KiB");
        this.assertExpr("human_readable_binary_byte_format(1024)", "1.00 KiB");
        this.assertExpr("human_readable_binary_byte_format(1024*1024)", "1.00 MiB");
        this.assertExpr("human_readable_binary_byte_format(1024*1024*1024)", "1.00 GiB");
        this.assertExpr("human_readable_binary_byte_format(1024*1024*1024*1024)", "1.00 TiB");
        this.assertExpr("human_readable_binary_byte_format(1024*1024*1024*1024*1024)", "1.00 PiB");
        this.assertExpr("human_readable_decimal_byte_format(-1000)", "-1.00 KB");
        this.assertExpr("human_readable_decimal_byte_format(1000)", "1.00 KB");
        this.assertExpr("human_readable_decimal_byte_format(1000*1000)", "1.00 MB");
        this.assertExpr("human_readable_decimal_byte_format(1000*1000*1000)", "1.00 GB");
        this.assertExpr("human_readable_decimal_byte_format(1000*1000*1000*1000)", "1.00 TB");
        this.assertExpr("human_readable_decimal_format(-1000)", "-1.00 K");
        this.assertExpr("human_readable_decimal_format(1000)", "1.00 K");
        this.assertExpr("human_readable_decimal_format(1000*1000)", "1.00 M");
        this.assertExpr("human_readable_decimal_format(1000*1000*1000)", "1.00 G");
        this.assertExpr("human_readable_decimal_format(1000*1000*1000*1000)", "1.00 T");
    }

    @Test
    public void testSizeFormatWithDifferentPrecision() {
        this.assertExpr("human_readable_binary_byte_format(1024, 0)", "1 KiB");
        this.assertExpr("human_readable_binary_byte_format(1024*1024, 1)", "1.0 MiB");
        this.assertExpr("human_readable_binary_byte_format(1024*1024*1024, 2)", "1.00 GiB");
        this.assertExpr("human_readable_binary_byte_format(1024*1024*1024*1024, 3)", "1.000 TiB");
        this.assertExpr("human_readable_decimal_byte_format(1234, 0)", "1 KB");
        this.assertExpr("human_readable_decimal_byte_format(1234*1000, 1)", "1.2 MB");
        this.assertExpr("human_readable_decimal_byte_format(1234*1000*1000, 2)", "1.23 GB");
        this.assertExpr("human_readable_decimal_byte_format(1234*1000*1000*1000, 3)", "1.234 TB");
        this.assertExpr("human_readable_decimal_format(1234, 0)", "1 K");
        this.assertExpr("human_readable_decimal_format(1234*1000,1)", "1.2 M");
        this.assertExpr("human_readable_decimal_format(1234*1000*1000,2)", "1.23 G");
        this.assertExpr("human_readable_decimal_format(1234*1000*1000*1000,3)", "1.234 T");
    }

    @Test
    public void testSizeFormatWithEdgeCases() {
        this.assertExpr("human_readable_binary_byte_format(nonexist)", NullHandling.sqlCompatible() ? null : "0 B");
        this.assertExpr("human_readable_binary_byte_format(f)", "12 B");
        this.assertExpr("human_readable_binary_byte_format(nan)", "0 B");
        this.assertExpr("human_readable_binary_byte_format(inf)", "8.00 EiB");
        this.assertExpr("human_readable_binary_byte_format(-inf)", "-8.00 EiB");
        this.assertExpr("human_readable_binary_byte_format(o)", "0 B");
        this.assertExpr("human_readable_binary_byte_format(od)", "0 B");
        this.assertExpr("human_readable_binary_byte_format(of)", "0 B");
    }

    @Test
    public void testSizeForatInvalidArgumentType() {
        try {
            Parser.parse((String)"human_readable_binary_byte_format(x)", (ExprMacroTable)ExprMacroTable.nil()).eval(this.bestEffortBindings);
            Assert.assertTrue((NullHandling.sqlCompatible() ? 1 : 0) != 0);
        }
        catch (ExpressionValidationException e) {
            Assert.assertEquals((Object)"Function[human_readable_binary_byte_format] needs a number as its first argument but got STRING instead", (Object)e.getMessage());
        }
        try {
            Parser.parse((String)"human_readable_binary_byte_format(1024, x)", (ExprMacroTable)ExprMacroTable.nil()).eval(this.bestEffortBindings);
            Assert.assertTrue((boolean)false);
        }
        catch (ExpressionValidationException e) {
            Assert.assertEquals((Object)"Function[human_readable_binary_byte_format] needs a LONG as its second argument but got STRING instead", (Object)e.getMessage());
        }
        try {
            Parser.parse((String)"human_readable_binary_byte_format(1024, of)", (ExprMacroTable)ExprMacroTable.nil()).eval(this.bestEffortBindings);
            Assert.assertTrue((boolean)false);
        }
        catch (ExpressionValidationException e) {
            Assert.assertEquals((Object)"Function[human_readable_binary_byte_format] needs a LONG as its second argument but got DOUBLE instead", (Object)e.getMessage());
        }
        try {
            Parser.parse((String)"human_readable_binary_byte_format(1024, nonexist)", (ExprMacroTable)ExprMacroTable.nil()).eval(this.bestEffortBindings);
            Assert.assertTrue((boolean)false);
        }
        catch (ExpressionValidationException e) {
            Assert.assertEquals((Object)"Function[human_readable_binary_byte_format] needs a LONG as its second argument but got STRING instead", (Object)e.getMessage());
        }
    }

    @Test
    public void testSizeFormatInvalidPrecision() {
        try {
            Parser.parse((String)"human_readable_binary_byte_format(1024, maxLong)", (ExprMacroTable)ExprMacroTable.nil()).eval(this.bestEffortBindings);
            Assert.assertTrue((boolean)false);
        }
        catch (ExpressionValidationException e) {
            Assert.assertEquals((Object)"Function[human_readable_binary_byte_format] given precision[9223372036854775807] must be in the range of [0,3]", (Object)e.getMessage());
        }
        try {
            Parser.parse((String)"human_readable_binary_byte_format(1024, minLong)", (ExprMacroTable)ExprMacroTable.nil()).eval(this.bestEffortBindings);
            Assert.assertTrue((boolean)false);
        }
        catch (ExpressionValidationException e) {
            Assert.assertEquals((Object)"Function[human_readable_binary_byte_format] given precision[-9223372036854775808] must be in the range of [0,3]", (Object)e.getMessage());
        }
        try {
            Parser.parse((String)"human_readable_binary_byte_format(1024, -1)", (ExprMacroTable)ExprMacroTable.nil()).eval(this.bestEffortBindings);
            Assert.assertTrue((boolean)false);
        }
        catch (ExpressionValidationException e) {
            Assert.assertEquals((Object)"Function[human_readable_binary_byte_format] given precision[-1] must be in the range of [0,3]", (Object)e.getMessage());
        }
        try {
            Parser.parse((String)"human_readable_binary_byte_format(1024, 4)", (ExprMacroTable)ExprMacroTable.nil()).eval(this.bestEffortBindings);
            Assert.assertTrue((boolean)false);
        }
        catch (ExpressionValidationException e) {
            Assert.assertEquals((Object)"Function[human_readable_binary_byte_format] given precision[4] must be in the range of [0,3]", (Object)e.getMessage());
        }
    }

    @Test
    public void testSizeFormatInvalidArgumentSize() {
        Throwable t = Assert.assertThrows(ExpressionValidationException.class, () -> Parser.parse((String)"human_readable_binary_byte_format(1024, 2, 3)", (ExprMacroTable)ExprMacroTable.nil()).eval(this.bestEffortBindings));
        Assert.assertEquals((Object)"Function[human_readable_binary_byte_format] requires 1 or 2 arguments", (Object)t.getMessage());
    }

    @Test
    public void testSafeDivide() {
        this.assertExpr("safe_divide(3, 1)", 3L);
        this.assertExpr("safe_divide(4.5, 2)", 2.25);
        this.assertExpr("safe_divide(3, 0)", null);
        this.assertExpr("safe_divide(1, 0.0)", null);
        this.assertExpr("safe_divide(NaN, 0.0)", null);
        this.assertExpr("safe_divide(0, NaN)", 0.0);
        this.assertExpr("safe_divide(0, POSITIVE_INFINITY)", NullHandling.defaultLongValue());
        this.assertExpr("safe_divide(POSITIVE_INFINITY,0)", NullHandling.defaultLongValue());
    }

    @Test
    public void testBitwise() {
        this.assertExpr("bitwiseAnd(3, 1)", 1L);
        this.assertExpr("bitwiseAnd(2, 1)", 0L);
        this.assertExpr("bitwiseOr(3, 1)", 3L);
        this.assertExpr("bitwiseOr(2, 1)", 3L);
        this.assertExpr("bitwiseXor(3, 1)", 2L);
        this.assertExpr("bitwiseXor(2, 1)", 3L);
        this.assertExpr("bitwiseShiftLeft(2, 1)", 4L);
        this.assertExpr("bitwiseShiftRight(2, 1)", 1L);
        this.assertExpr("bitwiseAnd(bitwiseComplement(1), 7)", 6L);
        this.assertExpr("bitwiseAnd('2', '1')", null);
        this.assertExpr("bitwiseAnd(3, '1')", 1L);
        this.assertExpr("bitwiseAnd(2, null)", NullHandling.replaceWithDefault() ? Long.valueOf(0L) : null);
        this.assertExpr("bitwiseComplement('1')", null);
        this.assertExpr("bitwiseComplement(null)", null);
        try {
            this.assertExpr("bitwiseComplement(461168601842738800000000000000.000000)", null);
            Assert.fail((String)"Did not throw IllegalArgumentException");
        }
        catch (ExpressionValidationException e) {
            Assert.assertEquals((Object)"Function[bitwiseComplement] Possible data truncation, param [461168601842738800000000000000.000000] is out of LONG value range", (Object)e.getMessage());
        }
        this.assertExpr("bitwiseOr(2.345, 1)", 3L);
        this.assertExpr("bitwiseOr(2, 1.3)", 3L);
        this.assertExpr("bitwiseAnd(2.345, 2.0)", 2L);
        this.assertExpr("bitwiseAnd(bitwiseConvertDoubleToLongBits(2.345), bitwiseConvertDoubleToLongBits(2.0))", 0x4000000000000000L);
        this.assertExpr("bitwiseConvertLongBitsToDouble(bitwiseAnd(bitwiseConvertDoubleToLongBits(2.345), bitwiseConvertDoubleToLongBits(2.0)))", 2.0);
        this.assertExpr("bitwiseConvertDoubleToLongBits(2.0)", 0x4000000000000000L);
        this.assertExpr("bitwiseConvertDoubleToLongBits(bitwiseConvertDoubleToLongBits(2.0))", 4886405595696988160L);
        this.assertExpr("bitwiseConvertLongBitsToDouble(4611686018427387904)", 2.0);
        this.assertExpr("bitwiseConvertLongBitsToDouble(bitwiseConvertLongBitsToDouble(4611686018427387904))", 9.9E-324);
        this.assertExpr("bitwiseConvertLongBitsToDouble('wat')", null);
        this.assertExpr("bitwiseConvertLongBitsToDouble('1')", null);
        this.assertExpr("bitwiseConvertLongBitsToDouble(null)", null);
        this.assertExpr("bitwiseConvertDoubleToLongBits('wat')", null);
        this.assertExpr("bitwiseConvertDoubleToLongBits('1.0')", null);
        this.assertExpr("bitwiseConvertDoubleToLongBits(null)", null);
    }

    @Test
    public void testRepeat() {
        this.assertExpr("repeat('hello', 2)", "hellohello");
        this.assertExpr("repeat('hello', -1)", null);
        this.assertExpr("repeat(null, 10)", null);
        this.assertExpr("repeat(nonexistent, 10)", null);
    }

    @Test
    public void testComplexDecode() {
        TypeStrategiesTest.NullableLongPair expected = new TypeStrategiesTest.NullableLongPair(1L, 2L);
        TypeStrategy strategy = TypeStrategiesTest.NULLABLE_TEST_PAIR_TYPE.getStrategy();
        byte[] bytes = new byte[strategy.estimateSizeBytes((Object)expected)];
        ByteBuffer buffer = ByteBuffer.wrap(bytes);
        int written = strategy.write(buffer, (Object)expected, bytes.length);
        Assert.assertEquals((long)bytes.length, (long)written);
        this.assertExpr(StringUtils.format((String)"complex_decode_base64('%s', '%s')", (Object[])new Object[]{TypeStrategiesTest.NULLABLE_TEST_PAIR_TYPE.getComplexTypeName(), StringUtils.encodeBase64String((byte[])bytes)}), expected);
    }

    @Test
    public void testComplexDecodeNull() {
        this.assertExpr(StringUtils.format((String)"complex_decode_base64('%s', null)", (Object[])new Object[]{TypeStrategiesTest.NULLABLE_TEST_PAIR_TYPE.getComplexTypeName()}), null);
    }

    @Test
    public void testComplexDecodeBaseWrongArgCount() {
        Throwable t = Assert.assertThrows(ExpressionValidationException.class, () -> this.assertExpr("complex_decode_base64(string)", null));
        Assert.assertEquals((Object)"Function[complex_decode_base64] requires 2 arguments", (Object)t.getMessage());
    }

    @Test
    public void testComplexDecodeBaseArg0Null() {
        Throwable t = Assert.assertThrows(ExpressionValidationException.class, () -> this.assertExpr("complex_decode_base64(null, string)", null));
        Assert.assertEquals((Object)"Function[complex_decode_base64] first argument must be constant STRING expression containing a valid complex type name but got NULL instead", (Object)t.getMessage());
    }

    @Test
    public void testComplexDecodeBaseArg0BadType() {
        Throwable t = Assert.assertThrows(ExpressionValidationException.class, () -> this.assertExpr("complex_decode_base64(1, string)", null));
        Assert.assertEquals((Object)"Function[complex_decode_base64] first argument must be constant STRING expression containing a valid complex type name but got '1' instead", (Object)t.getMessage());
    }

    @Test
    public void testComplexDecodeBaseArg0Unknown() {
        Throwable t = Assert.assertThrows(ExpressionValidationException.class, () -> this.assertExpr("complex_decode_base64('unknown', string)", null));
        Assert.assertEquals((Object)"Function[complex_decode_base64] first argument must be a valid COMPLEX type name, got unknown COMPLEX type [COMPLEX<unknown>]", (Object)t.getMessage());
    }

    @Test
    public void testMVToArrayWithValidInputs() {
        this.assertArrayExpr("mv_to_array(x)", new String[]{"foo"});
        this.assertArrayExpr("mv_to_array(a)", new String[]{"foo", "bar", "baz", "foobar"});
    }

    @Test
    public void testMVToArrayWithConstantLiteral() {
        Throwable t = Assert.assertThrows(ExpressionValidationException.class, () -> this.assertArrayExpr("mv_to_array('1')", null));
        Assert.assertEquals((Object)"Function[mv_to_array] argument 1 should be an identifier expression. Use array() instead", (Object)t.getMessage());
    }

    @Test
    public void testMVToArrayWithFunction() {
        Throwable t = Assert.assertThrows(ExpressionValidationException.class, () -> this.assertArrayExpr("mv_to_array(repeat('hello', 2))", null));
        Assert.assertEquals((Object)"Function[mv_to_array] argument (repeat [hello, 2]) should be an identifier expression. Use array() instead", (Object)t.getMessage());
    }

    @Test
    public void testMVToArrayWithMoreArgs() {
        Throwable t = Assert.assertThrows(ExpressionValidationException.class, () -> this.assertArrayExpr("mv_to_array(x,y)", null));
        Assert.assertEquals((Object)"Function[mv_to_array] requires 1 argument", (Object)t.getMessage());
    }

    @Test
    public void testMVToArrayWithNoArgs() {
        Throwable t = Assert.assertThrows(ExpressionValidationException.class, () -> this.assertArrayExpr("mv_to_array()", null));
        Assert.assertEquals((Object)"Function[mv_to_array] requires 1 argument", (Object)t.getMessage());
    }

    @Test
    public void testPlusOnString() {
        this.assertExpr("str1 + str2", "v1v2");
    }

    @Test
    public void testMultiplyOnString() {
        Throwable t = Assert.assertThrows(IAE.class, () -> this.assertExpr("str1 * str2", null));
        Assert.assertEquals((Object)"operator '*' in expression (\"str1\" * \"str2\") is not supported on type STRING.", (Object)t.getMessage());
    }

    @Test
    public void testMinusOnString() {
        Throwable t = Assert.assertThrows(IAE.class, () -> this.assertExpr("str1 - str2", null));
        Assert.assertEquals((Object)"operator '-' in expression (\"str1\" - \"str2\") is not supported on type STRING.", (Object)t.getMessage());
    }

    @Test
    public void testDivOnString() {
        Throwable t = Assert.assertThrows(IAE.class, () -> this.assertExpr("str1 / str2", null));
        Assert.assertEquals((Object)"operator '/' in expression (\"str1\" / \"str2\") is not supported on type STRING.", (Object)t.getMessage());
    }

    private void assertExpr(String expression, @Nullable Object expectedResult) {
        for (Expr.ObjectBinding toUse : this.allBindings) {
            this.assertExpr(expression, expectedResult, toUse);
        }
    }

    private void assertExpr(String expression, @Nullable Object expectedResult, Expr.ObjectBinding bindings) {
        Expr expr = Parser.parse((String)expression, (ExprMacroTable)ExprMacroTable.nil());
        Assert.assertEquals((String)expression, (Object)expectedResult, (Object)expr.eval(bindings).value());
        Expr exprNoFlatten = Parser.parse((String)expression, (ExprMacroTable)ExprMacroTable.nil(), (boolean)false);
        Expr roundTrip = Parser.parse((String)exprNoFlatten.stringify(), (ExprMacroTable)ExprMacroTable.nil());
        Assert.assertEquals((String)expr.stringify(), (Object)expectedResult, (Object)roundTrip.eval(bindings).value());
        Expr roundTripFlatten = Parser.parse((String)expr.stringify(), (ExprMacroTable)ExprMacroTable.nil());
        Assert.assertEquals((String)expr.stringify(), (Object)expectedResult, (Object)roundTripFlatten.eval(bindings).value());
        Assert.assertEquals((Object)expr.stringify(), (Object)roundTrip.stringify());
        Assert.assertEquals((Object)expr.stringify(), (Object)roundTripFlatten.stringify());
        Assert.assertArrayEquals((byte[])expr.getCacheKey(), (byte[])roundTrip.getCacheKey());
        Assert.assertArrayEquals((byte[])expr.getCacheKey(), (byte[])roundTripFlatten.getCacheKey());
    }

    private void assertArrayExpr(String expression, @Nullable Object[] expectedResult) {
        for (Expr.ObjectBinding toUse : this.allBindings) {
            this.assertArrayExpr(expression, expectedResult, toUse);
        }
    }

    private void assertArrayExpr(String expression, @Nullable Object[] expectedResult, Expr.ObjectBinding bindings) {
        Expr expr = Parser.parse((String)expression, (ExprMacroTable)ExprMacroTable.nil());
        Assert.assertArrayEquals((String)expression, (Object[])expectedResult, (Object[])expr.eval(bindings).asArray());
        Expr exprNoFlatten = Parser.parse((String)expression, (ExprMacroTable)ExprMacroTable.nil(), (boolean)false);
        Expr roundTrip = Parser.parse((String)exprNoFlatten.stringify(), (ExprMacroTable)ExprMacroTable.nil());
        Assert.assertArrayEquals((String)expression, (Object[])expectedResult, (Object[])roundTrip.eval(bindings).asArray());
        Expr roundTripFlatten = Parser.parse((String)expr.stringify(), (ExprMacroTable)ExprMacroTable.nil());
        Assert.assertArrayEquals((String)expression, (Object[])expectedResult, (Object[])roundTripFlatten.eval(bindings).asArray());
        Assert.assertEquals((Object)expr.stringify(), (Object)roundTrip.stringify());
        Assert.assertEquals((Object)expr.stringify(), (Object)roundTripFlatten.stringify());
        Assert.assertArrayEquals((byte[])expr.getCacheKey(), (byte[])roundTrip.getCacheKey());
        Assert.assertArrayEquals((byte[])expr.getCacheKey(), (byte[])roundTripFlatten.getCacheKey());
    }
}

