/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.math.expr;

import com.google.common.collect.ImmutableSet;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.TreeSet;
import java.util.function.BinaryOperator;
import java.util.function.DoubleBinaryOperator;
import java.util.function.LongBinaryOperator;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.apache.druid.common.config.NullHandling;
import org.apache.druid.java.util.common.DateTimes;
import org.apache.druid.java.util.common.HumanReadableBytes;
import org.apache.druid.java.util.common.StringUtils;
import org.apache.druid.java.util.common.UOE;
import org.apache.druid.math.expr.BinEqExpr;
import org.apache.druid.math.expr.Expr;
import org.apache.druid.math.expr.ExprEval;
import org.apache.druid.math.expr.ExprType;
import org.apache.druid.math.expr.ExpressionType;
import org.apache.druid.math.expr.ExpressionTypeConversion;
import org.apache.druid.math.expr.ExpressionTypeFactory;
import org.apache.druid.math.expr.Exprs;
import org.apache.druid.math.expr.IdentifierExpr;
import org.apache.druid.math.expr.NamedFunction;
import org.apache.druid.math.expr.vector.CastToTypeVectorProcessor;
import org.apache.druid.math.expr.vector.ExprVectorProcessor;
import org.apache.druid.math.expr.vector.VectorMathProcessors;
import org.apache.druid.math.expr.vector.VectorProcessors;
import org.apache.druid.math.expr.vector.VectorStringProcessors;
import org.apache.druid.segment.column.TypeSignature;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.format.DateTimeFormat;

public interface Function
extends NamedFunction {
    public ExprEval apply(List<Expr> var1, Expr.ObjectBinding var2);

    default public Set<Expr> getScalarInputs(List<Expr> args) {
        return ImmutableSet.copyOf(args);
    }

    default public Set<Expr> getArrayInputs(List<Expr> args) {
        return Collections.emptySet();
    }

    default public boolean hasArrayInputs() {
        return false;
    }

    default public boolean hasArrayOutput() {
        return false;
    }

    public void validateArguments(List<Expr> var1);

    @Nullable
    public ExpressionType getOutputType(Expr.InputBindingInspector var1, List<Expr> var2);

    default public boolean canVectorize(Expr.InputBindingInspector inspector, List<Expr> args) {
        return false;
    }

    default public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
        throw new UOE("Function[%s] is not vectorized", this.name());
    }

    public static class HumanReadableDecimalFormatFunc
    extends SizeFormatFunc {
        @Override
        public String name() {
            return "human_readable_decimal_format";
        }

        @Override
        protected HumanReadableBytes.UnitSystem getUnitSystem() {
            return HumanReadableBytes.UnitSystem.DECIMAL;
        }
    }

    public static class HumanReadableBinaryByteFormatFunc
    extends SizeFormatFunc {
        @Override
        public String name() {
            return "human_readable_binary_byte_format";
        }

        @Override
        protected HumanReadableBytes.UnitSystem getUnitSystem() {
            return HumanReadableBytes.UnitSystem.BINARY_BYTE;
        }
    }

    public static class HumanReadableDecimalByteFormatFunc
    extends SizeFormatFunc {
        @Override
        public String name() {
            return "human_readable_decimal_byte_format";
        }

        @Override
        protected HumanReadableBytes.UnitSystem getUnitSystem() {
            return HumanReadableBytes.UnitSystem.DECIMAL_BYTE;
        }
    }

    public static abstract class SizeFormatFunc
    implements Function {
        protected abstract HumanReadableBytes.UnitSystem getUnitSystem();

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            ExprEval valueParam = args.get(0).eval(bindings);
            if (NullHandling.sqlCompatible() && valueParam.isNumericNull()) {
                return ExprEval.of(null);
            }
            if (valueParam.value() != null && !valueParam.type().anyOf(new ExprType[]{ExprType.LONG, ExprType.DOUBLE})) {
                throw this.validationFailed("needs a number as its first argument but got %s instead", valueParam.type());
            }
            long precision = 2L;
            if (args.size() > 1) {
                ExprEval precisionParam = args.get(1).eval(bindings);
                if (!precisionParam.type().is(ExprType.LONG)) {
                    throw this.validationFailed("needs a LONG as its second argument but got %s instead", precisionParam.type());
                }
                precision = precisionParam.asLong();
                if (precision < 0L || precision > 3L) {
                    throw this.validationFailed("given precision[%d] must be in the range of [0,3]", precision);
                }
            }
            return ExprEval.of(HumanReadableBytes.format(valueParam.asLong(), precision, this.getUnitSystem()));
        }

        @Override
        public void validateArguments(List<Expr> args) {
            this.validationHelperCheckAnyOfArgumentCount(args, 1, 2);
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inputTypes, List<Expr> args) {
            return ExpressionType.STRING;
        }
    }

    public static class ArraySliceFunction
    implements Function {
        @Override
        public String name() {
            return "array_slice";
        }

        @Override
        public void validateArguments(List<Expr> args) {
            this.validationHelperCheckAnyOfArgumentCount(args, 2, 3);
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            ExpressionType arrayType = args.get(0).getOutputType(inspector);
            return Optional.ofNullable(ExpressionType.asArrayType(arrayType)).orElse(arrayType);
        }

        @Override
        public Set<Expr> getScalarInputs(List<Expr> args) {
            if (args.size() == 3) {
                return ImmutableSet.of((Object)args.get(1), (Object)args.get(2));
            }
            return ImmutableSet.of((Object)args.get(1));
        }

        @Override
        public Set<Expr> getArrayInputs(List<Expr> args) {
            return ImmutableSet.of((Object)args.get(0));
        }

        @Override
        public boolean hasArrayInputs() {
            return true;
        }

        @Override
        public boolean hasArrayOutput() {
            return true;
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            ExprEval expr = args.get(0).eval(bindings);
            Object[] array = expr.asArray();
            if (array == null) {
                return ExprEval.of(null);
            }
            int start = args.get(1).eval(bindings).asInt();
            int end = array.length;
            if (args.size() == 3) {
                end = args.get(2).eval(bindings).asInt();
            }
            if (start < 0 || start > array.length || start > end) {
                return ExprEval.of(null);
            }
            return ExprEval.ofArray(expr.asArrayType(), Arrays.copyOfRange(expr.asArray(), start, end));
        }
    }

    public static class ArrayOverlapFunction
    extends ArraysFunction {
        @Override
        public String name() {
            return "array_overlap";
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionType.LONG;
        }

        @Override
        ExprEval doApply(ExprEval lhsExpr, ExprEval rhsExpr) {
            Object[] array1 = lhsExpr.asArray();
            List<Object> array2 = Arrays.asList(rhsExpr.asArray());
            boolean any = false;
            for (Object check : array1) {
                any |= array2.contains(check);
            }
            return ExprEval.ofLongBoolean(any);
        }
    }

    public static class ArrayContainsFunction
    extends ArraysFunction {
        @Override
        public String name() {
            return "array_contains";
        }

        @Override
        public boolean hasArrayOutput() {
            return true;
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionType.LONG;
        }

        @Override
        ExprEval doApply(ExprEval lhsExpr, ExprEval rhsExpr) {
            Object[] array1 = lhsExpr.asArray();
            Object[] array2 = rhsExpr.asArray();
            return ExprEval.ofLongBoolean(Arrays.asList(array1).containsAll(Arrays.asList(array2)));
        }
    }

    public static class ArraySetAddAllFunction
    extends ArraysMergeFunction {
        @Override
        public String name() {
            return "array_set_add_all";
        }

        @Override
        <T> Object[] merge(TypeSignature<ExprType> elementType, T[] array1, T[] array2) {
            TreeSet l = new TreeSet(elementType.getNullableStrategy());
            l.addAll(Arrays.asList(array1));
            l.addAll(Arrays.asList(array2));
            return l.toArray();
        }
    }

    public static class ArraySetAddFunction
    extends ArrayAddElementFunction {
        @Override
        public String name() {
            return "array_set_add";
        }

        @Override
        <T> Object[] add(TypeSignature<ExprType> elementType, T[] array, @Nullable T val) {
            TreeSet set = new TreeSet(elementType.getNullableStrategy());
            set.addAll(Arrays.asList(array));
            set.add(val);
            return set.toArray();
        }
    }

    public static class ArrayConcatFunction
    extends ArraysMergeFunction {
        @Override
        public String name() {
            return "array_concat";
        }

        @Override
        <T> Object[] merge(TypeSignature<ExprType> elementType, T[] array1, T[] array2) {
            int i;
            Object[] output = new Object[array1.length + array2.length];
            for (i = 0; i < array1.length; ++i) {
                output[i] = array1[i];
            }
            i = array1.length;
            for (int j = 0; j < array2.length; ++j) {
                output[i] = array2[j];
                ++i;
            }
            return output;
        }
    }

    public static class ArrayPrependFunction
    extends ArrayAddElementFunction {
        @Override
        public String name() {
            return "array_prepend";
        }

        @Override
        Expr getScalarArgument(List<Expr> args) {
            return args.get(0);
        }

        @Override
        Expr getArrayArgument(List<Expr> args) {
            return args.get(1);
        }

        @Override
        <T> Object[] add(TypeSignature<ExprType> elementType, T[] array, @Nullable T val) {
            Object[] output = new Object[array.length + 1];
            output[0] = val;
            for (int i = 0; i < array.length; ++i) {
                output[i + 1] = array[i];
            }
            return output;
        }
    }

    public static class ArrayAppendFunction
    extends ArrayAddElementFunction {
        @Override
        public String name() {
            return "array_append";
        }

        @Override
        <T> Object[] add(TypeSignature<ExprType> elementType, T[] array, @Nullable T val) {
            Object[] output = new Object[array.length + 1];
            for (int i = 0; i < array.length; ++i) {
                output[i] = array[i];
            }
            output[array.length] = val;
            return output;
        }
    }

    public static class ArrayOrdinalOfFunction
    extends ArrayScalarFunction {
        @Override
        public String name() {
            return "array_ordinal_of";
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionType.LONG;
        }

        @Override
        ExprEval doApply(ExprEval arrayExpr, ExprEval scalarExpr) {
            Object[] array = arrayExpr.asArray();
            switch ((ExprType)scalarExpr.type().getType()) {
                case STRING: 
                case LONG: 
                case DOUBLE: {
                    int index = -1;
                    for (int i = 0; i < array.length; ++i) {
                        if (!Objects.equals(array[i], scalarExpr.value())) continue;
                        index = i;
                        break;
                    }
                    return index < 0 ? ExprEval.ofLong(NullHandling.replaceWithDefault() ? Integer.valueOf(-1) : null) : ExprEval.ofLong(index + 1);
                }
            }
            throw this.validationFailed("second argument must be a a scalar type but got %s instead", scalarExpr.type());
        }
    }

    public static class ArrayOffsetOfFunction
    extends ArrayScalarFunction {
        @Override
        public String name() {
            return "array_offset_of";
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionType.LONG;
        }

        @Override
        ExprEval doApply(ExprEval arrayExpr, ExprEval scalarExpr) {
            Object[] array = arrayExpr.asArray();
            switch ((ExprType)scalarExpr.type().getType()) {
                case STRING: 
                case LONG: 
                case DOUBLE: {
                    int index = -1;
                    for (int i = 0; i < array.length; ++i) {
                        if (!Objects.equals(array[i], scalarExpr.value())) continue;
                        index = i;
                        break;
                    }
                    return index < 0 ? ExprEval.ofLong(NullHandling.replaceWithDefault() ? Integer.valueOf(-1) : null) : ExprEval.ofLong(index);
                }
            }
            throw this.validationFailed("second argument must be a a scalar type but got %s instead", scalarExpr.type());
        }
    }

    public static class ArrayOrdinalFunction
    extends ArrayScalarFunction {
        @Override
        public String name() {
            return "array_ordinal";
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionType.elementType(args.get(0).getOutputType(inspector));
        }

        @Override
        ExprEval doApply(ExprEval arrayExpr, ExprEval scalarExpr) {
            int position;
            Object[] array = arrayExpr.asArray();
            if (array.length > (position = scalarExpr.asInt() - 1)) {
                return ExprEval.ofType(arrayExpr.elementType(), array[position]);
            }
            return ExprEval.of(null);
        }
    }

    public static class ArrayOffsetFunction
    extends ArrayScalarFunction {
        @Override
        public String name() {
            return "array_offset";
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionType.elementType(args.get(0).getOutputType(inspector));
        }

        @Override
        ExprEval doApply(ExprEval arrayExpr, ExprEval scalarExpr) {
            int position;
            Object[] array = arrayExpr.asArray();
            if (array.length > (position = scalarExpr.asInt())) {
                return ExprEval.ofType(arrayExpr.elementType(), array[position]);
            }
            return ExprEval.of(null);
        }
    }

    public static class ArrayToStringFunction
    extends ArrayScalarFunction {
        @Override
        public String name() {
            return "array_to_string";
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionType.STRING;
        }

        @Override
        ExprEval doApply(ExprEval arrayExpr, ExprEval scalarExpr) {
            String join = scalarExpr.asString();
            Object[] raw = arrayExpr.asArray();
            if (raw == null || raw.length == 1 && raw[0] == null) {
                return ExprEval.of(null);
            }
            return ExprEval.of(Arrays.stream(raw).map(String::valueOf).collect(Collectors.joining(join != null ? join : "")));
        }
    }

    public static class StringToArrayFunction
    implements Function {
        @Override
        public String name() {
            return "string_to_array";
        }

        @Override
        public void validateArguments(List<Expr> args) {
            this.validationHelperCheckArgumentCount(args, 2);
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionType.STRING_ARRAY;
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            ExprEval expr = args.get(0).eval(bindings);
            String arrayString = expr.asString();
            if (arrayString == null) {
                return ExprEval.of(null);
            }
            String split = args.get(1).eval(bindings).asString();
            return ExprEval.ofStringArray(arrayString.split(split != null ? split : ""));
        }

        @Override
        public Set<Expr> getScalarInputs(List<Expr> args) {
            return ImmutableSet.copyOf(args);
        }

        @Override
        public boolean hasArrayOutput() {
            return true;
        }
    }

    public static class ArrayLengthFunction
    implements Function {
        @Override
        public String name() {
            return "array_length";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            ExprEval expr = args.get(0).eval(bindings);
            Object[] array = expr.asArray();
            if (array == null) {
                return ExprEval.of(null);
            }
            return ExprEval.ofLong(array.length);
        }

        @Override
        public Set<Expr> getArrayInputs(List<Expr> args) {
            return ImmutableSet.of((Object)args.get(0));
        }

        @Override
        public boolean hasArrayInputs() {
            return true;
        }

        @Override
        public void validateArguments(List<Expr> args) {
            this.validationHelperCheckArgumentCount(args, 1);
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionType.LONG;
        }

        @Override
        public Set<Expr> getScalarInputs(List<Expr> args) {
            return Collections.emptySet();
        }
    }

    public static class ArrayConstructorFunction
    implements Function {
        @Override
        public String name() {
            return "array";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            int length = args.size();
            Object[] out = new Object[length];
            ExpressionType arrayType = null;
            for (int i = 0; i < length; ++i) {
                ExprEval evaluated = args.get(i).eval(bindings);
                arrayType = ArrayConstructorFunction.setArrayOutput(arrayType, out, i, evaluated);
            }
            return ExprEval.ofArray(arrayType, out);
        }

        @Override
        public Set<Expr> getScalarInputs(List<Expr> args) {
            return ImmutableSet.copyOf(args);
        }

        @Override
        public Set<Expr> getArrayInputs(List<Expr> args) {
            return Collections.emptySet();
        }

        @Override
        public boolean hasArrayOutput() {
            return true;
        }

        @Override
        public void validateArguments(List<Expr> args) {
            this.validationHelperCheckMinArgumentCount(args, 1);
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            ExpressionType type = ExpressionType.LONG;
            for (Expr arg : args) {
                type = ExpressionTypeConversion.function(type, arg.getOutputType(inspector));
            }
            return ExpressionType.asArrayType(type);
        }

        static ExpressionType setArrayOutput(@Nullable ExpressionType arrayType, Object[] out, int i, ExprEval evaluated) {
            if (arrayType == null) {
                arrayType = ExpressionTypeFactory.getInstance().ofArray(evaluated.type());
            }
            out[i] = arrayType.getElementType().isNumeric() && evaluated.isNumericNull() ? null : (!evaluated.asArrayType().equals(arrayType) ? evaluated.castTo((ExpressionType)arrayType.getElementType()).value() : evaluated.value());
            return arrayType;
        }
    }

    public static class MVToArrayFunction
    implements Function {
        @Override
        public String name() {
            return "mv_to_array";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            return args.get(0).eval(bindings).castTo(ExpressionType.STRING_ARRAY);
        }

        @Override
        public void validateArguments(List<Expr> args) {
            this.validationHelperCheckArgumentCount(args, 1);
            IdentifierExpr expr = args.get(0).getIdentifierExprIfIdentifierExpr();
            if (expr == null) {
                throw this.validationFailed("argument %s should be an identifier expression. Use array() instead", args.get(0).toString());
            }
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionType.STRING_ARRAY;
        }

        @Override
        public boolean hasArrayInputs() {
            return true;
        }

        @Override
        public boolean hasArrayOutput() {
            return true;
        }

        @Override
        public Set<Expr> getScalarInputs(List<Expr> args) {
            return Collections.emptySet();
        }

        @Override
        public Set<Expr> getArrayInputs(List<Expr> args) {
            return ImmutableSet.copyOf(args);
        }
    }

    public static class SubMonthFunc
    implements Function {
        @Override
        public String name() {
            return "subtract_months";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            Long left = args.get(0).eval(bindings).asLong();
            Long right = args.get(1).eval(bindings).asLong();
            DateTimeZone timeZone = DateTimes.inferTzFromString(args.get(2).eval(bindings).asString());
            if (left == null || right == null) {
                return ExprEval.of(null);
            }
            return ExprEval.of(DateTimes.subMonths(right, left, timeZone));
        }

        @Override
        public void validateArguments(List<Expr> args) {
            this.validationHelperCheckArgumentCount(args, 3);
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionType.LONG;
        }
    }

    public static class UnixTimestampFunc
    extends TimestampFromEpochFunc {
        @Override
        public String name() {
            return "unix_timestamp";
        }

        @Override
        protected final ExprEval toValue(DateTime date) {
            return ExprEval.of(date.getMillis() / 1000L);
        }
    }

    public static class TimestampFromEpochFunc
    implements Function {
        @Override
        public String name() {
            return "timestamp";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            DateTime date;
            ExprEval value = args.get(0).eval(bindings);
            if (!value.type().is(ExprType.STRING)) {
                throw this.validationFailed("first argument should be a STRING but got %s instead", value.type());
            }
            DateTimes.UtcFormatter formatter = DateTimes.ISO_DATE_OPTIONAL_TIME;
            if (args.size() > 1) {
                ExprEval format = args.get(1).eval(bindings);
                if (!format.type().is(ExprType.STRING)) {
                    throw this.validationFailed("second argument should be STRING but got %s instead", format.type());
                }
                formatter = DateTimes.wrapFormatter(DateTimeFormat.forPattern((String)format.asString()));
            }
            try {
                date = formatter.parse(value.asString());
            }
            catch (IllegalArgumentException e) {
                throw this.validationFailed(e, "invalid value %s", value.asString());
            }
            return this.toValue(date);
        }

        @Override
        public void validateArguments(List<Expr> args) {
            this.validationHelperCheckAnyOfArgumentCount(args, 1, 2);
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionType.LONG;
        }

        protected ExprEval toValue(DateTime date) {
            return ExprEval.of(date.getMillis());
        }
    }

    public static class RpadFunc
    implements Function {
        @Override
        public String name() {
            return "rpad";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            String base = args.get(0).eval(bindings).asString();
            int len = args.get(1).eval(bindings).asInt();
            String pad = args.get(2).eval(bindings).asString();
            if (base == null || pad == null) {
                return ExprEval.of(null);
            }
            return ExprEval.of(len == 0 ? NullHandling.defaultStringValue() : StringUtils.rpad(base, len, pad));
        }

        @Override
        public void validateArguments(List<Expr> args) {
            this.validationHelperCheckArgumentCount(args, 3);
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionType.STRING;
        }
    }

    public static class LpadFunc
    implements Function {
        @Override
        public String name() {
            return "lpad";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            String base = args.get(0).eval(bindings).asString();
            int len = args.get(1).eval(bindings).asInt();
            String pad = args.get(2).eval(bindings).asString();
            if (base == null || pad == null) {
                return ExprEval.of(null);
            }
            return ExprEval.of(len == 0 ? NullHandling.defaultStringValue() : StringUtils.lpad(base, len, pad));
        }

        @Override
        public void validateArguments(List<Expr> args) {
            this.validationHelperCheckArgumentCount(args, 3);
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionType.STRING;
        }
    }

    public static class RepeatFunc
    extends StringLongFunction {
        @Override
        public String name() {
            return "repeat";
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionType.STRING;
        }

        @Override
        protected ExprEval eval(String x, int y) {
            if (x == null) {
                return ExprEval.of(null);
            }
            return ExprEval.of(y < 1 ? NullHandling.defaultStringValue() : StringUtils.repeat(x, y));
        }
    }

    public static class ReverseFunc
    extends UnivariateFunction {
        @Override
        public String name() {
            return "reverse";
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionType.STRING;
        }

        @Override
        protected ExprEval eval(ExprEval param) {
            if (!param.type().is(ExprType.STRING)) {
                throw this.validationFailed("needs a STRING argument but got %s instead", param.type());
            }
            String arg = param.asString();
            return ExprEval.of(arg == null ? NullHandling.defaultStringValue() : new StringBuilder(arg).reverse().toString());
        }
    }

    public static class UpperFunc
    implements Function {
        @Override
        public String name() {
            return "upper";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            String arg = args.get(0).eval(bindings).asString();
            if (arg == null) {
                return ExprEval.of(NullHandling.defaultStringValue());
            }
            return ExprEval.of(StringUtils.toUpperCase(arg));
        }

        @Override
        public void validateArguments(List<Expr> args) {
            this.validationHelperCheckArgumentCount(args, 1);
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionType.STRING;
        }
    }

    public static class LowerFunc
    implements Function {
        @Override
        public String name() {
            return "lower";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            String arg = args.get(0).eval(bindings).asString();
            if (arg == null) {
                return ExprEval.of(NullHandling.defaultStringValue());
            }
            return ExprEval.of(StringUtils.toLowerCase(arg));
        }

        @Override
        public void validateArguments(List<Expr> args) {
            this.validationHelperCheckArgumentCount(args, 1);
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionType.STRING;
        }
    }

    public static class ReplaceFunc
    implements Function {
        @Override
        public String name() {
            return "replace";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            String arg = args.get(0).eval(bindings).asString();
            String pattern = NullHandling.nullToEmptyIfNeeded(args.get(1).eval(bindings).asString());
            String replacement = NullHandling.nullToEmptyIfNeeded(args.get(2).eval(bindings).asString());
            if (arg == null) {
                return ExprEval.of(NullHandling.defaultStringValue());
            }
            return ExprEval.of(StringUtils.replace(arg, pattern, replacement));
        }

        @Override
        public void validateArguments(List<Expr> args) {
            this.validationHelperCheckArgumentCount(args, 3);
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionType.STRING;
        }
    }

    public static class LeftFunc
    extends StringLongFunction {
        @Override
        public String name() {
            return "left";
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionType.STRING;
        }

        @Override
        protected ExprEval eval(@Nullable String x, int y) {
            if (y < 0) {
                throw this.validationFailed("needs a postive integer as second argument", new Object[0]);
            }
            if (x == null) {
                return ExprEval.of(null);
            }
            return ExprEval.of(y < x.length() ? x.substring(0, y) : x);
        }
    }

    public static class RightFunc
    extends StringLongFunction {
        @Override
        public String name() {
            return "right";
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionType.STRING;
        }

        @Override
        protected ExprEval eval(@Nullable String x, int y) {
            if (y < 0) {
                throw this.validationFailed("needs a positive integer as the second argument", new Object[0]);
            }
            if (x == null) {
                return ExprEval.of(null);
            }
            int len = x.length();
            return ExprEval.of(y < len ? x.substring(len - y) : x);
        }
    }

    public static class SubstringFunc
    implements Function {
        @Override
        public String name() {
            return "substring";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            String arg = args.get(0).eval(bindings).asString();
            if (arg == null) {
                return ExprEval.of(null);
            }
            int index = args.get(1).eval(bindings).asInt();
            int length = args.get(2).eval(bindings).asInt();
            if (index < arg.length()) {
                if (length >= 0) {
                    return ExprEval.of(arg.substring(index, Math.min(index + length, arg.length())));
                }
                return ExprEval.of(arg.substring(index));
            }
            return ExprEval.of(NullHandling.defaultStringValue());
        }

        @Override
        public void validateArguments(List<Expr> args) {
            this.validationHelperCheckArgumentCount(args, 3);
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionType.STRING;
        }
    }

    public static class StrposFunc
    implements Function {
        @Override
        public String name() {
            return "strpos";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            String haystack = NullHandling.nullToEmptyIfNeeded(args.get(0).eval(bindings).asString());
            String needle = NullHandling.nullToEmptyIfNeeded(args.get(1).eval(bindings).asString());
            if (haystack == null || needle == null) {
                return ExprEval.of(null);
            }
            int fromIndex = args.size() >= 3 ? args.get(2).eval(bindings).asInt() : 0;
            return ExprEval.of(haystack.indexOf(needle, fromIndex));
        }

        @Override
        public void validateArguments(List<Expr> args) {
            this.validationHelperCheckAnyOfArgumentCount(args, 2, 3);
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionType.LONG;
        }
    }

    public static class StringFormatFunc
    implements Function {
        @Override
        public String name() {
            return "format";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            String formatString = NullHandling.nullToEmptyIfNeeded(args.get(0).eval(bindings).asString());
            if (formatString == null) {
                return ExprEval.of(null);
            }
            Object[] formatArgs = new Object[args.size() - 1];
            for (int i = 1; i < args.size(); ++i) {
                formatArgs[i - 1] = args.get(i).eval(bindings).value();
            }
            return ExprEval.of(StringUtils.nonStrictFormat(formatString, formatArgs));
        }

        @Override
        public void validateArguments(List<Expr> args) {
            this.validationHelperCheckMinArgumentCount(args, 1);
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionType.STRING;
        }
    }

    public static class StrlenFunc
    implements Function {
        @Override
        public String name() {
            return "strlen";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            String arg = args.get(0).eval(bindings).asString();
            return arg == null ? ExprEval.ofLong(null) : ExprEval.of(arg.length());
        }

        @Override
        public void validateArguments(List<Expr> args) {
            this.validationHelperCheckArgumentCount(args, 1);
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionType.LONG;
        }
    }

    public static class ConcatFunc
    implements Function {
        @Override
        public String name() {
            return "concat";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            if (args.size() == 0) {
                return ExprEval.of(null);
            }
            String first = NullHandling.nullToEmptyIfNeeded(args.get(0).eval(bindings).asString());
            if (first == null) {
                return ExprEval.of(null);
            }
            StringBuilder builder = new StringBuilder(first);
            for (int i = 1; i < args.size(); ++i) {
                String s = NullHandling.nullToEmptyIfNeeded(args.get(i).eval(bindings).asString());
                if (s == null) {
                    return ExprEval.of(null);
                }
                builder.append(s);
            }
            return ExprEval.of(builder.toString());
        }

        @Override
        public void validateArguments(List<Expr> args) {
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionType.STRING;
        }

        @Override
        public boolean canVectorize(Expr.InputBindingInspector inspector, List<Expr> args) {
            return inspector.areScalar(args) && inspector.canVectorize(args);
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorStringProcessors.concat(inspector, args);
        }
    }

    public static class IsNotNullFunc
    implements Function {
        @Override
        public String name() {
            return "notnull";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            ExprEval expr = args.get(0).eval(bindings);
            return ExprEval.ofLongBoolean(expr.value() != null);
        }

        @Override
        public void validateArguments(List<Expr> args) {
            this.validationHelperCheckArgumentCount(args, 1);
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionType.LONG;
        }

        @Override
        public boolean canVectorize(Expr.InputBindingInspector inspector, List<Expr> args) {
            return args.get(0).canVectorize(inspector);
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorProcessors.isNotNull(inspector, args.get(0));
        }
    }

    public static class IsNullFunc
    implements Function {
        @Override
        public String name() {
            return "isnull";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            ExprEval expr = args.get(0).eval(bindings);
            return ExprEval.ofLongBoolean(expr.value() == null);
        }

        @Override
        public void validateArguments(List<Expr> args) {
            this.validationHelperCheckArgumentCount(args, 1);
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionType.LONG;
        }

        @Override
        public boolean canVectorize(Expr.InputBindingInspector inspector, List<Expr> args) {
            return args.get(0).canVectorize(inspector);
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorProcessors.isNull(inspector, args.get(0));
        }
    }

    public static class NvlFunc
    implements Function {
        @Override
        public String name() {
            return "nvl";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            ExprEval eval = args.get(0).eval(bindings);
            return eval.value() == null ? args.get(1).eval(bindings) : eval;
        }

        @Override
        public void validateArguments(List<Expr> args) {
            this.validationHelperCheckArgumentCount(args, 2);
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionTypeConversion.conditional(inspector, args);
        }

        @Override
        public boolean canVectorize(Expr.InputBindingInspector inspector, List<Expr> args) {
            return inspector.canVectorize(args);
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorProcessors.nvl(inspector, args.get(0), args.get(1));
        }
    }

    public static class CaseSimpleFunc
    implements Function {
        @Override
        public String name() {
            return "case_simple";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            for (int i = 1; i < args.size(); i += 2) {
                if (i == args.size() - 1) {
                    return args.get(i).eval(bindings);
                }
                if (!new BinEqExpr("==", args.get(0), args.get(i)).eval(bindings).asBoolean()) continue;
                return args.get(i + 1).eval(bindings);
            }
            return ExprEval.of(null);
        }

        @Override
        public void validateArguments(List<Expr> args) {
            this.validationHelperCheckMinArgumentCount(args, 3);
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            ArrayList<Expr> results = new ArrayList<Expr>();
            for (int i = 2; i < args.size(); i += 2) {
                results.add(args.get(i));
            }
            results.add(args.get(args.size() - 1));
            return ExpressionTypeConversion.conditional(inspector, results);
        }
    }

    public static class CaseSearchedFunc
    implements Function {
        @Override
        public String name() {
            return "case_searched";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            for (int i = 0; i < args.size(); i += 2) {
                if (i == args.size() - 1) {
                    return args.get(i).eval(bindings);
                }
                if (!args.get(i).eval(bindings).asBoolean()) continue;
                return args.get(i + 1).eval(bindings);
            }
            return ExprEval.of(null);
        }

        @Override
        public void validateArguments(List<Expr> args) {
            this.validationHelperCheckMinArgumentCount(args, 2);
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            ArrayList<Expr> results = new ArrayList<Expr>();
            for (int i = 1; i < args.size(); i += 2) {
                results.add(args.get(i));
            }
            results.add(args.get(args.size() - 1));
            return ExpressionTypeConversion.conditional(inspector, results);
        }
    }

    public static class ConditionFunc
    implements Function {
        @Override
        public String name() {
            return "if";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            ExprEval x = args.get(0).eval(bindings);
            return x.asBoolean() ? args.get(1).eval(bindings) : args.get(2).eval(bindings);
        }

        @Override
        public void validateArguments(List<Expr> args) {
            this.validationHelperCheckArgumentCount(args, 3);
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionTypeConversion.conditional(inspector, args.subList(1, 3));
        }
    }

    public static class LeastFunc
    extends ReduceFunction {
        public static final String NAME = "least";

        public LeastFunc() {
            super(Math::min, Math::min, BinaryOperator.minBy(Comparator.naturalOrder()));
        }

        @Override
        public String name() {
            return NAME;
        }
    }

    public static class GreatestFunc
    extends ReduceFunction {
        public static final String NAME = "greatest";

        public GreatestFunc() {
            super(Math::max, Math::max, BinaryOperator.maxBy(Comparator.naturalOrder()));
        }

        @Override
        public String name() {
            return NAME;
        }
    }

    public static class CastFunc
    extends BivariateFunction {
        @Override
        public String name() {
            return "cast";
        }

        @Override
        protected ExprEval eval(ExprEval x, ExprEval y) {
            ExpressionType castTo;
            if (NullHandling.sqlCompatible() && x.value() == null) {
                return ExprEval.of(null);
            }
            try {
                castTo = ExpressionType.fromString(StringUtils.toUpperCase(y.asString()));
            }
            catch (IllegalArgumentException e) {
                throw this.validationFailed("Invalid type [%s]", y.asString());
            }
            return x.castTo(castTo);
        }

        @Override
        public Set<Expr> getScalarInputs(List<Expr> args) {
            if (args.get(1).isLiteral()) {
                ExpressionType castTo = ExpressionType.fromString(StringUtils.toUpperCase(args.get(1).getLiteralValue().toString()));
                switch ((ExprType)castTo.getType()) {
                    case ARRAY: {
                        return Collections.emptySet();
                    }
                }
                return ImmutableSet.of((Object)args.get(0));
            }
            return Collections.emptySet();
        }

        @Override
        public Set<Expr> getArrayInputs(List<Expr> args) {
            if (args.get(1).isLiteral()) {
                ExpressionType castTo = ExpressionType.fromString(StringUtils.toUpperCase(args.get(1).getLiteralValue().toString()));
                switch ((ExprType)castTo.getType()) {
                    case STRING: 
                    case LONG: 
                    case DOUBLE: {
                        return Collections.emptySet();
                    }
                }
                return ImmutableSet.of((Object)args.get(0));
            }
            return Collections.emptySet();
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            if (args.get(1).isLiteral()) {
                return ExpressionType.fromString(StringUtils.toUpperCase(args.get(1).getLiteralValue().toString()));
            }
            return null;
        }

        @Override
        public boolean canVectorize(Expr.InputBindingInspector inspector, List<Expr> args) {
            return args.get(0).canVectorize(inspector) && args.get(1).isLiteral();
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return CastToTypeVectorProcessor.cast(args.get(0).asVectorProcessor(inspector), ExpressionType.fromString(StringUtils.toUpperCase(args.get(1).getLiteralValue().toString())));
        }
    }

    public static class Scalb
    extends BivariateFunction {
        @Override
        public String name() {
            return "scalb";
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionType.DOUBLE;
        }

        @Override
        protected ExprEval eval(ExprEval x, ExprEval y) {
            if (NullHandling.sqlCompatible() && (x.value() == null || y.value() == null)) {
                return ExprEval.of(null);
            }
            ExpressionType type = ExpressionTypeConversion.autoDetect(x, y);
            switch ((ExprType)type.getType()) {
                case STRING: {
                    return ExprEval.of(null);
                }
            }
            return ExprEval.of(Math.scalb(x.asDouble(), y.asInt()));
        }

        @Override
        public boolean canVectorize(Expr.InputBindingInspector inspector, List<Expr> args) {
            return inspector.areNumeric(args) && inspector.canVectorize(args);
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.scalb(inspector, args.get(0), args.get(1));
        }
    }

    public static class Pow
    extends DoubleBivariateMathFunction {
        @Override
        public String name() {
            return "pow";
        }

        @Override
        protected ExprEval eval(double x, double y) {
            return ExprEval.of(Math.pow(x, y));
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.doublePower(inspector, args.get(0), args.get(1));
        }
    }

    public static class NextAfter
    extends DoubleBivariateMathFunction {
        @Override
        public String name() {
            return "nextAfter";
        }

        @Override
        protected ExprEval eval(double x, double y) {
            return ExprEval.of(Math.nextAfter(x, y));
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.nextAfter(inspector, args.get(0), args.get(1));
        }
    }

    public static class Min
    extends BivariateMathFunction {
        @Override
        public String name() {
            return "min";
        }

        @Override
        protected ExprEval eval(long x, long y) {
            return ExprEval.of(Math.min(x, y));
        }

        @Override
        protected ExprEval eval(double x, double y) {
            return ExprEval.of(Math.min(x, y));
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.min(inspector, args.get(0), args.get(1));
        }
    }

    public static class Max
    extends BivariateMathFunction {
        @Override
        public String name() {
            return "max";
        }

        @Override
        protected ExprEval eval(long x, long y) {
            return ExprEval.of(Math.max(x, y));
        }

        @Override
        protected ExprEval eval(double x, double y) {
            return ExprEval.of(Math.max(x, y));
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.max(inspector, args.get(0), args.get(1));
        }
    }

    public static class Remainder
    extends DoubleBivariateMathFunction {
        @Override
        public String name() {
            return "remainder";
        }

        @Override
        protected ExprEval eval(double x, double y) {
            return ExprEval.of(Math.IEEEremainder(x, y));
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.remainder(inspector, args.get(0), args.get(1));
        }
    }

    public static class Hypot
    extends DoubleBivariateMathFunction {
        @Override
        public String name() {
            return "hypot";
        }

        @Override
        protected ExprEval eval(double x, double y) {
            return ExprEval.of(Math.hypot(x, y));
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.hypot(inspector, args.get(0), args.get(1));
        }
    }

    public static class CopySign
    extends DoubleBivariateMathFunction {
        @Override
        public String name() {
            return "copySign";
        }

        @Override
        protected ExprEval eval(double x, double y) {
            return ExprEval.of(Math.copySign(x, y));
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.copySign(inspector, args.get(0), args.get(1));
        }
    }

    public static class Atan2
    extends DoubleBivariateMathFunction {
        @Override
        public String name() {
            return "atan2";
        }

        @Override
        protected ExprEval eval(double y, double x) {
            return ExprEval.of(Math.atan2(y, x));
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.atan2(inspector, args.get(0), args.get(1));
        }
    }

    public static class Ulp
    extends DoubleUnivariateMathFunction {
        @Override
        public String name() {
            return "ulp";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.ulp(param));
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.ulp(inspector, args.get(0));
        }
    }

    public static class ToRadians
    extends DoubleUnivariateMathFunction {
        @Override
        public String name() {
            return "toRadians";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.toRadians(param));
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.toRadians(inspector, args.get(0));
        }
    }

    public static class ToDegrees
    extends DoubleUnivariateMathFunction {
        @Override
        public String name() {
            return "toDegrees";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.toDegrees(param));
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.toDegrees(inspector, args.get(0));
        }
    }

    public static class Tanh
    extends DoubleUnivariateMathFunction {
        @Override
        public String name() {
            return "tanh";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.tanh(param));
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.tanh(inspector, args.get(0));
        }
    }

    public static class Tan
    extends DoubleUnivariateMathFunction {
        @Override
        public String name() {
            return "tan";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.tan(param));
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.tan(inspector, args.get(0));
        }
    }

    public static class Sqrt
    extends DoubleUnivariateMathFunction {
        @Override
        public String name() {
            return "sqrt";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.sqrt(param));
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.sqrt(inspector, args.get(0));
        }
    }

    public static class Sinh
    extends DoubleUnivariateMathFunction {
        @Override
        public String name() {
            return "sinh";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.sinh(param));
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.sinh(inspector, args.get(0));
        }
    }

    public static class Sin
    extends DoubleUnivariateMathFunction {
        @Override
        public String name() {
            return "sin";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.sin(param));
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.sin(inspector, args.get(0));
        }
    }

    public static class Signum
    extends DoubleUnivariateMathFunction {
        @Override
        public String name() {
            return "signum";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.signum(param));
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.signum(inspector, args.get(0));
        }
    }

    public static class Round
    implements Function {
        private static final BigDecimal MAX_FINITE_VALUE = BigDecimal.valueOf(Double.MAX_VALUE);
        private static final BigDecimal MIN_FINITE_VALUE = BigDecimal.valueOf(-1.7976931348623157E308);

        @Override
        public String name() {
            return "round";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            ExprEval value1 = args.get(0).eval(bindings);
            if (NullHandling.sqlCompatible() && value1.isNumericNull()) {
                return ExprEval.of(null);
            }
            if (!value1.type().anyOf(new ExprType[]{ExprType.LONG, ExprType.DOUBLE})) {
                throw this.validationFailed("first argument should be a LONG or DOUBLE but got %s instead", value1.type());
            }
            if (args.size() == 1) {
                return this.eval(value1);
            }
            ExprEval value2 = args.get(1).eval(bindings);
            if (!value2.type().is(ExprType.LONG)) {
                throw this.validationFailed("second argument should be a LONG but got %s instead", value2.type());
            }
            return this.eval(value1, value2.asInt());
        }

        @Override
        public void validateArguments(List<Expr> args) {
            this.validationHelperCheckAnyOfArgumentCount(args, 1, 2);
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return args.get(0).getOutputType(inspector);
        }

        private ExprEval eval(ExprEval param) {
            return this.eval(param, 0);
        }

        private ExprEval eval(ExprEval param, int scale) {
            if (param.type().is(ExprType.LONG)) {
                return ExprEval.of(BigDecimal.valueOf(param.asLong()).setScale(scale, RoundingMode.HALF_UP).longValue());
            }
            if (param.type().is(ExprType.DOUBLE)) {
                BigDecimal decimal = Round.safeGetFromDouble(param.asDouble());
                return ExprEval.of(decimal.setScale(scale, RoundingMode.HALF_UP).doubleValue());
            }
            return ExprEval.of(null);
        }

        private static BigDecimal safeGetFromDouble(double val) {
            if (Double.isNaN(val)) {
                return BigDecimal.ZERO;
            }
            if (val == Double.POSITIVE_INFINITY) {
                return MAX_FINITE_VALUE;
            }
            if (val == Double.NEGATIVE_INFINITY) {
                return MIN_FINITE_VALUE;
            }
            return BigDecimal.valueOf(val);
        }
    }

    public static class Rint
    extends DoubleUnivariateMathFunction {
        @Override
        public String name() {
            return "rint";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.rint(param));
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.rint(inspector, args.get(0));
        }
    }

    public static class NextUp
    extends DoubleUnivariateMathFunction {
        @Override
        public String name() {
            return "nextUp";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.nextUp(param));
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.nextUp(inspector, args.get(0));
        }
    }

    public static class Log1p
    extends DoubleUnivariateMathFunction {
        @Override
        public String name() {
            return "log1p";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.log1p(param));
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.log1p(inspector, args.get(0));
        }
    }

    public static class Log10
    extends DoubleUnivariateMathFunction {
        @Override
        public String name() {
            return "log10";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.log10(param));
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.log10(inspector, args.get(0));
        }
    }

    public static class Log
    extends DoubleUnivariateMathFunction {
        @Override
        public String name() {
            return "log";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.log(param));
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.log(inspector, args.get(0));
        }
    }

    public static class GetExponent
    extends UnivariateMathFunction {
        @Override
        public String name() {
            return "getExponent";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.getExponent(param));
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionType.LONG;
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.getExponent(inspector, args.get(0));
        }
    }

    public static class Floor
    extends DoubleUnivariateMathFunction {
        @Override
        public String name() {
            return "floor";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.floor(param));
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.floor(inspector, args.get(0));
        }
    }

    public static class Expm1
    extends DoubleUnivariateMathFunction {
        @Override
        public String name() {
            return "expm1";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.expm1(param));
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.expm1(inspector, args.get(0));
        }
    }

    public static class Exp
    extends DoubleUnivariateMathFunction {
        @Override
        public String name() {
            return "exp";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.exp(param));
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.exp(inspector, args.get(0));
        }
    }

    public static class Div
    extends BivariateMathFunction {
        @Override
        public String name() {
            return "div";
        }

        @Override
        protected ExprEval eval(long x, long y) {
            return ExprEval.of(x / y);
        }

        @Override
        protected ExprEval eval(double x, double y) {
            return ExprEval.of((long)(x / y));
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionTypeConversion.integerMathFunction(args.get(0).getOutputType(inspector), args.get(1).getOutputType(inspector));
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.longDivide(inspector, args.get(0), args.get(1));
        }
    }

    public static class SafeDivide
    extends BivariateMathFunction {
        public static final String NAME = "safe_divide";

        @Override
        public String name() {
            return NAME;
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionTypeConversion.function(args.get(0).getOutputType(inspector), args.get(1).getOutputType(inspector));
        }

        @Override
        protected ExprEval eval(long x, long y) {
            if (y == 0L) {
                if (x != 0L) {
                    return ExprEval.ofLong(null);
                }
                return ExprEval.ofLong(0);
            }
            return ExprEval.ofLong(x / y);
        }

        @Override
        protected ExprEval eval(double x, double y) {
            if (y == 0.0 || Double.isNaN(y)) {
                if (x != 0.0) {
                    return ExprEval.ofDouble(null);
                }
                return ExprEval.ofDouble(0);
            }
            return ExprEval.ofDouble(x / y);
        }
    }

    public static class Cot
    extends DoubleUnivariateMathFunction {
        @Override
        public String name() {
            return "cot";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.cos(param) / Math.sin(param));
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.cot(inspector, args.get(0));
        }
    }

    public static class Cosh
    extends DoubleUnivariateMathFunction {
        @Override
        public String name() {
            return "cosh";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.cosh(param));
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.cosh(inspector, args.get(0));
        }
    }

    public static class Cos
    extends DoubleUnivariateMathFunction {
        @Override
        public String name() {
            return "cos";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.cos(param));
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.cos(inspector, args.get(0));
        }
    }

    public static class Ceil
    extends DoubleUnivariateMathFunction {
        @Override
        public String name() {
            return "ceil";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.ceil(param));
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.ceil(inspector, args.get(0));
        }
    }

    public static class Cbrt
    extends DoubleUnivariateMathFunction {
        @Override
        public String name() {
            return "cbrt";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.cbrt(param));
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.cbrt(inspector, args.get(0));
        }
    }

    public static class BitwiseXor
    extends BivariateBitwiseMathFunction {
        @Override
        public String name() {
            return "bitwiseXor";
        }

        @Override
        protected ExprEval eval(long x, long y) {
            return ExprEval.of(x ^ y);
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.bitwiseXor(inspector, args.get(0), args.get(1));
        }
    }

    public static class BitwiseShiftRight
    extends BivariateBitwiseMathFunction {
        @Override
        public String name() {
            return "bitwiseShiftRight";
        }

        @Override
        protected ExprEval eval(long x, long y) {
            return ExprEval.of(x >> (int)y);
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.bitwiseShiftRight(inspector, args.get(0), args.get(1));
        }
    }

    public static class BitwiseShiftLeft
    extends BivariateBitwiseMathFunction {
        @Override
        public String name() {
            return "bitwiseShiftLeft";
        }

        @Override
        protected ExprEval eval(long x, long y) {
            return ExprEval.of(x << (int)y);
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.bitwiseShiftLeft(inspector, args.get(0), args.get(1));
        }
    }

    public static class BitwiseOr
    extends BivariateBitwiseMathFunction {
        @Override
        public String name() {
            return "bitwiseOr";
        }

        @Override
        protected ExprEval eval(long x, long y) {
            return ExprEval.of(x | y);
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.bitwiseOr(inspector, args.get(0), args.get(1));
        }
    }

    public static class BitwiseAnd
    extends BivariateBitwiseMathFunction {
        @Override
        public String name() {
            return "bitwiseAnd";
        }

        @Override
        protected ExprEval eval(long x, long y) {
            return ExprEval.of(x & y);
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.bitwiseAnd(inspector, args.get(0), args.get(1));
        }
    }

    public static class BitwiseConvertDoubleToLongBits
    extends UnivariateMathFunction {
        @Override
        public String name() {
            return "bitwiseConvertDoubleToLongBits";
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            ExpressionType type = args.get(0).getOutputType(inspector);
            if (type == null) {
                return null;
            }
            return ExpressionType.LONG;
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Double.doubleToLongBits(param));
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.bitwiseConvertDoubleToLongBits(inspector, args.get(0));
        }
    }

    public static class BitwiseConvertLongBitsToDouble
    extends UnivariateMathFunction {
        @Override
        public String name() {
            return "bitwiseConvertLongBitsToDouble";
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            ExpressionType type = args.get(0).getOutputType(inspector);
            if (type == null) {
                return null;
            }
            return ExpressionType.DOUBLE;
        }

        @Override
        protected ExprEval eval(long param) {
            return ExprEval.of(Double.longBitsToDouble(param));
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.bitwiseConvertLongBitsToDouble(inspector, args.get(0));
        }
    }

    public static class BitwiseComplement
    extends UnivariateMathFunction {
        @Override
        public String name() {
            return "bitwiseComplement";
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionType.LONG;
        }

        @Override
        protected ExprEval eval(long param) {
            return ExprEval.of(param ^ 0xFFFFFFFFFFFFFFFFL);
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.bitwiseComplement(inspector, args.get(0));
        }
    }

    public static class Atan
    extends DoubleUnivariateMathFunction {
        @Override
        public String name() {
            return "atan";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.atan(param));
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.atan(inspector, args.get(0));
        }
    }

    public static class Asin
    extends DoubleUnivariateMathFunction {
        @Override
        public String name() {
            return "asin";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.asin(param));
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.asin(inspector, args.get(0));
        }
    }

    public static class Acos
    extends DoubleUnivariateMathFunction {
        @Override
        public String name() {
            return "acos";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.acos(param));
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.acos(inspector, args.get(0));
        }
    }

    public static class Abs
    extends UnivariateMathFunction {
        @Override
        public String name() {
            return "abs";
        }

        @Override
        protected ExprEval eval(long param) {
            return ExprEval.of(Math.abs(param));
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.abs(param));
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorMathProcessors.abs(inspector, args.get(0));
        }
    }

    public static class Pi
    implements Function {
        private static final double PI = Math.PI;

        @Override
        public String name() {
            return "pi";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            return ExprEval.of(Math.PI);
        }

        @Override
        public void validateArguments(List<Expr> args) {
            this.validationHelperCheckArgumentCount(args, 0);
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionType.DOUBLE;
        }

        @Override
        public boolean canVectorize(Expr.InputBindingInspector inspector, List<Expr> args) {
            return true;
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            return VectorProcessors.constant(Math.PI, inspector.getMaxVectorSize());
        }
    }

    public static class ParseLong
    implements Function {
        @Override
        public String name() {
            return "parse_long";
        }

        @Override
        public void validateArguments(List<Expr> args) {
            this.validationHelperCheckAnyOfArgumentCount(args, 1, 2);
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionType.LONG;
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            long retVal;
            int radix = args.size() == 1 ? 10 : args.get(1).eval(bindings).asInt();
            String input = NullHandling.nullToEmptyIfNeeded(args.get(0).eval(bindings).asString());
            if (input == null) {
                return ExprEval.ofLong(null);
            }
            try {
                retVal = radix == 16 && (input.startsWith("0x") || input.startsWith("0X")) ? Long.parseLong(input.substring(2), radix) : Long.parseLong(input, radix);
            }
            catch (NumberFormatException e) {
                return ExprEval.ofLong(null);
            }
            return ExprEval.of(retVal);
        }

        @Override
        public boolean canVectorize(Expr.InputBindingInspector inspector, List<Expr> args) {
            return (args.size() == 1 || args.get(1).isLiteral() && args.get(1).getLiteralValue() instanceof Number) && inspector.canVectorize(args);
        }

        @Override
        public <T> ExprVectorProcessor<T> asVectorProcessor(Expr.VectorInputBindingInspector inspector, List<Expr> args) {
            if (args.size() == 1 || args.get(1).isLiteral()) {
                int radix = args.size() == 1 ? 10 : ((Number)args.get(1).getLiteralValue()).intValue();
                return VectorProcessors.parseLong(inspector, args.get(0), radix);
            }
            throw Exprs.cannotVectorize(this);
        }
    }

    public static abstract class ReduceFunction
    implements Function {
        private final DoubleBinaryOperator doubleReducer;
        private final LongBinaryOperator longReducer;
        private final BinaryOperator<String> stringReducer;

        ReduceFunction(DoubleBinaryOperator doubleReducer, LongBinaryOperator longReducer, BinaryOperator<String> stringReducer) {
            this.doubleReducer = doubleReducer;
            this.longReducer = longReducer;
            this.stringReducer = stringReducer;
        }

        @Override
        public void validateArguments(List<Expr> args) {
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            ExpressionType outputType = ExpressionType.LONG;
            for (Expr expr : args) {
                outputType = ExpressionTypeConversion.function(outputType, expr.getOutputType(inspector));
            }
            return outputType;
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            if (args.isEmpty()) {
                return ExprEval.of(null);
            }
            ArrayList<ExprEval> evals = new ArrayList<ExprEval>();
            ExpressionType outputType = ExpressionType.LONG;
            for (Expr expr : args) {
                ExprEval exprEval = expr.eval(bindings);
                ExpressionType exprType = exprEval.type();
                if (this.isValidType(exprType)) {
                    outputType = ExpressionTypeConversion.function(outputType, exprType);
                }
                if (exprEval.value() == null) continue;
                evals.add(exprEval);
            }
            if (evals.isEmpty()) {
                return ExprEval.of(null);
            }
            switch ((ExprType)outputType.getType()) {
                case DOUBLE: {
                    return ExprEval.of(evals.stream().mapToDouble(ExprEval::asDouble).reduce(this.doubleReducer).getAsDouble());
                }
                case LONG: {
                    return ExprEval.of(evals.stream().mapToLong(ExprEval::asLong).reduce(this.longReducer).getAsLong());
                }
            }
            return ExprEval.of(evals.stream().map(ExprEval::asString).reduce(this.stringReducer).get());
        }

        private boolean isValidType(ExpressionType exprType) {
            switch ((ExprType)exprType.getType()) {
                case STRING: 
                case LONG: 
                case DOUBLE: {
                    return true;
                }
            }
            throw this.validationFailed("does not accept %s types", exprType);
        }
    }

    public static abstract class ArraysMergeFunction
    extends ArraysFunction {
        @Override
        public Set<Expr> getArrayInputs(List<Expr> args) {
            return ImmutableSet.copyOf(args);
        }

        @Override
        public boolean hasArrayOutput() {
            return true;
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            ExpressionType arrayType = args.get(0).getOutputType(inspector);
            return Optional.ofNullable(ExpressionType.asArrayType(arrayType)).orElse(arrayType);
        }

        @Override
        ExprEval doApply(ExprEval lhsExpr, ExprEval rhsExpr) {
            Object[] array1 = lhsExpr.asArray();
            Object[] array2 = rhsExpr.asArray();
            if (array1 == null) {
                return ExprEval.of(null);
            }
            if (array2 == null) {
                return lhsExpr;
            }
            ExpressionType arrayType = lhsExpr.asArrayType();
            if (!lhsExpr.asArrayType().equals(rhsExpr.asArrayType())) {
                ExprEval coerced = rhsExpr.castTo(arrayType);
                ExprEval.ofArray(arrayType, this.merge(arrayType.getElementType(), lhsExpr.asArray(), coerced.asArray()));
            }
            return ExprEval.ofArray(arrayType, this.merge(arrayType.getElementType(), lhsExpr.asArray(), rhsExpr.asArray()));
        }

        abstract <T> Object[] merge(TypeSignature<ExprType> var1, T[] var2, T[] var3);
    }

    public static abstract class ArrayAddElementFunction
    extends ArrayScalarFunction {
        @Override
        public boolean hasArrayOutput() {
            return true;
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            ExpressionType arrayType = this.getArrayArgument(args).getOutputType(inspector);
            return Optional.ofNullable(ExpressionType.asArrayType(arrayType)).orElse(arrayType);
        }

        @Override
        ExprEval doApply(ExprEval arrayExpr, ExprEval scalarExpr) {
            ExpressionType arrayType = arrayExpr.asArrayType();
            if (!scalarExpr.type().equals(arrayExpr.elementType())) {
                ExprEval coerced = scalarExpr.castTo(arrayExpr.elementType());
                return ExprEval.ofArray(arrayType, this.add(arrayType.getElementType(), arrayExpr.asArray(), coerced.value()));
            }
            return ExprEval.ofArray(arrayType, this.add(arrayType.getElementType(), arrayExpr.asArray(), scalarExpr.value()));
        }

        abstract <T> Object[] add(TypeSignature<ExprType> var1, T[] var2, @Nullable T var3);
    }

    public static abstract class ArraysFunction
    implements Function {
        @Override
        public void validateArguments(List<Expr> args) {
            this.validationHelperCheckArgumentCount(args, 2);
        }

        @Override
        public Set<Expr> getScalarInputs(List<Expr> args) {
            return Collections.emptySet();
        }

        @Override
        public Set<Expr> getArrayInputs(List<Expr> args) {
            return ImmutableSet.copyOf(args);
        }

        @Override
        public boolean hasArrayInputs() {
            return true;
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            ExprEval arrayExpr1 = args.get(0).eval(bindings);
            ExprEval arrayExpr2 = args.get(1).eval(bindings);
            if (arrayExpr1.asArray() == null) {
                return arrayExpr1;
            }
            if (arrayExpr2.asArray() == null) {
                return arrayExpr2;
            }
            return this.doApply(arrayExpr1, arrayExpr2);
        }

        abstract ExprEval doApply(ExprEval var1, ExprEval var2);
    }

    public static abstract class ArrayScalarFunction
    implements Function {
        @Override
        public void validateArguments(List<Expr> args) {
            this.validationHelperCheckArgumentCount(args, 2);
        }

        @Override
        public Set<Expr> getScalarInputs(List<Expr> args) {
            return ImmutableSet.of((Object)this.getScalarArgument(args));
        }

        @Override
        public Set<Expr> getArrayInputs(List<Expr> args) {
            return ImmutableSet.of((Object)this.getArrayArgument(args));
        }

        @Override
        public boolean hasArrayInputs() {
            return true;
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            ExprEval arrayExpr = this.getArrayArgument(args).eval(bindings);
            ExprEval scalarExpr = this.getScalarArgument(args).eval(bindings);
            if (arrayExpr.asArray() == null) {
                return ExprEval.of(null);
            }
            return this.doApply(arrayExpr, scalarExpr);
        }

        Expr getScalarArgument(List<Expr> args) {
            return args.get(1);
        }

        Expr getArrayArgument(List<Expr> args) {
            return args.get(0);
        }

        abstract ExprEval doApply(ExprEval var1, ExprEval var2);
    }

    public static abstract class StringLongFunction
    extends BivariateFunction {
        @Override
        protected final ExprEval eval(ExprEval x, ExprEval y) {
            if (!x.type().is(ExprType.STRING) || !y.type().is(ExprType.LONG)) {
                throw this.validationFailed("needs a STRING as first argument and a LONG as second argument", new Object[0]);
            }
            return this.eval(x.asString(), y.asInt());
        }

        protected abstract ExprEval eval(@Nullable String var1, int var2);
    }

    public static abstract class BivariateBitwiseMathFunction
    extends BivariateFunction {
        @Override
        protected final ExprEval eval(ExprEval x, ExprEval y) {
            if (NullHandling.sqlCompatible() && (x.value() == null || y.value() == null)) {
                return ExprEval.of(null);
            }
            ExpressionType type = ExpressionTypeConversion.autoDetect(x, y);
            if (type.is(ExprType.STRING)) {
                return ExprEval.of(null);
            }
            return this.eval(x.asLong(), y.asLong());
        }

        protected abstract ExprEval eval(long var1, long var3);

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionType.LONG;
        }

        @Override
        public boolean canVectorize(Expr.InputBindingInspector inspector, List<Expr> args) {
            return inspector.areNumeric(args) && inspector.canVectorize(args);
        }
    }

    public static abstract class DoubleBivariateMathFunction
    extends BivariateMathFunction {
        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionType.DOUBLE;
        }
    }

    public static abstract class BivariateMathFunction
    extends BivariateFunction {
        @Override
        protected final ExprEval eval(ExprEval x, ExprEval y) {
            if (NullHandling.sqlCompatible() && (x.value() == null || y.value() == null)) {
                return ExprEval.of(null);
            }
            ExpressionType type = ExpressionTypeConversion.autoDetect(x, y);
            switch ((ExprType)type.getType()) {
                case STRING: {
                    return ExprEval.of(null);
                }
                case LONG: {
                    return this.eval(x.asLong(), y.asLong());
                }
            }
            return this.eval(x.asDouble(), y.asDouble());
        }

        protected ExprEval eval(long x, long y) {
            return this.eval((double)x, (double)y);
        }

        protected ExprEval eval(double x, double y) {
            return this.eval((long)x, (long)y);
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionTypeConversion.function(args.get(0).getOutputType(inspector), args.get(1).getOutputType(inspector));
        }

        @Override
        public boolean canVectorize(Expr.InputBindingInspector inspector, List<Expr> args) {
            return inspector.areNumeric(args) && inspector.canVectorize(args);
        }
    }

    public static abstract class DoubleUnivariateMathFunction
    extends UnivariateMathFunction {
        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return ExpressionType.DOUBLE;
        }
    }

    public static abstract class UnivariateMathFunction
    extends UnivariateFunction {
        @Override
        protected final ExprEval eval(ExprEval param) {
            if (NullHandling.sqlCompatible() && param.isNumericNull()) {
                return ExprEval.of(null);
            }
            if (param.type().is(ExprType.LONG)) {
                return this.eval(param.asLong());
            }
            if (param.type().is(ExprType.DOUBLE)) {
                return this.eval(param.asDouble());
            }
            return ExprEval.of(null);
        }

        protected ExprEval eval(long param) {
            return this.eval((double)param);
        }

        protected ExprEval eval(double param) {
            if (param < -9.223372036854776E18 || param > 9.223372036854776E18) {
                throw this.validationFailed("Possible data truncation, param [%f] is out of LONG value range", param);
            }
            return this.eval((long)param);
        }

        @Override
        @Nullable
        public ExpressionType getOutputType(Expr.InputBindingInspector inspector, List<Expr> args) {
            return args.get(0).getOutputType(inspector);
        }

        @Override
        public boolean canVectorize(Expr.InputBindingInspector inspector, List<Expr> args) {
            return inspector.areNumeric(args) && inspector.canVectorize(args);
        }
    }

    public static abstract class BivariateFunction
    implements Function {
        @Override
        public void validateArguments(List<Expr> args) {
            this.validationHelperCheckArgumentCount(args, 2);
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            Expr expr1 = args.get(0);
            Expr expr2 = args.get(1);
            return this.eval(expr1.eval(bindings), expr2.eval(bindings));
        }

        protected abstract ExprEval eval(ExprEval var1, ExprEval var2);
    }

    public static abstract class UnivariateFunction
    implements Function {
        @Override
        public void validateArguments(List<Expr> args) {
            this.validationHelperCheckArgumentCount(args, 1);
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            Expr expr = args.get(0);
            return this.eval(expr.eval(bindings));
        }

        protected abstract ExprEval eval(ExprEval var1);
    }
}

