/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.segment.column;

import com.google.common.base.Preconditions;
import com.google.common.collect.Ordering;
import com.google.common.primitives.Floats;
import com.google.common.primitives.Longs;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Comparator;
import java.util.concurrent.ConcurrentHashMap;
import javax.annotation.Nullable;
import org.apache.druid.java.util.common.IAE;
import org.apache.druid.java.util.common.ISE;
import org.apache.druid.java.util.common.StringUtils;
import org.apache.druid.segment.column.ColumnType;
import org.apache.druid.segment.column.NullableTypeStrategy;
import org.apache.druid.segment.column.TypeSignature;
import org.apache.druid.segment.column.TypeStrategy;

public class TypeStrategies {
    public static final int VALUE_OFFSET = 1;
    public static final int NULLABLE_LONG_SIZE = 9;
    public static final int NULLABLE_DOUBLE_SIZE = 9;
    public static final int NULLABLE_FLOAT_SIZE = 5;
    public static final LongTypeStrategy LONG = new LongTypeStrategy();
    public static final FloatTypeStrategy FLOAT = new FloatTypeStrategy();
    public static final DoubleTypeStrategy DOUBLE = new DoubleTypeStrategy();
    public static final StringTypeStrategy STRING = new StringTypeStrategy();
    public static final ConcurrentHashMap<String, TypeStrategy<?>> COMPLEX_STRATEGIES = new ConcurrentHashMap();

    @Nullable
    public static TypeStrategy<?> getComplex(String typeName) {
        if (typeName == null) {
            return null;
        }
        return COMPLEX_STRATEGIES.get(typeName);
    }

    public static void registerComplex(String typeName, TypeStrategy<?> strategy) {
        Preconditions.checkNotNull((Object)typeName);
        COMPLEX_STRATEGIES.compute(typeName, (key, value) -> {
            if (value == null) {
                return strategy;
            }
            if (!value.getClass().getName().equals(strategy.getClass().getName())) {
                throw new ISE("Incompatible strategy for type[%s] already exists. Expected [%s], found [%s].", key, strategy.getClass().getName(), value.getClass().getName());
            }
            return value;
        });
    }

    public static int writeNull(ByteBuffer buffer, int offset) {
        buffer.put(offset, (byte)1);
        return 1;
    }

    public static boolean isNullableNull(ByteBuffer buffer, int offset) {
        return (buffer.get(offset) & 1) == 1;
    }

    public static int writeNotNullNullableLong(ByteBuffer buffer, int offset, long value) {
        buffer.put(offset++, (byte)0);
        buffer.putLong(offset, value);
        return 9;
    }

    public static long readNotNullNullableLong(ByteBuffer buffer, int offset) {
        return buffer.getLong(offset + 1);
    }

    public static int writeNotNullNullableDouble(ByteBuffer buffer, int offset, double value) {
        buffer.put(offset++, (byte)0);
        buffer.putDouble(offset, value);
        return 9;
    }

    public static double readNotNullNullableDouble(ByteBuffer buffer, int offset) {
        return buffer.getDouble(offset + 1);
    }

    public static int writeNotNullNullableFloat(ByteBuffer buffer, int offset, float value) {
        buffer.put(offset++, (byte)0);
        buffer.putFloat(offset, value);
        return 5;
    }

    public static float readNotNullNullableFloat(ByteBuffer buffer, int offset) {
        return buffer.getFloat(offset + 1);
    }

    public static void checkMaxSize(int available, int maxSizeBytes, TypeSignature<?> signature) {
        if (maxSizeBytes > available) {
            throw new IAE("Unable to write [%s], maxSizeBytes [%s] is greater than available [%s]", signature.asTypeString(), maxSizeBytes, available);
        }
    }

    public static final class ArrayTypeStrategy
    implements TypeStrategy<Object[]> {
        private final Comparator<Object> elementComparator;
        private final TypeSignature<?> arrayType;
        private final NullableTypeStrategy elementStrategy;

        public ArrayTypeStrategy(TypeSignature<?> type) {
            this.arrayType = type;
            this.elementStrategy = type.getElementType().getNullableStrategy();
            this.elementComparator = Comparator.nullsFirst(this.elementStrategy);
        }

        @Override
        public int estimateSizeBytes(Object[] value) {
            return 4 + Arrays.stream(value).mapToInt(this.elementStrategy::estimateSizeBytes).sum();
        }

        @Override
        public Object[] read(ByteBuffer buffer) {
            int arrayLength = buffer.getInt();
            Object[] array = new Object[arrayLength];
            for (int i = 0; i < arrayLength; ++i) {
                array[i] = this.elementStrategy.read(buffer);
            }
            return array;
        }

        @Override
        public boolean readRetainsBufferReference() {
            return this.elementStrategy.readRetainsBufferReference();
        }

        @Override
        public int write(ByteBuffer buffer, Object[] value, int maxSizeBytes) {
            TypeStrategies.checkMaxSize(buffer.remaining(), maxSizeBytes, this.arrayType);
            int sizeBytes = 4;
            int remaining = maxSizeBytes - sizeBytes;
            if (remaining < 0) {
                return remaining;
            }
            int extraNeeded = 0;
            buffer.putInt(value.length);
            for (Object o : value) {
                int written = this.elementStrategy.write(buffer, o, remaining);
                if (written < 0) {
                    extraNeeded += written;
                    remaining = 0;
                    continue;
                }
                sizeBytes += written;
                remaining -= written;
            }
            return extraNeeded < 0 ? extraNeeded : sizeBytes;
        }

        @Override
        public boolean groupable() {
            return this.elementStrategy.groupable();
        }

        @Override
        public int compare(@Nullable Object o1Obj, @Nullable Object o2Obj) {
            Object[] o1 = (Object[])o1Obj;
            Object[] o2 = (Object[])o2Obj;
            if (o1 == o2) {
                return 0;
            }
            if (o1 == null) {
                return -1;
            }
            if (o2 == null) {
                return 1;
            }
            int iter = Math.min(o1.length, o2.length);
            for (int i = 0; i < iter; ++i) {
                int cmp = this.elementComparator.compare(o1[i], o2[i]);
                if (cmp == 0) continue;
                return cmp;
            }
            return Integer.compare(o1.length, o2.length);
        }

        @Override
        public int hashCode(Object[] o) {
            if (o == null) {
                return 0;
            }
            int result = 1;
            for (Object element : o) {
                result = 31 * result + (element == null ? 0 : this.elementStrategy.hashCode(element));
            }
            return result;
        }

        @Override
        public boolean equals(@Nullable Object[] a, @Nullable Object[] b) {
            if (a == b) {
                return true;
            }
            if (a != null && b != null) {
                int length = a.length;
                if (b.length != length) {
                    return false;
                }
                for (int i = 0; i < length; ++i) {
                    if (this.elementStrategy.equals(a[i], b[i])) continue;
                    return false;
                }
                return true;
            }
            return false;
        }
    }

    public static final class StringTypeStrategy
    implements TypeStrategy<String> {
        private static final Ordering<String> ORDERING = Ordering.from(String::compareTo);

        @Override
        public int estimateSizeBytes(String value) {
            return 4 + StringUtils.toUtf8(value).length;
        }

        @Override
        public String read(ByteBuffer buffer) {
            int length = buffer.getInt();
            byte[] blob = new byte[length];
            buffer.get(blob, 0, length);
            return StringUtils.fromUtf8(blob);
        }

        @Override
        public boolean readRetainsBufferReference() {
            return false;
        }

        @Override
        public int write(ByteBuffer buffer, String value, int maxSizeBytes) {
            TypeStrategies.checkMaxSize(buffer.remaining(), maxSizeBytes, ColumnType.STRING);
            byte[] bytes = StringUtils.toUtf8(value);
            int sizeBytes = 4 + bytes.length;
            int remaining = maxSizeBytes - sizeBytes;
            if (remaining >= 0) {
                buffer.putInt(bytes.length);
                buffer.put(bytes, 0, bytes.length);
                return sizeBytes;
            }
            return remaining;
        }

        @Override
        public boolean groupable() {
            return true;
        }

        @Override
        public int compare(Object s, Object s2) {
            if (s == s2) {
                return 0;
            }
            return ORDERING.compare((Object)((String)s), (Object)((String)s2));
        }

        @Override
        public int hashCode(String o) {
            return o.hashCode();
        }

        @Override
        public boolean equals(String a, String b) {
            return a.equals(b);
        }
    }

    public static final class DoubleTypeStrategy
    implements TypeStrategy<Double> {
        @Override
        public int estimateSizeBytes(Double value) {
            return 8;
        }

        @Override
        public Double read(ByteBuffer buffer) {
            return buffer.getDouble();
        }

        @Override
        public Double read(ByteBuffer buffer, int offset) {
            return buffer.getDouble(offset);
        }

        @Override
        public boolean groupable() {
            return true;
        }

        @Override
        public boolean readRetainsBufferReference() {
            return false;
        }

        @Override
        public int write(ByteBuffer buffer, Double value, int maxSizeBytes) {
            TypeStrategies.checkMaxSize(buffer.remaining(), maxSizeBytes, ColumnType.DOUBLE);
            int sizeBytes = 8;
            int remaining = maxSizeBytes - 8;
            if (remaining >= 0) {
                buffer.putDouble(value);
                return 8;
            }
            return remaining;
        }

        @Override
        public int compare(Object o1, Object o2) {
            return Double.compare(((Number)o1).doubleValue(), ((Number)o2).doubleValue());
        }

        @Override
        public int hashCode(Double o) {
            return o.hashCode();
        }

        @Override
        public boolean equals(Double a, Double b) {
            return a.equals(b);
        }
    }

    public static final class FloatTypeStrategy
    implements TypeStrategy<Float> {
        @Override
        public int estimateSizeBytes(Float value) {
            return 4;
        }

        @Override
        public Float read(ByteBuffer buffer) {
            return Float.valueOf(buffer.getFloat());
        }

        @Override
        public Float read(ByteBuffer buffer, int offset) {
            return Float.valueOf(buffer.getFloat(offset));
        }

        @Override
        public boolean groupable() {
            return true;
        }

        @Override
        public boolean readRetainsBufferReference() {
            return false;
        }

        @Override
        public int write(ByteBuffer buffer, Float value, int maxSizeBytes) {
            TypeStrategies.checkMaxSize(buffer.remaining(), maxSizeBytes, ColumnType.FLOAT);
            int sizeBytes = 4;
            int remaining = maxSizeBytes - 4;
            if (remaining >= 0) {
                buffer.putFloat(value.floatValue());
                return 4;
            }
            return remaining;
        }

        @Override
        public int compare(Object o1, Object o2) {
            return Floats.compare((float)((Number)o1).floatValue(), (float)((Number)o2).floatValue());
        }

        @Override
        public int hashCode(Float o) {
            return o.hashCode();
        }

        @Override
        public boolean equals(Float a, Float b) {
            return a.equals(b);
        }
    }

    public static final class LongTypeStrategy
    implements TypeStrategy<Long> {
        @Override
        public int estimateSizeBytes(Long value) {
            return 8;
        }

        @Override
        public Long read(ByteBuffer buffer) {
            return buffer.getLong();
        }

        @Override
        public Long read(ByteBuffer buffer, int offset) {
            return buffer.getLong(offset);
        }

        @Override
        public boolean groupable() {
            return true;
        }

        @Override
        public boolean readRetainsBufferReference() {
            return false;
        }

        @Override
        public int write(ByteBuffer buffer, Long value, int maxSizeBytes) {
            TypeStrategies.checkMaxSize(buffer.remaining(), maxSizeBytes, ColumnType.LONG);
            int sizeBytes = 8;
            int remaining = maxSizeBytes - 8;
            if (remaining >= 0) {
                buffer.putLong(value);
                return 8;
            }
            return remaining;
        }

        @Override
        public int compare(Object o1, Object o2) {
            return Longs.compare((long)((Number)o1).longValue(), (long)((Number)o2).longValue());
        }

        @Override
        public int hashCode(Long o) {
            return o.hashCode();
        }

        @Override
        public boolean equals(Long a, Long b) {
            return a.equals(b);
        }
    }
}

