/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.math.expr;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import org.apache.druid.java.util.common.RE;
import org.apache.druid.java.util.common.StringUtils;
import org.apache.druid.math.expr.BinPlusExpr;
import org.apache.druid.math.expr.Expr;
import org.apache.druid.math.expr.ExprMacroTable;
import org.apache.druid.math.expr.ExpressionType;
import org.apache.druid.math.expr.IdentifierExpr;
import org.apache.druid.math.expr.InputBindings;
import org.apache.druid.math.expr.Parser;
import org.apache.druid.math.expr.VectorExprSanityTest;
import org.apache.druid.segment.column.TypeStrategies;
import org.apache.druid.segment.column.TypeStrategiesTest;
import org.apache.druid.segment.column.TypeStrategy;
import org.apache.druid.testing.InitializedNullHandlingTest;
import org.hamcrest.CoreMatchers;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

public class ParserTest
extends InitializedNullHandlingTest {
    @Rule
    public ExpectedException expectedException = ExpectedException.none();
    VectorExprSanityTest.SettableVectorInputBinding emptyBinding = new VectorExprSanityTest.SettableVectorInputBinding(8);

    @BeforeClass
    public static void setup() {
        TypeStrategies.registerComplex((String)TypeStrategiesTest.NULLABLE_TEST_PAIR_TYPE.getComplexTypeName(), (TypeStrategy)new TypeStrategiesTest.NullableLongPairTypeStrategy());
    }

    @Test
    public void testSimple() {
        String actual = Parser.parse((String)"1", (ExprMacroTable)ExprMacroTable.nil()).toString();
        String expected = "1";
        Assert.assertEquals((Object)expected, (Object)actual);
    }

    @Test
    public void testParseConstants() {
        this.validateLiteral("null", null, null);
        this.validateLiteral("'hello'", ExpressionType.STRING, "hello");
        this.validateLiteral("'hello \\uD83E\\uDD18'", ExpressionType.STRING, "hello \ud83e\udd18");
        this.validateLiteral("1", ExpressionType.LONG, BigInteger.valueOf(1L));
        this.validateLiteral(String.valueOf(Long.MAX_VALUE), ExpressionType.LONG, BigInteger.valueOf(Long.MAX_VALUE));
        this.validateLiteral("1.", ExpressionType.DOUBLE, 1.0, false);
        this.validateLiteral("1.234", ExpressionType.DOUBLE, 1.234);
        this.validateLiteral("1e10", ExpressionType.DOUBLE, 1.0E10, false);
        this.validateLiteral("1e-10", ExpressionType.DOUBLE, 1.0E-10, false);
        this.validateLiteral("1E10", ExpressionType.DOUBLE, 1.0E10, false);
        this.validateLiteral("1E-10", ExpressionType.DOUBLE, 1.0E-10, false);
        this.validateLiteral("1.E10", ExpressionType.DOUBLE, 1.0E10, false);
        this.validateLiteral("1.E-10", ExpressionType.DOUBLE, 1.0E-10, false);
        this.validateLiteral("1.e10", ExpressionType.DOUBLE, 1.0E10, false);
        this.validateLiteral("1.e-10", ExpressionType.DOUBLE, 1.0E-10, false);
        this.validateLiteral("1.1e10", ExpressionType.DOUBLE, 1.1E10, false);
        this.validateLiteral("1.1e-10", ExpressionType.DOUBLE, 1.1E-10, false);
        this.validateLiteral("1.1E10", ExpressionType.DOUBLE, 1.1E10);
        this.validateLiteral("1.1E-10", ExpressionType.DOUBLE, 1.1E-10);
        this.validateLiteral("Infinity", ExpressionType.DOUBLE, Double.POSITIVE_INFINITY);
        this.validateLiteral("NaN", ExpressionType.DOUBLE, Double.NaN);
    }

    @Test
    public void testParseOutOfRangeLong() {
        String s = "9223372036854775809";
        Expr expr = Parser.parse((String)"9223372036854775809", (ExprMacroTable)ExprMacroTable.nil(), (boolean)false);
        ArithmeticException e = (ArithmeticException)Assert.assertThrows(ArithmeticException.class, () -> expr.eval(InputBindings.nilBindings()));
        MatcherAssert.assertThat((Object)e.getMessage(), (Matcher)CoreMatchers.containsString((String)"BigInteger out of long range"));
        ArithmeticException e2 = (ArithmeticException)Assert.assertThrows(ArithmeticException.class, () -> Parser.parse((String)"9223372036854775809", (ExprMacroTable)ExprMacroTable.nil(), (boolean)true));
        MatcherAssert.assertThat((Object)e2.getMessage(), (Matcher)CoreMatchers.containsString((String)"BigInteger out of long range"));
    }

    @Test
    public void testFlattenBinaryOpConstantConstant() {
        Expr expr = Parser.parse((String)"(2 + -3)", (ExprMacroTable)ExprMacroTable.nil(), (boolean)true);
        Assert.assertTrue((boolean)expr.isLiteral());
        Assert.assertEquals((Object)-1L, (Object)expr.getLiteralValue());
    }

    @Test
    public void testFlattenBinaryOpIdentifierConstant() {
        Expr expr = Parser.parse((String)"(s + -3)", (ExprMacroTable)ExprMacroTable.nil(), (boolean)true);
        Assert.assertFalse((boolean)expr.isLiteral());
        MatcherAssert.assertThat((Object)expr, (Matcher)CoreMatchers.instanceOf(BinPlusExpr.class));
        Expr right = ((BinPlusExpr)expr).right;
        Assert.assertTrue((boolean)right.isLiteral());
        Assert.assertEquals((Object)-3L, (Object)right.getLiteralValue());
    }

    @Test
    public void testSimpleUnaryOps1() {
        String actual = Parser.parse((String)"-x", (ExprMacroTable)ExprMacroTable.nil()).toString();
        String expected = "-x";
        Assert.assertEquals((Object)expected, (Object)actual);
        actual = Parser.parse((String)"!x", (ExprMacroTable)ExprMacroTable.nil()).toString();
        expected = "!x";
        Assert.assertEquals((Object)expected, (Object)actual);
    }

    @Test
    public void testSimpleUnaryOps2() {
        this.validateFlatten(String.valueOf(Long.MIN_VALUE), String.valueOf(Long.MIN_VALUE), String.valueOf(Long.MIN_VALUE));
        this.validateFlatten("-1", "-1", "-1");
        this.validateFlatten("--1", "--1", "1");
        this.validateFlatten("-1+2", "(+ -1 2)", "1");
        this.validateFlatten("-1*2", "(* -1 2)", "-2");
        this.validateFlatten("-1^2", "(^ -1 2)", "1");
    }

    @Test
    public void testSimpleLogicalOps1() {
        this.validateParser("x>y", "(> x y)", (List<String>)ImmutableList.of((Object)"x", (Object)"y"));
        this.validateParser("x<y", "(< x y)", (List<String>)ImmutableList.of((Object)"x", (Object)"y"));
        this.validateParser("x<=y", "(<= x y)", (List<String>)ImmutableList.of((Object)"x", (Object)"y"));
        this.validateParser("x>=y", "(>= x y)", (List<String>)ImmutableList.of((Object)"x", (Object)"y"));
        this.validateParser("x==y", "(== x y)", (List<String>)ImmutableList.of((Object)"x", (Object)"y"));
        this.validateParser("x!=y", "(!= x y)", (List<String>)ImmutableList.of((Object)"x", (Object)"y"));
        this.validateParser("x && y", "(&& x y)", (List<String>)ImmutableList.of((Object)"x", (Object)"y"));
        this.validateParser("x || y", "(|| x y)", (List<String>)ImmutableList.of((Object)"x", (Object)"y"));
    }

    @Test
    public void testSimpleAdditivityOp1() {
        this.validateParser("x+y", "(+ x y)", (List<String>)ImmutableList.of((Object)"x", (Object)"y"));
        this.validateParser("x-y", "(- x y)", (List<String>)ImmutableList.of((Object)"x", (Object)"y"));
    }

    @Test
    public void testSimpleAdditivityOp2() {
        this.validateParser("x+y+z", "(+ (+ x y) z)", (List<String>)ImmutableList.of((Object)"x", (Object)"y", (Object)"z"));
        this.validateParser("x+y-z", "(- (+ x y) z)", (List<String>)ImmutableList.of((Object)"x", (Object)"y", (Object)"z"));
        this.validateParser("x-y+z", "(+ (- x y) z)", (List<String>)ImmutableList.of((Object)"x", (Object)"y", (Object)"z"));
        this.validateParser("x-y-z", "(- (- x y) z)", (List<String>)ImmutableList.of((Object)"x", (Object)"y", (Object)"z"));
        this.validateParser("x-y-x", "(- (- x y) x)", (List<String>)ImmutableList.of((Object)"x", (Object)"y"), (Set<String>)ImmutableSet.of((Object)"x", (Object)"x_0", (Object)"y"));
    }

    @Test
    public void testSimpleMultiplicativeOp1() {
        this.validateParser("x*y", "(* x y)", (List<String>)ImmutableList.of((Object)"x", (Object)"y"));
        this.validateParser("x/y", "(/ x y)", (List<String>)ImmutableList.of((Object)"x", (Object)"y"));
        this.validateParser("x%y", "(% x y)", (List<String>)ImmutableList.of((Object)"x", (Object)"y"));
    }

    @Test
    public void testSimpleMultiplicativeOp2() {
        this.validateFlatten("1*2*3", "(* (* 1 2) 3)", "6");
        this.validateFlatten("1*2/3", "(/ (* 1 2) 3)", "0");
        this.validateFlatten("1/2*3", "(* (/ 1 2) 3)", "0");
        this.validateFlatten("1/2/3", "(/ (/ 1 2) 3)", "0");
        this.validateFlatten("1.0*2*3", "(* (* 1.0 2) 3)", "6.0");
        this.validateFlatten("1.0*2/3", "(/ (* 1.0 2) 3)", "0.6666666666666666");
        this.validateFlatten("1.0/2*3", "(* (/ 1.0 2) 3)", "1.5");
        this.validateFlatten("1.0/2/3", "(/ (/ 1.0 2) 3)", "0.16666666666666666");
        this.validateFlatten("1.0*2*x", "(* (* 1.0 2) x)", "(* 2.0 x)");
        this.validateFlatten("1.0*2/x", "(/ (* 1.0 2) x)", "(/ 2.0 x)");
        this.validateFlatten("1.0/2*x", "(* (/ 1.0 2) x)", "(* 0.5 x)");
        this.validateFlatten("1.0/2/x", "(/ (/ 1.0 2) x)", "(/ 0.5 x)");
        this.validateFlatten("1.0*x*3", "(* (* 1.0 x) 3)", "(* (* 1.0 x) 3)");
    }

    @Test
    public void testSimpleCarrot1() {
        this.validateFlatten("1^2", "(^ 1 2)", "1");
    }

    @Test
    public void testSimpleCarrot2() {
        this.validateFlatten("1^2^3", "(^ 1 (^ 2 3))", "1");
    }

    @Test
    public void testMixed() {
        this.validateFlatten("1+2*3", "(+ 1 (* 2 3))", "7");
        this.validateFlatten("1+(2*3)", "(+ 1 (* 2 3))", "7");
        this.validateFlatten("(1+2)*3", "(* (+ 1 2) 3)", "9");
        this.validateFlatten("1*2+3", "(+ (* 1 2) 3)", "5");
        this.validateFlatten("(1*2)+3", "(+ (* 1 2) 3)", "5");
        this.validateFlatten("1*(2+3)", "(* 1 (+ 2 3))", "5");
        this.validateFlatten("1+2^3", "(+ 1 (^ 2 3))", "9");
        this.validateFlatten("1+(2^3)", "(+ 1 (^ 2 3))", "9");
        this.validateFlatten("(1+2)^3", "(^ (+ 1 2) 3)", "27");
        this.validateFlatten("1^2+3", "(+ (^ 1 2) 3)", "4");
        this.validateFlatten("(1^2)+3", "(+ (^ 1 2) 3)", "4");
        this.validateFlatten("1^(2+3)", "(^ 1 (+ 2 3))", "1");
        this.validateFlatten("1^2*3+4", "(+ (* (^ 1 2) 3) 4)", "7");
        this.validateFlatten("-1^2*-3+-4", "(+ (* (^ -1 2) -3) -4)", "-7");
        this.validateFlatten("max(3, 4)", "(max [3, 4])", "4");
        this.validateFlatten("min(1, max(3, 4))", "(min [1, (max [3, 4])])", "1");
    }

    @Test
    public void testIdentifiers() {
        this.validateParser("foo", "foo", (List<String>)ImmutableList.of((Object)"foo"), (Set<String>)ImmutableSet.of());
        this.validateParser("\"foo\"", "foo", (List<String>)ImmutableList.of((Object)"foo"), (Set<String>)ImmutableSet.of());
        this.validateParser("\"foo bar\"", "foo bar", (List<String>)ImmutableList.of((Object)"foo bar"), (Set<String>)ImmutableSet.of());
        this.validateParser("\"foo\\\"bar\"", "foo\"bar", (List<String>)ImmutableList.of((Object)"foo\"bar"), (Set<String>)ImmutableSet.of());
    }

    @Test
    public void testLiterals() {
        this.validateConstantExpression("'foo'", "foo");
        this.validateConstantExpression("'foo bar'", "foo bar");
        this.validateConstantExpression("'f\u00f6o bar'", "f\u00f6o bar");
        this.validateConstantExpression("'f\\u0040o bar'", "f@o bar");
        this.validateConstantExpression("'f\\u000Ao \\'b\\\\\\\"ar'", "f\no 'b\\\"ar");
    }

    @Test
    public void testLiteralArraysHomogeneousElements() {
        this.validateConstantExpression("[1.0, 2.345]", new Object[]{1.0, 2.345});
        this.validateConstantExpression("[1, 3]", new Object[]{1L, 3L});
        this.validateConstantExpression("['hello', 'world']", new Object[]{"hello", "world"});
    }

    @Test
    public void testLiteralArraysHomogeneousOrNullElements() {
        this.validateConstantExpression("[1.0, null, 2.345]", new Object[]{1.0, null, 2.345});
        this.validateConstantExpression("[null, 1, 3]", new Object[]{null, 1L, 3L});
        this.validateConstantExpression("['hello', 'world', null]", new Object[]{"hello", "world", null});
    }

    @Test
    public void testLiteralArraysEmptyAndAllNullImplicitAreString() {
        this.validateConstantExpression("[]", new Object[0]);
        this.validateConstantExpression("[null, null, null]", new Object[]{null, null, null});
    }

    @Test
    public void testLiteralArraysImplicitTypedNumericMixed() {
        this.validateConstantExpression("[1, null, 2000.0]", new Object[]{1.0, null, 2000.0});
        this.validateConstantExpression("[1.0, null, 2000]", new Object[]{1.0, null, 2000.0});
    }

    @Test
    public void testLiteralArraysExplicitTypedEmpties() {
        this.validateConstantExpression("ARRAY<STRING>[]", new Object[0]);
        this.validateConstantExpression("ARRAY<DOUBLE>[]", new Object[0]);
        this.validateConstantExpression("ARRAY<LONG>[]", new Object[0]);
    }

    @Test
    public void testLiteralArraysExplicitAllNull() {
        this.validateConstantExpression("ARRAY<DOUBLE>[null, null, null]", new Object[]{null, null, null});
        this.validateConstantExpression("ARRAY<LONG>[null, null, null]", new Object[]{null, null, null});
        this.validateConstantExpression("ARRAY<STRING>[null, null, null]", new Object[]{null, null, null});
    }

    @Test
    public void testLiteralArraysExplicitTypes() {
        this.validateConstantExpression("ARRAY<DOUBLE>[1.0, null, 2000.0]", new Object[]{1.0, null, 2000.0});
        this.validateConstantExpression("ARRAY<LONG>[3, null, 4]", new Object[]{3L, null, 4L});
        this.validateConstantExpression("ARRAY<STRING>['foo', 'bar', 'baz']", new Object[]{"foo", "bar", "baz"});
    }

    @Test
    public void testLiteralArraysExplicitTypesMixedElements() {
        this.validateConstantExpression("ARRAY<DOUBLE>[3, null, 4, 2.345]", new Object[]{3.0, null, 4.0, 2.345});
        this.validateConstantExpression("ARRAY<LONG>[1.0, null, 2000.0]", new Object[]{1L, null, 2000L});
        this.validateConstantExpression("ARRAY<STRING>['1', null, 2000, 1.1]", new Object[]{"1", null, "2000", "1.1"});
    }

    @Test
    public void testLiteralExplicitTypedArrays() {
        this.validateConstantExpression("ARRAY<DOUBLE>[1.0, 2.0, null, 3.0]", new Object[]{1.0, 2.0, null, 3.0});
        this.validateConstantExpression("ARRAY<LONG>[1, 2, null, 3]", new Object[]{1L, 2L, null, 3L});
        this.validateConstantExpression("ARRAY<STRING>['1', '2', null, '3.0']", new Object[]{"1", "2", null, "3.0"});
        this.validateConstantExpression("ARRAY<DOUBLE>[3, null, 4, 2.345]", new Object[]{3.0, null, 4.0, 2.345});
        this.validateConstantExpression("ARRAY<LONG>[1.0, null, 2000.0]", new Object[]{1L, null, 2000L});
        this.validateConstantExpression("ARRAY<STRING>['1', null, 2000, 1.1]", new Object[]{"1", null, "2000", "1.1"});
        this.validateConstantExpression("ARRAY<LONG>['1', null, 2000, 1.1]", new Object[]{1L, null, 2000L, 1L});
        this.validateConstantExpression("ARRAY<DOUBLE>['1', null, 2000, 1.1]", new Object[]{1.0, null, 2000.0, 1.1});
        this.validateConstantExpression("ARRAY<COMPLEX<nullableLongPair>>[]", new Object[0]);
        this.validateConstantExpression("ARRAY<ARRAY<LONG>>[]", new Object[0]);
    }

    @Test
    public void testConstantComplexAndNestedArrays() {
        this.validateConstantExpression("array(['foo', 'bar', 'baz'], ['baz','foo','bar'])", new Object[]{new Object[]{"foo", "bar", "baz"}, new Object[]{"baz", "foo", "bar"}});
        this.validateConstantExpression("array(['foo', 'bar', 'baz'], ARRAY<LONG>[1,2,3])", new Object[]{new Object[]{"foo", "bar", "baz"}, new Object[]{"1", "2", "3"}});
        TypeStrategiesTest.NullableLongPair l1 = new TypeStrategiesTest.NullableLongPair(1L, 2L);
        TypeStrategiesTest.NullableLongPair l2 = new TypeStrategiesTest.NullableLongPair(2L, 3L);
        TypeStrategy byteStrategy = TypeStrategiesTest.NULLABLE_TEST_PAIR_TYPE.getStrategy();
        byte[] b1 = new byte[byteStrategy.estimateSizeBytes((Object)l1)];
        byte[] b2 = new byte[byteStrategy.estimateSizeBytes((Object)l2)];
        ByteBuffer bb1 = ByteBuffer.wrap(b1);
        ByteBuffer bb2 = ByteBuffer.wrap(b2);
        int w1 = byteStrategy.write(bb1, (Object)l1, b1.length);
        int w2 = byteStrategy.write(bb2, (Object)l2, b2.length);
        Assert.assertTrue((w1 > 0 ? 1 : 0) != 0);
        Assert.assertTrue((w2 > 0 ? 1 : 0) != 0);
        String l1String = StringUtils.format((String)"complex_decode_base64('%s', '%s')", (Object[])new Object[]{TypeStrategiesTest.NULLABLE_TEST_PAIR_TYPE.getComplexTypeName(), StringUtils.encodeBase64String((byte[])b1)});
        String l2String = StringUtils.format((String)"complex_decode_base64('%s', '%s')", (Object[])new Object[]{TypeStrategiesTest.NULLABLE_TEST_PAIR_TYPE.getComplexTypeName(), StringUtils.encodeBase64String((byte[])b2)});
        this.validateConstantExpression(l1String, l1);
        this.validateConstantExpression(StringUtils.format((String)"array(%s,%s)", (Object[])new Object[]{l1String, l2String}), new Object[]{l1, l2});
    }

    @Test
    public void testLiteralArrayImplicitStringParseException() {
        this.expectedException.expect(RE.class);
        this.expectedException.expectMessage("Failed to parse array: element 2000 is not a string");
        this.validateConstantExpression("['1', null, 2000, 1.1]", new Object[]{"1", null, "2000", "1.1"});
    }

    @Test
    public void testLiteralArraysExplicitLongParseException() {
        this.expectedException.expect(RE.class);
        this.expectedException.expectMessage("Failed to parse array element '2000' as a long");
        this.validateConstantExpression("<LONG>[1, null, '2000']", new Object[]{1L, null, 2000L});
    }

    @Test
    public void testLiteralArraysExplicitDoubleParseException() {
        this.expectedException.expect(RE.class);
        this.expectedException.expectMessage("Failed to parse array element '2000.0' as a double");
        this.validateConstantExpression("<DOUBLE>[1.0, null, '2000.0']", new Object[]{1.0, null, 2000.0});
    }

    @Test
    public void testFunctions() {
        this.validateParser("sqrt(x)", "(sqrt [x])", (List<String>)ImmutableList.of((Object)"x"));
        this.validateParser("if(cond,then,else)", "(if [cond, then, else])", (List<String>)ImmutableList.of((Object)"then", (Object)"cond", (Object)"else"));
        this.validateParser("cast(x, 'STRING')", "(cast [x, STRING])", (List<String>)ImmutableList.of((Object)"x"));
        this.validateParser("cast(x, 'LONG')", "(cast [x, LONG])", (List<String>)ImmutableList.of((Object)"x"));
        this.validateParser("cast(x, 'DOUBLE')", "(cast [x, DOUBLE])", (List<String>)ImmutableList.of((Object)"x"));
        this.validateParser("cast(x, 'STRING_ARRAY')", "(cast [x, STRING_ARRAY])", (List<String>)ImmutableList.of((Object)"x"), (Set<String>)ImmutableSet.of(), (Set<String>)ImmutableSet.of((Object)"x"));
        this.validateParser("cast(x, 'LONG_ARRAY')", "(cast [x, LONG_ARRAY])", (List<String>)ImmutableList.of((Object)"x"), (Set<String>)ImmutableSet.of(), (Set<String>)ImmutableSet.of((Object)"x"));
        this.validateParser("cast(x, 'DOUBLE_ARRAY')", "(cast [x, DOUBLE_ARRAY])", (List<String>)ImmutableList.of((Object)"x"), (Set<String>)ImmutableSet.of(), (Set<String>)ImmutableSet.of((Object)"x"));
        this.validateParser("array_length(x)", "(array_length [x])", (List<String>)ImmutableList.of((Object)"x"), (Set<String>)ImmutableSet.of(), (Set<String>)ImmutableSet.of((Object)"x"));
        this.validateParser("array_concat(x, y)", "(array_concat [x, y])", (List<String>)ImmutableList.of((Object)"x", (Object)"y"), (Set<String>)ImmutableSet.of(), (Set<String>)ImmutableSet.of((Object)"x", (Object)"y"));
        this.validateParser("array_append(x, y)", "(array_append [x, y])", (List<String>)ImmutableList.of((Object)"x", (Object)"y"), (Set<String>)ImmutableSet.of((Object)"y"), (Set<String>)ImmutableSet.of((Object)"x"));
        this.validateFlatten("sqrt(4)", "(sqrt [4])", "2.0");
        this.validateFlatten("array_concat([1, 2], [3, 4])", "(array_concat [[1, 2], [3, 4]])", "[1, 2, 3, 4]");
    }

    @Test
    public void testApplyFunctions() {
        this.validateParser("map((x) -> 1, x)", "(map ([x] -> 1), [x])", (List<String>)ImmutableList.of((Object)"x"), (Set<String>)ImmutableSet.of(), (Set<String>)ImmutableSet.of((Object)"x"));
        this.validateParser("map((x) -> x + 1, x)", "(map ([x] -> (+ x 1)), [x])", (List<String>)ImmutableList.of((Object)"x"), (Set<String>)ImmutableSet.of(), (Set<String>)ImmutableSet.of((Object)"x"));
        this.validateParser("x + map((x) -> x + 1, y)", "(+ x (map ([x] -> (+ x 1)), [y]))", (List<String>)ImmutableList.of((Object)"x", (Object)"y"), (Set<String>)ImmutableSet.of((Object)"x"), (Set<String>)ImmutableSet.of((Object)"y"));
        this.validateParser("x + map((x) -> x + 1, x)", "(+ x (map ([x] -> (+ x 1)), [x]))", (List<String>)ImmutableList.of((Object)"x"), (Set<String>)ImmutableSet.of((Object)"x"), (Set<String>)ImmutableSet.of((Object)"x_0"));
        this.validateParser("map((x) -> concat(x, y), z)", "(map ([x] -> (concat [x, y])), [z])", (List<String>)ImmutableList.of((Object)"y", (Object)"z"), (Set<String>)ImmutableSet.of((Object)"y"), (Set<String>)ImmutableSet.of((Object)"z"));
        this.validateParser("fold((x, acc) -> acc + x, x, y)", "(fold ([x, acc] -> (+ acc x)), [x, y])", (List<String>)ImmutableList.of((Object)"x", (Object)"y"), (Set<String>)ImmutableSet.of(), (Set<String>)ImmutableSet.of((Object)"x"));
        this.validateParser("fold((x, acc) -> acc + x, map((x) -> x + 1, x), y)", "(fold ([x, acc] -> (+ acc x)), [(map ([x] -> (+ x 1)), [x]), y])", (List<String>)ImmutableList.of((Object)"x", (Object)"y"), (Set<String>)ImmutableSet.of(), (Set<String>)ImmutableSet.of((Object)"x"));
        this.validateParser("array_append(z, fold((x, acc) -> acc + x, map((x) -> x + 1, x), y))", "(array_append [z, (fold ([x, acc] -> (+ acc x)), [(map ([x] -> (+ x 1)), [x]), y])])", (List<String>)ImmutableList.of((Object)"x", (Object)"y", (Object)"z"), (Set<String>)ImmutableSet.of(), (Set<String>)ImmutableSet.of((Object)"x", (Object)"z"));
        this.validateParser("map(z -> z + 1, array_append(z, fold((x, acc) -> acc + x, map((x) -> x + 1, x), y)))", "(map ([z] -> (+ z 1)), [(array_append [z, (fold ([x, acc] -> (+ acc x)), [(map ([x] -> (+ x 1)), [x]), y])])])", (List<String>)ImmutableList.of((Object)"x", (Object)"y", (Object)"z"), (Set<String>)ImmutableSet.of(), (Set<String>)ImmutableSet.of((Object)"x", (Object)"z"));
        this.validateParser("array_append(map(z -> z + 1, array_append(z, fold((x, acc) -> acc + x, map((x) -> x + 1, x), y))), a)", "(array_append [(map ([z] -> (+ z 1)), [(array_append [z, (fold ([x, acc] -> (+ acc x)), [(map ([x] -> (+ x 1)), [x]), y])])]), a])", (List<String>)ImmutableList.of((Object)"x", (Object)"y", (Object)"a", (Object)"z"), (Set<String>)ImmutableSet.of((Object)"a"), (Set<String>)ImmutableSet.of((Object)"x", (Object)"z"));
        this.validateFlatten("map((x) -> x + 1, [1, 2, 3, 4])", "(map ([x] -> (+ x 1)), [[1, 2, 3, 4]])", "[2, 3, 4, 5]");
        this.validateFlatten("map((x) -> x + z, [1, 2, 3, 4])", "(map ([x] -> (+ x z)), [[1, 2, 3, 4]])", "(map ([x] -> (+ x z)), [[1, 2, 3, 4]])");
    }

    @Test
    public void testApplyUnapplied() {
        this.validateApplyUnapplied("x + 1", "(+ x 1)", "(+ x 1)", (List<String>)ImmutableList.of());
        this.validateApplyUnapplied("x + 1", "(+ x 1)", "(+ x 1)", (List<String>)ImmutableList.of((Object)"z"));
        this.validateApplyUnapplied("x + y", "(+ x y)", "(map ([x] -> (+ x y)), [x])", (List<String>)ImmutableList.of((Object)"x"));
        this.validateApplyUnapplied("x + y", "(+ x y)", "(cartesian_map ([x, y] -> (+ x y)), [x, y])", (List<String>)ImmutableList.of((Object)"x", (Object)"y"));
        this.validateApplyUnapplied("map(x -> x + y, x)", "(map ([x] -> (+ x y)), [x])", "(cartesian_map ([x, y] -> (+ x y)), [x, y])", (List<String>)ImmutableList.of((Object)"y"));
        this.validateApplyUnapplied("map(x -> x + 1, x + 1)", "(map ([x] -> (+ x 1)), [(+ x 1)])", "(map ([x] -> (+ x 1)), [(map ([x] -> (+ x 1)), [x])])", (List<String>)ImmutableList.of((Object)"x"));
        this.validateApplyUnapplied("fold((x, acc) -> acc + x + y, x, 0)", "(fold ([x, acc] -> (+ (+ acc x) y)), [x, 0])", "(cartesian_fold ([x, y, acc] -> (+ (+ acc x) y)), [x, y, 0])", (List<String>)ImmutableList.of((Object)"y"));
        this.validateApplyUnapplied("z + fold((x, acc) -> acc + x + y, x, 0)", "(+ z (fold ([x, acc] -> (+ (+ acc x) y)), [x, 0]))", "(+ z (cartesian_fold ([x, y, acc] -> (+ (+ acc x) y)), [x, y, 0]))", (List<String>)ImmutableList.of((Object)"y"));
        this.validateApplyUnapplied("z + fold((x, acc) -> acc + x + y, x, 0)", "(+ z (fold ([x, acc] -> (+ (+ acc x) y)), [x, 0]))", "(map ([z] -> (+ z (cartesian_fold ([x, y, acc] -> (+ (+ acc x) y)), [x, y, 0]))), [z])", (List<String>)ImmutableList.of((Object)"y", (Object)"z"));
        this.validateApplyUnapplied("array_to_string(concat(x, 'hello'), ',')", "(array_to_string [(concat [x, hello]), ,])", "(array_to_string [(map ([x] -> (concat [x, hello])), [x]), ,])", (List<String>)ImmutableList.of((Object)"x", (Object)"y"));
        this.validateApplyUnapplied("cast(x, 'LONG')", "(cast [x, LONG])", "(map ([x] -> (cast [x, LONG])), [x])", (List<String>)ImmutableList.of((Object)"x"));
        this.validateApplyUnapplied("cartesian_map((x,y) -> x + y, x, y)", "(cartesian_map ([x, y] -> (+ x y)), [x, y])", "(cartesian_map ([x, y] -> (+ x y)), [x, y])", (List<String>)ImmutableList.of((Object)"y"));
        this.validateApplyUnapplied("cast(x, 'LONG_ARRAY')", "(cast [x, LONG_ARRAY])", "(cast [x, LONG_ARRAY])", (List<String>)ImmutableList.of((Object)"x"));
        this.validateApplyUnapplied("case_searched((x == 'b'),'b',(x == 'g'),'g','Other')", "(case_searched [(== x b), b, (== x g), g, Other])", "(map ([x] -> (case_searched [(== x b), b, (== x g), g, Other])), [x])", (List<String>)ImmutableList.of((Object)"x"));
        this.validateApplyUnapplied("array_overlap(nvl(x, 'other'), ['a', 'b', 'other'])", "(array_overlap [(nvl [x, other]), [a, b, other]])", "(array_overlap [(map ([x] -> (nvl [x, other])), [x]), [a, b, other]])", (List<String>)ImmutableList.of((Object)"x"));
    }

    @Test
    public void testFoldUnapplied() {
        this.validateFoldUnapplied("x + __acc", "(+ x __acc)", "(+ x __acc)", (List<String>)ImmutableList.of(), "__acc");
        this.validateFoldUnapplied("x + __acc", "(+ x __acc)", "(+ x __acc)", (List<String>)ImmutableList.of((Object)"z"), "__acc");
        this.validateFoldUnapplied("x + __acc", "(+ x __acc)", "(fold ([x, __acc] -> (+ x __acc)), [x, __acc])", (List<String>)ImmutableList.of((Object)"x"), "__acc");
        this.validateFoldUnapplied("x + y + __acc", "(+ (+ x y) __acc)", "(cartesian_fold ([x, y, __acc] -> (+ (+ x y) __acc)), [x, y, __acc])", (List<String>)ImmutableList.of((Object)"x", (Object)"y"), "__acc");
        this.validateFoldUnapplied("__acc + z + fold((x, acc) -> acc + x + y, x, 0)", "(+ (+ __acc z) (fold ([x, acc] -> (+ (+ acc x) y)), [x, 0]))", "(fold ([z, __acc] -> (+ (+ __acc z) (fold ([x, acc] -> (+ (+ acc x) y)), [x, 0]))), [z, __acc])", (List<String>)ImmutableList.of((Object)"z"), "__acc");
        this.validateFoldUnapplied("__acc + z + fold((x, acc) -> acc + x + y, x, 0)", "(+ (+ __acc z) (fold ([x, acc] -> (+ (+ acc x) y)), [x, 0]))", "(fold ([z, __acc] -> (+ (+ __acc z) (cartesian_fold ([x, y, acc] -> (+ (+ acc x) y)), [x, y, 0]))), [z, __acc])", (List<String>)ImmutableList.of((Object)"y", (Object)"z"), "__acc");
        this.validateFoldUnapplied("__acc + fold((x, acc) -> x + y + acc, x, __acc)", "(+ __acc (fold ([x, acc] -> (+ (+ x y) acc)), [x, __acc]))", "(+ __acc (cartesian_fold ([x, y, acc] -> (+ (+ x y) acc)), [x, y, __acc]))", (List<String>)ImmutableList.of((Object)"y"), "__acc");
    }

    @Test
    public void testUniquify() {
        this.validateParser("x-x", "(- x x)", (List<String>)ImmutableList.of((Object)"x"), (Set<String>)ImmutableSet.of((Object)"x", (Object)"x_0"));
        this.validateParser("x - x + x", "(+ (- x x) x)", (List<String>)ImmutableList.of((Object)"x"), (Set<String>)ImmutableSet.of((Object)"x", (Object)"x_0", (Object)"x_1"));
        this.validateParser("map((x) -> x + x, x)", "(map ([x] -> (+ x x)), [x])", (List<String>)ImmutableList.of((Object)"x"), (Set<String>)ImmutableSet.of(), (Set<String>)ImmutableSet.of((Object)"x"));
        this.validateApplyUnapplied("x + x", "(+ x x)", "(map ([x] -> (+ x x)), [x])", (List<String>)ImmutableList.of((Object)"x"));
        this.validateApplyUnapplied("x + x + x", "(+ (+ x x) x)", "(map ([x] -> (+ (+ x x) x)), [x])", (List<String>)ImmutableList.of((Object)"x"));
        this.validateApplyUnapplied("x + x + x + y + y + y + y + z + z + z", "(+ (+ (+ (+ (+ (+ (+ (+ (+ x x) x) y) y) y) y) z) z) z)", "(cartesian_map ([x, y, z] -> (+ (+ (+ (+ (+ (+ (+ (+ (+ x x) x) y) y) y) y) z) z) z)), [x, y, z])", (List<String>)ImmutableList.of((Object)"x", (Object)"y", (Object)"z"));
    }

    private void validateLiteral(String expr, ExpressionType type, Object expected) {
        this.validateLiteral(expr, type, expected, true);
    }

    private void validateLiteral(String expr, ExpressionType type, Object expected, boolean roundTrip) {
        Expr parsed = Parser.parse((String)expr, (ExprMacroTable)ExprMacroTable.nil(), (boolean)false);
        Expr parsedFlat = Parser.parse((String)expr, (ExprMacroTable)ExprMacroTable.nil(), (boolean)true);
        Assert.assertTrue((boolean)parsed.isLiteral());
        Assert.assertTrue((boolean)parsedFlat.isLiteral());
        Assert.assertFalse((boolean)parsed.isIdentifier());
        Assert.assertEquals((Object)type, (Object)parsed.getOutputType((Expr.InputBindingInspector)this.emptyBinding));
        Assert.assertEquals((Object)type, (Object)parsedFlat.getOutputType((Expr.InputBindingInspector)this.emptyBinding));
        Assert.assertEquals((Object)expected, (Object)parsed.getLiteralValue());
        Assert.assertEquals((Object)(expected instanceof BigInteger ? Long.valueOf(((BigInteger)expected).longValueExact()) : expected), (Object)parsedFlat.getLiteralValue());
        if (roundTrip) {
            Assert.assertEquals((Object)expr, (Object)parsed.stringify());
            Assert.assertEquals((Object)expr, (Object)parsedFlat.stringify());
        }
    }

    private void validateFlatten(String expression, String withoutFlatten, String withFlatten) {
        Expr notFlat = Parser.parse((String)expression, (ExprMacroTable)ExprMacroTable.nil(), (boolean)false);
        Expr flat = Parser.parse((String)expression, (ExprMacroTable)ExprMacroTable.nil(), (boolean)true);
        Assert.assertEquals((String)expression, (Object)withoutFlatten, (Object)notFlat.toString());
        Assert.assertEquals((String)expression, (Object)withFlatten, (Object)flat.toString());
        Expr notFlatRoundTrip = Parser.parse((String)notFlat.stringify(), (ExprMacroTable)ExprMacroTable.nil(), (boolean)false);
        Expr flatRoundTrip = Parser.parse((String)flat.stringify(), (ExprMacroTable)ExprMacroTable.nil(), (boolean)true);
        Assert.assertEquals((String)expression, (Object)withoutFlatten, (Object)notFlatRoundTrip.toString());
        Assert.assertEquals((String)expression, (Object)withFlatten, (Object)flatRoundTrip.toString());
        Assert.assertEquals((Object)notFlat.stringify(), (Object)notFlatRoundTrip.stringify());
        Assert.assertEquals((Object)flat.stringify(), (Object)flatRoundTrip.stringify());
    }

    private void validateParser(String expression, String expected, List<String> identifiers) {
        this.validateParser(expression, expected, identifiers, (Set<String>)ImmutableSet.copyOf(identifiers), Collections.emptySet());
    }

    private void validateParser(String expression, String expected, List<String> identifiers, Set<String> scalars) {
        this.validateParser(expression, expected, identifiers, scalars, Collections.emptySet());
    }

    private void validateParser(String expression, String expected, List<String> identifiers, Set<String> scalars, Set<String> arrays) {
        Expr parsed = Parser.parse((String)expression, (ExprMacroTable)ExprMacroTable.nil());
        if (parsed instanceof IdentifierExpr) {
            Assert.assertTrue((boolean)parsed.isIdentifier());
        } else {
            Assert.assertFalse((boolean)parsed.isIdentifier());
        }
        Expr.BindingAnalysis deets = parsed.analyzeInputs();
        Assert.assertEquals((String)expression, (Object)expected, (Object)parsed.toString());
        Assert.assertEquals((String)expression, new HashSet<String>(identifiers), (Object)deets.getRequiredBindings());
        Assert.assertEquals((String)expression, scalars, (Object)deets.getScalarVariables());
        Assert.assertEquals((String)expression, arrays, (Object)deets.getArrayVariables());
        Expr parsedNoFlatten = Parser.parse((String)expression, (ExprMacroTable)ExprMacroTable.nil(), (boolean)false);
        Expr roundTrip = Parser.parse((String)parsedNoFlatten.stringify(), (ExprMacroTable)ExprMacroTable.nil());
        Assert.assertEquals((Object)parsed.stringify(), (Object)roundTrip.stringify());
        Expr.BindingAnalysis roundTripDeets = roundTrip.analyzeInputs();
        Assert.assertEquals((String)expression, new HashSet<String>(identifiers), (Object)roundTripDeets.getRequiredBindings());
        Assert.assertEquals((String)expression, scalars, (Object)roundTripDeets.getScalarVariables());
        Assert.assertEquals((String)expression, arrays, (Object)roundTripDeets.getArrayVariables());
    }

    private void validateApplyUnapplied(String expression, String unapplied, String applied, List<String> identifiers) {
        Expr parsed = Parser.parse((String)expression, (ExprMacroTable)ExprMacroTable.nil());
        Expr.BindingAnalysis deets = parsed.analyzeInputs();
        Parser.validateExpr((Expr)parsed, (Expr.BindingAnalysis)deets);
        Expr transformed = Parser.applyUnappliedBindings((Expr)parsed, (Expr.BindingAnalysis)deets, identifiers);
        Assert.assertEquals((String)expression, (Object)unapplied, (Object)parsed.toString());
        Assert.assertEquals((String)applied, (Object)applied, (Object)transformed.toString());
        Expr parsedNoFlatten = Parser.parse((String)expression, (ExprMacroTable)ExprMacroTable.nil(), (boolean)false);
        Expr parsedRoundTrip = Parser.parse((String)parsedNoFlatten.stringify(), (ExprMacroTable)ExprMacroTable.nil());
        Expr.BindingAnalysis roundTripDeets = parsedRoundTrip.analyzeInputs();
        Parser.validateExpr((Expr)parsedRoundTrip, (Expr.BindingAnalysis)roundTripDeets);
        Expr transformedRoundTrip = Parser.applyUnappliedBindings((Expr)parsedRoundTrip, (Expr.BindingAnalysis)roundTripDeets, identifiers);
        Assert.assertEquals((String)expression, (Object)unapplied, (Object)parsedRoundTrip.toString());
        Assert.assertEquals((String)applied, (Object)applied, (Object)transformedRoundTrip.toString());
        Assert.assertEquals((Object)parsed.stringify(), (Object)parsedRoundTrip.stringify());
        Assert.assertEquals((Object)transformed.stringify(), (Object)transformedRoundTrip.stringify());
    }

    private void validateFoldUnapplied(String expression, String unapplied, String applied, List<String> identifiers, String accumulator) {
        Expr parsed = Parser.parse((String)expression, (ExprMacroTable)ExprMacroTable.nil());
        Expr.BindingAnalysis deets = parsed.analyzeInputs();
        Parser.validateExpr((Expr)parsed, (Expr.BindingAnalysis)deets);
        Expr transformed = Parser.foldUnappliedBindings((Expr)parsed, (Expr.BindingAnalysis)deets, identifiers, (String)accumulator);
        Assert.assertEquals((String)expression, (Object)unapplied, (Object)parsed.toString());
        Assert.assertEquals((String)applied, (Object)applied, (Object)transformed.toString());
        Expr parsedNoFlatten = Parser.parse((String)expression, (ExprMacroTable)ExprMacroTable.nil(), (boolean)false);
        Expr parsedRoundTrip = Parser.parse((String)parsedNoFlatten.stringify(), (ExprMacroTable)ExprMacroTable.nil());
        Expr.BindingAnalysis roundTripDeets = parsedRoundTrip.analyzeInputs();
        Parser.validateExpr((Expr)parsedRoundTrip, (Expr.BindingAnalysis)roundTripDeets);
        Expr transformedRoundTrip = Parser.foldUnappliedBindings((Expr)parsedRoundTrip, (Expr.BindingAnalysis)roundTripDeets, identifiers, (String)accumulator);
        Assert.assertEquals((String)expression, (Object)unapplied, (Object)parsedRoundTrip.toString());
        Assert.assertEquals((String)applied, (Object)applied, (Object)transformedRoundTrip.toString());
        Assert.assertEquals((Object)parsed.stringify(), (Object)parsedRoundTrip.stringify());
        Assert.assertEquals((Object)transformed.stringify(), (Object)transformedRoundTrip.stringify());
    }

    private void validateConstantExpression(String expression, Object expected) {
        Expr parsed = Parser.parse((String)expression, (ExprMacroTable)ExprMacroTable.nil());
        Assert.assertEquals((String)expression, (Object)expected, (Object)parsed.eval(InputBindings.nilBindings()).value());
        Expr parsedNoFlatten = Parser.parse((String)expression, (ExprMacroTable)ExprMacroTable.nil(), (boolean)false);
        Expr parsedRoundTrip = Parser.parse((String)parsedNoFlatten.stringify(), (ExprMacroTable)ExprMacroTable.nil());
        Assert.assertEquals((String)expression, (Object)expected, (Object)parsedRoundTrip.eval(InputBindings.nilBindings()).value());
        Assert.assertEquals((Object)parsed.stringify(), (Object)parsedRoundTrip.stringify());
    }

    private void validateConstantExpression(String expression, Object[] expected) {
        Expr parsed = Parser.parse((String)expression, (ExprMacroTable)ExprMacroTable.nil());
        Object evaluated = parsed.eval(InputBindings.nilBindings()).value();
        Assert.assertArrayEquals((String)expression, (Object[])expected, (Object[])((Object[])evaluated));
        Assert.assertEquals(expected.getClass(), evaluated.getClass());
        Expr parsedNoFlatten = Parser.parse((String)expression, (ExprMacroTable)ExprMacroTable.nil(), (boolean)false);
        Expr roundTrip = Parser.parse((String)parsedNoFlatten.stringify(), (ExprMacroTable)ExprMacroTable.nil());
        Assert.assertArrayEquals((String)expression, (Object[])expected, (Object[])((Object[])roundTrip.eval(InputBindings.nilBindings()).value()));
        Assert.assertEquals((Object)parsed.stringify(), (Object)roundTrip.stringify());
    }
}

