/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.math.expr;

import com.google.common.base.Preconditions;
import com.google.common.primitives.Doubles;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import javax.annotation.Nullable;
import org.apache.druid.common.guava.GuavaUtils;
import org.apache.druid.java.util.common.IAE;
import org.apache.druid.java.util.common.ISE;
import org.apache.druid.java.util.common.NonnullPair;
import org.apache.druid.java.util.common.StringUtils;
import org.apache.druid.math.expr.ArrayExpr;
import org.apache.druid.math.expr.ComplexExpr;
import org.apache.druid.math.expr.DoubleExpr;
import org.apache.druid.math.expr.Evals;
import org.apache.druid.math.expr.Expr;
import org.apache.druid.math.expr.ExprType;
import org.apache.druid.math.expr.ExpressionType;
import org.apache.druid.math.expr.ExpressionTypeConversion;
import org.apache.druid.math.expr.ExpressionTypeFactory;
import org.apache.druid.math.expr.LongExpr;
import org.apache.druid.math.expr.NullDoubleExpr;
import org.apache.druid.math.expr.NullLongExpr;
import org.apache.druid.math.expr.StringExpr;
import org.apache.druid.segment.column.NullableTypeStrategy;
import org.apache.druid.segment.column.TypeStrategies;
import org.apache.druid.segment.column.TypeStrategy;
import org.apache.druid.segment.column.Types;
import org.apache.druid.segment.nested.StructuredData;

public abstract class ExprEval<T> {
    private boolean stringValueCached = false;
    @Nullable
    private String stringValue;
    @Nullable
    final T value;

    public static ExprEval deserialize(ByteBuffer buffer, int offset, int maxSize, ExpressionType type, boolean canRetainBufferReference) {
        switch ((ExprType)type.getType()) {
            case LONG: {
                if (TypeStrategies.isNullableNull(buffer, offset)) {
                    return ExprEval.ofLong(null);
                }
                return ExprEval.of(TypeStrategies.readNotNullNullableLong(buffer, offset));
            }
            case DOUBLE: {
                if (TypeStrategies.isNullableNull(buffer, offset)) {
                    return ExprEval.ofDouble(null);
                }
                return ExprEval.of(TypeStrategies.readNotNullNullableDouble(buffer, offset));
            }
        }
        NullableTypeStrategy strategy = type.getNullableStrategy();
        if (!canRetainBufferReference && strategy.readRetainsBufferReference()) {
            ByteBuffer dataCopyBuffer = ByteBuffer.allocate(maxSize);
            ByteBuffer mutationBuffer = buffer.duplicate();
            mutationBuffer.limit(offset + maxSize);
            mutationBuffer.position(offset);
            dataCopyBuffer.put(mutationBuffer);
            dataCopyBuffer.rewind();
            return ExprEval.ofType(type, strategy.read(dataCopyBuffer, 0));
        }
        return ExprEval.ofType(type, strategy.read(buffer, offset));
    }

    public static void serialize(ByteBuffer buffer, int position, ExpressionType type, ExprEval<?> eval, int maxSizeBytes) {
        int offset = position;
        switch ((ExprType)type.getType()) {
            case LONG: {
                if (eval.value() == null) {
                    TypeStrategies.writeNull(buffer, offset);
                    break;
                }
                TypeStrategies.writeNotNullNullableLong(buffer, offset, eval.asLong());
                break;
            }
            case DOUBLE: {
                if (eval.value() == null) {
                    TypeStrategies.writeNull(buffer, offset);
                    break;
                }
                TypeStrategies.writeNotNullNullableDouble(buffer, offset, eval.asDouble());
                break;
            }
            default: {
                int written;
                NullableTypeStrategy strategy = type.getNullableStrategy();
                if (type.equals(eval.type())) {
                    eval = eval.castTo(type);
                }
                if ((written = strategy.write(buffer, offset, eval.value(), maxSizeBytes)) >= 0) break;
                throw new ISE("Unable to serialize [%s], max size bytes is [%s], but need at least [%s] bytes to write entire value", type.asTypeString(), maxSizeBytes, maxSizeBytes - written);
            }
        }
    }

    @Nullable
    public static NonnullPair<ExpressionType, Object[]> coerceListToArray(@Nullable List<?> val, boolean homogenizeMultiValueStrings) {
        if (val != null && val.size() > 0) {
            Object[] array;
            Class coercedType = null;
            for (Object elem : val) {
                if (elem == null) continue;
                coercedType = ExprEval.convertType(coercedType, elem.getClass());
            }
            if (coercedType == Long.class || coercedType == Integer.class) {
                array = new Object[val.size()];
                int i = 0;
                for (Object o : val) {
                    array[i++] = o != null ? ExprEval.ofType(ExpressionType.LONG, o).value() : null;
                }
                return new NonnullPair<ExpressionType, Object[]>(ExpressionType.LONG_ARRAY, array);
            }
            if (coercedType == Float.class || coercedType == Double.class) {
                array = new Object[val.size()];
                int i = 0;
                for (Object o : val) {
                    array[i++] = o != null ? ExprEval.ofType(ExpressionType.DOUBLE, o).value() : null;
                }
                return new NonnullPair<ExpressionType, Object[]>(ExpressionType.DOUBLE_ARRAY, array);
            }
            if (coercedType == Object.class) {
                ExprEval[] evals = new ExprEval[val.size()];
                Object[] array2 = new Object[val.size()];
                int i = 0;
                ExpressionType elementType = null;
                for (Object o : val) {
                    if (o != null) {
                        ExprEval eval = ExprEval.bestEffortOf(o);
                        elementType = ExpressionTypeConversion.leastRestrictiveType(elementType, eval.type());
                        evals[i++] = eval;
                        continue;
                    }
                    evals[i++] = null;
                }
                i = 0;
                for (ExprEval eval : evals) {
                    array2[i++] = eval != null ? eval.castTo(elementType).value() : ExprEval.ofType(elementType, null).value();
                }
                ExpressionType arrayType = elementType == null ? ExpressionType.STRING_ARRAY : ExpressionTypeFactory.getInstance().ofArray(elementType);
                return new NonnullPair<ExpressionType, Object[]>(arrayType, array2);
            }
            array = new Object[val.size()];
            int i = 0;
            for (Object o : val) {
                array[i++] = o != null ? ExprEval.ofType(ExpressionType.STRING, o).value() : null;
            }
            return new NonnullPair<ExpressionType, Object[]>(ExpressionType.STRING_ARRAY, array);
        }
        if (homogenizeMultiValueStrings) {
            return new NonnullPair<ExpressionType, Object[]>(ExpressionType.STRING_ARRAY, new Object[]{null});
        }
        if (val != null) {
            return new NonnullPair<ExpressionType, Object[]>(ExpressionType.STRING_ARRAY, new Object[0]);
        }
        return null;
    }

    private static Class convertType(@Nullable Class existing, Class next) {
        if (existing != null && existing.equals(Object.class)) {
            return existing;
        }
        if (Number.class.isAssignableFrom(next) || next == String.class || next == Boolean.class) {
            if (next == Boolean.class) {
                next = Long.class;
            }
            if (existing == null) {
                return next;
            }
            if (existing == String.class) {
                return existing;
            }
            if (next == String.class) {
                return next;
            }
            if (existing == Integer.class) {
                return next;
            }
            if (existing == Float.class) {
                if (next == Double.class) {
                    return next;
                }
                return existing;
            }
            if (existing == Long.class) {
                if (next == Integer.class) {
                    return existing;
                }
                return next;
            }
            return Double.class;
        }
        return Object.class;
    }

    public static ExprEval of(long longValue) {
        return new LongExprEval(longValue);
    }

    public static ExprEval of(double doubleValue) {
        return new DoubleExprEval(doubleValue);
    }

    public static ExprEval of(@Nullable String stringValue) {
        if (stringValue == null) {
            return StringExprEval.OF_NULL;
        }
        return new StringExprEval(stringValue);
    }

    public static ExprEval ofLong(@Nullable Number longValue) {
        if (longValue == null) {
            return LongExprEval.OF_NULL;
        }
        return new LongExprEval(longValue);
    }

    public static ExprEval ofDouble(@Nullable Number doubleValue) {
        if (doubleValue == null) {
            return DoubleExprEval.OF_NULL;
        }
        return new DoubleExprEval(doubleValue);
    }

    public static ExprEval ofLongArray(@Nullable Object[] longValue) {
        if (longValue == null) {
            return ArrayExprEval.OF_NULL_LONG;
        }
        return new ArrayExprEval(ExpressionType.LONG_ARRAY, longValue);
    }

    public static ExprEval ofDoubleArray(@Nullable Object[] doubleValue) {
        if (doubleValue == null) {
            return ArrayExprEval.OF_NULL_DOUBLE;
        }
        return new ArrayExprEval(ExpressionType.DOUBLE_ARRAY, doubleValue);
    }

    public static ExprEval ofStringArray(@Nullable Object[] stringValue) {
        if (stringValue == null) {
            return ArrayExprEval.OF_NULL_STRING;
        }
        return new ArrayExprEval(ExpressionType.STRING_ARRAY, stringValue);
    }

    public static ExprEval ofArray(ExpressionType outputType, @Nullable Object[] value) {
        Preconditions.checkArgument((boolean)outputType.isArray(), (String)"Output type %s is not an array", (Object)outputType);
        return new ArrayExprEval(outputType, value);
    }

    public static ExprEval ofLongBoolean(boolean value) {
        return value ? LongExprEval.TRUE : LongExprEval.FALSE;
    }

    public static ExprEval ofComplex(ExpressionType outputType, @Nullable Object value) {
        if (ExpressionType.NESTED_DATA.equals(outputType)) {
            return new NestedDataExprEval(value);
        }
        return new ComplexExprEval(outputType, value);
    }

    public static ExprEval bestEffortArray(@Nullable List<?> theList) {
        NonnullPair<ExpressionType, Object[]> coerced = ExprEval.coerceListToArray(theList, false);
        if (coerced == null) {
            return ExprEval.bestEffortOf(null);
        }
        return ExprEval.ofArray((ExpressionType)coerced.lhs, (Object[])coerced.rhs);
    }

    public static ExprEval bestEffortOf(@Nullable Object val) {
        if (val == null) {
            return StringExprEval.OF_NULL;
        }
        if (val instanceof ExprEval) {
            return (ExprEval)val;
        }
        if (val instanceof String) {
            return new StringExprEval((String)val);
        }
        if (val instanceof Number) {
            if (val instanceof Float || val instanceof Double) {
                return new DoubleExprEval((Number)val);
            }
            return new LongExprEval((Number)val);
        }
        if (val instanceof Boolean) {
            return ExprEval.ofLongBoolean((Boolean)val);
        }
        if (val instanceof Long[]) {
            Long[] inputArray = (Long[])val;
            Object[] array = new Object[inputArray.length];
            for (int i = 0; i < inputArray.length; ++i) {
                array[i] = inputArray[i];
            }
            return new ArrayExprEval(ExpressionType.LONG_ARRAY, array);
        }
        if (val instanceof long[]) {
            long[] longArray = (long[])val;
            Object[] array = new Object[longArray.length];
            for (int i = 0; i < longArray.length; ++i) {
                array[i] = longArray[i];
            }
            return new ArrayExprEval(ExpressionType.LONG_ARRAY, array);
        }
        if (val instanceof Integer[]) {
            Integer[] inputArray = (Integer[])val;
            Object[] array = new Object[inputArray.length];
            for (int i = 0; i < inputArray.length; ++i) {
                array[i] = inputArray[i] != null ? Long.valueOf(inputArray[i].longValue()) : null;
            }
            return new ArrayExprEval(ExpressionType.LONG_ARRAY, array);
        }
        if (val instanceof int[]) {
            int[] longArray = (int[])val;
            Object[] array = new Object[longArray.length];
            for (int i = 0; i < longArray.length; ++i) {
                array[i] = (long)longArray[i];
            }
            return new ArrayExprEval(ExpressionType.LONG_ARRAY, array);
        }
        if (val instanceof Double[]) {
            Double[] inputArray = (Double[])val;
            Object[] array = new Object[inputArray.length];
            for (int i = 0; i < inputArray.length; ++i) {
                array[i] = inputArray[i] != null ? inputArray[i] : null;
            }
            return new ArrayExprEval(ExpressionType.DOUBLE_ARRAY, array);
        }
        if (val instanceof double[]) {
            double[] inputArray = (double[])val;
            Object[] array = new Object[inputArray.length];
            for (int i = 0; i < inputArray.length; ++i) {
                array[i] = inputArray[i];
            }
            return new ArrayExprEval(ExpressionType.DOUBLE_ARRAY, array);
        }
        if (val instanceof Float[]) {
            Float[] inputArray = (Float[])val;
            Object[] array = new Object[inputArray.length];
            for (int i = 0; i < inputArray.length; ++i) {
                array[i] = inputArray[i] != null ? Double.valueOf(inputArray[i].doubleValue()) : null;
            }
            return new ArrayExprEval(ExpressionType.DOUBLE_ARRAY, array);
        }
        if (val instanceof float[]) {
            float[] inputArray = (float[])val;
            Object[] array = new Object[inputArray.length];
            for (int i = 0; i < inputArray.length; ++i) {
                array[i] = (double)inputArray[i];
            }
            return new ArrayExprEval(ExpressionType.DOUBLE_ARRAY, array);
        }
        if (val instanceof String[]) {
            String[] inputArray = (String[])val;
            Object[] array = new Object[inputArray.length];
            for (int i = 0; i < inputArray.length; ++i) {
                array[i] = inputArray[i];
            }
            return new ArrayExprEval(ExpressionType.STRING_ARRAY, array);
        }
        if (val instanceof List || val instanceof Object[]) {
            List<Object> theList = val instanceof List ? (List<Object>)val : Arrays.asList((Object[])val);
            return ExprEval.bestEffortArray(theList);
        }
        if (val instanceof byte[]) {
            return new StringExprEval(StringUtils.encodeBase64String((byte[])val));
        }
        if (val instanceof Map) {
            return ExprEval.ofComplex(ExpressionType.NESTED_DATA, val);
        }
        return ExprEval.ofComplex(ExpressionType.UNKNOWN_COMPLEX, val);
    }

    public static ExprEval ofType(@Nullable ExpressionType type, @Nullable Object value) {
        if (type == null) {
            return ExprEval.bestEffortOf(value);
        }
        switch ((ExprType)type.getType()) {
            case STRING: {
                if (value instanceof String[]) {
                    String[] inputArray = (String[])value;
                    Object[] array = new Object[inputArray.length];
                    for (int i = 0; i < inputArray.length; ++i) {
                        array[i] = inputArray[i];
                    }
                    return new ArrayExprEval(ExpressionType.STRING_ARRAY, array);
                }
                if (value instanceof Object[]) {
                    return ExprEval.bestEffortOf(value);
                }
                if (value instanceof List) {
                    return ExprEval.bestEffortOf(value);
                }
                if (value instanceof byte[]) {
                    return new StringExprEval(StringUtils.encodeBase64String((byte[])value));
                }
                return ExprEval.of(Evals.asString(value));
            }
            case LONG: {
                if (value instanceof Number) {
                    return ExprEval.ofLong((Number)value);
                }
                if (value instanceof Boolean) {
                    return ExprEval.ofLongBoolean((Boolean)value);
                }
                if (value instanceof String) {
                    return ExprEval.ofLong(ExprEval.computeNumber((String)value));
                }
                return ExprEval.ofLong(null);
            }
            case DOUBLE: {
                if (value instanceof Number) {
                    return ExprEval.ofDouble((Number)value);
                }
                if (value instanceof Boolean) {
                    return ExprEval.ofDouble(Evals.asDouble((Boolean)value));
                }
                if (value instanceof String) {
                    return ExprEval.ofDouble(ExprEval.computeNumber((String)value));
                }
                return ExprEval.ofDouble(null);
            }
            case COMPLEX: {
                if (ExpressionType.NESTED_DATA.equals(type)) {
                    return ExprEval.ofComplex(type, StructuredData.unwrap(value));
                }
                byte[] bytes = null;
                if (value instanceof String) {
                    try {
                        bytes = StringUtils.decodeBase64String((String)value);
                    }
                    catch (IllegalArgumentException array) {}
                } else if (value instanceof byte[]) {
                    bytes = (byte[])value;
                }
                if (bytes != null) {
                    TypeStrategy strategy = type.getStrategy();
                    ByteBuffer bb = ByteBuffer.wrap(bytes);
                    return ExprEval.ofComplex(type, strategy.read(bb));
                }
                return ExprEval.ofComplex(type, value);
            }
            case ARRAY: {
                ExpressionType elementType = (ExpressionType)type.getElementType();
                if (value == null) {
                    return ExprEval.ofArray(type, null);
                }
                if (value instanceof List) {
                    List theList = (List)value;
                    Object[] array = new Object[theList.size()];
                    int i = 0;
                    for (Object o : theList) {
                        array[i++] = ExprEval.ofType(elementType, o).value();
                    }
                    return ExprEval.ofArray(type, array);
                }
                if (value instanceof Object[]) {
                    Object[] inputArray = (Object[])value;
                    Object[] array = new Object[inputArray.length];
                    int i = 0;
                    for (Object o : inputArray) {
                        array[i++] = ExprEval.ofType(elementType, o).value();
                    }
                    return ExprEval.ofArray(type, array);
                }
                return ExprEval.bestEffortOf(value).castTo(type);
            }
        }
        throw new IAE("Cannot create type [%s]", type);
    }

    @Nullable
    public static Number computeNumber(@Nullable String value) {
        if (value == null) {
            return null;
        }
        if (Evals.asBoolean(value)) {
            return 1.0;
        }
        if (value.equalsIgnoreCase("false")) {
            return 0.0;
        }
        Long v = GuavaUtils.tryParseLong(value);
        Number rv = v != null ? (Number)v : (Number)Doubles.tryParse((String)value);
        return rv;
    }

    @Nullable
    public static ExprEval<?> castForEqualityComparison(ExprEval<?> valueToCompare, ExpressionType typeToCompareWith) {
        Object[] array;
        if (valueToCompare.isArray() && !typeToCompareWith.isArray() && (array = valueToCompare.asArray()) != null && array.length != 1) {
            return null;
        }
        ExprEval cast = valueToCompare.castTo(typeToCompareWith);
        if (ExpressionType.LONG.equals(typeToCompareWith) && valueToCompare.asDouble() != cast.asDouble()) {
            return null;
        }
        if (ExpressionType.LONG_ARRAY.equals(typeToCompareWith)) {
            ExprEval doubleCast = valueToCompare.castTo(ExpressionType.DOUBLE_ARRAY);
            ExprEval castDoubleCast = cast.castTo(ExpressionType.DOUBLE_ARRAY);
            if (ExpressionType.DOUBLE_ARRAY.getStrategy().compare(doubleCast.value(), castDoubleCast.value()) != 0) {
                return null;
            }
        }
        if (valueToCompare.value() != null && cast.value() == null) {
            return null;
        }
        return cast;
    }

    private ExprEval(@Nullable T value) {
        this.value = value;
    }

    public abstract ExpressionType type();

    public ExpressionType elementType() {
        return this.type().isArray() ? (ExpressionType)this.type().getElementType() : this.type();
    }

    public ExpressionType asArrayType() {
        return this.type().isArray() ? this.type() : ExpressionTypeFactory.getInstance().ofArray(this.type());
    }

    @Nullable
    public T value() {
        return this.value;
    }

    @Nullable
    public T valueOrDefault() {
        return this.value;
    }

    void cacheStringValue(@Nullable String value) {
        this.stringValue = value;
        this.stringValueCached = true;
    }

    @Nullable
    String getCachedStringValue() {
        return this.stringValue;
    }

    boolean isStringValueCached() {
        return this.stringValueCached;
    }

    @Nullable
    public String asString() {
        if (!this.stringValueCached) {
            this.stringValue = Evals.asString(this.value);
            this.stringValueCached = true;
        }
        return this.stringValue;
    }

    public abstract boolean isNumericNull();

    public boolean isArray() {
        return false;
    }

    public abstract int asInt();

    public abstract long asLong();

    public abstract double asDouble();

    public abstract boolean asBoolean();

    @Nullable
    public abstract Object[] asArray();

    public abstract ExprEval castTo(ExpressionType var1);

    public abstract Expr toExpr();

    public static Types.InvalidCastException invalidCast(ExpressionType fromType, ExpressionType toType) {
        return new Types.InvalidCastException(fromType, toType);
    }

    private static class NestedDataExprEval
    extends ExprEval<Object> {
        @Nullable
        private Number number;
        private boolean computedNumber = false;

        private NestedDataExprEval(@Nullable Object value) {
            super(value);
        }

        @Override
        public ExpressionType type() {
            return ExpressionType.NESTED_DATA;
        }

        @Override
        public boolean isNumericNull() {
            this.computeNumber();
            return this.number == null;
        }

        @Override
        public int asInt() {
            this.computeNumber();
            if (this.number != null) {
                return this.number.intValue();
            }
            return 0;
        }

        @Override
        public long asLong() {
            this.computeNumber();
            if (this.number != null) {
                return this.number.longValue();
            }
            return 0L;
        }

        @Override
        public double asDouble() {
            this.computeNumber();
            if (this.number != null) {
                return this.number.doubleValue();
            }
            return 0.0;
        }

        @Override
        public boolean asBoolean() {
            Object val = StructuredData.unwrap(this.value);
            if (val != null) {
                return Evals.objectAsBoolean(val);
            }
            return false;
        }

        private void computeNumber() {
            if (!this.computedNumber && this.value != null) {
                this.computedNumber = true;
                Object val = StructuredData.unwrap(this.value);
                if (val instanceof Number) {
                    this.number = (Number)val;
                } else if (val instanceof Boolean) {
                    this.number = Evals.asLong((Boolean)val);
                } else if (val instanceof String) {
                    this.number = ExprEval.computeNumber((String)val);
                }
            }
        }

        @Override
        @Nullable
        public Object[] asArray() {
            Object val = StructuredData.unwrap(this.value);
            ExprEval maybeArray = ExprEval.bestEffortOf(val);
            if (maybeArray.type().isPrimitive() || maybeArray.isArray()) {
                return maybeArray.asArray();
            }
            return null;
        }

        @Override
        public ExprEval castTo(ExpressionType castTo) {
            if (ExpressionType.NESTED_DATA.equals(castTo)) {
                return this;
            }
            Object val = StructuredData.unwrap(this.value);
            ExprEval bestEffortOf = ExprEval.bestEffortOf(val);
            if (bestEffortOf.type().isPrimitive() || bestEffortOf.type().isArray()) {
                return bestEffortOf.castTo(castTo);
            }
            throw NestedDataExprEval.invalidCast(ExpressionType.NESTED_DATA, castTo);
        }

        @Override
        public Expr toExpr() {
            return new ComplexExpr(ExpressionType.NESTED_DATA, this.value);
        }
    }

    private static class ComplexExprEval
    extends ExprEval<Object> {
        private final ExpressionType expressionType;

        private ComplexExprEval(ExpressionType expressionType, @Nullable Object value) {
            super(value);
            this.expressionType = expressionType;
        }

        @Override
        public ExpressionType type() {
            return this.expressionType;
        }

        @Override
        public boolean isNumericNull() {
            return true;
        }

        @Override
        public int asInt() {
            return 0;
        }

        @Override
        public long asLong() {
            return 0L;
        }

        @Override
        public double asDouble() {
            return 0.0;
        }

        @Override
        public boolean asBoolean() {
            return false;
        }

        @Override
        @Nullable
        public Object[] asArray() {
            return null;
        }

        @Override
        public ExprEval castTo(ExpressionType castTo) {
            if (this.expressionType.equals(castTo)) {
                return this;
            }
            if (this.expressionType.getComplexTypeName() == null && ((ExprType)castTo.getType()).equals(ExprType.COMPLEX)) {
                return ComplexExprEval.ofComplex(castTo, this.value);
            }
            throw ComplexExprEval.invalidCast(this.expressionType, castTo);
        }

        @Override
        public Expr toExpr() {
            return new ComplexExpr(this.expressionType, this.value);
        }
    }

    static class ArrayExprEval
    extends ExprEval<Object[]> {
        public static final ExprEval OF_NULL_LONG = new ArrayExprEval(ExpressionType.LONG_ARRAY, null);
        public static final ExprEval OF_NULL_DOUBLE = new ArrayExprEval(ExpressionType.DOUBLE_ARRAY, null);
        public static final ExprEval OF_NULL_STRING = new ArrayExprEval(ExpressionType.STRING_ARRAY, null);
        private final ExpressionType arrayType;

        private ArrayExprEval(ExpressionType arrayType, @Nullable Object[] value) {
            super(value);
            this.arrayType = arrayType;
            Preconditions.checkArgument((boolean)arrayType.isArray(), (String)"Output type %s is not an array", (Object)arrayType);
        }

        @Override
        public ExpressionType type() {
            return this.arrayType;
        }

        @Override
        @Nullable
        public String asString() {
            if (!this.isStringValueCached()) {
                if (this.value == null) {
                    this.cacheStringValue(null);
                } else if (((Object[])this.value).length == 1) {
                    this.cacheStringValue(Evals.asString(((Object[])this.value)[0]));
                } else {
                    this.cacheStringValue(Arrays.deepToString((Object[])this.value));
                }
            }
            return this.getCachedStringValue();
        }

        @Override
        public boolean isNumericNull() {
            if (this.isScalar()) {
                if (this.arrayType.getElementType().is(ExprType.STRING)) {
                    Number n = ArrayExprEval.computeNumber((String)this.getScalarValue());
                    return n == null;
                }
                return this.getScalarValue() == null;
            }
            return true;
        }

        @Override
        public boolean isArray() {
            return true;
        }

        @Override
        public int asInt() {
            if (this.isScalar()) {
                Number scalar = null;
                if (this.arrayType.getElementType().isNumeric()) {
                    scalar = (Number)this.getScalarValue();
                } else if (this.arrayType.getElementType().is(ExprType.STRING)) {
                    scalar = ArrayExprEval.computeNumber((String)this.getScalarValue());
                }
                if (scalar == null) {
                    return 0;
                }
                return scalar.intValue();
            }
            return 0;
        }

        @Override
        public long asLong() {
            if (this.isScalar()) {
                Number scalar = null;
                if (this.arrayType.getElementType().isNumeric()) {
                    scalar = (Number)this.getScalarValue();
                } else if (this.arrayType.getElementType().is(ExprType.STRING)) {
                    scalar = ArrayExprEval.computeNumber((String)this.getScalarValue());
                }
                if (scalar == null) {
                    return 0L;
                }
                return scalar.longValue();
            }
            return 0L;
        }

        @Override
        public double asDouble() {
            if (this.isScalar()) {
                Number scalar = null;
                if (this.arrayType.getElementType().isNumeric()) {
                    scalar = (Number)this.getScalarValue();
                } else if (this.arrayType.getElementType().is(ExprType.STRING)) {
                    scalar = ArrayExprEval.computeNumber((String)this.getScalarValue());
                }
                if (scalar == null) {
                    return 0.0;
                }
                return scalar.doubleValue();
            }
            return 0.0;
        }

        @Override
        public boolean asBoolean() {
            if (this.isScalar()) {
                if (this.arrayType.getElementType().isNumeric()) {
                    Number scalarValue = (Number)this.getScalarValue();
                    if (scalarValue == null) {
                        return false;
                    }
                    return Evals.asBoolean(scalarValue.longValue());
                }
                if (this.arrayType.getElementType().is(ExprType.STRING)) {
                    return Evals.asBoolean((String)this.getScalarValue());
                }
            }
            return false;
        }

        @Override
        @Nullable
        public Object[] asArray() {
            return (Object[])this.value;
        }

        @Override
        public ExprEval castTo(ExpressionType castTo) {
            if (this.value == null) {
                if (castTo.isArray()) {
                    return new ArrayExprEval(castTo, null);
                }
                return ExprEval.ofType(castTo, null);
            }
            if (this.type().equals(castTo)) {
                return this;
            }
            switch ((ExprType)castTo.getType()) {
                case STRING: {
                    if (((Object[])this.value).length != 1) break;
                    return ExprEval.of(this.asString());
                }
                case LONG: {
                    if (((Object[])this.value).length != 1) break;
                    return this.isNumericNull() ? ExprEval.ofLong(null) : ExprEval.ofLong(this.asLong());
                }
                case DOUBLE: {
                    if (((Object[])this.value).length != 1) break;
                    return this.isNumericNull() ? ExprEval.ofDouble(null) : ExprEval.ofDouble(this.asDouble());
                }
                case ARRAY: {
                    ExpressionType elementType = (ExpressionType)castTo.getElementType();
                    Object[] cast = new Object[((Object[])this.value).length];
                    for (int i = 0; i < ((Object[])this.value).length; ++i) {
                        cast[i] = ExprEval.ofType(this.elementType(), ((Object[])this.value)[i]).castTo(elementType).value();
                    }
                    return ExprEval.ofArray(castTo, cast);
                }
                case COMPLEX: {
                    if (!ExpressionType.NESTED_DATA.equals(castTo)) break;
                    return new NestedDataExprEval(this.value);
                }
            }
            throw ArrayExprEval.invalidCast(this.type(), castTo);
        }

        @Override
        public Expr toExpr() {
            return new ArrayExpr(this.arrayType, (Object[])this.value);
        }

        protected boolean isScalar() {
            return this.value != null && ((Object[])this.value).length == 1;
        }

        @Nullable
        protected Object getScalarValue() {
            return ((Object[])this.value)[0];
        }
    }

    private static class StringExprEval
    extends ExprEval<String> {
        private static final StringExprEval OF_NULL = new StringExprEval(null);
        private boolean intValueValid = false;
        private boolean longValueValid = false;
        private boolean doubleValueValid = false;
        private boolean booleanValueValid = false;
        private int intValue;
        private long longValue;
        private double doubleValue;
        private boolean booleanValue;
        @Nullable
        private Number numericVal;

        private StringExprEval(@Nullable String value) {
            super(value);
        }

        @Override
        public final ExpressionType type() {
            return ExpressionType.STRING;
        }

        @Override
        public int asInt() {
            if (!this.intValueValid) {
                this.intValue = this.computeInt();
                this.intValueValid = true;
            }
            return this.intValue;
        }

        @Override
        public long asLong() {
            if (!this.longValueValid) {
                this.longValue = this.computeLong();
                this.longValueValid = true;
            }
            return this.longValue;
        }

        @Override
        public double asDouble() {
            if (!this.doubleValueValid) {
                this.doubleValue = this.computeDouble();
                this.doubleValueValid = true;
            }
            return this.doubleValue;
        }

        @Override
        @Nullable
        public String asString() {
            return (String)this.value;
        }

        @Override
        @Nullable
        public Object[] asArray() {
            Object[] objectArray;
            if (this.value == null) {
                objectArray = null;
            } else {
                Object[] objectArray2 = new Object[1];
                objectArray = objectArray2;
                objectArray2[0] = this.value;
            }
            return objectArray;
        }

        private int computeInt() {
            Number number = this.computeNumber();
            if (number == null) {
                return 0;
            }
            return number.intValue();
        }

        private long computeLong() {
            Number number = this.computeNumber();
            if (number == null) {
                return 0L;
            }
            return number.longValue();
        }

        private double computeDouble() {
            Number number = this.computeNumber();
            if (number == null) {
                return 0.0;
            }
            return number.doubleValue();
        }

        @Nullable
        Number computeNumber() {
            if (this.value == null) {
                return null;
            }
            if (this.numericVal != null) {
                return this.numericVal;
            }
            this.numericVal = StringExprEval.computeNumber((String)this.value);
            return this.numericVal;
        }

        @Override
        public boolean isNumericNull() {
            return this.computeNumber() == null;
        }

        @Override
        public final boolean asBoolean() {
            if (!this.booleanValueValid) {
                this.booleanValue = Evals.asBoolean((String)this.value);
                this.booleanValueValid = true;
            }
            return this.booleanValue;
        }

        @Override
        public final ExprEval castTo(ExpressionType castTo) {
            switch ((ExprType)castTo.getType()) {
                case DOUBLE: {
                    return ExprEval.ofDouble(this.computeNumber());
                }
                case LONG: {
                    return ExprEval.ofLong(this.computeNumber());
                }
                case STRING: {
                    return this;
                }
                case ARRAY: {
                    if (this.value == null) {
                        return new ArrayExprEval(castTo, null);
                    }
                    ExprType type = (ExprType)castTo.getElementType().getType();
                    if (type == ExprType.DOUBLE) {
                        Number number = this.computeNumber();
                        return ExprEval.ofDoubleArray(new Object[]{number == null ? null : Double.valueOf(number.doubleValue())});
                    }
                    if (type == ExprType.LONG) {
                        Number number = this.computeNumber();
                        return ExprEval.ofLongArray(new Object[]{number == null ? null : Long.valueOf(number.longValue())});
                    }
                    if (type == ExprType.STRING) {
                        return ExprEval.ofStringArray(new Object[]{this.value});
                    }
                    ExpressionType elementType = (ExpressionType)castTo.getElementType();
                    return new ArrayExprEval(castTo, new Object[]{this.castTo(elementType).value()});
                }
                case COMPLEX: {
                    if (!ExpressionType.NESTED_DATA.equals(castTo)) break;
                    return new NestedDataExprEval(this.value);
                }
            }
            throw StringExprEval.invalidCast(this.type(), castTo);
        }

        @Override
        public Expr toExpr() {
            return new StringExpr((String)this.value);
        }
    }

    private static class LongExprEval
    extends NumericExprEval {
        private static final LongExprEval TRUE = new LongExprEval(Evals.asLong(true));
        private static final LongExprEval FALSE = new LongExprEval(Evals.asLong(false));
        private static final LongExprEval OF_NULL = new LongExprEval(null);

        private LongExprEval(@Nullable Number value) {
            super(value == null ? null : Long.valueOf(value.longValue()));
        }

        @Override
        public final ExpressionType type() {
            return ExpressionType.LONG;
        }

        @Override
        public Number valueOrDefault() {
            if (this.value == null) {
                return null;
            }
            return ((Number)this.value).longValue();
        }

        @Override
        public final boolean asBoolean() {
            return Evals.asBoolean(this.asLong());
        }

        @Override
        @Nullable
        public Object[] asArray() {
            Object[] objectArray;
            if (this.value == null) {
                objectArray = null;
            } else {
                Object[] objectArray2 = new Object[1];
                objectArray = objectArray2;
                objectArray2[0] = this.valueOrDefault().longValue();
            }
            return objectArray;
        }

        @Override
        public final ExprEval castTo(ExpressionType castTo) {
            switch ((ExprType)castTo.getType()) {
                case DOUBLE: {
                    if (this.value == null) {
                        return ExprEval.ofDouble(null);
                    }
                    return ExprEval.of(this.asDouble());
                }
                case LONG: {
                    return this;
                }
                case STRING: {
                    return ExprEval.of(this.asString());
                }
                case ARRAY: {
                    if (this.value == null) {
                        return new ArrayExprEval(castTo, null);
                    }
                    switch ((ExprType)castTo.getElementType().getType()) {
                        case DOUBLE: {
                            return ExprEval.ofDoubleArray(new Object[]{((Number)this.value).doubleValue()});
                        }
                        case LONG: {
                            return ExprEval.ofLongArray(this.asArray());
                        }
                        case STRING: {
                            return ExprEval.ofStringArray(new Object[]{((Number)this.value).toString()});
                        }
                    }
                    ExpressionType elementType = (ExpressionType)castTo.getElementType();
                    return new ArrayExprEval(castTo, new Object[]{this.castTo(elementType).value()});
                }
                case COMPLEX: {
                    if (!ExpressionType.NESTED_DATA.equals(castTo)) break;
                    return new NestedDataExprEval(this.value);
                }
            }
            throw LongExprEval.invalidCast(this.type(), castTo);
        }

        @Override
        public Expr toExpr() {
            if (this.value == null) {
                return new NullLongExpr();
            }
            return new LongExpr(((Number)this.value).longValue());
        }
    }

    private static class DoubleExprEval
    extends NumericExprEval {
        private static final DoubleExprEval OF_NULL = new DoubleExprEval(null);

        private DoubleExprEval(@Nullable Number value) {
            super(value == null ? null : Double.valueOf(value.doubleValue()));
        }

        @Override
        public final ExpressionType type() {
            return ExpressionType.DOUBLE;
        }

        @Override
        public Number valueOrDefault() {
            if (this.value == null) {
                return null;
            }
            return ((Number)this.value).doubleValue();
        }

        @Override
        public final boolean asBoolean() {
            return Evals.asBoolean(this.asDouble());
        }

        @Override
        @Nullable
        public Object[] asArray() {
            Object[] objectArray;
            if (this.value == null) {
                objectArray = null;
            } else {
                Object[] objectArray2 = new Object[1];
                objectArray = objectArray2;
                objectArray2[0] = this.valueOrDefault().doubleValue();
            }
            return objectArray;
        }

        @Override
        public final ExprEval castTo(ExpressionType castTo) {
            switch ((ExprType)castTo.getType()) {
                case DOUBLE: {
                    return this;
                }
                case LONG: {
                    if (this.value == null) {
                        return ExprEval.ofLong(null);
                    }
                    return ExprEval.of(this.asLong());
                }
                case STRING: {
                    return ExprEval.of(this.asString());
                }
                case ARRAY: {
                    switch ((ExprType)castTo.getElementType().getType()) {
                        case DOUBLE: {
                            return ExprEval.ofDoubleArray(this.asArray());
                        }
                        case LONG: {
                            Object[] objectArray;
                            if (this.value == null) {
                                objectArray = null;
                            } else {
                                Object[] objectArray2 = new Object[1];
                                objectArray = objectArray2;
                                objectArray2[0] = ((Number)this.value).longValue();
                            }
                            return ExprEval.ofLongArray(objectArray);
                        }
                        case STRING: {
                            Object[] objectArray;
                            if (this.value == null) {
                                objectArray = null;
                            } else {
                                Object[] objectArray3 = new Object[1];
                                objectArray = objectArray3;
                                objectArray3[0] = ((Number)this.value).toString();
                            }
                            return ExprEval.ofStringArray(objectArray);
                        }
                    }
                    ExpressionType elementType = (ExpressionType)castTo.getElementType();
                    return new ArrayExprEval(castTo, new Object[]{this.castTo(elementType).value()});
                }
                case COMPLEX: {
                    if (!ExpressionType.NESTED_DATA.equals(castTo)) break;
                    return new NestedDataExprEval(this.value);
                }
            }
            throw DoubleExprEval.invalidCast(this.type(), castTo);
        }

        @Override
        public Expr toExpr() {
            if (this.value == null) {
                return new NullDoubleExpr();
            }
            return new DoubleExpr(((Number)this.value).doubleValue());
        }
    }

    private static abstract class NumericExprEval
    extends ExprEval<Number> {
        private NumericExprEval(@Nullable Number value) {
            super(value);
        }

        @Override
        public final int asInt() {
            if (this.value == null) {
                return 0;
            }
            return ((Number)this.value).intValue();
        }

        @Override
        public final long asLong() {
            if (this.value == null) {
                return 0L;
            }
            return ((Number)this.value).longValue();
        }

        @Override
        public final double asDouble() {
            if (this.value == null) {
                return 0.0;
            }
            return ((Number)this.value).doubleValue();
        }

        @Override
        public boolean isNumericNull() {
            return this.value == null;
        }
    }
}

