/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.frame.file;

import com.google.common.math.IntMath;
import it.unimi.dsi.fastutil.ints.IntObjectPair;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.math.RoundingMode;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.IntStream;
import org.apache.druid.frame.Frame;
import org.apache.druid.frame.FrameType;
import org.apache.druid.frame.TestArrayCursorFactory;
import org.apache.druid.frame.file.FrameFile;
import org.apache.druid.frame.read.FrameReader;
import org.apache.druid.frame.testutil.FrameSequenceBuilder;
import org.apache.druid.frame.testutil.FrameTestUtil;
import org.apache.druid.java.util.common.StringUtils;
import org.apache.druid.java.util.common.guava.Sequence;
import org.apache.druid.java.util.common.guava.Sequences;
import org.apache.druid.segment.CursorFactory;
import org.apache.druid.segment.QueryableIndexCursorFactory;
import org.apache.druid.segment.RowAdapters;
import org.apache.druid.segment.RowBasedCursorFactory;
import org.apache.druid.segment.TestIndex;
import org.apache.druid.segment.column.RowSignature;
import org.apache.druid.segment.incremental.IncrementalIndexCursorFactory;
import org.apache.druid.testing.InitializedNullHandlingTest;
import org.hamcrest.Matcher;
import org.hamcrest.Matchers;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Assume;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.rules.TemporaryFolder;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

@RunWith(value=Parameterized.class)
public class FrameFileTest
extends InitializedNullHandlingTest {
    private static final Map<FrameFileKey, byte[]> FRAME_FILES = new HashMap<FrameFileKey, byte[]>();
    private static final int PARTITION_SIZE = 99;
    private static final int SKIP_PARTITION = 13;
    @Rule
    public TemporaryFolder temporaryFolder = new TemporaryFolder();
    @Rule
    public ExpectedException expectedException = ExpectedException.none();
    private final FrameType frameType;
    private final int maxRowsPerFrame;
    private final boolean partitioned;
    private final AdapterType adapterType;
    private final int maxMmapSize;
    private CursorFactory cursorFactory;
    private int rowCount;
    private File file;

    public FrameFileTest(FrameType frameType, int maxRowsPerFrame, boolean partitioned, AdapterType adapterType, int maxMmapSize) {
        this.frameType = frameType;
        this.maxRowsPerFrame = maxRowsPerFrame;
        this.partitioned = partitioned;
        this.adapterType = adapterType;
        this.maxMmapSize = maxMmapSize;
    }

    @Parameterized.Parameters(name="frameType = {0}, maxRowsPerFrame = {1}, partitioned = {2}, adapter = {3}, maxMmapSize = {4}")
    public static Iterable<Object[]> constructorFeeder() {
        ArrayList<Object[]> constructors = new ArrayList<Object[]>();
        for (FrameType frameType : FrameType.values()) {
            for (int maxRowsPerFrame : new int[]{1, 17, 50, 99, Integer.MAX_VALUE}) {
                for (boolean partitioned : new boolean[]{true, false}) {
                    for (AdapterType adapterType : AdapterType.values()) {
                        int[] maxMmapSizes = maxRowsPerFrame == 1 ? new int[]{1000, 10000, Integer.MAX_VALUE} : new int[]{Integer.MAX_VALUE};
                        for (int maxMmapSize : maxMmapSizes) {
                            constructors.add(new Object[]{frameType, maxRowsPerFrame, partitioned, adapterType, maxMmapSize});
                        }
                    }
                }
            }
        }
        return constructors;
    }

    @Before
    public void setUp() throws IOException {
        this.cursorFactory = this.adapterType.getCursorFactory();
        this.rowCount = this.adapterType.getRowCount();
        this.file = this.temporaryFolder.newFile();
        try (OutputStream out = Files.newOutputStream(this.file.toPath(), new OpenOption[0]);){
            FrameFileKey frameFileKey = new FrameFileKey(this.adapterType, this.frameType, this.maxRowsPerFrame, this.partitioned);
            byte[] frameFileBytes = FRAME_FILES.computeIfAbsent(frameFileKey, FrameFileTest::computeFrameFile);
            out.write(frameFileBytes);
        }
    }

    @AfterClass
    public static void afterClass() {
        FRAME_FILES.clear();
    }

    @Test
    public void test_numFrames() throws IOException {
        try (FrameFile frameFile = FrameFile.open((File)this.file, (int)this.maxMmapSize, null, (FrameFile.Flag[])new FrameFile.Flag[0]);){
            Assert.assertEquals((long)this.computeExpectedNumFrames(), (long)frameFile.numFrames());
        }
    }

    @Test
    public void test_numPartitions() throws IOException {
        try (FrameFile frameFile = FrameFile.open((File)this.file, (int)this.maxMmapSize, null, (FrameFile.Flag[])new FrameFile.Flag[0]);){
            Assert.assertEquals((long)this.computeExpectedNumPartitions(), (long)frameFile.numPartitions());
        }
    }

    @Test
    public void test_frame_first() throws IOException {
        try (FrameFile frameFile = FrameFile.open((File)this.file, (int)this.maxMmapSize, null, (FrameFile.Flag[])new FrameFile.Flag[0]);){
            Assume.assumeThat((Object)frameFile.numFrames(), (Matcher)Matchers.greaterThan((Comparable)Integer.valueOf(0)));
            Frame firstFrame = frameFile.frame(0);
            Assert.assertEquals((long)Math.min(this.rowCount, this.maxRowsPerFrame), (long)firstFrame.numRows());
        }
    }

    @Test
    public void test_frame_last() throws IOException {
        try (FrameFile frameFile = FrameFile.open((File)this.file, (int)this.maxMmapSize, null, (FrameFile.Flag[])new FrameFile.Flag[0]);){
            Assume.assumeThat((Object)frameFile.numFrames(), (Matcher)Matchers.greaterThan((Comparable)Integer.valueOf(0)));
            Frame lastFrame = frameFile.frame(frameFile.numFrames() - 1);
            Assert.assertEquals((long)(this.rowCount % this.maxRowsPerFrame != 0 ? (long)(this.rowCount % this.maxRowsPerFrame) : (long)Math.min(this.rowCount, this.maxRowsPerFrame)), (long)lastFrame.numRows());
        }
    }

    @Test
    public void test_frame_outOfBoundsNegative() throws IOException {
        try (FrameFile frameFile = FrameFile.open((File)this.file, (int)this.maxMmapSize, null, (FrameFile.Flag[])new FrameFile.Flag[0]);){
            this.expectedException.expect(IllegalArgumentException.class);
            this.expectedException.expectMessage("Frame [-1] out of bounds");
            frameFile.frame(-1);
        }
    }

    @Test
    public void test_frame_outOfBoundsTooLarge() throws IOException {
        try (FrameFile frameFile = FrameFile.open((File)this.file, (int)this.maxMmapSize, null, (FrameFile.Flag[])new FrameFile.Flag[0]);){
            this.expectedException.expect(IllegalArgumentException.class);
            this.expectedException.expectMessage(StringUtils.format((String)"Frame [%,d] out of bounds", (Object[])new Object[]{frameFile.numFrames()}));
            frameFile.frame(frameFile.numFrames());
        }
    }

    @Test
    public void test_frame_readAllDataViaCursorFactory() throws IOException {
        FrameReader frameReader = FrameReader.create((RowSignature)this.cursorFactory.getRowSignature());
        try (FrameFile frameFile = FrameFile.open((File)this.file, (int)this.maxMmapSize, null, (FrameFile.Flag[])new FrameFile.Flag[0]);){
            Sequence frameFileRows = Sequences.concat(() -> IntStream.range(0, frameFile.numFrames()).mapToObj(arg_0 -> ((FrameFile)frameFile).frame(arg_0)).map(arg_0 -> ((FrameReader)frameReader).makeCursorFactory(arg_0)).map(FrameTestUtil::readRowsFromCursorFactoryWithRowNumber).iterator());
            Sequence<List<Object>> adapterRows = FrameTestUtil.readRowsFromCursorFactoryWithRowNumber(this.cursorFactory);
            FrameTestUtil.assertRowsEqual(adapterRows, (Sequence<List<Object>>)frameFileRows);
        }
    }

    @Test
    public void test_getPartitionStartFrame() throws IOException {
        try (FrameFile frameFile = FrameFile.open((File)this.file, (int)this.maxMmapSize, null, (FrameFile.Flag[])new FrameFile.Flag[0]);){
            if (this.partitioned) {
                for (int partitionNum = 0; partitionNum < frameFile.numPartitions(); ++partitionNum) {
                    Assert.assertEquals((String)("partition #" + partitionNum), (long)Math.min(IntMath.divide((int)((partitionNum >= 13 ? partitionNum + 1 : partitionNum) * 99), (int)this.maxRowsPerFrame, (RoundingMode)RoundingMode.CEILING), frameFile.numFrames()), (long)frameFile.getPartitionStartFrame(partitionNum));
                }
            } else {
                Assert.assertEquals((long)frameFile.numFrames(), (long)frameFile.getPartitionStartFrame(0));
            }
        }
    }

    @Test
    public void test_file() throws IOException {
        try (FrameFile frameFile = FrameFile.open((File)this.file, (int)this.maxMmapSize, null, (FrameFile.Flag[])new FrameFile.Flag[0]);){
            Assert.assertEquals((Object)this.file, (Object)frameFile.file());
        }
    }

    @Test
    public void test_open_withDeleteOnClose() throws IOException {
        FrameFile.open((File)this.file, (int)this.maxMmapSize, null, (FrameFile.Flag[])new FrameFile.Flag[0]).close();
        Assert.assertTrue((boolean)this.file.exists());
        FrameFile.open((File)this.file, null, (FrameFile.Flag[])new FrameFile.Flag[]{FrameFile.Flag.DELETE_ON_CLOSE}).close();
        Assert.assertFalse((boolean)this.file.exists());
    }

    @Test
    public void test_newReference() throws IOException {
        FrameFile frameFile1 = FrameFile.open((File)this.file, null, (FrameFile.Flag[])new FrameFile.Flag[]{FrameFile.Flag.DELETE_ON_CLOSE});
        FrameFile frameFile2 = frameFile1.newReference();
        FrameFile frameFile3 = frameFile2.newReference();
        frameFile1.close();
        Assert.assertTrue((boolean)this.file.exists());
        FrameFile frameFile4 = frameFile1.newReference();
        frameFile2.close();
        frameFile2.close();
        frameFile2.close();
        frameFile2.close();
        frameFile2.close();
        frameFile2.close();
        Assert.assertTrue((boolean)this.file.exists());
        frameFile3.close();
        Assert.assertTrue((boolean)this.file.exists());
        frameFile4.close();
        Assert.assertFalse((boolean)this.file.exists());
        this.expectedException.expect(IllegalStateException.class);
        this.expectedException.expectMessage("Frame file is closed");
        frameFile1.newReference();
    }

    private int computeExpectedNumFrames() {
        return IntMath.divide((int)FrameFileTest.countRows(this.cursorFactory), (int)this.maxRowsPerFrame, (RoundingMode)RoundingMode.CEILING);
    }

    private int computeExpectedNumPartitions() {
        if (this.partitioned) {
            return Math.min(this.computeExpectedNumFrames(), IntMath.divide((int)FrameFileTest.countRows(this.cursorFactory), (int)99, (RoundingMode)RoundingMode.CEILING));
        }
        return 0;
    }

    private static int countRows(CursorFactory cursorFactory) {
        return (Integer)FrameTestUtil.readRowsFromCursorFactory(cursorFactory, RowSignature.empty(), false).accumulate((Object)0, (i, in) -> i + 1);
    }

    private static byte[] computeFrameFile(FrameFileKey frameFileKey) {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        try {
            if (frameFileKey.partitioned) {
                FrameTestUtil.writeFrameFileWithPartitions((Sequence<IntObjectPair<Frame>>)FrameSequenceBuilder.fromCursorFactory(frameFileKey.adapterType.getCursorFactory()).frameType(frameFileKey.frameType).maxRowsPerFrame(frameFileKey.maxRowsPerFrame).frames().map((Function)new Function<Frame, IntObjectPair<Frame>>(){
                    private int rows = 0;

                    @Override
                    public IntObjectPair<Frame> apply(Frame frame) {
                        int partitionNum = this.rows / 99;
                        this.rows += frame.numRows();
                        return IntObjectPair.of((int)(partitionNum >= 13 ? partitionNum + 1 : partitionNum), (Object)frame);
                    }
                }), baos);
            } else {
                FrameTestUtil.writeFrameFile(FrameSequenceBuilder.fromCursorFactory(frameFileKey.adapterType.getCursorFactory()).frameType(frameFileKey.frameType).maxRowsPerFrame(frameFileKey.maxRowsPerFrame).frames(), baos);
            }
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        return baos.toByteArray();
    }

    static enum AdapterType {
        INCREMENTAL{

            @Override
            CursorFactory getCursorFactory() {
                return new IncrementalIndexCursorFactory(TestIndex.getNoRollupIncrementalTestIndex());
            }

            @Override
            int getRowCount() {
                return TestIndex.getNoRollupIncrementalTestIndex().numRows();
            }
        }
        ,
        MMAP{

            @Override
            CursorFactory getCursorFactory() {
                return new QueryableIndexCursorFactory(TestIndex.getNoRollupMMappedTestIndex());
            }

            @Override
            int getRowCount() {
                return TestIndex.getNoRollupMMappedTestIndex().getNumRows();
            }
        }
        ,
        MV_AS_STRING_ARRAYS{

            @Override
            CursorFactory getCursorFactory() {
                return new TestArrayCursorFactory(TestIndex.getNoRollupMMappedTestIndex());
            }

            @Override
            int getRowCount() {
                return TestIndex.getNoRollupMMappedTestIndex().getNumRows();
            }
        }
        ,
        EMPTY{

            @Override
            CursorFactory getCursorFactory() {
                return new RowBasedCursorFactory(Sequences.empty(), RowAdapters.standardRow(), RowSignature.empty());
            }

            @Override
            int getRowCount() {
                return 0;
            }
        };


        abstract CursorFactory getCursorFactory();

        abstract int getRowCount();
    }

    private static class FrameFileKey {
        final AdapterType adapterType;
        final FrameType frameType;
        final int maxRowsPerFrame;
        final boolean partitioned;

        public FrameFileKey(AdapterType adapterType, FrameType frameType, int maxRowsPerFrame, boolean partitioned) {
            this.adapterType = adapterType;
            this.frameType = frameType;
            this.maxRowsPerFrame = maxRowsPerFrame;
            this.partitioned = partitioned;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            FrameFileKey that = (FrameFileKey)o;
            return this.maxRowsPerFrame == that.maxRowsPerFrame && this.partitioned == that.partitioned && this.adapterType == that.adapterType && this.frameType == that.frameType;
        }

        public int hashCode() {
            return Objects.hash(new Object[]{this.adapterType, this.frameType, this.maxRowsPerFrame, this.partitioned});
        }

        public String toString() {
            return "FrameFileKey{adapterType=" + String.valueOf((Object)this.adapterType) + ", frameType=" + String.valueOf(this.frameType) + ", maxRowsPerFrame=" + this.maxRowsPerFrame + ", partitioned=" + this.partitioned + "}";
        }
    }
}

