/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.segment.loading;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.annotations.VisibleForTesting;
import com.google.inject.Inject;
import java.io.File;
import java.io.IOException;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.apache.druid.guice.annotations.Json;
import org.apache.druid.java.util.common.FileUtils;
import org.apache.druid.java.util.common.ISE;
import org.apache.druid.java.util.emitter.EmittingLogger;
import org.apache.druid.segment.loading.DataSegmentPusher;
import org.apache.druid.segment.loading.LeastBytesUsedStorageLocationSelectorStrategy;
import org.apache.druid.segment.loading.LoadSpec;
import org.apache.druid.segment.loading.SegmentCacheManager;
import org.apache.druid.segment.loading.SegmentLoaderConfig;
import org.apache.druid.segment.loading.SegmentLoadingException;
import org.apache.druid.segment.loading.StorageLocation;
import org.apache.druid.segment.loading.StorageLocationSelectorStrategy;
import org.apache.druid.timeline.DataSegment;

public class SegmentLocalCacheManager
implements SegmentCacheManager {
    @VisibleForTesting
    static final String DOWNLOAD_START_MARKER_FILE_NAME = "downloadStartMarker";
    private static final EmittingLogger log = new EmittingLogger(SegmentLocalCacheManager.class);
    private final SegmentLoaderConfig config;
    private final ObjectMapper jsonMapper;
    private final List<StorageLocation> locations;
    private final Object directoryWriteRemoveLock = new Object();
    private final ConcurrentHashMap<DataSegment, ReferenceCountingLock> segmentLocks = new ConcurrentHashMap();
    private final StorageLocationSelectorStrategy strategy;

    @Inject
    public SegmentLocalCacheManager(List<StorageLocation> locations, SegmentLoaderConfig config, @Nonnull StorageLocationSelectorStrategy strategy, @Json ObjectMapper mapper) {
        this.config = config;
        this.jsonMapper = mapper;
        this.locations = locations;
        this.strategy = strategy;
        log.info("Using storage location strategy: [%s]", new Object[]{this.strategy.getClass().getSimpleName()});
    }

    @VisibleForTesting
    SegmentLocalCacheManager(SegmentLoaderConfig config, @Nonnull StorageLocationSelectorStrategy strategy, @Json ObjectMapper mapper) {
        this(config.toStorageLocations(), config, strategy, mapper);
    }

    @VisibleForTesting
    public SegmentLocalCacheManager(SegmentLoaderConfig config, @Json ObjectMapper mapper) {
        this.config = config;
        this.jsonMapper = mapper;
        this.locations = config.toStorageLocations();
        this.strategy = new LeastBytesUsedStorageLocationSelectorStrategy(this.locations);
        log.info("Using storage location strategy: [%s]", new Object[]{this.strategy.getClass().getSimpleName()});
    }

    static String getSegmentDir(DataSegment segment) {
        return DataSegmentPusher.getDefaultStorageDir((DataSegment)segment, (boolean)false);
    }

    @Override
    public boolean isSegmentCached(DataSegment segment) {
        return this.findStoragePathIfCached(segment) != null;
    }

    @Nullable
    private File findStoragePathIfCached(DataSegment segment) {
        for (StorageLocation location : this.locations) {
            String storageDir;
            File localStorageDir = location.segmentDirectoryAsFile(storageDir = SegmentLocalCacheManager.getSegmentDir(segment));
            if (!localStorageDir.exists()) continue;
            if (this.checkSegmentFilesIntact(localStorageDir)) {
                log.warn("[%s] may be damaged. Delete all the segment files and pull from DeepStorage again.", new Object[]{localStorageDir.getAbsolutePath()});
                this.cleanupCacheFiles(location.getPath(), localStorageDir);
                location.removeSegmentDir(localStorageDir, segment);
                break;
            }
            location.maybeReserve(storageDir, segment);
            return localStorageDir;
        }
        return null;
    }

    private boolean checkSegmentFilesIntact(File dir) {
        return this.checkSegmentFilesIntactWithStartMarker(dir);
    }

    private boolean checkSegmentFilesIntactWithStartMarker(File localStorageDir) {
        File downloadStartMarker = new File(localStorageDir.getPath(), DOWNLOAD_START_MARKER_FILE_NAME);
        return downloadStartMarker.exists();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public File getSegmentFiles(DataSegment segment) throws SegmentLoadingException {
        ReferenceCountingLock lock;
        ReferenceCountingLock referenceCountingLock = lock = this.createOrGetLock(segment);
        synchronized (referenceCountingLock) {
            try {
                File segmentDir = this.findStoragePathIfCached(segment);
                if (segmentDir != null) {
                    File file = segmentDir;
                    return file;
                }
                File file = this.loadSegmentWithRetry(segment);
                return file;
            }
            finally {
                this.unlock(segment, lock);
            }
        }
    }

    private File loadSegmentWithRetry(DataSegment segment) throws SegmentLoadingException {
        String segmentDir = SegmentLocalCacheManager.getSegmentDir(segment);
        for (StorageLocation loc : this.locations) {
            if (!loc.isReserved(segmentDir)) continue;
            File storageDir = loc.segmentDirectoryAsFile(segmentDir);
            boolean success = this.loadInLocationWithStartMarkerQuietly(loc, segment, storageDir, false);
            if (!success) {
                throw new SegmentLoadingException("Failed to load segment %s in reserved location [%s]", new Object[]{segment.getId(), loc.getPath().getAbsolutePath()});
            }
            return storageDir;
        }
        Iterator<StorageLocation> locationsIterator = this.strategy.getLocations();
        while (locationsIterator.hasNext()) {
            boolean success;
            StorageLocation loc;
            loc = locationsIterator.next();
            File storageDir = loc.reserve(segmentDir, segment);
            if (storageDir == null || !(success = this.loadInLocationWithStartMarkerQuietly(loc, segment, storageDir, true))) continue;
            return storageDir;
        }
        throw new SegmentLoadingException("Failed to load segment %s in all locations.", new Object[]{segment.getId()});
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean loadInLocationWithStartMarkerQuietly(StorageLocation loc, DataSegment segment, File storageDir, boolean releaseLocation) {
        try {
            this.loadInLocationWithStartMarker(segment, storageDir);
            return true;
        }
        catch (SegmentLoadingException e) {
            try {
                log.makeAlert((Throwable)e, "Failed to load segment in current location [%s], try next location if any", new Object[]{loc.getPath().getAbsolutePath()}).addData("location", (Object)loc.getPath().getAbsolutePath()).emit();
            }
            finally {
                if (releaseLocation) {
                    loc.removeSegmentDir(storageDir, segment);
                }
                this.cleanupCacheFiles(loc.getPath(), storageDir);
            }
            return false;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void loadInLocationWithStartMarker(DataSegment segment, File storageDir) throws SegmentLoadingException {
        File downloadStartMarker = new File(storageDir, DOWNLOAD_START_MARKER_FILE_NAME);
        Object object = this.directoryWriteRemoveLock;
        synchronized (object) {
            if (!storageDir.mkdirs()) {
                log.debug("Unable to make parent file[%s]", new Object[]{storageDir});
            }
            try {
                if (!downloadStartMarker.createNewFile()) {
                    throw new SegmentLoadingException("Was not able to create new download marker for [%s]", new Object[]{storageDir});
                }
            }
            catch (IOException e) {
                throw new SegmentLoadingException((Throwable)e, "Unable to create marker file for [%s]", new Object[]{storageDir});
            }
        }
        this.loadInLocation(segment, storageDir);
        if (!downloadStartMarker.delete()) {
            throw new SegmentLoadingException("Unable to remove marker file for [%s]", new Object[]{storageDir});
        }
    }

    private void loadInLocation(DataSegment segment, File storageDir) throws SegmentLoadingException {
        LoadSpec loadSpec = (LoadSpec)this.jsonMapper.convertValue((Object)segment.getLoadSpec(), LoadSpec.class);
        LoadSpec.LoadSpecResult result = loadSpec.loadSegment(storageDir);
        if (result.getSize() != segment.getSize()) {
            log.warn("Segment [%s] is different than expected size. Expected [%d] found [%d]", new Object[]{segment.getId(), segment.getSize(), result.getSize()});
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public boolean reserve(DataSegment segment) {
        ReferenceCountingLock lock;
        ReferenceCountingLock referenceCountingLock = lock = this.createOrGetLock(segment);
        synchronized (referenceCountingLock) {
            try {
                StorageLocation location2;
                if (null != this.findStoragePathIfCached(segment)) {
                    boolean bl = true;
                    return bl;
                }
                String storageDirStr = SegmentLocalCacheManager.getSegmentDir(segment);
                for (StorageLocation location2 : this.locations) {
                    if (!location2.isReserved(storageDirStr)) continue;
                    boolean bl = true;
                    return bl;
                }
                Iterator<StorageLocation> it = this.strategy.getLocations();
                do {
                    if (!it.hasNext()) return false;
                } while (null == (location2 = it.next()).reserve(storageDirStr, segment));
                boolean bl = true;
                return bl;
            }
            finally {
                this.unlock(segment, lock);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public boolean release(DataSegment segment) {
        ReferenceCountingLock lock;
        ReferenceCountingLock referenceCountingLock = lock = this.createOrGetLock(segment);
        synchronized (referenceCountingLock) {
            try {
                StorageLocation location;
                String storageDir = SegmentLocalCacheManager.getSegmentDir(segment);
                Iterator<StorageLocation> iterator = this.locations.iterator();
                do {
                    if (!iterator.hasNext()) return false;
                } while (!(location = iterator.next()).isReserved(storageDir));
                File localStorageDir = location.segmentDirectoryAsFile(storageDir);
                if (localStorageDir.exists()) {
                    throw new ISE("Asking to release a location '%s' while the segment directory '%s' is present on disk. Any state on disk must be deleted before releasing", new Object[]{location.getPath().getAbsolutePath(), localStorageDir.getAbsolutePath()});
                }
                boolean bl = location.release(storageDir, segment.getSize());
                return bl;
            }
            finally {
                this.unlock(segment, lock);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void cleanup(DataSegment segment) {
        ReferenceCountingLock lock;
        if (!this.config.isDeleteOnRemove()) {
            return;
        }
        ReferenceCountingLock referenceCountingLock = lock = this.createOrGetLock(segment);
        synchronized (referenceCountingLock) {
            try {
                File loc = this.findStoragePathIfCached(segment);
                if (loc == null) {
                    log.warn("Asked to cleanup something[%s] that didn't exist.  Skipping.", new Object[]{segment.getId()});
                    return;
                }
                for (StorageLocation location : this.locations) {
                    File localStorageDir = new File(location.getPath(), SegmentLocalCacheManager.getSegmentDir(segment));
                    if (!localStorageDir.exists()) continue;
                    this.cleanupCacheFiles(location.getPath(), localStorageDir);
                    location.removeSegmentDir(localStorageDir, segment);
                }
            }
            finally {
                this.unlock(segment, lock);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void cleanupCacheFiles(File baseFile, File cacheFile) {
        File[] children;
        if (cacheFile.equals(baseFile)) {
            return;
        }
        Object object = this.directoryWriteRemoveLock;
        synchronized (object) {
            log.info("Deleting directory[%s]", new Object[]{cacheFile});
            try {
                FileUtils.deleteDirectory((File)cacheFile);
            }
            catch (Exception e) {
                log.error((Throwable)e, "Unable to remove directory[%s]", new Object[]{cacheFile});
            }
        }
        File parent = cacheFile.getParentFile();
        if (parent != null && ((children = parent.listFiles()) == null || children.length == 0)) {
            this.cleanupCacheFiles(baseFile, parent);
        }
    }

    private ReferenceCountingLock createOrGetLock(DataSegment dataSegment) {
        return this.segmentLocks.compute(dataSegment, (segment, lock) -> {
            ReferenceCountingLock nonNullLock = lock == null ? new ReferenceCountingLock() : lock;
            nonNullLock.increment();
            return nonNullLock;
        });
    }

    private void unlock(DataSegment dataSegment, ReferenceCountingLock lock) {
        this.segmentLocks.compute(dataSegment, (segment, existingLock) -> {
            if (existingLock == null) {
                throw new ISE("Lock has already been removed", new Object[0]);
            }
            if (existingLock != lock) {
                throw new ISE("Different lock instance", new Object[0]);
            }
            if (((ReferenceCountingLock)existingLock).numReferences == 1) {
                return null;
            }
            ((ReferenceCountingLock)existingLock).decrement();
            return existingLock;
        });
    }

    @VisibleForTesting
    public ConcurrentHashMap<DataSegment, ReferenceCountingLock> getSegmentLocks() {
        return this.segmentLocks;
    }

    @VisibleForTesting
    public List<StorageLocation> getLocations() {
        return this.locations;
    }

    @VisibleForTesting
    private static class ReferenceCountingLock {
        private int numReferences;

        private ReferenceCountingLock() {
        }

        private void increment() {
            ++this.numReferences;
        }

        private void decrement() {
            --this.numReferences;
        }
    }
}

