/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.server;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Ordering;
import com.google.inject.Inject;
import java.io.Closeable;
import java.io.IOException;
import java.util.Comparator;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import org.apache.druid.common.guava.SettableSupplier;
import org.apache.druid.java.util.common.ISE;
import org.apache.druid.java.util.common.Intervals;
import org.apache.druid.java.util.common.io.Closer;
import org.apache.druid.java.util.emitter.EmittingLogger;
import org.apache.druid.query.TableDataSource;
import org.apache.druid.query.planning.DataSourceAnalysis;
import org.apache.druid.segment.ReferenceCountingSegment;
import org.apache.druid.segment.SegmentLazyLoadFailCallback;
import org.apache.druid.segment.join.table.IndexedTable;
import org.apache.druid.segment.join.table.ReferenceCountingIndexedTable;
import org.apache.druid.segment.loading.SegmentLoader;
import org.apache.druid.segment.loading.SegmentLoadingException;
import org.apache.druid.timeline.DataSegment;
import org.apache.druid.timeline.SegmentId;
import org.apache.druid.timeline.VersionedIntervalTimeline;
import org.apache.druid.timeline.partition.PartitionChunk;
import org.apache.druid.timeline.partition.ShardSpec;
import org.apache.druid.utils.CollectionUtils;

public class SegmentManager {
    private static final EmittingLogger log = new EmittingLogger(SegmentManager.class);
    private final SegmentLoader segmentLoader;
    private final ConcurrentHashMap<String, DataSourceState> dataSources = new ConcurrentHashMap();

    @Inject
    public SegmentManager(SegmentLoader segmentLoader) {
        this.segmentLoader = segmentLoader;
    }

    @VisibleForTesting
    Map<String, DataSourceState> getDataSources() {
        return this.dataSources;
    }

    public Map<String, Long> getDataSourceSizes() {
        return CollectionUtils.mapValues(this.dataSources, DataSourceState::getTotalSegmentSize);
    }

    public Set<String> getDataSourceNames() {
        return this.dataSources.keySet();
    }

    public Map<String, Long> getDataSourceCounts() {
        return CollectionUtils.mapValues(this.dataSources, DataSourceState::getNumSegments);
    }

    public Optional<VersionedIntervalTimeline<String, ReferenceCountingSegment>> getTimeline(DataSourceAnalysis analysis) {
        TableDataSource tableDataSource = this.getTableDataSource(analysis);
        return Optional.ofNullable(this.dataSources.get(tableDataSource.getName())).map(DataSourceState::getTimeline);
    }

    public Optional<Stream<ReferenceCountingIndexedTable>> getIndexedTables(DataSourceAnalysis analysis) {
        return this.getTimeline(analysis).map(timeline -> {
            Stream<ReferenceCountingIndexedTable> segments = timeline.lookup(Intervals.ETERNITY).stream().flatMap(x -> StreamSupport.stream(x.getObject().payloads().spliterator(), false));
            TableDataSource tableDataSource = this.getTableDataSource(analysis);
            ConcurrentHashMap tables = Optional.ofNullable(this.dataSources.get(tableDataSource.getName())).map(DataSourceState::getTablesLookup).orElseThrow(() -> new ISE("Datasource %s does not have IndexedTables", new Object[]{tableDataSource.getName()}));
            return segments.map(segment -> (ReferenceCountingIndexedTable)tables.get(segment.getId())).filter(Objects::nonNull);
        });
    }

    public boolean hasIndexedTables(String dataSourceName) {
        if (this.dataSources.containsKey(dataSourceName)) {
            return this.dataSources.get(dataSourceName).tablesLookup.size() > 0;
        }
        return false;
    }

    private TableDataSource getTableDataSource(DataSourceAnalysis analysis) {
        return (TableDataSource)analysis.getBaseTableDataSource().orElseThrow(() -> new ISE("Cannot handle datasource: %s", new Object[]{analysis.getDataSource()}));
    }

    public boolean loadSegment(DataSegment segment, boolean lazy, SegmentLazyLoadFailCallback loadFailed) throws SegmentLoadingException {
        ReferenceCountingSegment adapter = this.getSegmentReference(segment, lazy, loadFailed);
        SettableSupplier resultSupplier = new SettableSupplier();
        this.dataSources.compute(segment.getDataSource(), (k, v) -> {
            DataSourceState dataSourceState = v == null ? new DataSourceState() : v;
            VersionedIntervalTimeline<String, ReferenceCountingSegment> loadedIntervals = dataSourceState.getTimeline();
            PartitionChunk entry = loadedIntervals.findChunk(segment.getInterval(), (Object)segment.getVersion(), segment.getShardSpec().getPartitionNum());
            if (entry != null) {
                log.warn("Told to load an adapter for segment[%s] that already exists", new Object[]{segment.getId()});
                resultSupplier.set((Object)false);
            } else {
                IndexedTable table = (IndexedTable)adapter.as(IndexedTable.class);
                if (table != null) {
                    if (dataSourceState.isEmpty() || dataSourceState.numSegments == (long)dataSourceState.tablesLookup.size()) {
                        dataSourceState.tablesLookup.put(segment.getId(), new ReferenceCountingIndexedTable(table));
                    } else {
                        log.error("Cannot load segment[%s] with IndexedTable, no existing segments are joinable", new Object[]{segment.getId()});
                    }
                } else if (dataSourceState.tablesLookup.size() > 0) {
                    log.error("Cannot load segment[%s] without IndexedTable, all existing segments are joinable", new Object[]{segment.getId()});
                }
                loadedIntervals.add(segment.getInterval(), (Object)segment.getVersion(), segment.getShardSpec().createChunk((Object)adapter));
                dataSourceState.addSegment(segment);
                resultSupplier.set((Object)true);
            }
            return dataSourceState;
        });
        return (Boolean)resultSupplier.get();
    }

    private ReferenceCountingSegment getSegmentReference(DataSegment dataSegment, boolean lazy, SegmentLazyLoadFailCallback loadFailed) throws SegmentLoadingException {
        ReferenceCountingSegment segment;
        try {
            segment = this.segmentLoader.getSegment(dataSegment, lazy, loadFailed);
        }
        catch (SegmentLoadingException e) {
            this.segmentLoader.cleanup(dataSegment);
            throw e;
        }
        if (segment == null) {
            throw new SegmentLoadingException("Null adapter from loadSpec[%s]", new Object[]{dataSegment.getLoadSpec()});
        }
        return segment;
    }

    public void dropSegment(DataSegment segment) {
        String dataSource = segment.getDataSource();
        this.dataSources.compute(dataSource, (dataSourceName, dataSourceState) -> {
            block18: {
                ReferenceCountingSegment oldQueryable;
                if (dataSourceState == null) {
                    log.info("Told to delete a queryable for a dataSource[%s] that doesn't exist.", new Object[]{dataSourceName});
                    return null;
                }
                VersionedIntervalTimeline<String, ReferenceCountingSegment> loadedIntervals = dataSourceState.getTimeline();
                ShardSpec shardSpec = segment.getShardSpec();
                PartitionChunk removed = loadedIntervals.remove(segment.getInterval(), (Object)segment.getVersion(), segment.getShardSpec().createChunk((Object)ReferenceCountingSegment.wrapSegment(null, (ShardSpec)shardSpec)));
                ReferenceCountingSegment referenceCountingSegment = oldQueryable = removed == null ? null : (ReferenceCountingSegment)removed.getObject();
                if (oldQueryable != null) {
                    try (Closer closer = Closer.create();){
                        ((DataSourceState)dataSourceState).removeSegment(segment);
                        closer.register((Closeable)oldQueryable);
                        log.info("Attempting to close segment %s", new Object[]{segment.getId()});
                        ReferenceCountingIndexedTable oldTable = (ReferenceCountingIndexedTable)((DataSourceState)dataSourceState).tablesLookup.remove(segment.getId());
                        if (oldTable != null) {
                            closer.register((Closeable)oldTable);
                        }
                        break block18;
                    }
                    catch (IOException e) {
                        throw new RuntimeException(e);
                    }
                }
                log.info("Told to delete a queryable on dataSource[%s] for interval[%s] and version[%s] that I don't have.", new Object[]{dataSourceName, segment.getInterval(), segment.getVersion()});
            }
            if (dataSourceState.isEmpty()) {
                return null;
            }
            DataSourceState dataSourceState2 = dataSourceState;
            return dataSourceState2;
        });
        this.segmentLoader.cleanup(segment);
    }

    public static class DataSourceState {
        private final VersionedIntervalTimeline<String, ReferenceCountingSegment> timeline = new VersionedIntervalTimeline((Comparator)Ordering.natural());
        private final ConcurrentHashMap<SegmentId, ReferenceCountingIndexedTable> tablesLookup = new ConcurrentHashMap();
        private long totalSegmentSize;
        private long numSegments;

        private void addSegment(DataSegment segment) {
            this.totalSegmentSize += segment.getSize();
            ++this.numSegments;
        }

        private void removeSegment(DataSegment segment) {
            this.totalSegmentSize -= segment.getSize();
            --this.numSegments;
        }

        public VersionedIntervalTimeline<String, ReferenceCountingSegment> getTimeline() {
            return this.timeline;
        }

        public ConcurrentHashMap<SegmentId, ReferenceCountingIndexedTable> getTablesLookup() {
            return this.tablesLookup;
        }

        public long getTotalSegmentSize() {
            return this.totalSegmentSize;
        }

        public long getNumSegments() {
            return this.numSegments;
        }

        public boolean isEmpty() {
            return this.numSegments == 0L;
        }
    }
}

