/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.server.coordinator.duty;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.inject.Inject;
import java.util.List;
import java.util.Set;
import javax.annotation.Nullable;
import org.apache.druid.client.indexing.IndexingServiceClient;
import org.apache.druid.java.util.common.DateTimes;
import org.apache.druid.java.util.common.JodaUtils;
import org.apache.druid.java.util.common.logger.Logger;
import org.apache.druid.metadata.SegmentsMetadataManager;
import org.apache.druid.server.coordinator.DruidCoordinatorConfig;
import org.apache.druid.server.coordinator.DruidCoordinatorRuntimeParams;
import org.apache.druid.server.coordinator.duty.CoordinatorDuty;
import org.apache.druid.utils.CollectionUtils;
import org.joda.time.DateTime;
import org.joda.time.Interval;

public class KillUnusedSegments
implements CoordinatorDuty {
    private static final Logger log = new Logger(KillUnusedSegments.class);
    private final long period;
    private final long retainDuration;
    private final boolean ignoreRetainDuration;
    private final int maxSegmentsToKill;
    private long lastKillTime = 0L;
    private final SegmentsMetadataManager segmentsMetadataManager;
    private final IndexingServiceClient indexingServiceClient;

    @Inject
    public KillUnusedSegments(SegmentsMetadataManager segmentsMetadataManager, IndexingServiceClient indexingServiceClient, DruidCoordinatorConfig config) {
        this.period = config.getCoordinatorKillPeriod().getMillis();
        Preconditions.checkArgument((this.period > config.getCoordinatorIndexingPeriod().getMillis() ? 1 : 0) != 0, (Object)"coordinator kill period must be greater than druid.coordinator.period.indexingPeriod");
        this.ignoreRetainDuration = config.getCoordinatorKillIgnoreDurationToRetain();
        this.retainDuration = config.getCoordinatorKillDurationToRetain().getMillis();
        if (this.ignoreRetainDuration) {
            log.debug("druid.coordinator.kill.durationToRetain [%s] will be ignored when discovering segments to kill because you have set druid.coordinator.kill.ignoreDurationToRetain to True.", new Object[]{this.retainDuration});
        }
        this.maxSegmentsToKill = config.getCoordinatorKillMaxSegments();
        Preconditions.checkArgument((this.maxSegmentsToKill > 0 ? 1 : 0) != 0, (Object)"coordinator kill maxSegments must be > 0");
        log.info("Kill Task scheduling enabled with period [%s], retainDuration [%s], maxSegmentsToKill [%s]", new Object[]{this.period, this.ignoreRetainDuration ? "IGNORING" : Long.valueOf(this.retainDuration), this.maxSegmentsToKill});
        this.segmentsMetadataManager = segmentsMetadataManager;
        this.indexingServiceClient = indexingServiceClient;
    }

    @Override
    public DruidCoordinatorRuntimeParams run(DruidCoordinatorRuntimeParams params) {
        Set<String> dataSourcesToKill = params.getCoordinatorDynamicConfig().getSpecificDataSourcesToKillUnusedSegmentsIn();
        if (CollectionUtils.isNullOrEmpty(dataSourcesToKill)) {
            dataSourcesToKill = this.segmentsMetadataManager.retrieveAllDataSourceNames();
        }
        if (dataSourcesToKill != null && dataSourcesToKill.size() > 0 && this.lastKillTime + this.period < System.currentTimeMillis()) {
            this.lastKillTime = System.currentTimeMillis();
            for (String dataSource : dataSourcesToKill) {
                Interval intervalToKill = this.findIntervalForKill(dataSource, this.maxSegmentsToKill);
                if (intervalToKill == null) continue;
                try {
                    this.indexingServiceClient.killUnusedSegments("coordinator-issued", dataSource, intervalToKill);
                }
                catch (Exception ex) {
                    log.error((Throwable)ex, "Failed to submit kill task for dataSource [%s]", new Object[]{dataSource});
                    if (!Thread.currentThread().isInterrupted()) continue;
                    log.warn("skipping kill task scheduling because thread is interrupted.", new Object[0]);
                    break;
                }
            }
        }
        return params;
    }

    @Nullable
    @VisibleForTesting
    Interval findIntervalForKill(String dataSource, int limit) {
        List<Interval> unusedSegmentIntervals = this.segmentsMetadataManager.getUnusedSegmentIntervals(dataSource, this.getEndTimeUpperLimit(), limit);
        if (unusedSegmentIntervals != null && unusedSegmentIntervals.size() > 0) {
            return JodaUtils.umbrellaInterval(unusedSegmentIntervals);
        }
        return null;
    }

    @VisibleForTesting
    DateTime getEndTimeUpperLimit() {
        return this.ignoreRetainDuration ? DateTimes.COMPARE_DATE_AS_STRING_MAX : DateTimes.nowUtc().minus(this.retainDuration);
    }

    @VisibleForTesting
    Long getRetainDuration() {
        return this.retainDuration;
    }
}

