/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.server.coordinator.duty;

import com.google.common.base.Preconditions;
import com.google.inject.Inject;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import org.apache.druid.client.indexing.IndexingServiceClient;
import org.apache.druid.java.util.common.DateTimes;
import org.apache.druid.java.util.common.JodaUtils;
import org.apache.druid.java.util.common.logger.Logger;
import org.apache.druid.metadata.SegmentsMetadataManager;
import org.apache.druid.server.coordinator.DruidCoordinatorConfig;
import org.apache.druid.server.coordinator.DruidCoordinatorRuntimeParams;
import org.apache.druid.server.coordinator.duty.CoordinatorDuty;
import org.apache.druid.utils.CollectionUtils;
import org.joda.time.DateTime;
import org.joda.time.Interval;

public class KillUnusedSegments
implements CoordinatorDuty {
    private static final Logger log = new Logger(KillUnusedSegments.class);
    private final long period;
    private final long retainDuration;
    private final boolean ignoreRetainDuration;
    private final int maxSegmentsToKill;
    private long lastKillTime = 0L;
    private final SegmentsMetadataManager segmentsMetadataManager;
    private final IndexingServiceClient indexingServiceClient;

    @Inject
    public KillUnusedSegments(SegmentsMetadataManager segmentsMetadataManager, IndexingServiceClient indexingServiceClient, DruidCoordinatorConfig config) {
        this.period = config.getCoordinatorKillPeriod().getMillis();
        Preconditions.checkArgument((this.period > config.getCoordinatorIndexingPeriod().getMillis() ? 1 : 0) != 0, (Object)"coordinator kill period must be greater than druid.coordinator.period.indexingPeriod");
        this.ignoreRetainDuration = config.getCoordinatorKillIgnoreDurationToRetain();
        this.retainDuration = config.getCoordinatorKillDurationToRetain().getMillis();
        if (this.ignoreRetainDuration) {
            log.debug("druid.coordinator.kill.durationToRetain [%s] will be ignored when discovering segments to kill because you have set druid.coordinator.kill.ignoreDurationToRetain to True.", new Object[]{this.retainDuration});
        }
        this.maxSegmentsToKill = config.getCoordinatorKillMaxSegments();
        Preconditions.checkArgument((this.maxSegmentsToKill > 0 ? 1 : 0) != 0, (Object)"coordinator kill maxSegments must be > 0");
        log.info("Kill Task scheduling enabled with period [%s], retainDuration [%s], maxSegmentsToKill [%s]", new Object[]{this.period, this.ignoreRetainDuration ? "IGNORING" : Long.valueOf(this.retainDuration), this.maxSegmentsToKill});
        this.segmentsMetadataManager = segmentsMetadataManager;
        this.indexingServiceClient = indexingServiceClient;
    }

    @Override
    public DruidCoordinatorRuntimeParams run(DruidCoordinatorRuntimeParams params) {
        Set<String> dataSourcesToKill = params.getCoordinatorDynamicConfig().getSpecificDataSourcesToKillUnusedSegmentsIn();
        if (CollectionUtils.isNullOrEmpty(dataSourcesToKill)) {
            dataSourcesToKill = this.segmentsMetadataManager.retrieveAllDataSourceNames();
        }
        long currentTimeMillis = System.currentTimeMillis();
        if (CollectionUtils.isNullOrEmpty(dataSourcesToKill)) {
            log.debug("No eligible datasource to kill unused segments.", new Object[0]);
        } else if (this.lastKillTime + this.period > currentTimeMillis) {
            log.debug("Skipping kill of unused segments as kill period has not elapsed yet.", new Object[0]);
        } else {
            log.debug("Killing unused segments in datasources: %s", new Object[]{dataSourcesToKill});
            this.lastKillTime = currentTimeMillis;
            this.killUnusedSegments(dataSourcesToKill);
        }
        return params;
    }

    private void killUnusedSegments(Collection<String> dataSourcesToKill) {
        int submittedTasks = 0;
        for (String dataSource : dataSourcesToKill) {
            Interval intervalToKill = this.findIntervalForKill(dataSource);
            if (intervalToKill == null) continue;
            try {
                this.indexingServiceClient.killUnusedSegments("coordinator-issued", dataSource, intervalToKill);
                ++submittedTasks;
            }
            catch (Exception ex) {
                log.error((Throwable)ex, "Failed to submit kill task for dataSource [%s]", new Object[]{dataSource});
                if (!Thread.currentThread().isInterrupted()) continue;
                log.warn("skipping kill task scheduling because thread is interrupted.", new Object[0]);
                break;
            }
        }
        log.debug("Submitted kill tasks for [%d] datasources.", new Object[]{submittedTasks});
    }

    private Interval findIntervalForKill(String dataSource) {
        DateTime maxEndTime = this.ignoreRetainDuration ? DateTimes.COMPARE_DATE_AS_STRING_MAX : DateTimes.nowUtc().minus(this.retainDuration);
        List<Interval> unusedSegmentIntervals = this.segmentsMetadataManager.getUnusedSegmentIntervals(dataSource, maxEndTime, this.maxSegmentsToKill);
        if (CollectionUtils.isNullOrEmpty(unusedSegmentIntervals)) {
            return null;
        }
        if (unusedSegmentIntervals.size() == 1) {
            return unusedSegmentIntervals.get(0);
        }
        return JodaUtils.umbrellaInterval(unusedSegmentIntervals);
    }
}

