/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.query.aggregation.histogram;

import com.fasterxml.jackson.annotation.JsonValue;
import com.google.common.base.Preconditions;
import com.google.common.primitives.Floats;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import javax.annotation.Nullable;
import org.apache.druid.query.aggregation.histogram.ArrayUtils;
import org.apache.druid.query.aggregation.histogram.Histogram;

public class ApproximateHistogram {
    public static final int DEFAULT_HISTOGRAM_SIZE = 50;
    public static final int DEFAULT_BUCKET_SIZE = 7;
    int size;
    public float[] positions;
    public long[] bins;
    int binCount;
    float min;
    float max;
    transient long count;
    transient float lowerLimit;
    transient float upperLimit;
    private static final long APPROX_FLAG_BIT = Long.MIN_VALUE;
    private static final long COUNT_BITS = Long.MAX_VALUE;

    public boolean equals(Object o) {
        int i;
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        ApproximateHistogram that = (ApproximateHistogram)o;
        if (this.size != that.size) {
            return false;
        }
        if (this.binCount != that.binCount) {
            return false;
        }
        if (Float.compare(that.max, this.max) != 0) {
            return false;
        }
        if (Float.compare(that.min, this.min) != 0) {
            return false;
        }
        for (i = 0; i < this.binCount; ++i) {
            if (this.positions[i] == that.positions[i]) continue;
            return false;
        }
        for (i = 0; i < this.binCount; ++i) {
            if (this.bins[i] == that.bins[i]) continue;
            return false;
        }
        return true;
    }

    public int hashCode() {
        int result = this.size;
        result = 31 * result + (this.positions != null ? ArrayUtils.hashCode(this.positions, 0, this.binCount) : 0);
        result = 31 * result + (this.bins != null ? ArrayUtils.hashCode(this.bins, 0, this.binCount) : 0);
        result = 31 * result + this.binCount;
        result = 31 * result + (this.min != 0.0f ? Float.floatToIntBits(this.min) : 0);
        result = 31 * result + (this.max != 0.0f ? Float.floatToIntBits(this.max) : 0);
        return result;
    }

    public ApproximateHistogram(int size, float[] positions, long[] bins, int binCount, float min, float max, long count, float lowerLimit, float upperLimit) {
        Preconditions.checkArgument((positions.length == bins.length ? 1 : 0) != 0, (Object)"position and bin array must have same size");
        Preconditions.checkArgument((binCount <= size ? 1 : 0) != 0, (Object)"binCount must be less or equal to size");
        this.size = size;
        this.positions = positions;
        this.bins = bins;
        this.binCount = binCount;
        this.min = min;
        this.max = max;
        this.count = count;
        this.lowerLimit = lowerLimit;
        this.upperLimit = upperLimit;
    }

    public ApproximateHistogram() {
        this(50);
    }

    public ApproximateHistogram(int size) {
        this(size, new float[size], new long[size], 0, Float.POSITIVE_INFINITY, Float.NEGATIVE_INFINITY, 0L, Float.NEGATIVE_INFINITY, Float.POSITIVE_INFINITY);
    }

    public ApproximateHistogram(int size, float lowerLimit, float upperLimit) {
        this(size, new float[size], new long[size], 0, Float.POSITIVE_INFINITY, Float.NEGATIVE_INFINITY, 0L, lowerLimit, upperLimit);
    }

    public ApproximateHistogram(int binCount, float[] positions, long[] bins, float min, float max) {
        this(positions.length, positions, bins, binCount, min, max, ApproximateHistogram.sumBins(bins, binCount), Float.NEGATIVE_INFINITY, Float.POSITIVE_INFINITY);
    }

    public long count() {
        return this.count;
    }

    public float min() {
        return this.min;
    }

    public float max() {
        return this.max;
    }

    public int binCount() {
        return this.binCount;
    }

    public float[] positions() {
        return Arrays.copyOfRange(this.positions, 0, this.binCount);
    }

    public long[] bins() {
        long[] counts = new long[this.binCount];
        for (int i = 0; i < this.binCount; ++i) {
            counts[i] = this.bins[i] & Long.MAX_VALUE;
        }
        return counts;
    }

    public String toString() {
        return "ApproximateHistogram{size=" + this.size + ", lowerLimit=" + this.lowerLimit + ", upperLimit=" + this.upperLimit + ", positions=" + Arrays.toString(this.positions()) + ", bins=" + this.getBinsString() + ", binCount=" + this.binCount + ", min=" + this.min + ", max=" + this.max + ", count=" + this.count + '}';
    }

    public long getExactCount() {
        long exactCount = 0L;
        for (int i = 0; i < this.binCount; ++i) {
            if ((this.bins[i] & Long.MIN_VALUE) != 0L) continue;
            exactCount += this.bins[i] & Long.MAX_VALUE;
        }
        return exactCount;
    }

    public float getMin() {
        return this.min;
    }

    public float getMax() {
        return this.max;
    }

    private static long sumBins(long[] bins, int binCount) {
        long count = 0L;
        for (int i = 0; i < binCount; ++i) {
            count += bins[i] & Long.MAX_VALUE;
        }
        return count;
    }

    protected String getBinsString() {
        StringBuilder s = new StringBuilder();
        s.append('[');
        for (int i = 0; i < this.bins.length; ++i) {
            if (i > 0) {
                s.append(", ");
            }
            if ((this.bins[i] & Long.MIN_VALUE) != 0L) {
                s.append("*");
            }
            s.append(this.bins[i] & Long.MAX_VALUE);
        }
        s.append(']');
        return s.toString();
    }

    public void setLowerLimit(float lowerLimit) {
        this.lowerLimit = lowerLimit;
    }

    public void setUpperLimit(float upperLimit) {
        this.upperLimit = upperLimit;
    }

    public void offer(float value) {
        if (value < this.min) {
            this.min = value;
        }
        if (value > this.max) {
            this.max = value;
        }
        if (this.binCount == 0) {
            this.positions[0] = value;
            this.bins[0] = 1L;
            ++this.count;
            ++this.binCount;
            return;
        }
        int index = Arrays.binarySearch(this.positions, 0, this.binCount, value);
        if (index >= 0) {
            this.bins[index] = this.bins[index] & Long.MIN_VALUE | (this.bins[index] & Long.MAX_VALUE) + 1L;
            ++this.count;
            return;
        }
        int insertAt = -(index + 1);
        if (this.binCount < this.size) {
            this.shiftRight(insertAt, this.binCount);
            this.positions[insertAt] = value;
            this.bins[insertAt] = 1L;
            ++this.count;
            ++this.binCount;
            return;
        }
        int minPos = this.minDeltaIndex();
        float minDelta = minPos >= 0 ? this.positions[minPos + 1] - this.positions[minPos] : Float.POSITIVE_INFINITY;
        float deltaRight = insertAt < this.binCount ? this.positions[insertAt] - value : Float.POSITIVE_INFINITY;
        float deltaLeft = insertAt > 0 ? value - this.positions[insertAt - 1] : Float.POSITIVE_INFINITY;
        boolean mergeValue = false;
        if (deltaRight < minDelta) {
            minDelta = deltaRight;
            minPos = insertAt;
            mergeValue = true;
        }
        if (deltaLeft < minDelta) {
            minPos = insertAt - 1;
            mergeValue = true;
        }
        if (mergeValue) {
            long k = this.bins[minPos] & Long.MAX_VALUE;
            this.positions[minPos] = (this.positions[minPos] * (float)k + value) / (float)(k + 1L);
            this.bins[minPos] = k + 1L | Long.MIN_VALUE;
            ++this.count;
        } else {
            this.mergeInsert(minPos, insertAt, value, 1L);
        }
    }

    protected int minDeltaIndex() {
        float minDelta = Float.POSITIVE_INFINITY;
        int minPos = -1;
        for (int i = 0; i < this.binCount - 1; ++i) {
            float delta = this.positions[i + 1] - this.positions[i];
            if (!(delta < minDelta)) continue;
            minDelta = delta;
            minPos = i;
        }
        return minPos;
    }

    protected void mergeInsert(int mergeAt, int insertAt, float v, long c) {
        long k0 = this.bins[mergeAt] & Long.MAX_VALUE;
        long k1 = this.bins[mergeAt + 1] & Long.MAX_VALUE;
        long sum = k0 + k1;
        this.positions[mergeAt] = (float)(((double)this.positions[mergeAt] * (double)k0 + (double)this.positions[mergeAt + 1] * (double)k1) / (double)sum);
        this.bins[mergeAt] = sum | Long.MIN_VALUE;
        int unusedIndex = mergeAt + 1;
        if (insertAt >= 0) {
            if (insertAt < unusedIndex) {
                this.shiftRight(insertAt, unusedIndex);
            } else {
                this.shiftLeft(unusedIndex, insertAt - 1);
                --insertAt;
            }
            this.positions[insertAt] = v;
            this.bins[insertAt] = c;
            ++this.count;
        } else {
            this.shiftLeft(unusedIndex, this.binCount - 1);
            --this.binCount;
        }
    }

    protected void shiftRight(int start, int end) {
        float prevVal = this.positions[start];
        long prevCnt = this.bins[start];
        for (int i = start + 1; i <= end; ++i) {
            float tmpVal = this.positions[i];
            long tmpCnt = this.bins[i];
            this.positions[i] = prevVal;
            this.bins[i] = prevCnt;
            prevVal = tmpVal;
            prevCnt = tmpCnt;
        }
    }

    protected void shiftLeft(int start, int end) {
        for (int i = start; i < end; ++i) {
            this.positions[i] = this.positions[i + 1];
            this.bins[i] = this.bins[i + 1];
        }
    }

    public ApproximateHistogram fold(ApproximateHistogram h, float[] mergedPositions, long[] mergedBins, float[] deltas) {
        if (this.size == 0) {
            return this.copy(h);
        }
        return this.foldMin(h, mergedPositions, mergedBins, deltas);
    }

    public ApproximateHistogram foldFast(ApproximateHistogram h) {
        return this.foldFast(h, null, null);
    }

    public ApproximateHistogram foldFast(ApproximateHistogram h, @Nullable float[] mergedPositions, @Nullable long[] mergedBins) {
        if (this.size == 0) {
            return this.copy(h);
        }
        return this.foldRule(h, mergedPositions, mergedBins);
    }

    public ApproximateHistogram copy(ApproximateHistogram h) {
        if (h.size > this.size) {
            this.size = h.size;
            this.positions = new float[this.size];
            this.bins = new long[this.size];
        }
        System.arraycopy(h.positions, 0, this.positions, 0, h.binCount);
        System.arraycopy(h.bins, 0, this.bins, 0, h.binCount);
        this.min = h.min;
        this.max = h.max;
        this.binCount = h.binCount;
        this.count = h.count;
        return this;
    }

    protected ApproximateHistogram foldMin(ApproximateHistogram h, float[] mergedPositions, long[] mergedBins, float[] deltas) {
        float mergedMin = this.min < h.min ? this.min : h.min;
        float mergedMax = this.max > h.max ? this.max : h.max;
        long mergedCount = this.count + h.count;
        int maxSize = this.binCount + h.binCount;
        int[] next = new int[maxSize];
        int[] prev = new int[maxSize];
        if (mergedPositions == null || mergedBins == null || deltas == null) {
            mergedPositions = new float[maxSize];
            mergedBins = new long[maxSize];
            deltas = new float[maxSize];
        } else {
            Preconditions.checkArgument((mergedPositions.length >= maxSize ? 1 : 0) != 0, (String)"temp buffer [mergedPositions] too small: length must be at least [%s], got [%s]", (int)maxSize, (int)mergedPositions.length);
            Preconditions.checkArgument((mergedBins.length >= maxSize ? 1 : 0) != 0, (String)"temp buffer [mergedBins] too small: length must be at least [%s], got [%s]", (int)maxSize, (int)mergedPositions.length);
            Preconditions.checkArgument((deltas.length >= maxSize ? 1 : 0) != 0, (String)"temp buffer [deltas] too small: length must be at least [%s], got [%s]", (int)maxSize, (int)mergedPositions.length);
        }
        int mergedBinCount = ApproximateHistogram.combineBins(this.binCount, this.positions, this.bins, h.binCount, h.positions, h.bins, mergedPositions, mergedBins, deltas);
        if (mergedBinCount == 0) {
            return this;
        }
        int numMerge = mergedBinCount - this.size;
        if (numMerge < 0) {
            numMerge = 0;
        }
        ApproximateHistogram.mergeBins(mergedBinCount, mergedPositions, mergedBins, deltas, numMerge, next, prev);
        int i = 0;
        int k = 0;
        while (i < mergedBinCount) {
            this.positions[k] = mergedPositions[i];
            this.bins[k] = mergedBins[i];
            ++k;
            i = next[i];
        }
        this.binCount = mergedBinCount - numMerge;
        this.min = mergedMin;
        this.max = mergedMax;
        this.count = mergedCount;
        return this;
    }

    protected ApproximateHistogram foldRule(ApproximateHistogram h, @Nullable float[] mergedPositions, @Nullable long[] mergedBins) {
        if (h.binCount == 0) {
            return this;
        }
        float mergedMin = this.min < h.min ? this.min : h.min;
        float mergedMax = this.max > h.max ? this.max : h.max;
        long mergedCount = this.count + h.count;
        this.min = mergedMin;
        this.max = mergedMax;
        if (mergedPositions == null) {
            mergedPositions = new float[this.size];
            mergedBins = new long[this.size];
        }
        int mergedBinCount = this.binCount + h.binCount <= this.size ? ApproximateHistogram.combineBins(this.binCount, this.positions, this.bins, h.binCount, h.positions, h.bins, mergedPositions, mergedBins, null) : this.ruleCombineBins(this.binCount, this.positions, this.bins, h.binCount, h.positions, h.bins, mergedPositions, mergedBins);
        for (int i = 0; i < mergedBinCount; ++i) {
            this.positions[i] = mergedPositions[i];
            this.bins[i] = mergedBins[i];
        }
        this.binCount = mergedBinCount;
        this.count = mergedCount;
        return this;
    }

    protected int ruleCombineBins(int leftBinCount, float[] leftPositions, long[] leftBins, int rightBinCount, float[] rightPositions, long[] rightBins, float[] mergedPositions, long[] mergedBins) {
        float w;
        long sum;
        long k0;
        float delta;
        long k1;
        float m1;
        int j;
        float cutoff = this.upperLimit != Float.POSITIVE_INFINITY && this.lowerLimit != Float.NEGATIVE_INFINITY ? (this.upperLimit - this.lowerLimit) / (float)(this.size - 2 - 1) : (this.upperLimit != Float.POSITIVE_INFINITY ? (this.upperLimit - this.min) / (float)(this.size - 2) : (this.lowerLimit != Float.NEGATIVE_INFINITY ? (this.max - this.lowerLimit) / (float)(this.size - 2) : (this.max - this.min) / (float)(this.size - 1)));
        float lowerPosition = 0.0f;
        long lowerBin = 0L;
        float upperPosition = 0.0f;
        long upperBin = 0L;
        int k = 0;
        int pos = 0;
        for (j = 0; j != leftBinCount && (m1 = leftPositions[j]) < this.lowerLimit; ++j) {
            k1 = leftBins[j] & Long.MAX_VALUE;
            delta = m1 - lowerPosition;
            k0 = lowerBin & Long.MAX_VALUE;
            sum = k0 + k1;
            w = (float)k0 / (float)sum;
            lowerPosition = -delta * w + m1;
            lowerBin = sum | Long.MIN_VALUE;
        }
        while (k != rightBinCount && (m1 = rightPositions[k]) < this.lowerLimit) {
            k1 = rightBins[k] & Long.MAX_VALUE;
            delta = m1 - lowerPosition;
            k0 = lowerBin & Long.MAX_VALUE;
            sum = k0 + k1;
            w = (float)k0 / (float)sum;
            lowerPosition = -delta * w + m1;
            lowerBin = sum | Long.MIN_VALUE;
            ++k;
        }
        if ((lowerBin & Long.MAX_VALUE) > 0L) {
            mergedPositions[0] = lowerPosition;
            mergedBins[0] = lowerBin;
            pos = 1;
        }
        if (j != leftBinCount || k != rightBinCount) {
            if (j != leftBinCount && (k == rightBinCount || leftPositions[j] < rightPositions[k])) {
                mergedPositions[pos] = leftPositions[j];
                mergedBins[pos] = leftBins[j];
                ++j;
            } else {
                mergedPositions[pos] = rightPositions[k];
                mergedBins[pos] = rightBins[k];
                ++k;
            }
        }
        while (j != leftBinCount || k != rightBinCount) {
            if (j != leftBinCount && (k == rightBinCount || leftPositions[j] < rightPositions[k])) {
                m1 = leftPositions[j];
                k1 = leftBins[j] & Long.MAX_VALUE;
                if (m1 > this.upperLimit) {
                    delta = m1 - upperPosition;
                    k0 = upperBin & Long.MAX_VALUE;
                    sum = k0 + k1;
                    w = (float)k0 / (float)sum;
                    upperPosition = -delta * w + m1;
                    upperBin = sum | Long.MIN_VALUE;
                    ++j;
                    continue;
                }
                delta = m1 - mergedPositions[pos];
                if (delta <= cutoff) {
                    k0 = mergedBins[pos] & Long.MAX_VALUE;
                    sum = k0 + k1;
                    w = (float)k0 / (float)sum;
                    mergedPositions[pos] = -delta * w + m1;
                    mergedBins[pos] = sum | Long.MIN_VALUE;
                } else {
                    mergedPositions[++pos] = m1;
                    mergedBins[pos] = k1;
                }
                ++j;
                continue;
            }
            m1 = rightPositions[k];
            k1 = rightBins[k] & Long.MAX_VALUE;
            if (m1 > this.upperLimit) {
                delta = m1 - upperPosition;
                k0 = upperBin & Long.MAX_VALUE;
                sum = k0 + k1;
                w = (float)k0 / (float)sum;
                upperPosition = -delta * w + m1;
                upperBin = sum | Long.MIN_VALUE;
                ++k;
                continue;
            }
            delta = m1 - mergedPositions[pos];
            if (delta <= cutoff) {
                k0 = mergedBins[pos] & Long.MAX_VALUE;
                sum = k0 + k1;
                w = (float)k0 / (float)sum;
                mergedPositions[pos] = -delta * w + m1;
                mergedBins[pos] = sum | Long.MIN_VALUE;
            } else {
                mergedPositions[++pos] = m1;
                mergedBins[pos] = k1;
            }
            ++k;
        }
        if ((upperBin & Long.MAX_VALUE) > 0L) {
            mergedPositions[++pos] = upperPosition;
            mergedBins[pos] = upperBin;
        }
        return pos + 1;
    }

    private static void mergeBins(int mergedBinCount, float[] mergedPositions, long[] mergedBins, float[] deltas, int numMerge, int[] next, int[] prev) {
        int i;
        int i2;
        int lastValidIndex = mergedBinCount - 1;
        for (i2 = 0; i2 < mergedBinCount; ++i2) {
            next[i2] = i2 + 1;
        }
        for (i2 = 0; i2 < mergedBinCount; ++i2) {
            prev[i2] = i2 - 1;
        }
        int heapSize = mergedBinCount - 1;
        int[] heap = new int[heapSize];
        int[] reverseIndex = new int[heapSize];
        for (i = 0; i < heapSize; ++i) {
            heap[i] = i;
        }
        for (i = 0; i < heapSize; ++i) {
            reverseIndex[i] = i;
        }
        ApproximateHistogram.heapify(heap, reverseIndex, heapSize, deltas);
        for (i = 0; i < numMerge; ++i) {
            float mm0;
            int currentIndex = heap[0];
            int nextIndex = next[currentIndex];
            int prevIndex = prev[currentIndex];
            long k0 = mergedBins[currentIndex] & Long.MAX_VALUE;
            long k1 = mergedBins[nextIndex] & Long.MAX_VALUE;
            float m0 = mergedPositions[currentIndex];
            float m1 = mergedPositions[nextIndex];
            float d1 = deltas[nextIndex];
            long sum = k0 + k1;
            float w = (float)k0 / (float)sum;
            mergedPositions[currentIndex] = mm0 = (m0 - m1) * w + m1;
            mergedBins[currentIndex] = sum | Long.MIN_VALUE;
            if (nextIndex == lastValidIndex) {
                heapSize = ApproximateHistogram.heapDelete(heap, reverseIndex, heapSize, reverseIndex[currentIndex], deltas);
            } else {
                heapSize = ApproximateHistogram.heapDelete(heap, reverseIndex, heapSize, reverseIndex[nextIndex], deltas);
                deltas[currentIndex] = m1 - mm0 + d1;
                ApproximateHistogram.siftDown(heap, reverseIndex, reverseIndex[currentIndex], heapSize - 1, deltas);
            }
            if (prevIndex >= 0) {
                deltas[prevIndex] = mm0 - mergedPositions[prevIndex];
                ApproximateHistogram.siftDown(heap, reverseIndex, reverseIndex[prevIndex], heapSize - 1, deltas);
            }
            if (nextIndex == lastValidIndex) {
                lastValidIndex = currentIndex;
            }
            next[currentIndex] = next[nextIndex];
            if (nextIndex >= lastValidIndex) continue;
            prev[next[nextIndex]] = currentIndex;
        }
    }

    private static void heapify(int[] heap, int[] reverseIndex, int count, float[] values) {
        for (int start = (count - 2) / 2; start >= 0; --start) {
            ApproximateHistogram.siftDown(heap, reverseIndex, start, count - 1, values);
        }
    }

    private static void siftDown(int[] heap, int[] reverseIndex, int start, int end, float[] values) {
        int root = start;
        while (root * 2 + 1 <= end) {
            int swap = root;
            int child = root * 2 + 1;
            if (values[heap[swap]] > values[heap[child]]) {
                swap = child;
            }
            if (child + 1 <= end && values[heap[swap]] > values[heap[child + 1]]) {
                swap = child + 1;
            }
            if (swap != root) {
                int tmp = heap[swap];
                heap[swap] = heap[root];
                heap[root] = tmp;
                reverseIndex[heap[swap]] = swap;
                reverseIndex[heap[root]] = root;
                root = swap;
                continue;
            }
            return;
        }
    }

    private static int heapDelete(int[] heap, int[] reverseIndex, int count, int heapIndex, float[] values) {
        int end = count - 1;
        reverseIndex[heap[heapIndex]] = -1;
        heap[heapIndex] = heap[end];
        reverseIndex[heap[heapIndex]] = heapIndex;
        ApproximateHistogram.siftDown(heap, reverseIndex, heapIndex, --end, values);
        return count - 1;
    }

    private static int combineBins(int leftBinCount, float[] leftPositions, long[] leftBins, int rightBinCount, float[] rightPositions, long[] rightBins, float[] mergedPositions, @Nullable long[] mergedBins, @Nullable float[] deltas) {
        int i = 0;
        int j = 0;
        int k = 0;
        while (j < leftBinCount || k < rightBinCount) {
            if (j < leftBinCount && (k == rightBinCount || leftPositions[j] < rightPositions[k])) {
                mergedPositions[i] = leftPositions[j];
                mergedBins[i] = leftBins[j];
                ++j;
            } else if (k < rightBinCount && (j == leftBinCount || leftPositions[j] > rightPositions[k])) {
                mergedPositions[i] = rightPositions[k];
                mergedBins[i] = rightBins[k];
                ++k;
            } else {
                mergedPositions[i] = leftPositions[j];
                mergedBins[i] = leftBins[j] + rightBins[k];
                ++j;
                ++k;
            }
            if (deltas != null && i > 0) {
                deltas[i - 1] = mergedPositions[i] - mergedPositions[i - 1];
            }
            ++i;
        }
        return i;
    }

    @JsonValue
    public byte[] toBytes() {
        ByteBuffer buf = ByteBuffer.allocate(this.getMinStorageSize());
        this.toBytes(buf);
        return buf.array();
    }

    public int getDenseStorageSize() {
        return 8 + 4 * this.size + 8 * this.size + 8;
    }

    public int getSparseStorageSize() {
        return 8 + 4 * this.binCount + 8 * this.binCount + 8;
    }

    public int getCompactStorageSize() {
        Preconditions.checkState((boolean)this.canStoreCompact(), (Object)"Approximate histogram cannot be stored in compact form");
        long exactCount = this.getExactCount();
        if (exactCount == this.count) {
            return 3 + 4 * (int)exactCount;
        }
        return 3 + 4 * (int)exactCount + 1 + 4 * (int)(this.count - exactCount) + 8;
    }

    public int getMaxStorageSize() {
        return this.getDenseStorageSize();
    }

    public int getMinStorageSize() {
        if (this.canStoreCompact() && this.getCompactStorageSize() < this.getSparseStorageSize()) {
            return this.getCompactStorageSize();
        }
        return this.getSparseStorageSize();
    }

    public boolean canStoreCompact() {
        long exactCount = this.getExactCount();
        return this.size <= Short.MAX_VALUE && exactCount <= 127L && this.count - exactCount <= 127L;
    }

    public void toBytes(ByteBuffer buf) {
        if (this.canStoreCompact() && this.getCompactStorageSize() < this.getSparseStorageSize()) {
            this.toBytesCompact(buf);
        } else {
            this.toBytesSparse(buf);
        }
    }

    public void toBytesDense(ByteBuffer buf) {
        buf.putInt(this.size);
        buf.putInt(this.binCount);
        buf.asFloatBuffer().put(this.positions);
        buf.position(buf.position() + 4 * this.positions.length);
        buf.asLongBuffer().put(this.bins);
        buf.position(buf.position() + 8 * this.bins.length);
        buf.putFloat(this.min);
        buf.putFloat(this.max);
    }

    public void toBytesSparse(ByteBuffer buf) {
        int i;
        buf.putInt(this.size);
        buf.putInt(-1 * this.binCount);
        for (i = 0; i < this.binCount; ++i) {
            buf.putFloat(this.positions[i]);
        }
        for (i = 0; i < this.binCount; ++i) {
            buf.putLong(this.bins[i]);
        }
        buf.putFloat(this.min);
        buf.putFloat(this.max);
    }

    public void toBytesCompact(ByteBuffer buf) {
        int k;
        int i;
        Preconditions.checkState((boolean)this.canStoreCompact(), (Object)"Approximate histogram cannot be stored in compact form");
        buf.putShort((short)(-1 * this.size));
        long exactCount = this.getExactCount();
        if (exactCount != this.count) {
            buf.put((byte)(-1L * (this.count - exactCount)));
            for (i = 0; i < this.binCount; ++i) {
                if ((this.bins[i] & Long.MIN_VALUE) == 0L) continue;
                k = 0;
                while ((long)k < (this.bins[i] & Long.MAX_VALUE)) {
                    buf.putFloat(this.positions[i]);
                    ++k;
                }
            }
            buf.putFloat(this.min);
            buf.putFloat(this.max);
        }
        buf.put((byte)exactCount);
        for (i = 0; i < this.binCount; ++i) {
            if ((this.bins[i] & Long.MIN_VALUE) != 0L) continue;
            k = 0;
            while ((long)k < (this.bins[i] & Long.MAX_VALUE)) {
                buf.putFloat(this.positions[i]);
                ++k;
            }
        }
    }

    public static ApproximateHistogram fromBytes(byte[] bytes) {
        ByteBuffer buf = ByteBuffer.wrap(bytes);
        return ApproximateHistogram.fromBytes(buf);
    }

    public static ApproximateHistogram fromBytesDense(ByteBuffer buf) {
        int size = buf.getInt();
        int binCount = buf.getInt();
        float[] positions = new float[size];
        long[] bins = new long[size];
        buf.asFloatBuffer().get(positions);
        buf.position(buf.position() + 4 * positions.length);
        buf.asLongBuffer().get(bins);
        buf.position(buf.position() + 8 * bins.length);
        float min = buf.getFloat();
        float max = buf.getFloat();
        return new ApproximateHistogram(binCount, positions, bins, min, max);
    }

    public static ApproximateHistogram fromBytesSparse(ByteBuffer buf) {
        int i;
        int size = buf.getInt();
        int binCount = -1 * buf.getInt();
        float[] positions = new float[size];
        long[] bins = new long[size];
        for (i = 0; i < binCount; ++i) {
            positions[i] = buf.getFloat();
        }
        for (i = 0; i < binCount; ++i) {
            bins[i] = buf.getLong();
        }
        float min = buf.getFloat();
        float max = buf.getFloat();
        return new ApproximateHistogram(binCount, positions, bins, min, max);
    }

    public static ApproximateHistogram fromBytesCompact(ByteBuffer buf) {
        int i;
        short size = (short)(-1 * buf.getShort());
        int count = buf.get();
        if (count >= 0) {
            ApproximateHistogram histogram = new ApproximateHistogram(size);
            for (int i2 = 0; i2 < count; ++i2) {
                histogram.offer(buf.getFloat());
            }
            return histogram;
        }
        int approxCount = -1 * count;
        HashMap<Float, Long> approx = new HashMap<Float, Long>();
        for (int i3 = 0; i3 < approxCount; ++i3) {
            float value = buf.getFloat();
            if (approx.containsKey(Float.valueOf(value))) {
                approx.put(Float.valueOf(value), (Long)approx.get(Float.valueOf(value)) + 1L);
                continue;
            }
            approx.put(Float.valueOf(value), 1L);
        }
        float min = buf.getFloat();
        float max = buf.getFloat();
        int exactCount = buf.get();
        HashMap<Float, Long> exact = new HashMap<Float, Long>();
        for (int i4 = 0; i4 < exactCount; ++i4) {
            float value = buf.getFloat();
            if (exact.containsKey(Float.valueOf(value))) {
                exact.put(Float.valueOf(value), (Long)exact.get(Float.valueOf(value)) + 1L);
                continue;
            }
            exact.put(Float.valueOf(value), 1L);
        }
        int binCount = exact.size() + approx.size();
        ArrayList pos = new ArrayList();
        pos.addAll(exact.keySet());
        pos.addAll(approx.keySet());
        Collections.sort(pos);
        float[] positions = new float[size];
        long[] bins = new long[size];
        for (i = 0; i < pos.size(); ++i) {
            positions[i] = ((Float)pos.get(i)).floatValue();
        }
        for (i = 0; i < pos.size(); ++i) {
            float value = ((Float)pos.get(i)).floatValue();
            bins[i] = exact.containsKey(Float.valueOf(value)) ? (Long)exact.get(Float.valueOf(value)) : (Long)approx.get(Float.valueOf(value)) | Long.MIN_VALUE;
        }
        return new ApproximateHistogram(binCount, positions, bins, min, max);
    }

    public static ApproximateHistogram fromBytes(ByteBuffer buf) {
        if (buf.getShort(buf.position()) < 0) {
            return ApproximateHistogram.fromBytesCompact(buf);
        }
        if (buf.getInt(buf.position() + 4) < 0) {
            return ApproximateHistogram.fromBytesSparse(buf);
        }
        return ApproximateHistogram.fromBytesDense(buf);
    }

    public double sum(float b) {
        if (b < this.min) {
            return 0.0;
        }
        if (b >= this.max) {
            return this.count;
        }
        int index = Arrays.binarySearch(this.positions, 0, this.binCount, b);
        boolean exactMatch = index >= 0;
        int n = index = exactMatch ? index : -(index + 1);
        if (!exactMatch) {
            --index;
        }
        boolean outerLeft = index < 0;
        boolean outerRight = index >= this.binCount - 1;
        long m0 = outerLeft ? 0L : this.bins[index] & Long.MAX_VALUE;
        long m1 = outerRight ? 0L : this.bins[index + 1] & Long.MAX_VALUE;
        double p0 = outerLeft ? (double)this.min : (double)this.positions[index];
        double p1 = outerRight ? (double)this.max : (double)this.positions[index + 1];
        boolean exact0 = !outerLeft && (this.bins[index] & Long.MIN_VALUE) == 0L;
        boolean exact1 = !outerRight && (this.bins[index + 1] & Long.MIN_VALUE) == 0L;
        double l = p1 == p0 ? 0.0 : ((double)b - p0) / (p1 - p0);
        long tm0 = m0;
        long tm1 = m1;
        if (exact0) {
            tm0 = 0L;
        }
        if (exact1) {
            tm1 = 0L;
        }
        double mb = (double)tm0 + (double)(tm1 - tm0) * l;
        double s = 0.5 * ((double)tm0 + mb) * l;
        for (int i = 0; i < index; ++i) {
            s += (double)(this.bins[i] & Long.MAX_VALUE);
        }
        if (exact0) {
            return s + (double)m0;
        }
        return s + 0.5 * (double)m0;
    }

    public float[] getQuantiles(float[] probabilities) {
        for (float p : probabilities) {
            Preconditions.checkArgument((0.0f < p && p < 1.0f ? 1 : 0) != 0, (Object)"quantile probabilities must be strictly between 0 and 1");
        }
        float[] quantiles = new float[probabilities.length];
        Arrays.fill(quantiles, Float.NaN);
        if (this.count() == 0L) {
            return quantiles;
        }
        long[] bins = this.bins();
        for (int j = 0; j < probabilities.length; ++j) {
            double s = probabilities[j] * (float)this.count();
            int i = 0;
            long sum = 0L;
            for (int k = 1; k <= this.binCount(); ++k) {
                long count = bins[k - 1];
                if ((double)(sum + count) > s) {
                    i = k - 1;
                    break;
                }
                sum += count;
            }
            if (i == 0) {
                quantiles[j] = this.min();
                continue;
            }
            double d = s - (double)sum;
            double c = -2.0 * d;
            long a = bins[i] - bins[i - 1];
            long b = 2L * bins[i - 1];
            double z = a == 0L ? -c / (double)b : ((double)(-b) + Math.sqrt((double)(b * b) - (double)(4L * a) * c)) / (double)(2L * a);
            double uj = (double)this.positions[i - 1] + (double)(this.positions[i] - this.positions[i - 1]) * z;
            quantiles[j] = (float)uj < this.max() ? (float)uj : this.max();
        }
        return quantiles;
    }

    public Histogram toHistogram(float[] breaks) {
        double[] approximateBins = new double[breaks.length - 1];
        double prev = this.sum(breaks[0]);
        for (int i = 1; i < breaks.length; ++i) {
            double s = this.sum(breaks[i]);
            approximateBins[i - 1] = (float)(s - prev);
            prev = s;
        }
        return new Histogram(breaks, approximateBins);
    }

    public Histogram toHistogram(int size) {
        Preconditions.checkArgument((size > 1 ? 1 : 0) != 0, (Object)"histogram size must be greater than 1");
        float[] breaks = new float[size + 1];
        float delta = (this.max - this.min) / (float)(size - 1);
        breaks[0] = this.min - delta;
        for (int i = 1; i < breaks.length - 1; ++i) {
            breaks[i] = breaks[i - 1] + delta;
        }
        breaks[breaks.length - 1] = this.max;
        return this.toHistogram(breaks);
    }

    public Histogram toHistogram(float bucketSize, float offset) {
        float minFloor = (float)Math.floor((this.min() - offset) / bucketSize) * bucketSize + offset;
        float lowerLimitFloor = (float)Math.floor((this.lowerLimit - offset) / bucketSize) * bucketSize + offset;
        float firstBreak = Math.max(minFloor, lowerLimitFloor);
        float maxCeil = (float)Math.ceil((this.max() - offset) / bucketSize) * bucketSize + offset;
        float upperLimitCeil = (float)Math.ceil((this.upperLimit - offset) / bucketSize) * bucketSize + offset;
        float lastBreak = Math.min(maxCeil, upperLimitCeil);
        float cutoff = 0.1f;
        ArrayList<Float> breaks = new ArrayList<Float>();
        float bottomBreak = minFloor - bucketSize;
        if (bottomBreak != firstBreak && this.sum(firstBreak) - this.sum(bottomBreak) > (double)0.1f) {
            breaks.add(Float.valueOf(bottomBreak));
        }
        float left = firstBreak;
        boolean leftSet = false;
        while (left + bucketSize <= lastBreak + bucketSize / 10.0f) {
            float right = left + bucketSize;
            if (this.sum(right) - this.sum(left) > (double)0.1f) {
                if (!leftSet) {
                    breaks.add(Float.valueOf(left));
                }
                breaks.add(Float.valueOf(right));
                leftSet = true;
            } else {
                leftSet = false;
            }
            left = right;
        }
        if (((Float)breaks.get(breaks.size() - 1)).floatValue() != maxCeil && this.sum(maxCeil) - this.sum(((Float)breaks.get(breaks.size() - 1)).floatValue()) > (double)0.1f) {
            breaks.add(Float.valueOf(maxCeil));
        }
        return this.toHistogram(Floats.toArray(breaks));
    }
}

