/*
 * Decompiled with CFR 0.152.
 */
package org.apache.dubbo.common.utils;

import java.io.IOException;
import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.MulticastSocket;
import java.net.NetworkInterface;
import java.net.ServerSocket;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.util.BitSet;
import java.util.Collections;
import java.util.Enumeration;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ThreadLocalRandom;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import org.apache.dubbo.common.URL;
import org.apache.dubbo.common.config.ConfigurationUtils;
import org.apache.dubbo.common.logger.Logger;
import org.apache.dubbo.common.logger.LoggerFactory;
import org.apache.dubbo.common.logger.support.FailsafeLogger;
import org.apache.dubbo.common.utils.CIDRUtils;
import org.apache.dubbo.common.utils.CollectionUtils;
import org.apache.dubbo.common.utils.LRUCache;
import org.apache.dubbo.common.utils.StringUtils;
import org.apache.dubbo.common.utils.SystemPropertyConfigUtils;
import org.apache.dubbo.rpc.model.ScopeModel;

public final class NetUtils {
    private static Logger logger = LoggerFactory.getLogger(NetUtils.class);
    private static final int RND_PORT_START = 30000;
    private static final int RND_PORT_RANGE = 10000;
    private static final int MIN_PORT = 1;
    private static final int MAX_PORT = 65535;
    private static final Pattern ADDRESS_PATTERN;
    private static final Pattern LOCAL_IP_PATTERN;
    private static final Pattern IP_PATTERN;
    private static final Map<String, String> HOST_NAME_CACHE;
    private static volatile InetAddress LOCAL_ADDRESS;
    private static volatile Inet6Address LOCAL_ADDRESS_V6;
    private static final String SPLIT_IPV4_CHARACTER = "\\.";
    private static final String SPLIT_IPV6_CHARACTER = ":";
    private static BitSet USED_PORT;
    private static volatile String HOST_ADDRESS;
    private static volatile String HOST_NAME;
    private static volatile String HOST_ADDRESS_V6;

    private NetUtils() {
        throw new UnsupportedOperationException("No instance of 'NetUtils' for you! ");
    }

    public static int getRandomPort() {
        return 30000 + ThreadLocalRandom.current().nextInt(10000);
    }

    public static synchronized int getAvailablePort() {
        int randomPort = NetUtils.getRandomPort();
        return NetUtils.getAvailablePort(randomPort);
    }

    public static synchronized int getAvailablePort(int port) {
        if (port < 1) {
            return 1;
        }
        for (int i = port; i < 65535; ++i) {
            if (USED_PORT.get(i)) continue;
            try (ServerSocket ignored = new ServerSocket(i);){
                USED_PORT.set(i);
                port = i;
                break;
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
        return port;
    }

    public static boolean isPortInUsed(int port) {
        boolean bl;
        ServerSocket ignored = new ServerSocket(port);
        try {
            bl = false;
        }
        catch (Throwable throwable) {
            try {
                try {
                    ignored.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException iOException) {
                return true;
            }
        }
        ignored.close();
        return bl;
    }

    public static boolean isInvalidPort(int port) {
        return port < 1 || port > 65535;
    }

    public static boolean isValidAddress(String address) {
        return ADDRESS_PATTERN.matcher(address).matches();
    }

    public static boolean isLocalHost(String host) {
        return host != null && (LOCAL_IP_PATTERN.matcher(host).matches() || host.equalsIgnoreCase("localhost"));
    }

    public static boolean isAnyHost(String host) {
        return "0.0.0.0".equals(host);
    }

    public static boolean isInvalidLocalHost(String host) {
        return host == null || host.length() == 0 || host.equalsIgnoreCase("localhost") || host.equals("0.0.0.0") || host.startsWith("127.");
    }

    public static boolean isValidLocalHost(String host) {
        return !NetUtils.isInvalidLocalHost(host);
    }

    public static InetSocketAddress getLocalSocketAddress(String host, int port) {
        return NetUtils.isInvalidLocalHost(host) ? new InetSocketAddress(port) : new InetSocketAddress(host, port);
    }

    static boolean isValidV4Address(InetAddress address) {
        if (address == null || address.isLoopbackAddress()) {
            return false;
        }
        String name = address.getHostAddress();
        return name != null && IP_PATTERN.matcher(name).matches() && !"0.0.0.0".equals(name) && !"127.0.0.1".equals(name);
    }

    static boolean isPreferIPV6Address() {
        return Boolean.getBoolean("java.net.preferIPv6Addresses");
    }

    static InetAddress normalizeV6Address(Inet6Address address) {
        String addr = address.getHostAddress();
        int i = addr.lastIndexOf(37);
        if (i > 0) {
            try {
                return InetAddress.getByName(addr.substring(0, i) + '%' + address.getScopeId());
            }
            catch (UnknownHostException e) {
                logger.debug("Unknown IPV6 address: ", e);
            }
        }
        return address;
    }

    public static String getLocalHost() {
        if (HOST_ADDRESS != null) {
            return HOST_ADDRESS;
        }
        InetAddress address = NetUtils.getLocalAddress();
        if (address != null) {
            if (address instanceof Inet6Address) {
                String ipv6AddressString = address.getHostAddress();
                if (ipv6AddressString.contains("%")) {
                    ipv6AddressString = ipv6AddressString.substring(0, ipv6AddressString.indexOf("%"));
                }
                HOST_ADDRESS = ipv6AddressString;
                return HOST_ADDRESS;
            }
            HOST_ADDRESS = address.getHostAddress();
            return HOST_ADDRESS;
        }
        return "127.0.0.1";
    }

    public static String getLocalHostV6() {
        if (StringUtils.isNotEmpty(HOST_ADDRESS_V6)) {
            return HOST_ADDRESS_V6;
        }
        if ("".equals(HOST_ADDRESS_V6)) {
            return null;
        }
        Inet6Address address = NetUtils.getLocalAddressV6();
        if (address != null) {
            String ipv6AddressString = address.getHostAddress();
            if (ipv6AddressString.contains("%")) {
                ipv6AddressString = ipv6AddressString.substring(0, ipv6AddressString.indexOf("%"));
            }
            HOST_ADDRESS_V6 = ipv6AddressString;
            return HOST_ADDRESS_V6;
        }
        HOST_ADDRESS_V6 = "";
        return null;
    }

    public static String filterLocalHost(String host) {
        if (host == null || host.length() == 0) {
            return host;
        }
        if (host.contains("://")) {
            URL u = URL.valueOf(host);
            if (NetUtils.isInvalidLocalHost(u.getHost())) {
                return u.setHost(NetUtils.getLocalHost()).toFullString();
            }
        } else if (host.contains(SPLIT_IPV6_CHARACTER)) {
            int i = host.lastIndexOf(58);
            if (NetUtils.isInvalidLocalHost(host.substring(0, i))) {
                return NetUtils.getLocalHost() + host.substring(i);
            }
        } else if (NetUtils.isInvalidLocalHost(host)) {
            return NetUtils.getLocalHost();
        }
        return host;
    }

    public static String getIpByConfig(ScopeModel scopeModel) {
        String configIp = ConfigurationUtils.getProperty(scopeModel, "DUBBO_IP_TO_BIND");
        if (configIp != null) {
            return configIp;
        }
        return NetUtils.getLocalHost();
    }

    public static InetAddress getLocalAddress() {
        InetAddress localAddress;
        if (LOCAL_ADDRESS != null) {
            return LOCAL_ADDRESS;
        }
        LOCAL_ADDRESS = localAddress = NetUtils.getLocalAddress0();
        return localAddress;
    }

    public static Inet6Address getLocalAddressV6() {
        Inet6Address localAddress;
        if (LOCAL_ADDRESS_V6 != null) {
            return LOCAL_ADDRESS_V6;
        }
        LOCAL_ADDRESS_V6 = localAddress = NetUtils.getLocalAddress0V6();
        return localAddress;
    }

    private static Optional<InetAddress> toValidAddress(InetAddress address) {
        if (address instanceof Inet6Address) {
            Inet6Address v6Address = (Inet6Address)address;
            if (NetUtils.isPreferIPV6Address()) {
                return Optional.ofNullable(NetUtils.normalizeV6Address(v6Address));
            }
        }
        if (NetUtils.isValidV4Address(address)) {
            return Optional.of(address);
        }
        return Optional.empty();
    }

    private static InetAddress getLocalAddress0() {
        InetAddress localAddress = null;
        try {
            NetworkInterface networkInterface = NetUtils.findNetworkInterface();
            Enumeration<InetAddress> addresses = networkInterface.getInetAddresses();
            while (addresses.hasMoreElements()) {
                Optional<InetAddress> addressOp = NetUtils.toValidAddress(addresses.nextElement());
                if (!addressOp.isPresent()) continue;
                try {
                    if (!addressOp.get().isReachable(100)) continue;
                    return addressOp.get();
                }
                catch (IOException iOException) {
                }
            }
        }
        catch (Throwable e) {
            logger.warn(e);
        }
        try {
            localAddress = InetAddress.getLocalHost();
            Optional<InetAddress> addressOp = NetUtils.toValidAddress(localAddress);
            if (addressOp.isPresent()) {
                return addressOp.get();
            }
        }
        catch (Throwable e) {
            logger.warn(e);
        }
        localAddress = NetUtils.getLocalAddressV6();
        return localAddress;
    }

    private static Inet6Address getLocalAddress0V6() {
        try {
            NetworkInterface networkInterface = NetUtils.findNetworkInterface();
            Enumeration<InetAddress> addresses = networkInterface.getInetAddresses();
            while (addresses.hasMoreElements()) {
                InetAddress address = addresses.nextElement();
                if (!(address instanceof Inet6Address) || address.isLoopbackAddress() || address.isAnyLocalAddress() || address.isLinkLocalAddress() || address.isSiteLocalAddress() || NetUtils.isUniqueLocalAddress(address) || !address.getHostAddress().contains(SPLIT_IPV6_CHARACTER)) continue;
                return (Inet6Address)address;
            }
        }
        catch (Throwable e) {
            logger.warn(e);
        }
        return null;
    }

    private static boolean isUniqueLocalAddress(InetAddress address) {
        byte[] ip = address.getAddress();
        return (ip[0] & 0xFF) == 253;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static boolean ignoreNetworkInterface(NetworkInterface networkInterface) throws SocketException {
        String networkInterfaceDisplayName;
        if (networkInterface == null || networkInterface.isLoopback() || networkInterface.isVirtual() || !networkInterface.isUp()) {
            return true;
        }
        if (Boolean.parseBoolean(SystemPropertyConfigUtils.getSystemProperty("dubbo.network.interface.point-to-point.ignored", "false")) && networkInterface.isPointToPoint()) {
            return true;
        }
        String ignoredInterfaces = SystemPropertyConfigUtils.getSystemProperty("dubbo.network.interface.ignored");
        if (StringUtils.isNotEmpty(ignoredInterfaces) && StringUtils.isNotEmpty(networkInterfaceDisplayName = networkInterface.getDisplayName())) {
            for (String ignoredInterface : ignoredInterfaces.split(",")) {
                String trimIgnoredInterface = ignoredInterface.trim();
                boolean matched = false;
                try {
                    matched = networkInterfaceDisplayName.matches(trimIgnoredInterface);
                }
                catch (PatternSyntaxException e) {
                    logger.warn("exception occurred: " + networkInterfaceDisplayName + " matches " + trimIgnoredInterface, e);
                }
                finally {
                    if (matched) {
                        return true;
                    }
                    if (!networkInterfaceDisplayName.equals(trimIgnoredInterface)) continue;
                    return true;
                }
            }
        }
        return false;
    }

    private static List<NetworkInterface> getValidNetworkInterfaces() throws SocketException {
        LinkedList<NetworkInterface> validNetworkInterfaces = new LinkedList<NetworkInterface>();
        Enumeration<NetworkInterface> interfaces = NetworkInterface.getNetworkInterfaces();
        while (interfaces.hasMoreElements()) {
            NetworkInterface networkInterface = interfaces.nextElement();
            if (NetUtils.ignoreNetworkInterface(networkInterface)) continue;
            validNetworkInterfaces.add(networkInterface);
        }
        return validNetworkInterfaces;
    }

    public static boolean isPreferredNetworkInterface(NetworkInterface networkInterface) {
        String preferredNetworkInterface = SystemPropertyConfigUtils.getSystemProperty("dubbo.network.interface.preferred");
        return Objects.equals(networkInterface.getDisplayName(), preferredNetworkInterface);
    }

    public static NetworkInterface findNetworkInterface() {
        List<Object> validNetworkInterfaces = Collections.emptyList();
        try {
            validNetworkInterfaces = NetUtils.getValidNetworkInterfaces();
        }
        catch (Throwable e) {
            logger.warn(e);
        }
        NetworkInterface result = null;
        for (NetworkInterface networkInterface : validNetworkInterfaces) {
            if (!NetUtils.isPreferredNetworkInterface(networkInterface)) continue;
            result = networkInterface;
            break;
        }
        if (result == null) {
            for (NetworkInterface networkInterface : validNetworkInterfaces) {
                Enumeration<InetAddress> addresses = networkInterface.getInetAddresses();
                while (addresses.hasMoreElements()) {
                    Optional<InetAddress> addressOp = NetUtils.toValidAddress(addresses.nextElement());
                    if (!addressOp.isPresent()) continue;
                    try {
                        if (!addressOp.get().isReachable(100)) continue;
                        return networkInterface;
                    }
                    catch (IOException iOException) {
                    }
                }
            }
        }
        if (result == null) {
            result = (NetworkInterface)CollectionUtils.first(validNetworkInterfaces);
        }
        return result;
    }

    public static String getHostName(String address) {
        try {
            String hostname;
            int i = address.indexOf(58);
            if (i > -1) {
                address = address.substring(0, i);
            }
            if ((hostname = HOST_NAME_CACHE.get(address)) != null && hostname.length() > 0) {
                return hostname;
            }
            InetAddress inetAddress = InetAddress.getByName(address);
            if (inetAddress != null) {
                hostname = inetAddress.getHostName();
                HOST_NAME_CACHE.put(address, hostname);
                return hostname;
            }
        }
        catch (Throwable throwable) {
            // empty catch block
        }
        return address;
    }

    public static String getLocalHostName() {
        if (HOST_NAME != null) {
            return HOST_NAME;
        }
        try {
            HOST_NAME = InetAddress.getLocalHost().getHostName();
        }
        catch (UnknownHostException e) {
            HOST_NAME = Optional.ofNullable(NetUtils.getLocalAddress()).map(k -> k.getHostName()).orElse(null);
        }
        return HOST_NAME;
    }

    public static String getIpByHost(String hostName) {
        try {
            return InetAddress.getByName(hostName).getHostAddress();
        }
        catch (UnknownHostException e) {
            return hostName;
        }
    }

    public static String toAddressString(InetSocketAddress address) {
        return address.getAddress().getHostAddress() + SPLIT_IPV6_CHARACTER + address.getPort();
    }

    public static InetSocketAddress toAddress(String address) {
        int port;
        String host;
        int i = address.indexOf(58);
        if (i > -1) {
            host = address.substring(0, i);
            port = Integer.parseInt(address.substring(i + 1));
        } else {
            host = address;
            port = 0;
        }
        return new InetSocketAddress(host, port);
    }

    public static String toURL(String protocol, String host, int port, String path) {
        StringBuilder sb = new StringBuilder();
        sb.append(protocol).append("://");
        sb.append(host).append(':').append(port);
        if (path.charAt(0) != '/') {
            sb.append('/');
        }
        sb.append(path);
        return sb.toString();
    }

    public static void joinMulticastGroup(MulticastSocket multicastSocket, InetAddress multicastAddress) throws IOException {
        NetUtils.setInterface(multicastSocket, multicastAddress instanceof Inet6Address);
        multicastSocket.setLoopbackMode(false);
        multicastSocket.joinGroup(multicastAddress);
    }

    public static void setInterface(MulticastSocket multicastSocket, boolean preferIpv6) throws IOException {
        boolean interfaceSet = false;
        for (NetworkInterface networkInterface : NetUtils.getValidNetworkInterfaces()) {
            Enumeration<InetAddress> addresses = networkInterface.getInetAddresses();
            while (addresses.hasMoreElements()) {
                InetAddress address = addresses.nextElement();
                if (preferIpv6 && address instanceof Inet6Address) {
                    try {
                        if (!address.isReachable(100)) continue;
                        multicastSocket.setInterface(address);
                        interfaceSet = true;
                        break;
                    }
                    catch (IOException iOException) {
                        continue;
                    }
                }
                if (preferIpv6 || !(address instanceof Inet4Address)) continue;
                try {
                    if (!address.isReachable(100)) continue;
                    multicastSocket.setInterface(address);
                    interfaceSet = true;
                    break;
                }
                catch (IOException iOException) {
                }
            }
            if (!interfaceSet) continue;
            break;
        }
    }

    public static boolean matchIpExpression(String pattern, String address) throws UnknownHostException {
        if (address == null) {
            return false;
        }
        String host = address;
        int port = 0;
        if (address.endsWith(SPLIT_IPV6_CHARACTER)) {
            String[] hostPort = address.split(SPLIT_IPV6_CHARACTER);
            host = hostPort[0];
            port = StringUtils.parseInteger(hostPort[1]);
        }
        if (pattern.contains("/")) {
            CIDRUtils utils = new CIDRUtils(pattern);
            return utils.isInRange(host);
        }
        return NetUtils.matchIpRange(pattern, host, port);
    }

    public static boolean matchIpExpression(String pattern, String host, int port) throws UnknownHostException {
        if (pattern.contains("/")) {
            CIDRUtils utils = new CIDRUtils(pattern);
            return utils.isInRange(host);
        }
        return NetUtils.matchIpRange(pattern, host, port);
    }

    public static boolean matchIpRange(String pattern, String host, int port) throws UnknownHostException {
        if (pattern == null || host == null) {
            throw new IllegalArgumentException("Illegal Argument pattern or hostName. Pattern:" + pattern + ", Host:" + host);
        }
        if ("*.*.*.*".equals(pattern = pattern.trim()) || "*".equals(pattern)) {
            return true;
        }
        InetAddress inetAddress = InetAddress.getByName(host);
        boolean isIpv4 = NetUtils.isValidV4Address(inetAddress);
        String[] hostAndPort = NetUtils.getPatternHostAndPort(pattern, isIpv4);
        if (hostAndPort[1] != null && !hostAndPort[1].equals(String.valueOf(port))) {
            return false;
        }
        pattern = hostAndPort[0];
        String splitCharacter = SPLIT_IPV4_CHARACTER;
        if (!isIpv4) {
            splitCharacter = SPLIT_IPV6_CHARACTER;
        }
        String[] mask = pattern.split(splitCharacter);
        NetUtils.checkHostPattern(pattern, mask, isIpv4);
        host = inetAddress.getHostAddress();
        if (pattern.equals(host)) {
            return true;
        }
        if (!NetUtils.ipPatternContainExpression(pattern)) {
            InetAddress patternAddress = InetAddress.getByName(pattern);
            return patternAddress.getHostAddress().equals(host);
        }
        String[] ipAddress = host.split(splitCharacter);
        for (int i = 0; i < mask.length; ++i) {
            if ("*".equals(mask[i]) || mask[i].equals(ipAddress[i])) continue;
            if (mask[i].contains("-")) {
                String[] rangeNumStrs = StringUtils.split(mask[i], '-');
                if (rangeNumStrs.length != 2) {
                    throw new IllegalArgumentException("There is wrong format of ip Address: " + mask[i]);
                }
                Integer min = NetUtils.getNumOfIpSegment(rangeNumStrs[0], isIpv4);
                Integer max = NetUtils.getNumOfIpSegment(rangeNumStrs[1], isIpv4);
                Integer ip = NetUtils.getNumOfIpSegment(ipAddress[i], isIpv4);
                if (ip >= min && ip <= max) continue;
                return false;
            }
            if ("0".equals(ipAddress[i]) && ("0".equals(mask[i]) || "00".equals(mask[i]) || "000".equals(mask[i]) || "0000".equals(mask[i])) || mask[i].equals(ipAddress[i])) continue;
            return false;
        }
        return true;
    }

    public static boolean isMulticastAddress(String host) {
        String prefix;
        int i = host.indexOf(46);
        if (i > 0 && StringUtils.isNumber(prefix = host.substring(0, i))) {
            int p = Integer.parseInt(prefix);
            return p >= 224 && p <= 239;
        }
        return false;
    }

    private static boolean ipPatternContainExpression(String pattern) {
        return pattern.contains("*") || pattern.contains("-");
    }

    private static void checkHostPattern(String pattern, String[] mask, boolean isIpv4) {
        if (!isIpv4) {
            if (mask.length != 8 && NetUtils.ipPatternContainExpression(pattern)) {
                throw new IllegalArgumentException("If you config ip expression that contains '*' or '-', please fill qualified ip pattern like 234e:0:4567:0:0:0:3d:*. ");
            }
            if (mask.length != 8 && !pattern.contains("::")) {
                throw new IllegalArgumentException("The host is ipv6, but the pattern is not ipv6 pattern : " + pattern);
            }
        } else if (mask.length != 4) {
            throw new IllegalArgumentException("The host is ipv4, but the pattern is not ipv4 pattern : " + pattern);
        }
    }

    private static String[] getPatternHostAndPort(String pattern, boolean isIpv4) {
        String[] result = new String[2];
        if (pattern.startsWith("[") && pattern.contains("]:")) {
            int end = pattern.indexOf("]:");
            result[0] = pattern.substring(1, end);
            result[1] = pattern.substring(end + 2);
            return result;
        }
        if (pattern.startsWith("[") && pattern.endsWith("]")) {
            result[0] = pattern.substring(1, pattern.length() - 1);
            result[1] = null;
            return result;
        }
        if (isIpv4 && pattern.contains(SPLIT_IPV6_CHARACTER)) {
            int end = pattern.indexOf(SPLIT_IPV6_CHARACTER);
            result[0] = pattern.substring(0, end);
            result[1] = pattern.substring(end + 1);
            return result;
        }
        result[0] = pattern;
        return result;
    }

    private static Integer getNumOfIpSegment(String ipSegment, boolean isIpv4) {
        if (isIpv4) {
            return Integer.parseInt(ipSegment);
        }
        return Integer.parseInt(ipSegment, 16);
    }

    public static boolean isIPV6URLStdFormat(String ip) {
        if (ip.charAt(0) == '[' && ip.indexOf(93) > 2) {
            return true;
        }
        return ip.indexOf(SPLIT_IPV6_CHARACTER) != ip.lastIndexOf(SPLIT_IPV6_CHARACTER);
    }

    public static String getLegalIP(String ip) {
        int ind;
        if (ip.charAt(0) == '[' && (ind = ip.indexOf(93)) > 2) {
            String nhost = ip;
            ip = nhost.substring(0, ind + 1);
            ip = ip.substring(1, ind);
            return ip;
        }
        return ip;
    }

    static {
        if (logger instanceof FailsafeLogger) {
            logger = ((FailsafeLogger)logger).getLogger();
        }
        ADDRESS_PATTERN = Pattern.compile("^\\d{1,3}(\\.\\d{1,3}){3}\\:\\d{1,5}$");
        LOCAL_IP_PATTERN = Pattern.compile("127(\\.\\d{1,3}){3}$");
        IP_PATTERN = Pattern.compile("\\d{1,3}(\\.\\d{1,3}){3,5}$");
        HOST_NAME_CACHE = new LRUCache<String, String>(1000);
        LOCAL_ADDRESS = null;
        LOCAL_ADDRESS_V6 = null;
        USED_PORT = new BitSet(65536);
    }
}

