/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.felix.ipojo;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Dictionary;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.felix.ipojo.util.Logger;
import org.osgi.framework.BundleContext;

/**
 * The instance creator creates instances and tracks their factories.
 * It allows creating instances from external factories.
 * @author <a href="mailto:dev@felix.apache.org">Felix Project Team</a>
 */
public class InstanceCreator implements FactoryStateListener {

    /**
     * The logger to log messages if errors occur.
     */
    private Logger m_logger;

    /**
     * The configurations to create and to maintain.
     */
    private List m_idle = new ArrayList();

    /**
     * The map storing created instances.
     * This map contains [AbstractFactory, List [ManagedInstance]] couples.
     */
    private Map m_attached = new HashMap();

    /**
     * The abstract factory list.
     */
    private List m_factories = new ArrayList();

    /**
     * Creates the instance creator.
     * This object is generally a singleton.
     * @param context the bundle context of the iPOJO bundle.
     */
    public InstanceCreator(BundleContext context) {
        m_logger = new Logger(context, "iPOJO Instance Creator");
    }

    /**
     * Adds an instance to manage.
     * @param instance the instance configuration
     * @param bundle the bundle id declaring the instance
     */
    synchronized void addInstance(Dictionary instance, long bundle) {
        if (instance.get("factory.version") != null) {
            m_logger.log(Logger.DEBUG, "New instance to managed, looking for " + instance.get("component") + "-" + instance.get("factory.version"));
        } else {
            m_logger.log(Logger.DEBUG, "New instance to managed, looking for " + instance.get("component"));
        }

        ManagedInstance managed = new ManagedInstance(instance, bundle);
        for (int i = 0; i < m_factories.size(); i++) {
            IPojoFactory factory = (IPojoFactory) m_factories.get(i);
            if (managed.matchNameAndVersion(factory)) {
                // Subscribe to the factory state change
                m_logger.log(Logger.DEBUG, "Listen factory " + factory.getName() + " events");
                factory.addFactoryStateListener(this);
                if (factory.getState() == Factory.VALID && managed.match(factory)) {
                    managed.create(factory);
                    List list = (List) m_attached.get(factory);
                    if (list == null) {
                        list = new ArrayList();
                        list.add(managed);
                        m_attached.put(factory, list);
                    } else {
                        list.add(managed);
                    }
                    return;
                }
            }
        }
        // If there is no matching factory, add the instance to the idle list
        m_idle.add(managed);
    }

    /**
     * Disposes all instances declared by the given (leaving) bundle.
     * @param bundle the bundle.
     */
    void removeInstancesFromBundle(long bundle) {
        // Disposes instance from attached instances
        Map copy = null;

        // To avoid concurrent modification exception, we create a copy.
        synchronized (this) {
            copy = new HashMap(m_attached);
        }

        Collection col = copy.keySet();
        Iterator iterator = col.iterator();
        List instanceToRemove = new ArrayList();
        List factoryToRemove = new ArrayList();
        while (iterator.hasNext()) {
            IPojoFactory factory = (IPojoFactory) iterator.next();
            List list = (List) copy.get(factory);
            for (int i = 0; i < list.size(); i++) {
                ManagedInstance managed = (ManagedInstance) list.get(i);
                if (managed.m_bundleId == bundle) {
                    managed.dispose();
                    instanceToRemove.add(managed);
                }
            }
            if (!instanceToRemove.isEmpty()) {
                list.removeAll(instanceToRemove);
                if (list.isEmpty()) {
                    factory.removeFactoryStateListener(this);
                    factoryToRemove.add(factory);
                }
            }
        }

        // We remove from the original map
        for (int i = 0; i < factoryToRemove.size(); i++) {
            m_attached.remove(factoryToRemove.get(i));
        }

        // Delete idle instances
        List list = null;
        synchronized (this) {
            list = new ArrayList(m_idle);
        }
        instanceToRemove.clear();
        for (int i = 0; i < list.size(); i++) {
            ManagedInstance managed = (ManagedInstance) list.get(i);
            if (managed.m_bundleId == bundle) {
                instanceToRemove.add(managed);
            }
        }
        m_idle.removeAll(instanceToRemove);
    }

    /**
     * This method is called when a factory appears.
     * @param factory the new factory.
     */
    public synchronized void addFactory(IPojoFactory factory) {
        List createdInstances = new ArrayList(1);
        m_logger.log(Logger.DEBUG, "Add the factory " + factory.getName());
        m_factories.add(factory);
        for (int i = 0; i < m_idle.size(); i++) {
            ManagedInstance managed = (ManagedInstance) m_idle.get(i);
            if (managed.matchNameAndVersion(factory)) {
                // We have to subscribe to the factory.
                factory.addFactoryStateListener(this);
                if (factory.getState() == Factory.VALID && managed.match(factory)) {
                    managed.create(factory);
                    List list = (List) m_attached.get(factory);
                    if (list == null) {
                        list = new ArrayList();
                        list.add(managed);
                        m_attached.put(factory, list);
                    } else {
                        list.add(managed);
                    }
                    createdInstances.add(managed);
                }
            }
        }
        if (!createdInstances.isEmpty()) {
            m_idle.removeAll(createdInstances);
        }
    }

    /**
     * This method is called when a factory is leaving.
     * @param factory the leaving factory
     */
    void removeFactory(IPojoFactory factory) {
        factory.removeFactoryStateListener(this);
        m_factories.remove(factory);
        onInvalidation(factory);
        m_attached.remove(factory);
    }

    /**
     * This method is called when the given factory becomes valid.
     * @param factory the factory becoming valid.
     */
    private void onValidation(IPojoFactory factory) {
        List toRemove = new ArrayList();
        for (int i = 0; i < m_idle.size(); i++) {
            ManagedInstance managed = (ManagedInstance) m_idle.get(i);
            if (managed.match(factory)) {
                managed.create(factory);
                List list = (List) m_attached.get(factory);
                if (list == null) {
                    list = new ArrayList();
                    list.add(managed);
                    m_attached.put(factory, list);
                } else {
                    list.add(managed);
                }
                toRemove.add(managed);
            }
        }
        if (!toRemove.isEmpty()) {
            m_idle.removeAll(toRemove);
        }
    }

    /**
     * This method is called when the given factory becomes invalid.
     * @param factory the factory becoming invalid.
     */
    private void onInvalidation(IPojoFactory factory) {
        List instances = (List) m_attached.remove(factory);
        if (instances != null) {
            for (int i = 0; i < instances.size(); i++) {
                ManagedInstance managed = (ManagedInstance) instances.get(i);
                managed.dispose();
                m_idle.add(managed);
            }
        }
    }

    /**
     * This method is called when the state of a factory changes.
     * @param factory the factory.
     * @param newState the new state.
     * @see org.apache.felix.ipojo.FactoryStateListener#stateChanged(org.apache.felix.ipojo.Factory, int)
     */
    public void stateChanged(Factory factory, int newState) {
        if (newState == Factory.VALID) {
            m_logger.log(Logger.DEBUG, "A factory is becoming valid : " + factory.getName());
            onValidation((IPojoFactory) factory);
        } else {
            m_logger.log(Logger.DEBUG, "A factory is becoming invalid : " + factory.getName());
            onInvalidation((IPojoFactory) factory);
        }
    }

    /**
     * This structure aims to manage a configuration.
     * It stores all necessary information to create an instance
     * and to track the factory.
     */
    private class ManagedInstance {
        /**
         * The configuration of the instance to create.
         */
        private Dictionary m_configuration;

        /**
         * The bundle which creates the instance.
         */
        private long m_bundleId;

        /**
         * The factory used to create the instance.
         */
        private IPojoFactory m_factory;

        /**
         * The created instance.
         */
        private ComponentInstance m_instance;

        /**
         * Creates a ManagedInstance.
         * @param conf the configuration to create.
         * @param bundle the bundle in which the instance is declared.
         */
        ManagedInstance(Dictionary conf, long bundle) {
            m_configuration = conf;
            m_bundleId = bundle;
        }

        /**
         * Checks if the required factory name match with the given factory.
         * This methods checks only the name, and not the configuration.
         * @param factory the factory to test
         * @return <code>true</code> if the factory name and the version (if set) match, <code>false</code>
         * otherwise.
         */
        public boolean matchNameAndVersion(IPojoFactory factory) {
            String component = (String) m_configuration.get("component");
            String v = (String) m_configuration.get("factory.version");
            if (v == null) {
                return factory.getName().equals(component) || factory.getClassName().equalsIgnoreCase(component);
            } else {
                return (factory.getName().equals(component) || factory.getClassName().equalsIgnoreCase(component))
                    && v.equals(factory.getVersion());
            }
        }

        /**
         * Checks if the given factory match with the factory
         * required by this instance. A factory matches if its
         * name or its class name is equals to the 'component'
         * property of the instance. Then the acceptability of
         * the configuration is checked.
         * @param factory the factory to confront against the current instance.
         * @return <code>true</code> if the factory matches.
         */
        public boolean match(IPojoFactory factory) {
            // Test factory name (and classname)
            if (matchNameAndVersion(factory)) {
                // Test factory accessibility
                if (factory.m_isPublic || factory.getBundleContext().getBundle().getBundleId() == m_bundleId) {
                    // Test the configuration validity.
                    try {
                        factory.checkAcceptability(m_configuration);
                        return true;
                    } catch (UnacceptableConfiguration e) {
                        m_logger.log(Logger.ERROR, "An instance can be bound to a matching factory, however the configuration seems unacceptable : "
                                + e.getMessage());
                        return false;
                    } catch (MissingHandlerException e) {
                        m_logger.log(Logger.ERROR, "An instance can be bound to a matching factory, but this factory cannot be used : "
                                + e.getMessage());
                        return false;
                    }
                }
            }
            return false;
        }

        /**
         * Creates the instance by using the given factory.
         * @param factory the factory to use to create the instance. The factory must match.
         */
        public void create(IPojoFactory factory) {
            try {
                m_factory = factory;
                m_instance = m_factory.createComponentInstance(m_configuration);
                m_logger.log(Logger.INFO, "Instance created");
            } catch (UnacceptableConfiguration e) {
                m_logger.log(Logger.ERROR, "A matching factory was found for " + m_configuration + ", but the instantiation failed : "
                        + e.getMessage());
            } catch (MissingHandlerException e) {
                m_logger.log(Logger.ERROR, "A matching factory was found for " + m_configuration + ", but the instantiation failed : "
                        + e.getMessage());
            } catch (ConfigurationException e) {
                m_logger.log(Logger.ERROR, "A matching factory was found for " + m_configuration + ", but the instantiation failed : "
                        + e.getMessage());
            }
        }

        /**
         * Disposes the current instance if not <code>null</code>.
         */
        public void dispose() {
            if (m_instance != null) {
                m_instance.dispose();
            }
            m_instance = null;
            m_factory = null;
        }
    }

}
