/*
 * Decompiled with CFR 0.152.
 */
package org.apache.geode.internal;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.Serializable;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.io.FileUtils;
import org.apache.geode.internal.ClassPathLoader;
import org.apache.geode.internal.DeployedJar;
import org.apache.geode.internal.logging.LogService;
import org.apache.logging.log4j.Logger;

public class JarDeployer
implements Serializable {
    private static final long serialVersionUID = 1L;
    private static final Logger logger = LogService.getLogger();
    public static final String JAR_PREFIX_FOR_REGEX = "";
    private static final Lock lock = new ReentrantLock();
    private final Map<String, DeployedJar> deployedJars = new ConcurrentHashMap<String, DeployedJar>();
    public static final Pattern versionedPattern = Pattern.compile("(.*)\\.v(\\d++).jar$");
    private final File deployDirectory;
    final Pattern oldNamingPattern = Pattern.compile("^vf\\.gf#(.*)\\.jar#(\\d+)$");

    public JarDeployer() {
        this.deployDirectory = new File(System.getProperty("user.dir"));
    }

    public JarDeployer(File deployDirectory) {
        this.deployDirectory = deployDirectory;
    }

    public File getDeployDirectory() {
        return this.deployDirectory;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public DeployedJar deployWithoutRegistering(String jarName, byte[] jarBytes) throws IOException {
        lock.lock();
        try {
            boolean shouldDeployNewVersion = this.shouldDeployNewVersion(jarName, jarBytes);
            if (!shouldDeployNewVersion) {
                logger.debug("No need to deploy a new version of {}", (Object)jarName);
                DeployedJar deployedJar = null;
                return deployedJar;
            }
            this.verifyWritableDeployDirectory();
            File newVersionedJarFile = this.getNextVersionedJarFile(jarName);
            this.writeJarBytesToFile(newVersionedJarFile, jarBytes);
            DeployedJar deployedJar = new DeployedJar(newVersionedJarFile, jarName, jarBytes);
            return deployedJar;
        }
        finally {
            lock.unlock();
        }
    }

    public List<DeployedJar> findDeployedJars() {
        return this.getDeployedJars().values().stream().collect(Collectors.toList());
    }

    public void suspendAll() {
        lock.lock();
    }

    public void resumeAll() {
        lock.unlock();
    }

    protected File getNextVersionedJarFile(String unversionedJarName) {
        String nextVersionedJarName;
        File[] oldVersions = this.findSortedOldVersionsOfJar(unversionedJarName);
        if (oldVersions == null || oldVersions.length == 0) {
            nextVersionedJarName = this.removeJarExtension(unversionedJarName) + ".v1.jar";
        } else {
            String latestVersionedJarName = oldVersions[0].getName();
            int nextVersion = JarDeployer.extractVersionFromFilename(latestVersionedJarName) + 1;
            nextVersionedJarName = this.removeJarExtension(unversionedJarName) + ".v" + nextVersion + ".jar";
        }
        logger.debug("Next versioned jar name for {} is {}", (Object)unversionedJarName, (Object)nextVersionedJarName);
        return new File(this.deployDirectory, nextVersionedJarName);
    }

    private boolean writeJarBytesToFile(File file, byte[] jarBytes) throws IOException {
        boolean isDebugEnabled = logger.isDebugEnabled();
        if (file.createNewFile()) {
            if (isDebugEnabled) {
                logger.debug("Successfully created new JAR file: {}", (Object)file.getAbsolutePath());
            }
            FileOutputStream outStream = new FileOutputStream(file);
            ((OutputStream)outStream).write(jarBytes);
            ((OutputStream)outStream).close();
            return true;
        }
        return this.doesFileMatchBytes(file, jarBytes);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean doesFileMatchBytes(File file, byte[] bytes) throws IOException {
        String absolutePath = file.getAbsolutePath();
        boolean keepTrying = true;
        boolean isDebugEnabled = logger.isDebugEnabled();
        while (file.length() < (long)bytes.length && keepTrying) {
            if (isDebugEnabled) {
                logger.debug("Loop waiting for another to write file: {}", (Object)absolutePath);
            }
            long startingFileLength = file.length();
            try {
                Thread.sleep(500L);
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
            if (startingFileLength != file.length()) continue;
            if (isDebugEnabled) {
                logger.debug("Done waiting for another to write file: {}", (Object)absolutePath);
            }
            keepTrying = false;
        }
        if (file.length() != (long)bytes.length) {
            if (isDebugEnabled) {
                logger.debug("Unmatching file length when waiting for another to write file: {}", (Object)absolutePath);
            }
            return false;
        }
        try (BufferedInputStream inStream = new BufferedInputStream(new FileInputStream(file));){
            for (int index = 0; index < bytes.length; ++index) {
                if ((byte)inStream.read() == bytes[index]) continue;
                if (isDebugEnabled) {
                    logger.debug("Did not find a match when waiting for another to write file: {}", (Object)absolutePath);
                }
                boolean bl = false;
                return bl;
            }
        }
        return true;
    }

    public static int extractVersionFromFilename(String filename) {
        Matcher matcher = versionedPattern.matcher(filename);
        if (matcher.find()) {
            return Integer.parseInt(matcher.group(2));
        }
        return 0;
    }

    protected Set<String> findDistinctDeployedJarsOnDisk() {
        File[] oldFiles = this.deployDirectory.listFiles((file, name) -> versionedPattern.matcher(name).matches());
        HashSet<String> jarNames = new HashSet<String>();
        for (File oldFile : oldFiles) {
            Matcher matcher = versionedPattern.matcher(oldFile.getName());
            matcher.find();
            jarNames.add(matcher.group(1) + ".jar");
        }
        return jarNames;
    }

    protected File[] findSortedOldVersionsOfJar(String unversionedJarName) {
        logger.debug("Finding sorted old versions of {}", (Object)unversionedJarName);
        Pattern pattern = Pattern.compile(JAR_PREFIX_FOR_REGEX + this.removeJarExtension(unversionedJarName) + "\\.v\\d++\\.jar$");
        File[] oldJarFiles = this.deployDirectory.listFiles((file, name) -> pattern.matcher(name).matches());
        Arrays.sort(oldJarFiles, (file1, file2) -> {
            int file1Version = JarDeployer.extractVersionFromFilename(file1.getName());
            int file2Version = JarDeployer.extractVersionFromFilename(file2.getName());
            return file2Version - file1Version;
        });
        logger.debug("Found [{}]", (Object)Arrays.stream(oldJarFiles).map(File::getAbsolutePath).collect(Collectors.joining(",")));
        return oldJarFiles;
    }

    protected String removeJarExtension(String jarName) {
        if (jarName != null && jarName.endsWith(".jar")) {
            return jarName.replaceAll("\\.jar$", JAR_PREFIX_FOR_REGEX);
        }
        return jarName;
    }

    public void verifyWritableDeployDirectory() throws IOException {
        Exception exception = null;
        int tryCount = 0;
        do {
            try {
                if (this.deployDirectory.canWrite()) {
                    return;
                }
            }
            catch (Exception ex) {
                exception = ex;
            }
            try {
                Thread.sleep(100L);
            }
            catch (InterruptedException iex) {
                logger.error("Interrupted while testing writable deploy directory", (Throwable)iex);
            }
        } while (tryCount++ < 20);
        if (exception != null) {
            throw new IOException("Unable to write to deploy directory", exception);
        }
        throw new IOException("Unable to write to deploy directory: " + this.deployDirectory.getCanonicalPath());
    }

    protected void renameJarsWithOldNamingConvention() throws IOException {
        Set<File> jarsWithOldNamingConvention = this.findJarsWithOldNamingConvention();
        if (jarsWithOldNamingConvention.isEmpty()) {
            return;
        }
        for (File jar : jarsWithOldNamingConvention) {
            this.renameJarWithOldNamingConvention(jar);
        }
    }

    protected Set<File> findJarsWithOldNamingConvention() {
        return Stream.of(this.deployDirectory.listFiles()).filter(file -> this.isOldNamingConvention(file.getName())).collect(Collectors.toSet());
    }

    protected boolean isOldNamingConvention(String fileName) {
        return this.oldNamingPattern.matcher(fileName).matches();
    }

    private void renameJarWithOldNamingConvention(File oldJar) throws IOException {
        Matcher matcher = this.oldNamingPattern.matcher(oldJar.getName());
        if (!matcher.matches()) {
            throw new IllegalArgumentException("The given jar " + oldJar.getCanonicalPath() + " does not match the old naming convention");
        }
        String unversionedJarNameWithoutExtension = matcher.group(1);
        String jarVersion = matcher.group(2);
        String newJarName = unversionedJarNameWithoutExtension + ".v" + jarVersion + ".jar";
        File newJar = new File(this.deployDirectory, newJarName);
        logger.debug("Renaming deployed jar from {} to {}", (Object)oldJar.getCanonicalPath(), (Object)newJar.getCanonicalPath());
        FileUtils.moveFile((File)oldJar, (File)newJar);
    }

    public void loadPreviouslyDeployedJarsFromDisk() {
        logger.info("Loading previously deployed jars");
        lock.lock();
        try {
            this.verifyWritableDeployDirectory();
            this.renameJarsWithOldNamingConvention();
            Set<String> jarNames = this.findDistinctDeployedJarsOnDisk();
            if (jarNames.isEmpty()) {
                return;
            }
            ArrayList<DeployedJar> latestVersionOfEachJar = new ArrayList<DeployedJar>();
            for (String jarName : jarNames) {
                DeployedJar deployedJar = this.findLatestValidDeployedJarFromDisk(jarName);
                if (deployedJar == null) continue;
                latestVersionOfEachJar.add(deployedJar);
                this.deleteOtherVersionsOfJar(deployedJar);
            }
            this.registerNewVersions(latestVersionOfEachJar);
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
        finally {
            lock.unlock();
        }
    }

    public void deleteOtherVersionsOfJar(DeployedJar deployedJar) {
        logger.info("Deleting all versions of " + deployedJar.getJarName() + " other than " + deployedJar.getFileName());
        File[] jarFiles = this.findSortedOldVersionsOfJar(deployedJar.getJarName());
        Stream.of(jarFiles).filter(jarFile -> !jarFile.equals(deployedJar.getFile())).forEach(jarFile -> {
            logger.info("Deleting old version of jar: " + jarFile.getAbsolutePath());
            FileUtils.deleteQuietly((File)jarFile);
        });
    }

    public DeployedJar findLatestValidDeployedJarFromDisk(String unversionedJarName) throws IOException {
        File[] jarFiles = this.findSortedOldVersionsOfJar(unversionedJarName);
        Optional<File> latestValidDeployedJarOptional = Arrays.stream(jarFiles).filter(Objects::nonNull).filter(jarFile -> {
            try {
                return DeployedJar.hasValidJarContent(FileUtils.readFileToByteArray((File)jarFile));
            }
            catch (IOException e) {
                return false;
            }
        }).findFirst();
        if (!latestValidDeployedJarOptional.isPresent()) {
            return null;
        }
        File latestValidDeployedJar = latestValidDeployedJarOptional.get();
        return new DeployedJar(latestValidDeployedJar, unversionedJarName);
    }

    public URL[] getDeployedJarURLs() {
        return (URL[])this.deployedJars.values().stream().map(DeployedJar::getFileURL).toArray(URL[]::new);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<DeployedJar> registerNewVersions(List<DeployedJar> deployedJars) throws ClassNotFoundException {
        lock.lock();
        try {
            HashMap<DeployedJar, DeployedJar> newVersionToOldVersion = new HashMap<DeployedJar, DeployedJar>();
            for (DeployedJar deployedJar : deployedJars) {
                if (deployedJar == null) continue;
                logger.info("Registering new version of jar: {}", (Object)deployedJar);
                DeployedJar oldJar = this.deployedJars.put(deployedJar.getJarName(), deployedJar);
                newVersionToOldVersion.put(deployedJar, oldJar);
            }
            ClassPathLoader.getLatest().rebuildClassLoaderForDeployedJars();
            for (Map.Entry entry : newVersionToOldVersion.entrySet()) {
                DeployedJar newjar = (DeployedJar)entry.getKey();
                DeployedJar oldJar = (DeployedJar)entry.getValue();
                newjar.registerFunctions();
                if (oldJar == null) continue;
                oldJar.cleanUp(newjar);
            }
        }
        finally {
            lock.unlock();
        }
        return deployedJars;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<DeployedJar> deploy(String[] jarNames, byte[][] jarBytes) throws IOException, ClassNotFoundException {
        int i;
        DeployedJar[] deployedJars = new DeployedJar[jarNames.length];
        for (i = 0; i < jarNames.length; ++i) {
            if (DeployedJar.hasValidJarContent(jarBytes[i])) continue;
            throw new IllegalArgumentException("File does not contain valid JAR content: " + jarNames[i]);
        }
        lock.lock();
        try {
            for (i = 0; i < jarNames.length; ++i) {
                String jarName = jarNames[i];
                byte[] newJarBytes = jarBytes[i];
                deployedJars[i] = this.deployWithoutRegistering(jarName, newJarBytes);
            }
            List<DeployedJar> list = this.registerNewVersions(Arrays.asList(deployedJars));
            return list;
        }
        finally {
            lock.unlock();
        }
    }

    private boolean shouldDeployNewVersion(String jarName, byte[] newJarBytes) throws IOException {
        DeployedJar oldDeployedJar = this.deployedJars.get(jarName);
        if (oldDeployedJar == null) {
            return true;
        }
        if (oldDeployedJar.hasSameContentAs(newJarBytes)) {
            logger.warn("Jar is identical to the latest deployed version: {}", (Object)oldDeployedJar.getFileCanonicalPath());
            return false;
        }
        return true;
    }

    public DeployedJar findDeployedJar(String jarName) {
        return this.deployedJars.get(jarName);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public DeployedJar deploy(String jarName, byte[] jarBytes) throws IOException, ClassNotFoundException {
        lock.lock();
        try {
            List<DeployedJar> deployedJars = this.deploy(new String[]{jarName}, new byte[][]{jarBytes});
            if (deployedJars == null || deployedJars.size() == 0) {
                DeployedJar deployedJar = null;
                return deployedJar;
            }
            DeployedJar deployedJar = deployedJars.get(0);
            return deployedJar;
        }
        finally {
            lock.unlock();
        }
    }

    public Map<String, DeployedJar> getDeployedJars() {
        return Collections.unmodifiableMap(this.deployedJars);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public String undeploy(String jarName) throws IOException {
        lock.lock();
        try {
            DeployedJar deployedJar = this.deployedJars.remove(jarName);
            if (deployedJar == null) {
                throw new IllegalArgumentException("JAR not deployed");
            }
            ClassPathLoader.getLatest().rebuildClassLoaderForDeployedJars();
            deployedJar.cleanUp(null);
            this.deleteAllVersionsOfJar(jarName);
            String string = deployedJar.getFileCanonicalPath();
            return string;
        }
        finally {
            lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void deleteAllVersionsOfJar(String unversionedJarName) {
        lock.lock();
        try {
            File[] jarFiles;
            for (File jarFile : jarFiles = this.findSortedOldVersionsOfJar(unversionedJarName)) {
                logger.info("Deleting: {}", (Object)jarFile.getAbsolutePath());
                FileUtils.deleteQuietly((File)jarFile);
            }
        }
        finally {
            lock.unlock();
        }
    }
}

