/*
 * Decompiled with CFR 0.152.
 */
package org.apache.geode.internal.offheap;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.SortedSet;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReferenceArray;
import org.apache.geode.OutOfOffHeapMemoryException;
import org.apache.geode.internal.logging.LogService;
import org.apache.geode.internal.offheap.Fragment;
import org.apache.geode.internal.offheap.MemoryAllocatorImpl;
import org.apache.geode.internal.offheap.MemoryBlock;
import org.apache.geode.internal.offheap.MemoryBlockNode;
import org.apache.geode.internal.offheap.OffHeapMemoryStats;
import org.apache.geode.internal.offheap.OffHeapStoredObject;
import org.apache.geode.internal.offheap.OffHeapStoredObjectAddressStack;
import org.apache.geode.internal.offheap.Slab;
import org.apache.logging.log4j.Logger;

public class FreeListManager {
    static final Logger logger = LogService.getLogger();
    private final Slab[] slabs;
    private final long totalSlabSize;
    private final AtomicReferenceArray<OffHeapStoredObjectAddressStack> tinyFreeLists = new AtomicReferenceArray(TINY_FREE_LIST_COUNT);
    private final ConcurrentSkipListSet<OffHeapStoredObject> hugeChunkSet = new ConcurrentSkipListSet();
    private final AtomicLong allocatedSize = new AtomicLong(0L);
    private final AtomicInteger lastFragmentAllocation = new AtomicInteger(0);
    private final CopyOnWriteArrayList<Fragment> fragmentList;
    private final MemoryAllocatorImpl ma;
    protected final AtomicInteger defragmentationCount = new AtomicInteger();
    final boolean validateMemoryWithFill = Boolean.getBoolean("gemfire.validateOffHeapWithFill");
    public static final int TINY_MULTIPLE = Integer.getInteger("gemfire.OFF_HEAP_ALIGNMENT", 8);
    public static final int TINY_FREE_LIST_COUNT;
    public static final int HUGE_MULTIPLE = 256;
    public static final int MAX_TINY;

    private int getNearestTinyMultiple(int size) {
        return (size - 1) / TINY_MULTIPLE;
    }

    List<OffHeapStoredObject> getLiveChunks() {
        ArrayList<OffHeapStoredObject> result = new ArrayList<OffHeapStoredObject>();
        for (int i = 0; i < this.slabs.length; ++i) {
            this.getLiveChunks(this.slabs[i], result);
        }
        return result;
    }

    private void getLiveChunks(Slab slab, List<OffHeapStoredObject> result) {
        long addr = slab.getMemoryAddress();
        while (addr <= slab.getMemoryAddress() + (long)slab.getSize() - 16L) {
            Fragment f = this.isAddrInFragmentFreeSpace(addr);
            if (f != null) {
                addr = f.getAddress() + (long)f.getSize();
                continue;
            }
            int curChunkSize = OffHeapStoredObject.getSize(addr);
            int refCount = OffHeapStoredObject.getRefCount(addr);
            if (refCount > 0) {
                result.add(new OffHeapStoredObject(addr));
            }
            addr += (long)curChunkSize;
        }
    }

    private Fragment isAddrInFragmentFreeSpace(long addr) {
        for (Fragment f : this.fragmentList) {
            if (addr < f.getAddress() + (long)f.getFreeIndex() || addr >= f.getAddress() + (long)f.getSize()) continue;
            return f;
        }
        return null;
    }

    public long getUsedMemory() {
        return this.allocatedSize.get();
    }

    public long getFreeMemory() {
        return this.getTotalMemory() - this.getUsedMemory();
    }

    long getFreeFragmentMemory() {
        long result = 0L;
        for (Fragment f : this.fragmentList) {
            int freeSpace = f.freeSpace();
            if (freeSpace < 16) continue;
            result += (long)freeSpace;
        }
        return result;
    }

    long getFreeTinyMemory() {
        long tinyFree = 0L;
        for (int i = 0; i < this.tinyFreeLists.length(); ++i) {
            OffHeapStoredObjectAddressStack cl = this.tinyFreeLists.get(i);
            if (cl == null) continue;
            tinyFree += cl.computeTotalSize();
        }
        return tinyFree;
    }

    long getFreeHugeMemory() {
        long hugeFree = 0L;
        for (OffHeapStoredObject c : this.hugeChunkSet) {
            hugeFree += (long)c.getSize();
        }
        return hugeFree;
    }

    public FreeListManager(MemoryAllocatorImpl ma, Slab[] slabs) {
        this.ma = ma;
        this.slabs = slabs;
        long total = 0L;
        Fragment[] tmp = new Fragment[slabs.length];
        for (int i = 0; i < slabs.length; ++i) {
            tmp[i] = this.createFragment(slabs[i].getMemoryAddress(), slabs[i].getSize());
            total += (long)slabs[i].getSize();
        }
        this.fragmentList = new CopyOnWriteArrayList<Fragment>(tmp);
        this.totalSlabSize = total;
        this.fillFragments();
    }

    protected Fragment createFragment(long addr, int size) {
        return new Fragment(addr, size);
    }

    private void fillFragments() {
        if (!this.validateMemoryWithFill) {
            return;
        }
        for (Fragment fragment : this.fragmentList) {
            fragment.fill();
        }
    }

    public OffHeapStoredObject allocate(int size) {
        assert (size > 0);
        OffHeapStoredObject result = this.basicAllocate(size, true);
        result.setDataSize(size);
        this.allocatedSize.addAndGet(result.getSize());
        result.initializeUseCount();
        return result;
    }

    private OffHeapStoredObject basicAllocate(int size, boolean useSlabs) {
        if (useSlabs) {
            size += 8;
        }
        if (size <= MAX_TINY) {
            return this.allocateTiny(size, useSlabs);
        }
        return this.allocateHuge(size, useSlabs);
    }

    private OffHeapStoredObject allocateFromFragments(int chunkSize) {
        do {
            OffHeapStoredObject result;
            int lastAllocationId;
            int i;
            for (i = lastAllocationId = this.lastFragmentAllocation.get(); i < this.fragmentList.size(); ++i) {
                result = this.allocateFromFragment(i, chunkSize);
                if (result == null) continue;
                return result;
            }
            for (i = 0; i < lastAllocationId; ++i) {
                result = this.allocateFromFragment(i, chunkSize);
                if (result == null) continue;
                return result;
            }
        } while (this.defragment(chunkSize));
        this.logOffHeapState(chunkSize);
        OutOfOffHeapMemoryException failure = new OutOfOffHeapMemoryException("Out of off-heap memory. Could not allocate size of " + chunkSize);
        try {
            throw failure;
        }
        catch (Throwable throwable) {
            this.ma.getOutOfOffHeapMemoryListener().outOfOffHeapMemory(failure);
            throw throwable;
        }
    }

    private void logOffHeapState(int chunkSize) {
        this.logOffHeapState(logger, chunkSize);
    }

    void logOffHeapState(Logger lw, int chunkSize) {
        OffHeapMemoryStats stats = this.ma.getStats();
        lw.info("OutOfOffHeapMemory allocating size of " + chunkSize + ". allocated=" + this.allocatedSize.get() + " defragmentations=" + this.defragmentationCount.get() + " objects=" + stats.getObjects() + " free=" + stats.getFreeMemory() + " fragments=" + stats.getFragments() + " largestFragment=" + stats.getLargestFragment() + " fragmentation=" + stats.getFragmentation());
        this.logFragmentState(lw);
        this.logTinyState(lw);
        this.logHugeState(lw);
    }

    private void logHugeState(Logger lw) {
        for (OffHeapStoredObject c : this.hugeChunkSet) {
            lw.info("Free huge of size " + c.getSize());
        }
    }

    private void logTinyState(Logger lw) {
        for (int i = 0; i < this.tinyFreeLists.length(); ++i) {
            OffHeapStoredObjectAddressStack cl = this.tinyFreeLists.get(i);
            if (cl == null) continue;
            cl.logSizes(lw, "Free tiny of size ");
        }
    }

    private void logFragmentState(Logger lw) {
        for (Fragment f : this.fragmentList) {
            int freeSpace = f.freeSpace();
            if (freeSpace <= 0) continue;
            lw.info("Fragment at " + f.getAddress() + " of size " + f.getSize() + " has " + freeSpace + " bytes free.");
        }
    }

    boolean combineIfAdjacentAndSmallEnough(long lowAddr, long highAddr) {
        int highSize;
        int combinedSize;
        assert (lowAddr <= highAddr);
        int lowSize = OffHeapStoredObject.getSize(lowAddr);
        if (this.isAdjacent(lowAddr, lowSize, highAddr) && this.isSmallEnough(combinedSize = lowSize + (highSize = OffHeapStoredObject.getSize(highAddr)))) {
            OffHeapStoredObject.setSize(lowAddr, combinedSize);
            return true;
        }
        return false;
    }

    boolean isAdjacent(long lowAddr, int lowSize, long highAddr) {
        return lowAddr + (long)lowSize == highAddr;
    }

    boolean isSmallEnough(long size) {
        return size <= Integer.MAX_VALUE;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    boolean defragment(int chunkSize) {
        boolean bl;
        boolean result;
        long startDefragmentationTime = this.ma.getStats().startDefragmentation();
        int countPreSync = this.defragmentationCount.get();
        this.afterDefragmentationCountFetched();
        try {
            FreeListManager freeListManager = this;
            synchronized (freeListManager) {
                if (this.defragmentationCount.get() != countPreSync) {
                    boolean bl2 = true;
                    // MONITOREXIT @DISABLED, blocks:[0, 4, 6] lbl9 : MonitorExitStatement: MONITOREXIT : var5_4
                    this.ma.getStats().endDefragmentation(startDefragmentationTime);
                    return bl2;
                }
                result = this.doDefragment(chunkSize);
                this.defragmentationCount.incrementAndGet();
            }
        }
        catch (Throwable throwable) {
            this.ma.getStats().endDefragmentation(startDefragmentationTime);
            throw throwable;
        }
        {
            bl = result;
        }
        this.ma.getStats().endDefragmentation(startDefragmentationTime);
        return bl;
    }

    boolean doDefragment(int chunkSize) {
        boolean result = false;
        ArrayList<LongStack> freeChunks = new ArrayList<LongStack>();
        this.collectFreeChunks(freeChunks);
        ResizableLongArray sorted = new ResizableLongArray();
        for (LongStack l : freeChunks) {
            long addr = l.poll();
            while (addr != 0L) {
                int idx = sorted.binarySearch(addr);
                idx = -idx;
                int sortedSize = sorted.size();
                if (--idx == sortedSize) {
                    if (sortedSize == 0) {
                        sorted.add(addr);
                    } else if (!this.combineIfAdjacentAndSmallEnough(sorted.get(idx - 1), addr)) {
                        sorted.add(addr);
                    }
                } else if (this.combineIfAdjacentAndSmallEnough(addr, sorted.get(idx))) {
                    sorted.set(idx, addr);
                } else if (idx == 0 || !this.combineIfAdjacentAndSmallEnough(sorted.get(idx - 1), addr)) {
                    sorted.insert(idx, addr);
                }
                addr = l.poll();
            }
        }
        for (int i = sorted.size() - 1; i > 0; --i) {
            if (!this.combineIfAdjacentAndSmallEnough(sorted.get(i - 1), sorted.get(i))) continue;
            sorted.set(i, 0L);
        }
        int largestFragment = 0;
        this.lastFragmentAllocation.set(0);
        ArrayList<Fragment> tmp = new ArrayList<Fragment>();
        for (int i = sorted.size() - 1; i >= 0; --i) {
            long addr = sorted.get(i);
            if (addr == 0L) continue;
            int addrSize = OffHeapStoredObject.getSize(addr);
            Fragment f = this.createFragment(addr, addrSize);
            if (addrSize >= chunkSize) {
                result = true;
            }
            if (addrSize > largestFragment) {
                largestFragment = addrSize;
                tmp.add(0, f);
                continue;
            }
            tmp.add(f);
        }
        this.fragmentList.addAll(tmp);
        this.fillFragments();
        this.ma.getStats().setLargestFragment(largestFragment);
        this.ma.getStats().setFragments(tmp.size());
        this.ma.getStats().setFragmentation(this.getFragmentation());
        return result;
    }

    protected void afterDefragmentationCountFetched() {
    }

    static void verifyOffHeapAlignment(int tinyMultiple) {
        if (tinyMultiple <= 0 || (tinyMultiple & 3) != 0) {
            throw new IllegalStateException("gemfire.OFF_HEAP_ALIGNMENT must be a multiple of 8.");
        }
        if (tinyMultiple > 256) {
            throw new IllegalStateException("gemfire.OFF_HEAP_ALIGNMENT must be <= 256 and a multiple of 8.");
        }
    }

    static void verifyOffHeapFreeListCount(int tinyFreeListCount) {
        if (tinyFreeListCount <= 0) {
            throw new IllegalStateException("gemfire.OFF_HEAP_FREE_LIST_COUNT must be >= 1.");
        }
    }

    static void verifyHugeMultiple(int hugeMultiple) {
        if (hugeMultiple > 256 || hugeMultiple < 0) {
            throw new IllegalStateException("HUGE_MULTIPLE must be >= 0 and <= 256 but it was " + hugeMultiple);
        }
    }

    protected int getFragmentCount() {
        return this.fragmentList.size();
    }

    protected int getFragmentation() {
        if (this.getUsedMemory() == 0L) {
            return 0;
        }
        int availableFragments = this.getFragmentCount();
        if (availableFragments == 0) {
            return 0;
        }
        if (availableFragments == 1) {
            return 0;
        }
        long freeMemory = this.getFreeMemory();
        assert (freeMemory > 16L);
        long maxPossibleFragments = freeMemory / 16L;
        double fragmentation = (double)availableFragments / (double)maxPossibleFragments * 100.0;
        return (int)Math.rint(fragmentation);
    }

    private void collectFreeChunks(List<LongStack> l) {
        this.collectFreeFragmentChunks(l);
        this.collectFreeHugeChunks(l);
        this.collectFreeTinyChunks(l);
    }

    List<Fragment> getFragmentList() {
        return this.fragmentList;
    }

    private void collectFreeFragmentChunks(List<LongStack> l) {
        if (this.fragmentList.size() == 0) {
            return;
        }
        OffHeapStoredObjectAddressStack result = new OffHeapStoredObjectAddressStack();
        for (Fragment f : this.fragmentList) {
            int offset;
            int diff;
            do {
                offset = f.getFreeIndex();
            } while ((diff = f.getSize() - offset) >= 16 && !f.allocate(offset, offset + diff));
            if (diff < 16) {
                assert (diff == 0);
                continue;
            }
            long chunkAddr = f.getAddress() + (long)offset;
            OffHeapStoredObject.setSize(chunkAddr, diff);
            result.offer(chunkAddr);
        }
        this.fragmentList.clear();
        if (!result.isEmpty()) {
            l.add(result);
        }
    }

    private void collectFreeTinyChunks(List<LongStack> l) {
        for (int i = 0; i < this.tinyFreeLists.length(); ++i) {
            long head;
            OffHeapStoredObjectAddressStack cl = this.tinyFreeLists.get(i);
            if (cl == null || (head = cl.clear()) == 0L) continue;
            l.add(new OffHeapStoredObjectAddressStack(head));
        }
    }

    private void collectFreeHugeChunks(List<LongStack> l) {
        OffHeapStoredObject c = this.hugeChunkSet.pollFirst();
        OffHeapStoredObjectAddressStack result = null;
        while (c != null) {
            if (result == null) {
                result = new OffHeapStoredObjectAddressStack();
                l.add(result);
            }
            result.offer(c.getAddress());
            c = this.hugeChunkSet.pollFirst();
        }
    }

    OffHeapStoredObject allocateFromFragment(int fragIdx, int chunkSize) {
        boolean retryFragment;
        Fragment fragment;
        if (fragIdx >= this.fragmentList.size()) {
            return null;
        }
        try {
            fragment = this.fragmentList.get(fragIdx);
        }
        catch (IndexOutOfBoundsException ignore) {
            return null;
        }
        do {
            OffHeapStoredObject result;
            retryFragment = false;
            int oldOffset = fragment.getFreeIndex();
            int fragmentSize = fragment.getSize();
            int fragmentFreeSize = fragmentSize - oldOffset;
            if (fragmentFreeSize < chunkSize) continue;
            int newOffset = oldOffset + chunkSize;
            int extraSize = fragmentSize - newOffset;
            if (extraSize < 16) {
                newOffset += extraSize;
            } else {
                extraSize = 0;
            }
            if (fragment.allocate(oldOffset, newOffset)) {
                this.lastFragmentAllocation.set(fragIdx);
                result = new OffHeapStoredObject(fragment.getAddress() + (long)oldOffset, chunkSize + extraSize);
                this.checkDataIntegrity(result);
                return result;
            }
            result = this.basicAllocate(chunkSize, false);
            if (result != null) {
                return result;
            }
            retryFragment = true;
        } while (retryFragment);
        return null;
    }

    private int round(int multiple, int value) {
        return (int)(((long)value + (long)(multiple - 1)) / (long)multiple * (long)multiple);
    }

    private OffHeapStoredObject allocateTiny(int size, boolean useFragments) {
        return this.basicAllocate(this.getNearestTinyMultiple(size), TINY_MULTIPLE, 0, this.tinyFreeLists, useFragments);
    }

    private OffHeapStoredObject basicAllocate(int idx, int multiple, int offset, AtomicReferenceArray<OffHeapStoredObjectAddressStack> freeLists, boolean useFragments) {
        long memAddr;
        OffHeapStoredObjectAddressStack clq = freeLists.get(idx);
        if (clq != null && (memAddr = clq.poll()) != 0L) {
            OffHeapStoredObject result = new OffHeapStoredObject(memAddr);
            this.checkDataIntegrity(result);
            result.readyForAllocation();
            return result;
        }
        if (useFragments) {
            return this.allocateFromFragments((idx + 1) * multiple + offset);
        }
        return null;
    }

    private OffHeapStoredObject allocateHuge(int size, boolean useFragments) {
        SearchMarker sizeHolder = new SearchMarker(size);
        SortedSet ts = this.hugeChunkSet.tailSet((Object)sizeHolder);
        OffHeapStoredObject result = (OffHeapStoredObject)ts.pollFirst();
        if (result != null) {
            if (result.getSize() - 248 < size) {
                this.checkDataIntegrity(result);
                result.readyForAllocation();
                return result;
            }
            this.hugeChunkSet.add(result);
        }
        if (useFragments) {
            return this.allocateFromFragments(this.round(TINY_MULTIPLE, size));
        }
        return null;
    }

    private void checkDataIntegrity(OffHeapStoredObject data) {
        if (this.validateMemoryWithFill) {
            data.validateFill();
        }
    }

    public void free(long addr) {
        if (this.validateMemoryWithFill) {
            OffHeapStoredObject.fill(addr);
        }
        this.free(addr, true);
    }

    private void free(long addr, boolean updateStats) {
        int cSize = OffHeapStoredObject.getSize(addr);
        if (updateStats) {
            OffHeapMemoryStats stats = this.ma.getStats();
            stats.incObjects(-1);
            this.allocatedSize.addAndGet(-cSize);
            stats.incUsedMemory(-cSize);
            stats.incFreeMemory(cSize);
            this.ma.notifyListeners();
        }
        if (cSize <= MAX_TINY) {
            this.freeTiny(addr, cSize);
        } else {
            this.freeHuge(addr, cSize);
        }
    }

    private void freeTiny(long addr, int cSize) {
        this.basicFree(addr, this.getNearestTinyMultiple(cSize), this.tinyFreeLists);
    }

    private void basicFree(long addr, int idx, AtomicReferenceArray<OffHeapStoredObjectAddressStack> freeLists) {
        OffHeapStoredObjectAddressStack clq = freeLists.get(idx);
        if (clq != null) {
            clq.offer(addr);
        } else {
            clq = this.createFreeListForEmptySlot(freeLists, idx);
            clq.offer(addr);
            if (!freeLists.compareAndSet(idx, null, clq)) {
                clq = freeLists.get(idx);
                clq.offer(addr);
            }
        }
    }

    protected OffHeapStoredObjectAddressStack createFreeListForEmptySlot(AtomicReferenceArray<OffHeapStoredObjectAddressStack> freeLists, int idx) {
        return new OffHeapStoredObjectAddressStack();
    }

    private void freeHuge(long addr, int cSize) {
        this.hugeChunkSet.add(new OffHeapStoredObject(addr));
    }

    List<MemoryBlock> getOrderedBlocks() {
        ArrayList<MemoryBlock> value = new ArrayList<MemoryBlock>();
        this.addBlocksFromFragments(this.fragmentList, value);
        this.addBlocksFromChunks(this.getLiveChunks(), value);
        this.addBlocksFromChunks(this.hugeChunkSet, value);
        this.addMemoryBlocks(this.getTinyFreeBlocks(), value);
        Collections.sort(value, new Comparator<MemoryBlock>(){

            @Override
            public int compare(MemoryBlock o1, MemoryBlock o2) {
                return Long.valueOf(o1.getAddress()).compareTo(o2.getAddress());
            }
        });
        return value;
    }

    private void addBlocksFromFragments(Collection<Fragment> src, List<MemoryBlock> dest) {
        for (MemoryBlock memoryBlock : src) {
            dest.add(new MemoryBlockNode(this.ma, memoryBlock));
        }
    }

    private void addBlocksFromChunks(Collection<OffHeapStoredObject> src, List<MemoryBlock> dest) {
        for (OffHeapStoredObject chunk : src) {
            dest.add(new MemoryBlockNode(this.ma, chunk));
        }
    }

    private void addMemoryBlocks(Collection<MemoryBlock> src, List<MemoryBlock> dest) {
        for (MemoryBlock block : src) {
            dest.add(new MemoryBlockNode(this.ma, block));
        }
    }

    private List<MemoryBlock> getTinyFreeBlocks() {
        ArrayList<MemoryBlock> value = new ArrayList<MemoryBlock>();
        MemoryAllocatorImpl sma = this.ma;
        for (int i = 0; i < this.tinyFreeLists.length(); ++i) {
            if (this.tinyFreeLists.get(i) == null) continue;
            long addr = this.tinyFreeLists.get(i).getTopAddress();
            while (addr != 0L) {
                value.add(new MemoryBlockNode(sma, new TinyMemoryBlock(addr, i)));
                addr = OffHeapStoredObject.getNext(addr);
            }
        }
        return value;
    }

    List<MemoryBlock> getAllocatedBlocks() {
        ArrayList<MemoryBlock> value = new ArrayList<MemoryBlock>();
        this.addBlocksFromChunks(this.getLiveChunks(), value);
        Collections.sort(value, new Comparator<MemoryBlock>(){

            @Override
            public int compare(MemoryBlock o1, MemoryBlock o2) {
                return Long.valueOf(o1.getAddress()).compareTo(o2.getAddress());
            }
        });
        return value;
    }

    long getTotalMemory() {
        return this.totalSlabSize;
    }

    void freeSlabs() {
        for (int i = 0; i < this.slabs.length; ++i) {
            this.slabs[i].free();
        }
    }

    boolean okToReuse(Slab[] newSlabs) {
        return newSlabs == null || newSlabs == this.slabs;
    }

    int getLargestSlabSize() {
        return this.slabs[0].getSize();
    }

    int findSlab(long addr) {
        for (int i = 0; i < this.slabs.length; ++i) {
            Slab slab = this.slabs[i];
            long slabAddr = slab.getMemoryAddress();
            if (addr < slabAddr || addr >= slabAddr + (long)slab.getSize()) continue;
            return i;
        }
        throw new IllegalStateException("could not find a slab for addr " + addr);
    }

    void getSlabDescriptions(StringBuilder sb) {
        for (int i = 0; i < this.slabs.length; ++i) {
            long startAddr = this.slabs[i].getMemoryAddress();
            long endAddr = startAddr + (long)this.slabs[i].getSize();
            sb.append("[").append(Long.toString(startAddr, 16)).append("..").append(Long.toString(endAddr, 16)).append("] ");
        }
    }

    boolean validateAddressAndSizeWithinSlab(long addr, int size) {
        for (int i = 0; i < this.slabs.length; ++i) {
            if (this.slabs[i].getMemoryAddress() > addr || addr >= this.slabs[i].getMemoryAddress() + (long)this.slabs[i].getSize()) continue;
            if (size != -1 && (this.slabs[i].getMemoryAddress() > addr + (long)size - 1L || addr + (long)size - 1L >= this.slabs[i].getMemoryAddress() + (long)this.slabs[i].getSize())) {
                throw new IllegalStateException(" address 0x" + Long.toString(addr + (long)size - 1L, 16) + " does not address the original slab memory");
            }
            return true;
        }
        return false;
    }

    static {
        FreeListManager.verifyOffHeapAlignment(TINY_MULTIPLE);
        TINY_FREE_LIST_COUNT = Integer.getInteger("gemfire.OFF_HEAP_FREE_LIST_COUNT", 65536);
        FreeListManager.verifyOffHeapFreeListCount(TINY_FREE_LIST_COUNT);
        FreeListManager.verifyHugeMultiple(256);
        MAX_TINY = TINY_MULTIPLE * TINY_FREE_LIST_COUNT;
    }

    protected static class TinyMemoryBlock
    implements MemoryBlock {
        private final long address;
        private final int freeListId;

        protected TinyMemoryBlock(long address, int freeListId) {
            this.address = address;
            this.freeListId = freeListId;
        }

        @Override
        public MemoryBlock.State getState() {
            return MemoryBlock.State.DEALLOCATED;
        }

        @Override
        public long getAddress() {
            return this.address;
        }

        @Override
        public int getBlockSize() {
            return OffHeapStoredObject.getSize(this.address);
        }

        @Override
        public MemoryBlock getNextBlock() {
            throw new UnsupportedOperationException();
        }

        @Override
        public int getSlabId() {
            throw new UnsupportedOperationException();
        }

        @Override
        public int getFreeListId() {
            return this.freeListId;
        }

        @Override
        public int getRefCount() {
            return 0;
        }

        @Override
        public String getDataType() {
            return "N/A";
        }

        @Override
        public boolean isSerialized() {
            return false;
        }

        @Override
        public boolean isCompressed() {
            return false;
        }

        @Override
        public Object getDataValue() {
            return null;
        }

        public boolean equals(Object o) {
            if (o instanceof TinyMemoryBlock) {
                return this.getAddress() == ((TinyMemoryBlock)o).getAddress();
            }
            return false;
        }

        public int hashCode() {
            long value = this.getAddress();
            return (int)(value ^ value >>> 32);
        }
    }

    private static class SearchMarker
    extends OffHeapStoredObject {
        private final int size;

        public SearchMarker(int size) {
            this.size = size;
        }

        @Override
        public int getSize() {
            return this.size;
        }
    }

    public static class ResizableLongArray {
        private static final int SORT_ARRAY_BLOCK_SIZE = 128;
        long[] data = new long[128];
        int size = 0;

        public int binarySearch(long l) {
            return Arrays.binarySearch(this.data, 0, this.size, l);
        }

        public int size() {
            return this.size;
        }

        public long get(int idx) {
            return this.data[idx];
        }

        public void set(int idx, long l) {
            this.data[idx] = l;
        }

        public void add(long l) {
            if (this.size >= this.data.length) {
                long[] newData = new long[this.data.length + 128];
                System.arraycopy(this.data, 0, newData, 0, this.data.length);
                this.data = newData;
            }
            this.data[this.size] = l;
            ++this.size;
        }

        public void insert(int idx, long l) {
            if (this.size >= this.data.length) {
                long[] newData = new long[this.data.length + 128];
                System.arraycopy(this.data, 0, newData, 0, idx);
                newData[idx] = l;
                System.arraycopy(this.data, idx, newData, idx + 1, this.size - idx);
                this.data = newData;
            } else {
                System.arraycopy(this.data, idx, this.data, idx + 1, this.size - idx);
                this.data[idx] = l;
            }
            ++this.size;
        }
    }

    public static interface LongStack {
        public long poll();
    }
}

