/*
 * Decompiled with CFR 0.152.
 */
package org.apache.gobblin.fsm;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.SetMultimap;
import java.io.Closeable;
import java.io.IOException;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.gobblin.fsm.StateWithCallbacks;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FiniteStateMachine<T> {
    private static final Logger log = LoggerFactory.getLogger(FiniteStateMachine.class);
    private final SetMultimap<T, T> allowedTransitions;
    private final Set<T> universalEnds;
    private final T errorState;
    private final ReentrantReadWriteLock lock;
    private final Condition condition;
    private final T initialState;
    private volatile T currentState;
    private volatile Transition currentTransition;

    protected FiniteStateMachine(SetMultimap<T, T> allowedTransitions, Set<T> universalEnds, T errorState, T initialState) {
        this.allowedTransitions = allowedTransitions;
        this.universalEnds = universalEnds;
        this.errorState = errorState;
        this.lock = new ReentrantReadWriteLock();
        this.condition = this.lock.writeLock().newCondition();
        this.initialState = initialState;
        this.currentState = initialState;
        if (this.currentState instanceof StateWithCallbacks) {
            ((StateWithCallbacks)this.currentState).onEnterState(null);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Transition startTransition(T endState) throws UnallowedTransitionException, InterruptedException {
        try {
            Transition transition;
            this.lock.writeLock().lock();
            while (this.isTransitioning()) {
                this.condition.await();
            }
            if (!this.isAllowedTransition(this.currentState, endState)) {
                throw new UnallowedTransitionException(this.currentState, endState);
            }
            this.currentTransition = transition = new Transition(endState);
            Transition transition2 = transition;
            return transition2;
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    public void transitionImmediately(T endState) throws UnallowedTransitionException, InterruptedException, FailedTransitionCallbackException {
        Transition transition = this.startTransition(endState);
        transition.close();
    }

    public boolean transitionIfAllowed(T endState) throws InterruptedException, FailedTransitionCallbackException {
        try {
            this.transitionImmediately(endState);
        }
        catch (UnallowedTransitionException exc) {
            return false;
        }
        return true;
    }

    public T getCurrentState() throws InterruptedException {
        try {
            this.lock.readLock().lock();
            this.waitForNonTransitioningReadLock();
            T t = this.currentState;
            return t;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    @VisibleForTesting
    T getCurrentStateEvenIfTransitioning() {
        return this.currentState;
    }

    public FiniteStateMachine<T> cloneAtInitialState() {
        return new FiniteStateMachine<T>(this.allowedTransitions, this.universalEnds, this.errorState, this.initialState);
    }

    public FiniteStateMachine<T> cloneAtCurrentState() throws InterruptedException {
        try {
            this.lock.readLock().lock();
            this.waitForNonTransitioningReadLock();
            FiniteStateMachine<T> finiteStateMachine = new FiniteStateMachine<T>(this.allowedTransitions, this.universalEnds, this.errorState, this.currentState);
            return finiteStateMachine;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    private void waitForNonTransitioningReadLock() throws InterruptedException {
        if (this.isTransitioning()) {
            this.lock.readLock().unlock();
            this.lock.writeLock().lock();
            try {
                while (this.isTransitioning()) {
                    this.condition.await();
                }
                this.lock.readLock().lock();
            }
            finally {
                this.lock.writeLock().unlock();
            }
        }
    }

    private boolean isTransitioning() {
        if (this.currentTransition != null && Thread.currentThread().equals(this.currentTransition.ownerThread)) {
            throw new ReentrantStableStateWait("Tried to check for non-transitioning state from a thread that had already initiated a transition, this may indicate a deadlock. To change end state use Transition.changeEndState() instead.");
        }
        if (this.currentTransition != null && !this.currentTransition.ownerThread.isAlive()) {
            throw new AbandonedTransitionException(this.currentTransition.ownerThread);
        }
        return this.currentTransition != null;
    }

    protected boolean isAllowedTransition(T startState, T endState) {
        if (endState.equals(this.errorState)) {
            return true;
        }
        if (this.universalEnds.contains(endState)) {
            return true;
        }
        Set endStates = this.allowedTransitions.get(startState);
        return endStates != null && endStates.contains(endState);
    }

    public static class FailedTransitionCallbackException
    extends IOException {
        private final Transition transition;
        private final FailedCallback failedCallback;
        private final Throwable originalException;

        public FailedTransitionCallbackException(Transition transition, FailedCallback failedCallback, Throwable originalException) {
            super("Failed callbacks when ending transition.", originalException);
            this.transition = transition;
            this.failedCallback = failedCallback;
            this.originalException = originalException;
        }

        public Transition getTransition() {
            return this.transition;
        }

        public FailedCallback getFailedCallback() {
            return this.failedCallback;
        }

        public Throwable getOriginalException() {
            return this.originalException;
        }
    }

    public static enum FailedCallback {
        START_STATE,
        END_STATE;

    }

    public static class AbandonedTransitionException
    extends RuntimeException {
        private final Thread startingThread;

        public AbandonedTransitionException(Thread startingThread) {
            super(String.format("Thread %s initiated a transition but became inactive before closing it.", startingThread));
            this.startingThread = startingThread;
        }
    }

    public static class ReentrantStableStateWait
    extends RuntimeException {
        public ReentrantStableStateWait(String message) {
            super(message);
        }
    }

    public static class UnallowedTransitionException
    extends Exception {
        private final Object startState;
        private final Object endState;

        public UnallowedTransitionException(Object startState, Object endState) {
            super(String.format("Unallowed transition: %s -> %s", startState, endState));
            this.startState = startState;
            this.endState = endState;
        }

        public Object getStartState() {
            return this.startState;
        }

        public Object getEndState() {
            return this.endState;
        }
    }

    public class Transition
    implements Closeable {
        private final Thread ownerThread = Thread.currentThread();
        private volatile T endState;
        private volatile boolean closed;

        private Transition(T endState) {
            this.endState = endState;
            this.closed = false;
        }

        public T getStartState() {
            if (this.closed) {
                throw new IllegalStateException("Transition already closed.");
            }
            return FiniteStateMachine.this.currentState;
        }

        public synchronized void changeEndState(T endState) throws UnallowedTransitionException {
            if (this.closed) {
                throw new IllegalStateException("Transition already closed.");
            }
            if (!FiniteStateMachine.this.isAllowedTransition(FiniteStateMachine.this.currentState, endState)) {
                throw new UnallowedTransitionException(FiniteStateMachine.this.currentState, endState);
            }
            this.endState = endState;
        }

        public synchronized void switchEndStateToErrorState() {
            this.endState = FiniteStateMachine.this.errorState;
        }

        @Override
        public void close() throws FailedTransitionCallbackException {
            this.doClose(true);
        }

        public void closeWithoutCallbacks() {
            try {
                this.doClose(false);
            }
            catch (FailedTransitionCallbackException exc) {
                throw new IllegalStateException(String.format("Close without callbacks threw a %s. This is an error in code.", FailedTransitionCallbackException.class), exc);
            }
        }

        private synchronized void doClose(boolean withCallbacks) throws FailedTransitionCallbackException {
            if (this.closed) {
                return;
            }
            try {
                FiniteStateMachine.this.lock.writeLock().lock();
                try {
                    if (withCallbacks && this.getStartState() instanceof StateWithCallbacks) {
                        ((StateWithCallbacks)this.getStartState()).onLeaveState(this.endState);
                    }
                }
                catch (Throwable t) {
                    throw new FailedTransitionCallbackException(this, FailedCallback.START_STATE, t);
                }
                try {
                    if (withCallbacks && this.endState instanceof StateWithCallbacks) {
                        ((StateWithCallbacks)this.endState).onEnterState(this.getStartState());
                    }
                }
                catch (Throwable t) {
                    throw new FailedTransitionCallbackException(this, FailedCallback.END_STATE, t);
                }
                this.closed = true;
                FiniteStateMachine.this.currentState = this.endState;
                FiniteStateMachine.this.currentTransition = null;
                FiniteStateMachine.this.condition.signalAll();
            }
            finally {
                FiniteStateMachine.this.lock.writeLock().unlock();
            }
        }
    }

    public static class Builder<T> {
        private final SetMultimap<T, T> allowedTransitions = HashMultimap.create();
        private final Set<T> universalEnds = new HashSet<T>();
        private T errorState;

        public Builder<T> addTransition(T startState, T endState) {
            this.allowedTransitions.put(startState, endState);
            return this;
        }

        public Builder<T> addUniversalEnd(T state) {
            this.universalEnds.add(state);
            return this;
        }

        public Builder<T> errorState(T state) {
            this.errorState = state;
            return this;
        }

        public FiniteStateMachine<T> build(T initialState) {
            return new FiniteStateMachine<T>(this.allowedTransitions, this.universalEnds, this.errorState, initialState);
        }
    }
}

