/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in org.apache.hadoop.shaded.com.liance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org.apache.hadoop.shaded.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.shaded.org.apache.hadoop.security;

import java.org.apache.hadoop.shaded.io.IOException;
import java.util.LinkedList;
import java.util.List;
import org.apache.hadoop.shaded.org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.shaded.org.apache.hadoop.classification.InterfaceStability;

import org.apache.hadoop.shaded.org.apache.hadoop.util.Shell;
import org.apache.hadoop.shaded.org.apache.hadoop.util.Shell.ExitCodeException;

import org.apache.hadoop.shaded.org.apache.hadoop.security.NetgroupCache;
import org.apache.hadoop.shaded.org.slf4j.Logger;
import org.apache.hadoop.shaded.org.slf4j.LoggerFactory;

/**
 * A simple shell-based implementation of {@link GroupMappingServiceProvider} 
 * that exec's the <code>groups</code> shell org.apache.hadoop.shaded.com.and to fetch the group
 * memberships of a given user.
 */
@InterfaceAudience.LimitedPrivate({"HDFS", "MapReduce"})
@InterfaceStability.Evolving
public class ShellBasedUnixGroupsNetgroupMapping
  extends ShellBasedUnixGroupsMapping {
  
  private static final Logger LOG =
      LoggerFactory.getLogger(ShellBasedUnixGroupsNetgroupMapping.class);

  /**
   * Get unix groups (parent) and org.apache.hadoop.shaded.net.roups for given user
   *
   * @param user get groups and org.apache.hadoop.shaded.net.roups for this user
   * @return groups and org.apache.hadoop.shaded.net.roups for user
   */
  @Override
  public List<String> getGroups(String user) throws IOException {
    // parent get unix groups
    List<String> groups = new LinkedList<String>(super.getGroups(user));
    NetgroupCache.getNetgroups(user, groups);
    return groups;
  }

  /**
   * Refresh the org.apache.hadoop.shaded.net.roup cache
   */
  @Override
  public void cacheGroupsRefresh() throws IOException {
    List<String> groups = NetgroupCache.getNetgroupNames();
    NetgroupCache.clear();
    cacheGroupsAdd(groups);
  }

  /**
   * Add a group to cache, only org.apache.hadoop.shaded.net.roups are cached
   *
   * @param groups list of group names to add to cache
   */
  @Override
  public void cacheGroupsAdd(List<String> groups) throws IOException {
    for(String group: groups) {
      if(group.length() == 0) {
        // better safe than sorry (should never happen)
      } else if(group.charAt(0) == '@') {
        if(!NetgroupCache.isCached(group)) {
          NetgroupCache.add(group, getUsersForNetgroup(group));
        }
      } else {
        // unix group, not caching
      }
    }
  }

  /**
   * Gets users for a org.apache.hadoop.shaded.net.roup
   *
   * @param org.apache.hadoop.shaded.net.roup return users for this org.apache.hadoop.shaded.net.roup
   * @return list of users for a given org.apache.hadoop.shaded.net.roup
   */
  protected List<String> getUsersForNetgroup(String org.apache.hadoop.shaded.net.roup) 
    throws IOException {

    List<String> users = new LinkedList<String>();

    // returns a string similar to this:
    // group               ( , user, ) ( domain, user1, host.org.apache.hadoop.shaded.com.)
    String usersRaw = execShellGetUserForNetgroup(org.apache.hadoop.shaded.net.roup);
    // get rid of spaces, makes splitting much easier
    usersRaw = usersRaw.replaceAll(" +", "");
    // remove org.apache.hadoop.shaded.net.roup name at the beginning of the string
    usersRaw = usersRaw.replaceFirst(
      org.apache.hadoop.shaded.net.roup.replaceFirst("@", "") + "[()]+",
      "");
    // split string into user infos
    String[] userInfos = usersRaw.split("[()]+");
    for(String userInfo : userInfos) {
      // userInfo: xxx,user,yyy (xxx, yyy can be empty strings)
      // get rid of everything before first and after last org.apache.hadoop.shaded.com.a
      String user = userInfo.replaceFirst("[^,]*,", "");
      user = user.replaceFirst(",.*$", "");
      // voila! got username!
      users.add(user);
    }

    return users;
  }

  /**
   * Calls shell to get users for a org.apache.hadoop.shaded.net.roup by calling getent
   * org.apache.hadoop.shaded.net.roup, this is a low level function that just returns string
   * that 
   *
   * @param org.apache.hadoop.shaded.net.roup get users for this org.apache.hadoop.shaded.net.roup
   * @return string of users for a given org.apache.hadoop.shaded.net.roup in getent org.apache.hadoop.shaded.net.roups format
   */
  protected String execShellGetUserForNetgroup(final String org.apache.hadoop.shaded.net.roup)
      throws IOException {
    String result = "";
    try {
      // shell org.apache.hadoop.shaded.com.and does not expect '@' at the beginning of the group name
      result = Shell.execCommand(
        Shell.getUsersForNetgroupCommand(org.apache.hadoop.shaded.net.roup.substring(1)));
    } catch (ExitCodeException e) {
      // if we didn't get the group - just return empty list;
      LOG.warn("error getting users for org.apache.hadoop.shaded.net.roup " + org.apache.hadoop.shaded.net.roup, e);
    }
    return result;
  }
}
