/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.yarn.server.resourcemanager;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.net.NetUtils;
import org.apache.hadoop.net.Node;
import org.apache.hadoop.service.AbstractService;
import org.apache.hadoop.service.CompositeService;
import org.apache.hadoop.shaded.com.google.common.annotations.VisibleForTesting;
import org.apache.hadoop.util.HostsFileReader;
import org.apache.hadoop.util.StringUtils;
import org.apache.hadoop.util.Time;
import org.apache.hadoop.yarn.api.records.NodeId;
import org.apache.hadoop.yarn.api.records.NodeState;
import org.apache.hadoop.yarn.api.records.Resource;
import org.apache.hadoop.yarn.conf.YarnConfiguration;
import org.apache.hadoop.yarn.event.Event;
import org.apache.hadoop.yarn.event.EventHandler;
import org.apache.hadoop.yarn.exceptions.YarnException;
import org.apache.hadoop.yarn.exceptions.YarnRuntimeException;
import org.apache.hadoop.yarn.server.resourcemanager.ClusterMetrics;
import org.apache.hadoop.yarn.server.resourcemanager.NodesListManagerEvent;
import org.apache.hadoop.yarn.server.resourcemanager.NodesListManagerEventType;
import org.apache.hadoop.yarn.server.resourcemanager.RMContext;
import org.apache.hadoop.yarn.server.resourcemanager.rmapp.RMApp;
import org.apache.hadoop.yarn.server.resourcemanager.rmapp.RMAppNodeUpdateEvent;
import org.apache.hadoop.yarn.server.resourcemanager.rmnode.RMNode;
import org.apache.hadoop.yarn.server.resourcemanager.rmnode.RMNodeDecommissioningEvent;
import org.apache.hadoop.yarn.server.resourcemanager.rmnode.RMNodeEvent;
import org.apache.hadoop.yarn.server.resourcemanager.rmnode.RMNodeEventType;
import org.apache.hadoop.yarn.server.resourcemanager.rmnode.RMNodeImpl;
import org.apache.hadoop.yarn.util.Clock;
import org.apache.hadoop.yarn.util.SystemClock;

public class NodesListManager
extends CompositeService
implements EventHandler<NodesListManagerEvent> {
    private static final Log LOG = LogFactory.getLog(NodesListManager.class);
    private HostsFileReader hostsReader;
    private Configuration conf;
    private final RMContext rmContext;
    private int defaultDecTimeoutSecs = 3600;
    private String includesFile;
    private String excludesFile;
    private Resolver resolver;
    private Timer removalTimer;
    private int nodeRemovalCheckInterval;

    public NodesListManager(RMContext rmContext) {
        super(NodesListManager.class.getName());
        this.rmContext = rmContext;
    }

    protected void serviceInit(Configuration conf) throws Exception {
        this.conf = conf;
        int nodeIpCacheTimeout = conf.getInt("yarn.resourcemanager.node-ip-cache.expiry-interval-secs", -1);
        if (nodeIpCacheTimeout <= 0) {
            this.resolver = new DirectResolver();
        } else {
            this.resolver = new CachedResolver((Clock)SystemClock.getInstance(), nodeIpCacheTimeout);
            this.addIfService(this.resolver);
        }
        try {
            this.includesFile = conf.get("yarn.resourcemanager.nodes.include-path", "");
            this.excludesFile = conf.get("yarn.resourcemanager.nodes.exclude-path", "");
            this.hostsReader = this.createHostsFileReader(this.includesFile, this.excludesFile);
            this.setDecomissionedNMs();
            this.printConfiguredHosts();
        }
        catch (YarnException ex) {
            this.disableHostsFileReader((Exception)((Object)ex));
        }
        catch (IOException ioe) {
            this.disableHostsFileReader(ioe);
        }
        final int nodeRemovalTimeout = conf.getInt("yarn.resourcemanager.node-removal-untracked.timeout-ms", 60000);
        this.nodeRemovalCheckInterval = Math.min(nodeRemovalTimeout / 2, 600000);
        this.removalTimer = new Timer("Node Removal Timer");
        this.removalTimer.schedule(new TimerTask(){

            @Override
            public void run() {
                long now = Time.monotonicNow();
                for (Map.Entry entry : NodesListManager.this.rmContext.getInactiveRMNodes().entrySet()) {
                    NodeId nodeId = (NodeId)entry.getKey();
                    RMNode rmNode = (RMNode)entry.getValue();
                    if (NodesListManager.this.isUntrackedNode(rmNode.getHostName())) {
                        RMNode result;
                        if (rmNode.getUntrackedTimeStamp() == 0L) {
                            rmNode.setUntrackedTimeStamp(now);
                            continue;
                        }
                        if (now - rmNode.getUntrackedTimeStamp() <= (long)nodeRemovalTimeout || (result = (RMNode)NodesListManager.this.rmContext.getInactiveRMNodes().remove(nodeId)) == null) continue;
                        NodesListManager.this.decrInactiveNMMetrics(rmNode);
                        LOG.info((Object)("Removed " + result.getState().toString() + " node " + result.getHostName() + " from inactive nodes list"));
                        continue;
                    }
                    rmNode.setUntrackedTimeStamp(0L);
                }
            }
        }, this.nodeRemovalCheckInterval, (long)this.nodeRemovalCheckInterval);
        super.serviceInit(conf);
    }

    private void decrInactiveNMMetrics(RMNode rmNode) {
        ClusterMetrics clusterMetrics = ClusterMetrics.getMetrics();
        switch (rmNode.getState()) {
            case SHUTDOWN: {
                clusterMetrics.decrNumShutdownNMs();
                break;
            }
            case DECOMMISSIONED: {
                clusterMetrics.decrDecommisionedNMs();
                break;
            }
            case LOST: {
                clusterMetrics.decrNumLostNMs();
                break;
            }
            case REBOOTED: {
                clusterMetrics.decrNumRebootedNMs();
                break;
            }
            default: {
                LOG.debug((Object)"Unexpected node state");
            }
        }
    }

    public void serviceStop() {
        this.removalTimer.cancel();
    }

    private void printConfiguredHosts() {
        if (!LOG.isDebugEnabled()) {
            return;
        }
        LOG.debug((Object)("hostsReader: in=" + this.conf.get("yarn.resourcemanager.nodes.include-path", "") + " out=" + this.conf.get("yarn.resourcemanager.nodes.exclude-path", "")));
        HostsFileReader.HostDetails hostDetails = this.hostsReader.getHostDetails();
        for (String include : hostDetails.getIncludedHosts()) {
            LOG.debug((Object)("include: " + include));
        }
        for (String exclude : hostDetails.getExcludedHosts()) {
            LOG.debug((Object)("exclude: " + exclude));
        }
    }

    public void refreshNodes(Configuration yarnConf) throws IOException, YarnException {
        this.refreshNodes(yarnConf, false);
    }

    public void refreshNodes(Configuration yarnConf, boolean graceful) throws IOException, YarnException {
        this.refreshHostsReader(yarnConf, graceful, null);
    }

    private void refreshHostsReader(Configuration yarnConf, boolean graceful, Integer timeout) throws IOException, YarnException {
        if (null == timeout) {
            timeout = this.readDecommissioningTimeout(yarnConf);
        }
        if (null == yarnConf) {
            yarnConf = new YarnConfiguration();
        }
        this.includesFile = yarnConf.get("yarn.resourcemanager.nodes.include-path", "");
        this.excludesFile = yarnConf.get("yarn.resourcemanager.nodes.exclude-path", "");
        LOG.info((Object)("refreshNodes excludesFile " + this.excludesFile));
        this.hostsReader.refresh(this.includesFile, this.excludesFile);
        this.printConfiguredHosts();
        LOG.info((Object)("hostsReader include:{" + StringUtils.join((CharSequence)",", (Iterable)this.hostsReader.getHosts()) + "} exclude:{" + StringUtils.join((CharSequence)",", (Iterable)this.hostsReader.getExcludedHosts()) + "}"));
        this.handleExcludeNodeList(graceful, timeout);
    }

    private void setDecomissionedNMs() {
        Set excludeList = this.hostsReader.getExcludedHosts();
        for (String host : excludeList) {
            NodeId nodeId = NodesListManager.createUnknownNodeId(host);
            RMNodeImpl rmNode = new RMNodeImpl(nodeId, this.rmContext, host, -1, -1, new UnknownNode(host), Resource.newInstance((int)0, (int)0), "unknown");
            this.rmContext.getInactiveRMNodes().put(nodeId, rmNode);
            rmNode.handle(new RMNodeEvent(nodeId, RMNodeEventType.DECOMMISSION));
        }
    }

    private void handleExcludeNodeList(boolean graceful, int timeout) {
        RMNodeEvent e;
        ArrayList<RMNode> nodesToRecom = new ArrayList<RMNode>();
        ArrayList<RMNode> nodesToDecom = new ArrayList<RMNode>();
        HostsFileReader.HostDetails hostDetails = this.hostsReader.getHostDetails();
        Set includes = hostDetails.getIncludedHosts();
        Set excludes = hostDetails.getExcludedHosts();
        for (RMNode n : this.rmContext.getRMNodes().values()) {
            NodeState s = n.getState();
            boolean isExcluded = !this.isValidNode(n.getHostName(), includes, excludes);
            String nodeStr = "node " + n.getNodeID() + " with state " + s;
            if (!isExcluded) {
                if (s != NodeState.DECOMMISSIONING) continue;
                LOG.info((Object)("Recommission " + nodeStr));
                nodesToRecom.add(n);
                continue;
            }
            if (graceful) {
                if (s != NodeState.DECOMMISSIONED && s != NodeState.DECOMMISSIONING) {
                    LOG.info((Object)("Gracefully decommission " + nodeStr));
                    nodesToDecom.add(n);
                    continue;
                }
                if (s == NodeState.DECOMMISSIONING && !Objects.equals(n.getDecommissioningTimeout(), timeout)) {
                    LOG.info((Object)("Update " + nodeStr + " timeout to be " + timeout));
                    nodesToDecom.add(n);
                    continue;
                }
                LOG.info((Object)("No action for " + nodeStr));
                continue;
            }
            if (s == NodeState.DECOMMISSIONED) continue;
            LOG.info((Object)("Forcefully decommission " + nodeStr));
            nodesToDecom.add(n);
        }
        for (RMNode n : nodesToRecom) {
            e = new RMNodeEvent(n.getNodeID(), RMNodeEventType.RECOMMISSION);
            this.rmContext.getDispatcher().getEventHandler().handle((Event)e);
        }
        for (RMNode n : nodesToDecom) {
            if (graceful) {
                e = new RMNodeDecommissioningEvent(n.getNodeID(), timeout);
            } else {
                RMNodeEventType eventType = this.isUntrackedNode(n.getHostName()) ? RMNodeEventType.SHUTDOWN : RMNodeEventType.DECOMMISSION;
                e = new RMNodeEvent(n.getNodeID(), eventType);
            }
            this.rmContext.getDispatcher().getEventHandler().handle((Event)e);
        }
        this.updateInactiveNodes();
    }

    @VisibleForTesting
    public int getNodeRemovalCheckInterval() {
        return this.nodeRemovalCheckInterval;
    }

    @VisibleForTesting
    public void setNodeRemovalCheckInterval(int interval) {
        this.nodeRemovalCheckInterval = interval;
    }

    @VisibleForTesting
    public Resolver getResolver() {
        return this.resolver;
    }

    public boolean isValidNode(String hostName) {
        HostsFileReader.HostDetails hostDetails = this.hostsReader.getHostDetails();
        return this.isValidNode(hostName, hostDetails.getIncludedHosts(), hostDetails.getExcludedHosts());
    }

    private boolean isValidNode(String hostName, Set<String> hostsList, Set<String> excludeList) {
        String ip = this.resolver.resolve(hostName);
        return (hostsList.isEmpty() || hostsList.contains(hostName) || hostsList.contains(ip)) && !excludeList.contains(hostName) && !excludeList.contains(ip);
    }

    private void sendRMAppNodeUpdateEventToNonFinalizedApps(RMNode eventNode, RMAppNodeUpdateEvent.RMAppNodeUpdateType appNodeUpdateType) {
        for (RMApp app : this.rmContext.getRMApps().values()) {
            if (app.isAppFinalStateStored()) continue;
            this.rmContext.getDispatcher().getEventHandler().handle((Event)new RMAppNodeUpdateEvent(app.getApplicationId(), eventNode, appNodeUpdateType));
        }
    }

    public void handle(NodesListManagerEvent event) {
        RMNode eventNode = event.getNode();
        switch ((NodesListManagerEventType)event.getType()) {
            case NODE_UNUSABLE: {
                LOG.debug((Object)(eventNode + " reported unusable"));
                this.sendRMAppNodeUpdateEventToNonFinalizedApps(eventNode, RMAppNodeUpdateEvent.RMAppNodeUpdateType.NODE_UNUSABLE);
                break;
            }
            case NODE_USABLE: {
                LOG.debug((Object)(eventNode + " reported usable"));
                this.sendRMAppNodeUpdateEventToNonFinalizedApps(eventNode, RMAppNodeUpdateEvent.RMAppNodeUpdateType.NODE_USABLE);
                break;
            }
            case NODE_DECOMMISSIONING: {
                LOG.debug((Object)(eventNode + " reported decommissioning"));
                this.sendRMAppNodeUpdateEventToNonFinalizedApps(eventNode, RMAppNodeUpdateEvent.RMAppNodeUpdateType.NODE_DECOMMISSIONING);
                break;
            }
            default: {
                LOG.error((Object)("Ignoring invalid eventtype " + event.getType()));
            }
        }
        if (this.resolver instanceof CachedResolver) {
            ((CachedResolver)this.resolver).removeFromCache(eventNode.getNodeID().getHost());
        }
    }

    private void disableHostsFileReader(Exception ex) {
        LOG.warn((Object)"Failed to init hostsReader, disabling", (Throwable)ex);
        try {
            this.includesFile = this.conf.get("");
            this.excludesFile = this.conf.get("");
            this.hostsReader = this.createHostsFileReader(this.includesFile, this.excludesFile);
            this.setDecomissionedNMs();
        }
        catch (IOException ioe2) {
            this.hostsReader = null;
            throw new YarnRuntimeException((Throwable)ioe2);
        }
        catch (YarnException e) {
            this.hostsReader = null;
            throw new YarnRuntimeException((Throwable)e);
        }
    }

    @VisibleForTesting
    public HostsFileReader getHostsReader() {
        return this.hostsReader;
    }

    private HostsFileReader createHostsFileReader(String includesFile, String excludesFile) throws IOException, YarnException {
        HostsFileReader hostsReader = new HostsFileReader(includesFile, includesFile == null || includesFile.isEmpty() ? null : this.rmContext.getConfigurationProvider().getConfigurationInputStream(this.conf, includesFile), excludesFile, excludesFile == null || excludesFile.isEmpty() ? null : this.rmContext.getConfigurationProvider().getConfigurationInputStream(this.conf, excludesFile));
        return hostsReader;
    }

    private void updateInactiveNodes() {
        long now = Time.monotonicNow();
        for (Map.Entry entry : this.rmContext.getInactiveRMNodes().entrySet()) {
            NodeId nodeId = (NodeId)entry.getKey();
            RMNode rmNode = (RMNode)entry.getValue();
            if (!this.isUntrackedNode(nodeId.getHost()) || rmNode.getUntrackedTimeStamp() != 0L) continue;
            rmNode.setUntrackedTimeStamp(now);
        }
    }

    public boolean isUntrackedNode(String hostName) {
        String ip = this.resolver.resolve(hostName);
        HostsFileReader.HostDetails hostDetails = this.hostsReader.getHostDetails();
        Set hostsList = hostDetails.getIncludedHosts();
        Set excludeList = hostDetails.getExcludedHosts();
        return !hostsList.isEmpty() && !hostsList.contains(hostName) && !hostsList.contains(ip) && !excludeList.contains(hostName) && !excludeList.contains(ip);
    }

    public void refreshNodesGracefully(Configuration yarnConf, Integer timeout) throws IOException, YarnException {
        this.refreshHostsReader(yarnConf, true, timeout);
    }

    public Set<NodeId> checkForDecommissioningNodes() {
        HashSet<NodeId> decommissioningNodes = new HashSet<NodeId>();
        for (Map.Entry entry : this.rmContext.getRMNodes().entrySet()) {
            if (((RMNode)entry.getValue()).getState() != NodeState.DECOMMISSIONING) continue;
            decommissioningNodes.add((NodeId)entry.getKey());
        }
        return decommissioningNodes;
    }

    public void refreshNodesForcefully() {
        for (Map.Entry entry : this.rmContext.getRMNodes().entrySet()) {
            if (((RMNode)entry.getValue()).getState() != NodeState.DECOMMISSIONING) continue;
            RMNodeEventType nodeEventType = this.isUntrackedNode(((NodeId)entry.getKey()).getHost()) ? RMNodeEventType.SHUTDOWN : RMNodeEventType.DECOMMISSION;
            this.rmContext.getDispatcher().getEventHandler().handle((Event)new RMNodeEvent((NodeId)entry.getKey(), nodeEventType));
        }
    }

    private int readDecommissioningTimeout(Configuration pConf) {
        try {
            int configuredDefaultDecTimeoutSecs;
            if (pConf == null) {
                pConf = new YarnConfiguration();
            }
            if (this.defaultDecTimeoutSecs != (configuredDefaultDecTimeoutSecs = pConf.getInt("yarn.resourcemanager.nodemanager-graceful-decommission-timeout-secs", 3600))) {
                this.defaultDecTimeoutSecs = configuredDefaultDecTimeoutSecs;
                LOG.info((Object)("Use new decommissioningTimeoutSecs: " + this.defaultDecTimeoutSecs));
            }
        }
        catch (Exception e) {
            LOG.warn((Object)("Error readDecommissioningTimeout " + e.getMessage()));
        }
        return this.defaultDecTimeoutSecs;
    }

    public static NodeId createUnknownNodeId(String host) {
        return NodeId.newInstance((String)host, (int)-1);
    }

    private static class UnknownNode
    implements Node {
        private String host;

        public UnknownNode(String host) {
            this.host = host;
        }

        public String getNetworkLocation() {
            return null;
        }

        public void setNetworkLocation(String location) {
        }

        public String getName() {
            return this.host;
        }

        public Node getParent() {
            return null;
        }

        public void setParent(Node parent) {
        }

        public int getLevel() {
            return 0;
        }

        public void setLevel(int i) {
        }

        public String getHost() {
            return this.host;
        }

        public void setHost(String hst) {
            this.host = hst;
        }
    }

    @VisibleForTesting
    public static class CachedResolver
    extends AbstractService
    implements Resolver {
        private Map<String, CacheEntry> cache = new ConcurrentHashMap<String, CacheEntry>();
        private int expiryIntervalMs;
        private int checkIntervalMs;
        private final Clock clock;
        private Timer checkingTimer;
        private TimerTask expireChecker = new ExpireChecker();

        public CachedResolver(Clock clock, int expiryIntervalSecs) {
            super("NodesListManager.CachedResolver");
            this.clock = clock;
            this.expiryIntervalMs = expiryIntervalSecs * 1000;
            this.checkIntervalMs = this.expiryIntervalMs / 3;
            this.checkingTimer = new Timer("Timer-NodesListManager.CachedResolver.ExpireChecker", true);
        }

        protected void serviceStart() throws Exception {
            this.checkingTimer.scheduleAtFixedRate(this.expireChecker, this.checkIntervalMs, (long)this.checkIntervalMs);
            super.serviceStart();
        }

        protected void serviceStop() throws Exception {
            this.checkingTimer.cancel();
            super.serviceStop();
        }

        @VisibleForTesting
        public void addToCache(String hostName, String ip) {
            this.cache.put(hostName, new CacheEntry(ip, this.clock.getTime()));
        }

        public void removeFromCache(String hostName) {
            this.cache.remove(hostName);
        }

        private String reload(String hostName) {
            String ip = NetUtils.normalizeHostName((String)hostName);
            this.addToCache(hostName, ip);
            return ip;
        }

        @Override
        public String resolve(String hostName) {
            CacheEntry e = this.cache.get(hostName);
            if (e != null) {
                return e.ip;
            }
            return this.reload(hostName);
        }

        @VisibleForTesting
        public TimerTask getExpireChecker() {
            return this.expireChecker;
        }

        private class ExpireChecker
        extends TimerTask {
            private ExpireChecker() {
            }

            @Override
            public void run() {
                long currentTime = CachedResolver.this.clock.getTime();
                Iterator iterator = CachedResolver.this.cache.entrySet().iterator();
                while (iterator.hasNext()) {
                    Map.Entry entry = iterator.next();
                    if (currentTime <= ((CacheEntry)entry.getValue()).resolveTime + (long)CachedResolver.this.expiryIntervalMs) continue;
                    iterator.remove();
                    if (!LOG.isDebugEnabled()) continue;
                    LOG.debug((Object)("[" + (String)entry.getKey() + ":" + ((CacheEntry)entry.getValue()).ip + "] Expired after " + CachedResolver.this.expiryIntervalMs / 1000 + " secs"));
                }
            }
        }

        private static class CacheEntry {
            public String ip;
            public long resolveTime;

            public CacheEntry(String ip, long resolveTime) {
                this.ip = ip;
                this.resolveTime = resolveTime;
            }
        }
    }

    @VisibleForTesting
    public static class DirectResolver
    implements Resolver {
        @Override
        public String resolve(String hostName) {
            return NetUtils.normalizeHostName((String)hostName);
        }
    }

    @VisibleForTesting
    public static interface Resolver {
        public String resolve(String var1);
    }
}

