/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.hdfs.server.common.HdfsServerConstants;
import org.apache.hadoop.hdfs.server.common.Storage;
import org.apache.hadoop.hdfs.server.common.StorageErrorReporter;
import org.apache.hadoop.hdfs.server.common.StorageInfo;
import org.apache.hadoop.hdfs.server.namenode.EditLogFileInputStream;
import org.apache.hadoop.hdfs.server.namenode.EditLogFileOutputStream;
import org.apache.hadoop.hdfs.server.namenode.EditLogInputStream;
import org.apache.hadoop.hdfs.server.namenode.EditLogOutputStream;
import org.apache.hadoop.hdfs.server.namenode.FSEditLogLoader;
import org.apache.hadoop.hdfs.server.namenode.JournalManager;
import org.apache.hadoop.hdfs.server.namenode.NNStorage;
import org.apache.hadoop.hdfs.server.namenode.NNStorageRetentionManager;
import org.apache.hadoop.hdfs.server.namenode.NNUpgradeUtil;
import org.apache.hadoop.hdfs.server.protocol.NamespaceInfo;
import org.apache.hadoop.hdfs.server.protocol.RemoteEditLog;
import org.apache.hadoop.io.nativeio.NativeIO;
import org.apache.hadoop.shaded.com.google.common.annotations.VisibleForTesting;
import org.apache.hadoop.shaded.com.google.common.base.Joiner;
import org.apache.hadoop.shaded.com.google.common.base.Preconditions;
import org.apache.hadoop.shaded.com.google.common.collect.ComparisonChain;
import org.apache.hadoop.shaded.com.google.common.collect.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class FileJournalManager
implements JournalManager {
    private static final Logger LOG = LoggerFactory.getLogger(FileJournalManager.class);
    private final Configuration conf;
    private final Storage.StorageDirectory sd;
    private final StorageErrorReporter errorReporter;
    private int outputBufferCapacity = 524288;
    private static final Pattern EDITS_REGEX = Pattern.compile(NNStorage.NameNodeFile.EDITS.getName() + "_(\\d+)-(\\d+)");
    private static final Pattern EDITS_INPROGRESS_REGEX = Pattern.compile(NNStorage.NameNodeFile.EDITS_INPROGRESS.getName() + "_(\\d+)");
    private static final Pattern EDITS_INPROGRESS_STALE_REGEX = Pattern.compile(NNStorage.NameNodeFile.EDITS_INPROGRESS.getName() + "_(\\d+).*(\\S+)");
    @VisibleForTesting
    File currentInProgress = null;
    private long lastReadableTxId = Long.MAX_VALUE;
    @VisibleForTesting
    NNStorageRetentionManager.StoragePurger purger = new NNStorageRetentionManager.DeletionStoragePurger();

    public FileJournalManager(Configuration conf, Storage.StorageDirectory sd, StorageErrorReporter errorReporter) {
        this.conf = conf;
        this.sd = sd;
        this.errorReporter = errorReporter;
    }

    @Override
    public void close() throws IOException {
    }

    @Override
    public void format(NamespaceInfo ns, boolean force) throws IOException {
        throw new UnsupportedOperationException();
    }

    @Override
    public boolean hasSomeData() {
        throw new UnsupportedOperationException();
    }

    @Override
    public synchronized EditLogOutputStream startLogSegment(long txid, int layoutVersion) throws IOException {
        try {
            this.currentInProgress = NNStorage.getInProgressEditsFile(this.sd, txid);
            EditLogFileOutputStream stm = new EditLogFileOutputStream(this.conf, this.currentInProgress, this.outputBufferCapacity);
            ((EditLogOutputStream)stm).create(layoutVersion);
            return stm;
        }
        catch (IOException e) {
            LOG.warn("Unable to start log segment " + txid + " at " + this.currentInProgress + ": " + e.getLocalizedMessage());
            this.errorReporter.reportErrorOnFile(this.currentInProgress);
            throw e;
        }
    }

    @Override
    public synchronized void finalizeLogSegment(long firstTxId, long lastTxId) throws IOException {
        File inprogressFile = NNStorage.getInProgressEditsFile(this.sd, firstTxId);
        File dstFile = NNStorage.getFinalizedEditsFile(this.sd, firstTxId, lastTxId);
        LOG.info("Finalizing edits file " + inprogressFile + " -> " + dstFile);
        Preconditions.checkState((!dstFile.exists() ? 1 : 0) != 0, (Object)("Can't finalize edits file " + inprogressFile + " since finalized file already exists"));
        try {
            NativeIO.renameTo((File)inprogressFile, (File)dstFile);
        }
        catch (IOException e) {
            this.errorReporter.reportErrorOnFile(dstFile);
            throw new IllegalStateException("Unable to finalize edits file " + inprogressFile, e);
        }
        if (inprogressFile.equals(this.currentInProgress)) {
            this.currentInProgress = null;
        }
    }

    @VisibleForTesting
    public Storage.StorageDirectory getStorageDirectory() {
        return this.sd;
    }

    @Override
    public synchronized void setOutputBufferCapacity(int size) {
        this.outputBufferCapacity = size;
    }

    public long getLastReadableTxId() {
        return this.lastReadableTxId;
    }

    public void setLastReadableTxId(long id) {
        this.lastReadableTxId = id;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void purgeLogsOlderThan(long minTxIdToKeep) throws IOException {
        LOG.info("Purging logs older than " + minTxIdToKeep);
        File[] files = FileUtil.listFiles((File)this.sd.getCurrentDir());
        List<EditLogFile> editLogs = FileJournalManager.matchEditLogs(files, true);
        FileJournalManager fileJournalManager = this;
        synchronized (fileJournalManager) {
            for (EditLogFile log : editLogs) {
                if (log.getFirstTxId() < minTxIdToKeep && log.getLastTxId() < minTxIdToKeep) {
                    this.purger.purgeLog(log);
                    continue;
                }
                if (!this.isStaleInProgressLog(minTxIdToKeep, log)) continue;
                this.purger.markStale(log);
            }
        }
    }

    private boolean isStaleInProgressLog(long minTxIdToKeep, EditLogFile log) {
        return log.isInProgress() && !log.getFile().equals(this.currentInProgress) && log.getFirstTxId() >= minTxIdToKeep && EDITS_INPROGRESS_REGEX.matcher(log.getFile().getName()).matches();
    }

    public List<RemoteEditLog> getRemoteEditLogs(long firstTxId, boolean inProgressOk) throws IOException {
        File currentDir = this.sd.getCurrentDir();
        List<EditLogFile> allLogFiles = FileJournalManager.matchEditLogs(currentDir);
        ArrayList ret = Lists.newArrayListWithCapacity((int)allLogFiles.size());
        for (EditLogFile elf : allLogFiles) {
            if (elf.hasCorruptHeader() || !inProgressOk && elf.isInProgress()) continue;
            if (elf.isInProgress()) {
                try {
                    elf.scanLog(this.getLastReadableTxId(), true);
                }
                catch (IOException e) {
                    LOG.error("got IOException while trying to validate header of " + elf + ".  Skipping.", (Throwable)e);
                    continue;
                }
            }
            if (elf.getFirstTxId() >= firstTxId) {
                ret.add(new RemoteEditLog(elf.firstTxId, elf.lastTxId, elf.isInProgress()));
                continue;
            }
            if (elf.getFirstTxId() >= firstTxId || firstTxId > elf.getLastTxId()) continue;
            ret.add(new RemoteEditLog(elf.firstTxId, elf.lastTxId, elf.isInProgress()));
        }
        Collections.sort(ret);
        return ret;
    }

    private void discardEditLogSegments(long startTxId) throws IOException {
        File currentDir = this.sd.getCurrentDir();
        List<EditLogFile> allLogFiles = FileJournalManager.matchEditLogs(currentDir);
        ArrayList toTrash = Lists.newArrayList();
        LOG.info("Discard the EditLog files, the given start txid is " + startTxId);
        for (EditLogFile elf : allLogFiles) {
            if (elf.getFirstTxId() >= startTxId) {
                toTrash.add(elf);
                continue;
            }
            Preconditions.checkState((elf.getLastTxId() < startTxId ? 1 : 0) != 0);
        }
        for (EditLogFile elf : toTrash) {
            elf.moveAsideTrashFile(startTxId);
            LOG.info("Trash the EditLog file " + elf);
        }
    }

    public static List<EditLogFile> matchEditLogs(File logDir) throws IOException {
        return FileJournalManager.matchEditLogs(FileUtil.listFiles((File)logDir));
    }

    static List<EditLogFile> matchEditLogs(File[] filesInStorage) {
        return FileJournalManager.matchEditLogs(filesInStorage, false);
    }

    private static List<EditLogFile> matchEditLogs(File[] filesInStorage, boolean forPurging) {
        ArrayList ret = Lists.newArrayList();
        for (File f : filesInStorage) {
            Matcher staleInprogressEditsMatch;
            Matcher inProgressEditsMatch;
            String name = f.getName();
            Matcher editsMatch = EDITS_REGEX.matcher(name);
            if (editsMatch.matches()) {
                try {
                    long startTxId = Long.parseLong(editsMatch.group(1));
                    long endTxId = Long.parseLong(editsMatch.group(2));
                    ret.add(new EditLogFile(f, startTxId, endTxId));
                    continue;
                }
                catch (NumberFormatException nfe) {
                    LOG.error("Edits file " + f + " has improperly formatted transaction ID");
                }
            }
            if ((inProgressEditsMatch = EDITS_INPROGRESS_REGEX.matcher(name)).matches()) {
                try {
                    long startTxId = Long.parseLong(inProgressEditsMatch.group(1));
                    ret.add(new EditLogFile(f, startTxId, -12345L, true));
                    continue;
                }
                catch (NumberFormatException nfe) {
                    LOG.error("In-progress edits file " + f + " has improperly formatted transaction ID");
                }
            }
            if (!forPurging || !(staleInprogressEditsMatch = EDITS_INPROGRESS_STALE_REGEX.matcher(name)).matches()) continue;
            try {
                long startTxId = Long.parseLong(staleInprogressEditsMatch.group(1));
                ret.add(new EditLogFile(f, startTxId, -12345L, true));
            }
            catch (NumberFormatException nfe) {
                LOG.error("In-progress stale edits file " + f + " has improperly formatted transaction ID");
            }
        }
        return ret;
    }

    public synchronized void selectInputStreams(Collection<EditLogInputStream> streams, long fromTxnId, boolean inProgressOk) throws IOException {
        this.selectInputStreams(streams, fromTxnId, inProgressOk, false);
    }

    @Override
    public synchronized void selectInputStreams(Collection<EditLogInputStream> streams, long fromTxId, boolean inProgressOk, boolean onlyDurableTxns) throws IOException {
        List<EditLogFile> elfs = FileJournalManager.matchEditLogs(this.sd.getCurrentDir());
        if (LOG.isDebugEnabled()) {
            LOG.debug(this + ": selecting input streams starting at " + fromTxId + (inProgressOk ? " (inProgress ok) " : " (excluding inProgress) ") + "from among " + elfs.size() + " candidate file(s)");
        }
        FileJournalManager.addStreamsToCollectionFromFiles(elfs, streams, fromTxId, this.getLastReadableTxId(), inProgressOk);
    }

    static void addStreamsToCollectionFromFiles(Collection<EditLogFile> elfs, Collection<EditLogInputStream> streams, long fromTxId, long maxTxIdToScan, boolean inProgressOk) {
        for (EditLogFile elf : elfs) {
            if (elf.isInProgress()) {
                if (!inProgressOk) {
                    if (!LOG.isDebugEnabled()) continue;
                    LOG.debug("passing over " + elf + " because it is in progress and we are ignoring in-progress logs.");
                    continue;
                }
                try {
                    elf.scanLog(maxTxIdToScan, true);
                }
                catch (IOException e) {
                    LOG.error("got IOException while trying to validate header of " + elf + ".  Skipping.", (Throwable)e);
                    continue;
                }
            }
            if (elf.lastTxId < fromTxId) {
                assert (elf.lastTxId != -12345L);
                if (!LOG.isDebugEnabled()) continue;
                LOG.debug("passing over " + elf + " because it ends at " + elf.lastTxId + ", but we only care about transactions as new as " + fromTxId);
                continue;
            }
            EditLogFileInputStream elfis = new EditLogFileInputStream(elf.getFile(), elf.getFirstTxId(), elf.getLastTxId(), elf.isInProgress());
            LOG.debug("selecting edit log stream " + elf);
            streams.add(elfis);
        }
    }

    @Override
    public synchronized void recoverUnfinalizedSegments() throws IOException {
        File currentDir = this.sd.getCurrentDir();
        LOG.info("Recovering unfinalized segments in " + currentDir);
        List<EditLogFile> allLogFiles = FileJournalManager.matchEditLogs(currentDir);
        for (EditLogFile elf : allLogFiles) {
            if (elf.getFile().equals(this.currentInProgress) || !elf.isInProgress()) continue;
            if (elf.getFile().length() == 0L) {
                LOG.info("Deleting zero-length edit log file " + elf);
                if (elf.getFile().delete()) continue;
                throw new IOException("Unable to delete file " + elf.getFile());
            }
            elf.scanLog(this.getLastReadableTxId(), true);
            if (elf.hasCorruptHeader()) {
                elf.moveAsideCorruptFile();
                throw new JournalManager.CorruptionException("In-progress edit log file is corrupt: " + elf);
            }
            if (elf.getLastTxId() == -12345L) {
                LOG.info("Moving aside edit log file that seems to have zero transactions " + elf);
                elf.moveAsideEmptyFile();
                continue;
            }
            this.finalizeLogSegment(elf.getFirstTxId(), elf.getLastTxId());
        }
    }

    public List<EditLogFile> getLogFiles(long fromTxId) throws IOException {
        File currentDir = this.sd.getCurrentDir();
        List<EditLogFile> allLogFiles = FileJournalManager.matchEditLogs(currentDir);
        ArrayList logFiles = Lists.newArrayList();
        for (EditLogFile elf : allLogFiles) {
            if (fromTxId > elf.getFirstTxId() && !elf.containsTxId(fromTxId)) continue;
            logFiles.add(elf);
        }
        Collections.sort(logFiles, EditLogFile.COMPARE_BY_START_TXID);
        return logFiles;
    }

    public EditLogFile getLogFile(long startTxId) throws IOException {
        return FileJournalManager.getLogFile(this.sd.getCurrentDir(), startTxId, true);
    }

    public EditLogFile getLogFile(long startTxId, boolean inProgressOk) throws IOException {
        return FileJournalManager.getLogFile(this.sd.getCurrentDir(), startTxId, inProgressOk);
    }

    public static EditLogFile getLogFile(File dir, long startTxId) throws IOException {
        return FileJournalManager.getLogFile(dir, startTxId, true);
    }

    public static EditLogFile getLogFile(File dir, long startTxId, boolean inProgressOk) throws IOException {
        List<EditLogFile> files = FileJournalManager.matchEditLogs(dir);
        LinkedList ret = Lists.newLinkedList();
        for (EditLogFile elf : files) {
            if (elf.getFirstTxId() != startTxId || !inProgressOk && elf.isInProgress()) continue;
            ret.add(elf);
        }
        if (ret.isEmpty()) {
            return null;
        }
        if (ret.size() == 1) {
            return (EditLogFile)ret.get(0);
        }
        throw new IllegalStateException("More than one log segment in " + dir + " starting at txid " + startTxId + ": " + Joiner.on((String)", ").join((Iterable)ret));
    }

    @Override
    public String toString() {
        return String.format("FileJournalManager(root=%s)", this.sd.getRoot());
    }

    @Override
    public void doPreUpgrade() throws IOException {
        LOG.info("Starting upgrade of edits directory " + this.sd.getRoot());
        try {
            NNUpgradeUtil.doPreUpgrade(this.conf, this.sd);
        }
        catch (IOException ioe) {
            LOG.error("Failed to move aside pre-upgrade storage in image directory " + this.sd.getRoot(), (Throwable)ioe);
            throw ioe;
        }
    }

    @Override
    public void doUpgrade(Storage storage) throws IOException {
        NNUpgradeUtil.doUpgrade(this.sd, storage);
    }

    @Override
    public void doFinalize() throws IOException {
        NNUpgradeUtil.doFinalize(this.sd);
    }

    @Override
    public boolean canRollBack(StorageInfo storage, StorageInfo prevStorage, int targetLayoutVersion) throws IOException {
        return NNUpgradeUtil.canRollBack(this.sd, storage, prevStorage, targetLayoutVersion);
    }

    @Override
    public void doRollback() throws IOException {
        NNUpgradeUtil.doRollBack(this.sd);
    }

    @Override
    public void discardSegments(long startTxid) throws IOException {
        this.discardEditLogSegments(startTxid);
    }

    @Override
    public long getJournalCTime() throws IOException {
        StorageInfo sInfo = new StorageInfo((HdfsServerConstants.NodeType)null);
        sInfo.readProperties(this.sd);
        return sInfo.getCTime();
    }

    @InterfaceAudience.Private
    public static class EditLogFile {
        private File file;
        private final long firstTxId;
        private long lastTxId;
        private boolean hasCorruptHeader = false;
        private final boolean isInProgress;
        static final Comparator<EditLogFile> COMPARE_BY_START_TXID = new Comparator<EditLogFile>(){

            @Override
            public int compare(EditLogFile a, EditLogFile b) {
                return ComparisonChain.start().compare(a.getFirstTxId(), b.getFirstTxId()).compare(a.getLastTxId(), b.getLastTxId()).result();
            }
        };

        EditLogFile(File file, long firstTxId, long lastTxId) {
            this(file, firstTxId, lastTxId, false);
            assert (lastTxId != -12345L && lastTxId >= firstTxId);
        }

        EditLogFile(File file, long firstTxId, long lastTxId, boolean isInProgress) {
            assert (lastTxId == -12345L && isInProgress || lastTxId != -12345L && lastTxId >= firstTxId);
            assert (firstTxId > 0L || firstTxId == -12345L);
            assert (file != null);
            Preconditions.checkArgument((!isInProgress || lastTxId == -12345L ? 1 : 0) != 0);
            this.firstTxId = firstTxId;
            this.lastTxId = lastTxId;
            this.file = file;
            this.isInProgress = isInProgress;
        }

        public long getFirstTxId() {
            return this.firstTxId;
        }

        public long getLastTxId() {
            return this.lastTxId;
        }

        boolean containsTxId(long txId) {
            return this.firstTxId <= txId && txId <= this.lastTxId;
        }

        public void scanLog(long maxTxIdToScan, boolean verifyVersion) throws IOException {
            FSEditLogLoader.EditLogValidation val = EditLogFileInputStream.scanEditLog(this.file, maxTxIdToScan, verifyVersion);
            this.lastTxId = val.getEndTxId();
            this.hasCorruptHeader = val.hasCorruptHeader();
        }

        public boolean isInProgress() {
            return this.isInProgress;
        }

        public File getFile() {
            return this.file;
        }

        boolean hasCorruptHeader() {
            return this.hasCorruptHeader;
        }

        void moveAsideCorruptFile() throws IOException {
            assert (this.hasCorruptHeader);
            this.renameSelf(".corrupt");
        }

        void moveAsideTrashFile(long markerTxid) throws IOException {
            assert (this.getFirstTxId() >= markerTxid);
            this.renameSelf(".trash");
        }

        public void moveAsideEmptyFile() throws IOException {
            assert (this.lastTxId == -12345L);
            this.renameSelf(".empty");
        }

        public void moveAsideStaleInprogressFile() throws IOException {
            assert (this.isInProgress);
            this.renameSelf(".stale");
        }

        private void renameSelf(String newSuffix) throws IOException {
            File src = this.file;
            File dst = new File(src.getParent(), src.getName() + newSuffix);
            try {
                if (dst.exists() && !dst.delete()) {
                    throw new IOException("Couldn't delete " + dst);
                }
                NativeIO.renameTo((File)src, (File)dst);
            }
            catch (IOException e) {
                throw new IOException("Couldn't rename log " + src + " to " + dst, e);
            }
            this.file = dst;
        }

        public String toString() {
            return String.format("EditLogFile(file=%s,first=%019d,last=%019d,inProgress=%b,hasCorruptHeader=%b)", this.file.toString(), this.firstTxId, this.lastTxId, this.isInProgress(), this.hasCorruptHeader);
        }
    }
}

