/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode.snapshot;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;
import javax.management.ObjectName;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.DFSUtil;
import org.apache.hadoop.hdfs.DFSUtilClient;
import org.apache.hadoop.hdfs.protocol.HdfsFileStatus;
import org.apache.hadoop.hdfs.protocol.SnapshotDiffReport;
import org.apache.hadoop.hdfs.protocol.SnapshotDiffReportListing;
import org.apache.hadoop.hdfs.protocol.SnapshotException;
import org.apache.hadoop.hdfs.protocol.SnapshotInfo;
import org.apache.hadoop.hdfs.protocol.SnapshottableDirectoryStatus;
import org.apache.hadoop.hdfs.server.namenode.FSDirectory;
import org.apache.hadoop.hdfs.server.namenode.FSImageFormat;
import org.apache.hadoop.hdfs.server.namenode.INode;
import org.apache.hadoop.hdfs.server.namenode.INodeDirectory;
import org.apache.hadoop.hdfs.server.namenode.INodesInPath;
import org.apache.hadoop.hdfs.server.namenode.LeaseManager;
import org.apache.hadoop.hdfs.server.namenode.snapshot.DirectoryDiffListFactory;
import org.apache.hadoop.hdfs.server.namenode.snapshot.DirectorySnapshottableFeature;
import org.apache.hadoop.hdfs.server.namenode.snapshot.Snapshot;
import org.apache.hadoop.hdfs.server.namenode.snapshot.SnapshotDiffInfo;
import org.apache.hadoop.hdfs.server.namenode.snapshot.SnapshotDiffListingInfo;
import org.apache.hadoop.hdfs.server.namenode.snapshot.SnapshotStatsMXBean;
import org.apache.hadoop.metrics2.util.MBeans;
import org.apache.hadoop.shaded.com.google.common.annotations.VisibleForTesting;
import org.apache.hadoop.shaded.com.google.common.base.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SnapshotManager
implements SnapshotStatsMXBean {
    public static final Logger LOG = LoggerFactory.getLogger(SnapshotManager.class);
    private final FSDirectory fsdir;
    private boolean captureOpenFiles;
    private boolean skipCaptureAccessTimeOnlyChange = false;
    private final boolean snapshotDiffAllowSnapRootDescendant;
    private final AtomicInteger numSnapshots = new AtomicInteger();
    private static final int SNAPSHOT_ID_BIT_WIDTH = 28;
    private boolean allowNestedSnapshots = false;
    private int snapshotCounter = 0;
    private final int maxSnapshotLimit;
    private final Map<Long, INodeDirectory> snapshottables = new HashMap<Long, INodeDirectory>();
    private ObjectName mxBeanName;

    public SnapshotManager(Configuration conf, FSDirectory fsdir) {
        this.fsdir = fsdir;
        this.captureOpenFiles = conf.getBoolean("dfs.namenode.snapshot.capture.openfiles", false);
        this.skipCaptureAccessTimeOnlyChange = conf.getBoolean("dfs.namenode.snapshot.skip.capture.accesstime-only-change", false);
        this.snapshotDiffAllowSnapRootDescendant = conf.getBoolean("dfs.namenode.snapshotdiff.allow.snap-root-descendant", true);
        this.maxSnapshotLimit = conf.getInt("dfs.namenode.snapshot.max.limit", 65536);
        LOG.info("Loaded config captureOpenFiles: " + this.captureOpenFiles + ", skipCaptureAccessTimeOnlyChange: " + this.skipCaptureAccessTimeOnlyChange + ", snapshotDiffAllowSnapRootDescendant: " + this.snapshotDiffAllowSnapRootDescendant + ", maxSnapshotLimit: " + this.maxSnapshotLimit);
        int maxLevels = conf.getInt("dfs.namenode.snapshot.skiplist.max.levels", 0);
        int skipInterval = conf.getInt("dfs.namenode.snapshot.skiplist.interval", 10);
        DirectoryDiffListFactory.init(skipInterval, maxLevels, LOG);
    }

    @VisibleForTesting
    void setCaptureOpenFiles(boolean captureOpenFiles) {
        this.captureOpenFiles = captureOpenFiles;
    }

    public boolean getSkipCaptureAccessTimeOnlyChange() {
        return this.skipCaptureAccessTimeOnlyChange;
    }

    void setAllowNestedSnapshots(boolean allowNestedSnapshots) {
        this.allowNestedSnapshots = allowNestedSnapshots;
    }

    public boolean isAllowNestedSnapshots() {
        return this.allowNestedSnapshots;
    }

    private void checkNestedSnapshottable(INodeDirectory dir, String path) throws SnapshotException {
        if (this.allowNestedSnapshots) {
            return;
        }
        for (INodeDirectory s : this.snapshottables.values()) {
            if (s.isAncestorDirectory(dir)) {
                throw new SnapshotException("Nested snapshottable directories not allowed: path=" + path + ", the subdirectory " + s.getFullPathName() + " is already a snapshottable directory.");
            }
            if (!dir.isAncestorDirectory(s)) continue;
            throw new SnapshotException("Nested snapshottable directories not allowed: path=" + path + ", the ancestor " + s.getFullPathName() + " is already a snapshottable directory.");
        }
    }

    public void setSnapshottable(String path, boolean checkNestedSnapshottable) throws IOException {
        INodesInPath iip = this.fsdir.getINodesInPath(path, FSDirectory.DirOp.WRITE);
        INodeDirectory d = INodeDirectory.valueOf(iip.getLastINode(), path);
        if (checkNestedSnapshottable) {
            this.checkNestedSnapshottable(d, path);
        }
        if (d.isSnapshottable()) {
            d.setSnapshotQuota(65536);
        } else {
            d.addSnapshottableFeature();
        }
        this.addSnapshottable(d);
    }

    public void addSnapshottable(INodeDirectory dir) {
        Preconditions.checkArgument((boolean)dir.isSnapshottable());
        this.snapshottables.put(dir.getId(), dir);
    }

    private void removeSnapshottable(INodeDirectory s) {
        this.snapshottables.remove(s.getId());
    }

    public void removeSnapshottable(List<INodeDirectory> toRemove) {
        if (toRemove != null) {
            for (INodeDirectory s : toRemove) {
                this.removeSnapshottable(s);
            }
        }
    }

    public void resetSnapshottable(String path) throws IOException {
        INodesInPath iip = this.fsdir.getINodesInPath(path, FSDirectory.DirOp.WRITE);
        INodeDirectory d = INodeDirectory.valueOf(iip.getLastINode(), path);
        DirectorySnapshottableFeature sf = d.getDirectorySnapshottableFeature();
        if (sf == null) {
            return;
        }
        if (sf.getNumSnapshots() > 0) {
            throw new SnapshotException("The directory " + path + " has snapshot(s). Please redo the operation after removing all the snapshots.");
        }
        if (d == this.fsdir.getRoot()) {
            d.setSnapshotQuota(0);
        } else {
            d.removeSnapshottableFeature();
        }
        this.removeSnapshottable(d);
    }

    public INodeDirectory getSnapshottableRoot(INodesInPath iip) throws IOException {
        String path = iip.getPath();
        INodeDirectory dir = INodeDirectory.valueOf(iip.getLastINode(), path);
        if (!dir.isSnapshottable()) {
            throw new SnapshotException("Directory is not a snapshottable directory: " + path);
        }
        return dir;
    }

    public INodeDirectory getSnapshottableAncestorDir(INodesInPath iip) throws IOException {
        String path = iip.getPath();
        INodeDirectory dir = INodeDirectory.valueOf(iip.getLastINode(), path);
        if (dir.isSnapshottable()) {
            return dir;
        }
        for (INodeDirectory snapRoot : this.snapshottables.values()) {
            if (!dir.isAncestorDirectory(snapRoot)) continue;
            return snapRoot;
        }
        throw new SnapshotException("Directory is neither snapshottable nor under a snap root!");
    }

    public boolean isDescendantOfSnapshotRoot(INodeDirectory dir) {
        if (dir.isSnapshottable()) {
            return true;
        }
        for (INodeDirectory p = dir; p != null; p = p.getParent()) {
            if (!this.snapshottables.containsValue(p)) continue;
            return true;
        }
        return false;
    }

    public String createSnapshot(LeaseManager leaseManager, INodesInPath iip, String snapshotRoot, String snapshotName, long mtime) throws IOException {
        INodeDirectory srcRoot = this.getSnapshottableRoot(iip);
        if (this.snapshotCounter == this.getMaxSnapshotID()) {
            throw new SnapshotException("Failed to create the snapshot. The FileSystem has run out of snapshot IDs and ID rollover is not supported.");
        }
        srcRoot.addSnapshot(this.snapshotCounter, snapshotName, leaseManager, this.captureOpenFiles, this.maxSnapshotLimit, mtime);
        ++this.snapshotCounter;
        this.numSnapshots.getAndIncrement();
        return Snapshot.getSnapshotPath(snapshotRoot, snapshotName);
    }

    public void deleteSnapshot(INodesInPath iip, String snapshotName, INode.ReclaimContext reclaimContext, long now) throws IOException {
        INodeDirectory srcRoot = this.getSnapshottableRoot(iip);
        srcRoot.removeSnapshot(reclaimContext, snapshotName, now);
        this.numSnapshots.getAndDecrement();
    }

    public void renameSnapshot(INodesInPath iip, String snapshotRoot, String oldSnapshotName, String newSnapshotName, long now) throws IOException {
        INodeDirectory srcRoot = this.getSnapshottableRoot(iip);
        srcRoot.renameSnapshot(snapshotRoot, oldSnapshotName, newSnapshotName, now);
    }

    public int getNumSnapshottableDirs() {
        return this.snapshottables.size();
    }

    public int getNumSnapshots() {
        return this.numSnapshots.get();
    }

    void setNumSnapshots(int num) {
        this.numSnapshots.set(num);
    }

    int getSnapshotCounter() {
        return this.snapshotCounter;
    }

    void setSnapshotCounter(int counter) {
        this.snapshotCounter = counter;
    }

    INodeDirectory[] getSnapshottableDirs() {
        return this.snapshottables.values().toArray(new INodeDirectory[this.snapshottables.size()]);
    }

    public void write(DataOutput out) throws IOException {
        out.writeInt(this.snapshotCounter);
        out.writeInt(this.numSnapshots.get());
        for (INodeDirectory snapshottableDir : this.snapshottables.values()) {
            for (Snapshot s : snapshottableDir.getDirectorySnapshottableFeature().getSnapshotList()) {
                s.write(out);
            }
        }
    }

    public Map<Integer, Snapshot> read(DataInput in, FSImageFormat.Loader loader) throws IOException {
        this.snapshotCounter = in.readInt();
        this.numSnapshots.set(in.readInt());
        HashMap<Integer, Snapshot> snapshotMap = new HashMap<Integer, Snapshot>();
        for (int i = 0; i < this.numSnapshots.get(); ++i) {
            Snapshot s = Snapshot.read(in, loader);
            snapshotMap.put(s.getId(), s);
        }
        return snapshotMap;
    }

    public SnapshottableDirectoryStatus[] getSnapshottableDirListing(String userName) {
        if (this.snapshottables.isEmpty()) {
            return null;
        }
        ArrayList<SnapshottableDirectoryStatus> statusList = new ArrayList<SnapshottableDirectoryStatus>();
        for (INodeDirectory dir : this.snapshottables.values()) {
            if (userName != null && !userName.equals(dir.getUserName())) continue;
            SnapshottableDirectoryStatus status = new SnapshottableDirectoryStatus(dir.getModificationTime(), dir.getAccessTime(), dir.getFsPermission(), EnumSet.noneOf(HdfsFileStatus.Flags.class), dir.getUserName(), dir.getGroupName(), dir.getLocalNameBytes(), dir.getId(), dir.getChildrenNum(0x7FFFFFFE), dir.getDirectorySnapshottableFeature().getNumSnapshots(), dir.getDirectorySnapshottableFeature().getSnapshotQuota(), dir.getParent() == null ? DFSUtilClient.EMPTY_BYTES : DFSUtil.string2Bytes(dir.getParent().getFullPathName()));
            statusList.add(status);
        }
        Collections.sort(statusList, SnapshottableDirectoryStatus.COMPARATOR);
        return statusList.toArray(new SnapshottableDirectoryStatus[statusList.size()]);
    }

    public SnapshotDiffReport diff(INodesInPath iip, String snapshotPath, String from, String to) throws IOException {
        INodeDirectory snapshotRootDir = this.snapshotDiffAllowSnapRootDescendant ? this.getSnapshottableAncestorDir(iip) : this.getSnapshottableRoot(iip);
        Preconditions.checkNotNull((Object)snapshotRootDir);
        INodeDirectory snapshotDescendantDir = INodeDirectory.valueOf(iip.getLastINode(), snapshotPath);
        if ((from == null || from.isEmpty()) && (to == null || to.isEmpty())) {
            return new SnapshotDiffReport(snapshotPath, from, to, Collections.emptyList());
        }
        SnapshotDiffInfo diffs = snapshotRootDir.getDirectorySnapshottableFeature().computeDiff(snapshotRootDir, snapshotDescendantDir, from, to);
        return diffs != null ? diffs.generateReport() : new SnapshotDiffReport(snapshotPath, from, to, Collections.emptyList());
    }

    public SnapshotDiffReportListing diff(INodesInPath iip, String snapshotPath, String from, String to, byte[] startPath, int index, int snapshotDiffReportLimit) throws IOException {
        INodeDirectory snapshotRootDir = this.snapshotDiffAllowSnapRootDescendant ? this.getSnapshottableAncestorDir(iip) : this.getSnapshottableRoot(iip);
        Preconditions.checkNotNull((Object)snapshotRootDir);
        INodeDirectory snapshotDescendantDir = INodeDirectory.valueOf(iip.getLastINode(), snapshotPath);
        SnapshotDiffListingInfo diffs = snapshotRootDir.getDirectorySnapshottableFeature().computeDiff(snapshotRootDir, snapshotDescendantDir, from, to, startPath, index, snapshotDiffReportLimit);
        return diffs != null ? diffs.generateReport() : new SnapshotDiffReportListing();
    }

    public void clearSnapshottableDirs() {
        this.snapshottables.clear();
    }

    public int getMaxSnapshotID() {
        return 0xFFFFFFF;
    }

    public void registerMXBean() {
        this.mxBeanName = MBeans.register((String)"NameNode", (String)"SnapshotInfo", (Object)this);
    }

    public void shutdown() {
        MBeans.unregister((ObjectName)this.mxBeanName);
        this.mxBeanName = null;
    }

    @Override
    public SnapshottableDirectoryStatus.Bean[] getSnapshottableDirectories() {
        ArrayList<SnapshottableDirectoryStatus.Bean> beans = new ArrayList<SnapshottableDirectoryStatus.Bean>();
        for (INodeDirectory d : this.getSnapshottableDirs()) {
            beans.add(SnapshotManager.toBean(d));
        }
        return beans.toArray(new SnapshottableDirectoryStatus.Bean[beans.size()]);
    }

    @Override
    public SnapshotInfo.Bean[] getSnapshots() {
        ArrayList<SnapshotInfo.Bean> beans = new ArrayList<SnapshotInfo.Bean>();
        for (INodeDirectory d : this.getSnapshottableDirs()) {
            for (Snapshot s : d.getDirectorySnapshottableFeature().getSnapshotList()) {
                beans.add(SnapshotManager.toBean(s));
            }
        }
        return beans.toArray(new SnapshotInfo.Bean[beans.size()]);
    }

    public static SnapshottableDirectoryStatus.Bean toBean(INodeDirectory d) {
        return new SnapshottableDirectoryStatus.Bean(d.getFullPathName(), d.getDirectorySnapshottableFeature().getNumSnapshots(), d.getDirectorySnapshottableFeature().getSnapshotQuota(), d.getModificationTime(), Short.valueOf(Integer.toOctalString(d.getFsPermissionShort())).shortValue(), d.getUserName(), d.getGroupName());
    }

    public static SnapshotInfo.Bean toBean(Snapshot s) {
        return new SnapshotInfo.Bean(s.getRoot().getLocalName(), s.getRoot().getFullPathName(), s.getRoot().getModificationTime());
    }
}

