/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.BlockLocation;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FSInputStream;
import org.apache.hadoop.fs.FileChecksum;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.fs.FilterFileSystem;
import org.apache.hadoop.fs.LocatedFileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.util.LineReader;
import org.apache.hadoop.util.Progressable;

public class HarFileSystem
extends FilterFileSystem {
    private static final Log LOG = LogFactory.getLog(HarFileSystem.class);
    public static final String METADATA_CACHE_ENTRIES_KEY = "fs.har.metadatacache.entries";
    public static final int METADATA_CACHE_ENTRIES_DEFAULT = 10;
    public static final int VERSION = 3;
    private static Map<URI, HarMetaData> harMetaCache;
    private URI uri;
    private Path archivePath;
    private String harAuth;
    private HarMetaData metadata;

    public HarFileSystem() {
    }

    public HarFileSystem(FileSystem fs) {
        super(fs);
    }

    private synchronized void initializeMetadataCache(Configuration conf) {
        if (harMetaCache == null) {
            int cacheSize = conf.getInt(METADATA_CACHE_ENTRIES_KEY, 10);
            harMetaCache = Collections.synchronizedMap(new LruCache(cacheSize));
        }
    }

    @Override
    public void initialize(URI name, Configuration conf) throws IOException {
        this.initializeMetadataCache(conf);
        URI underLyingURI = this.decodeHarURI(name, conf);
        Path harPath = this.archivePath(new Path(name.getScheme(), name.getAuthority(), name.getPath()));
        if (harPath == null) {
            throw new IOException("Invalid path for the Har Filesystem. " + name.toString());
        }
        if (this.fs == null) {
            this.fs = FileSystem.get(underLyingURI, conf);
        }
        this.uri = harPath.toUri();
        this.archivePath = new Path(this.uri.getPath());
        this.harAuth = this.getHarAuth(underLyingURI);
        Path masterIndexPath = new Path(this.archivePath, "_masterindex");
        Path archiveIndexPath = new Path(this.archivePath, "_index");
        if (!this.fs.exists(masterIndexPath) || !this.fs.exists(archiveIndexPath)) {
            throw new IOException("Invalid path for the Har Filesystem. No index file in " + harPath);
        }
        this.metadata = harMetaCache.get(this.uri);
        if (this.metadata != null) {
            FileStatus mStat = this.fs.getFileStatus(masterIndexPath);
            FileStatus aStat = this.fs.getFileStatus(archiveIndexPath);
            if (mStat.getModificationTime() != this.metadata.getMasterIndexTimestamp() || aStat.getModificationTime() != this.metadata.getArchiveIndexTimestamp()) {
                this.metadata = null;
                harMetaCache.remove(this.uri);
            }
        }
        if (this.metadata == null) {
            this.metadata = new HarMetaData(this.fs, masterIndexPath, archiveIndexPath);
            this.metadata.parseMetaData();
            harMetaCache.put(this.uri, this.metadata);
        }
    }

    public int getHarVersion() throws IOException {
        if (this.metadata != null) {
            return this.metadata.getVersion();
        }
        throw new IOException("Invalid meta data for the Har Filesystem");
    }

    private Path archivePath(Path p) {
        Path retPath = null;
        Path tmp = p;
        for (int i = 0; i < p.depth(); ++i) {
            if (tmp.toString().endsWith(".har")) {
                retPath = tmp;
                break;
            }
            tmp = tmp.getParent();
        }
        return retPath;
    }

    private URI decodeHarURI(URI rawURI, Configuration conf) throws IOException {
        String tmpAuth = rawURI.getAuthority();
        if (tmpAuth == null) {
            return FileSystem.getDefaultUri(conf);
        }
        String host = rawURI.getHost();
        if (host == null) {
            throw new IOException("URI: " + rawURI + " is an invalid Har URI since host==null." + "  Expecting har://<scheme>-<host>/<path>.");
        }
        int i = host.indexOf(45);
        if (i < 0) {
            throw new IOException("URI: " + rawURI + " is an invalid Har URI since '-' not found." + "  Expecting har://<scheme>-<host>/<path>.");
        }
        String underLyingScheme = host.substring(0, i);
        String underLyingHost = ++i == host.length() ? null : host.substring(i);
        int underLyingPort = rawURI.getPort();
        String auth = underLyingHost == null && underLyingPort == -1 ? null : underLyingHost + ":" + underLyingPort;
        URI tmp = null;
        if (rawURI.getQuery() != null) {
            throw new IOException("query component in Path not supported  " + rawURI);
        }
        try {
            tmp = new URI(underLyingScheme, auth, rawURI.getPath(), rawURI.getQuery(), rawURI.getFragment());
        }
        catch (URISyntaxException e) {
            // empty catch block
        }
        return tmp;
    }

    private static String decodeString(String str) throws UnsupportedEncodingException {
        return URLDecoder.decode(str, "UTF-8");
    }

    private String decodeFileName(String fname) throws UnsupportedEncodingException {
        int version = this.metadata.getVersion();
        if (version == 2 || version == 3) {
            return HarFileSystem.decodeString(fname);
        }
        return fname;
    }

    @Override
    public Path getWorkingDirectory() {
        return new Path(this.uri.toString());
    }

    private String getHarAuth(URI underLyingUri) {
        String auth = underLyingUri.getScheme() + "-";
        if (underLyingUri.getHost() != null) {
            auth = auth + underLyingUri.getHost() + ":";
            if (underLyingUri.getPort() != -1) {
                auth = auth + underLyingUri.getPort();
            }
        } else {
            auth = auth + ":";
        }
        return auth;
    }

    @Override
    public URI getUri() {
        return this.uri;
    }

    private Path getPathInHar(Path path) {
        Path harPath = new Path(path.toUri().getPath());
        if (this.archivePath.compareTo(harPath) == 0) {
            return new Path("/");
        }
        Path tmp = new Path(harPath.getName());
        Path parent = harPath.getParent();
        while (parent.compareTo(this.archivePath) != 0) {
            if (parent.toString().equals("/")) {
                tmp = null;
                break;
            }
            tmp = new Path(parent.getName(), tmp);
            parent = parent.getParent();
        }
        if (tmp != null) {
            tmp = new Path("/", tmp);
        }
        return tmp;
    }

    private Path makeRelative(String initial, Path p) {
        String scheme = this.uri.getScheme();
        String authority = this.uri.getAuthority();
        Path root = new Path("/");
        if (root.compareTo(p) == 0) {
            return new Path(scheme, authority, initial);
        }
        Path retPath = new Path(p.getName());
        Path parent = p.getParent();
        for (int i = 0; i < p.depth() - 1; ++i) {
            retPath = new Path(parent.getName(), retPath);
            parent = parent.getParent();
        }
        return new Path(new Path(scheme, authority, initial), retPath.toString());
    }

    @Override
    public Path makeQualified(Path path) {
        Path fsPath = path;
        if (!path.isAbsolute()) {
            fsPath = new Path(this.archivePath, path);
        }
        URI tmpURI = fsPath.toUri();
        return new Path(this.uri.getScheme(), this.harAuth, tmpURI.getPath());
    }

    static BlockLocation[] fixBlockLocations(BlockLocation[] locations, long start, long len, long fileOffsetInHar) {
        long end = start + len;
        for (BlockLocation location : locations) {
            long harBlockStart = location.getOffset() - fileOffsetInHar;
            long harBlockEnd = harBlockStart + location.getLength();
            if (start > harBlockStart) {
                location.setOffset(start);
                location.setLength(location.getLength() - (start - harBlockStart));
            } else {
                location.setOffset(harBlockStart);
            }
            if (harBlockEnd <= end) continue;
            location.setLength(location.getLength() - (harBlockEnd - end));
        }
        return locations;
    }

    @Override
    public BlockLocation[] getFileBlockLocations(FileStatus file, long start, long len) throws IOException {
        HarStatus hstatus = this.getFileHarStatus(file.getPath());
        Path partPath = new Path(this.archivePath, hstatus.getPartName());
        FileStatus partStatus = this.metadata.getPartFileStatus(partPath);
        BlockLocation[] locations = this.fs.getFileBlockLocations(partStatus, hstatus.getStartIndex() + start, len);
        return HarFileSystem.fixBlockLocations(locations, start, len, hstatus.getStartIndex());
    }

    public static int getHarHash(Path p) {
        return p.toString().hashCode() & Integer.MAX_VALUE;
    }

    private void fileStatusesInIndex(HarStatus parent, List<FileStatus> statuses, List<String> children) throws IOException {
        String parentString = parent.getName();
        if (!parentString.endsWith("/")) {
            parentString = parentString + "/";
        }
        Path harPath = new Path(parentString);
        int harlen = harPath.depth();
        TreeMap<String, FileStatus> cache = new TreeMap<String, FileStatus>();
        for (HarStatus hstatus : this.metadata.archive.values()) {
            Path thisPath;
            String child = hstatus.getName();
            if (!child.startsWith(parentString) || (thisPath = new Path(child)).depth() != harlen + 1) continue;
            statuses.add(this.toFileStatus(hstatus, cache));
        }
    }

    private FileStatus toFileStatus(HarStatus h, Map<String, FileStatus> cache) throws IOException {
        FileStatus underlying = null;
        if (cache != null) {
            underlying = cache.get(h.partName);
        }
        if (underlying == null) {
            Path p = h.isDir ? this.archivePath : new Path(this.archivePath, h.partName);
            underlying = this.fs.getFileStatus(p);
            if (cache != null) {
                cache.put(h.partName, underlying);
            }
        }
        long modTime = 0L;
        int version = this.metadata.getVersion();
        if (version < 3) {
            modTime = underlying.getModificationTime();
        } else if (version == 3) {
            modTime = h.getModificationTime();
        }
        return new FileStatus(h.isDir() ? 0L : h.getLength(), h.isDir(), underlying.getReplication(), underlying.getBlockSize(), modTime, underlying.getAccessTime(), underlying.getPermission(), underlying.getOwner(), underlying.getGroup(), this.makeRelative(this.uri.getPath(), new Path(h.name)));
    }

    @Override
    public FileStatus getFileStatus(Path f) throws IOException {
        HarStatus hstatus = this.getFileHarStatus(f);
        return this.toFileStatus(hstatus, null);
    }

    private HarStatus getFileHarStatus(Path f) throws IOException {
        Path p = this.makeQualified(f);
        Path harPath = this.getPathInHar(p);
        if (harPath == null) {
            throw new IOException("Invalid file name: " + f + " in " + this.uri);
        }
        HarStatus hstatus = this.metadata.archive.get(harPath);
        if (hstatus == null) {
            throw new FileNotFoundException("File: " + f + " does not exist in " + this.uri);
        }
        return hstatus;
    }

    @Override
    public FileChecksum getFileChecksum(Path f) {
        return null;
    }

    @Override
    public FSDataInputStream open(Path f, int bufferSize) throws IOException {
        HarStatus hstatus = this.getFileHarStatus(f);
        if (hstatus.isDir()) {
            throw new FileNotFoundException(f + " : not a file in " + this.archivePath);
        }
        return new HarFSDataInputStream(this.fs, new Path(this.archivePath, hstatus.getPartName()), hstatus.getStartIndex(), hstatus.getLength(), bufferSize);
    }

    @Override
    public FSDataOutputStream create(Path f, FsPermission permission, boolean overwrite, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        throw new IOException("Har: create not allowed.");
    }

    @Override
    public void close() throws IOException {
        if (this.fs != null) {
            try {
                this.fs.close();
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
    }

    @Override
    public boolean setReplication(Path src, short replication) throws IOException {
        throw new IOException("Har: setreplication not allowed");
    }

    @Override
    public boolean delete(Path f, boolean recursive) throws IOException {
        throw new IOException("Har: delete not allowed");
    }

    @Override
    public RemoteIterator<LocatedFileStatus> listLocatedStatus(Path f) throws IOException {
        return this.listLocatedStatus(f, DEFAULT_FILTER);
    }

    @Override
    public FileStatus[] listStatus(Path f) throws IOException {
        ArrayList<FileStatus> statuses = new ArrayList<FileStatus>();
        Path tmpPath = this.makeQualified(f);
        Path harPath = this.getPathInHar(tmpPath);
        HarStatus hstatus = this.metadata.archive.get(harPath);
        if (hstatus == null) {
            throw new FileNotFoundException("File " + f + " not found in " + this.archivePath);
        }
        if (hstatus.isDir()) {
            this.fileStatusesInIndex(hstatus, statuses, hstatus.children);
        } else {
            statuses.add(this.toFileStatus(hstatus, null));
        }
        return statuses.toArray(new FileStatus[statuses.size()]);
    }

    @Override
    public Path getHomeDirectory() {
        return new Path(this.uri.toString());
    }

    @Override
    public void setWorkingDirectory(Path newDir) {
    }

    @Override
    public boolean mkdirs(Path f, FsPermission permission) throws IOException {
        throw new IOException("Har: mkdirs not allowed");
    }

    @Override
    public void copyFromLocalFile(boolean delSrc, Path src, Path dst) throws IOException {
        throw new IOException("Har: copyfromlocalfile not allowed");
    }

    @Override
    public void copyToLocalFile(boolean delSrc, Path src, Path dst) throws IOException {
        FileUtil.copy(this, src, HarFileSystem.getLocal(this.getConf()), dst, false, this.getConf());
    }

    @Override
    public Path startLocalOutput(Path fsOutputFile, Path tmpLocalFile) throws IOException {
        throw new IOException("Har: startLocalOutput not allowed");
    }

    @Override
    public void completeLocalOutput(Path fsOutputFile, Path tmpLocalFile) throws IOException {
        throw new IOException("Har: completeLocalOutput not allowed");
    }

    @Override
    public void setOwner(Path p, String username, String groupname) throws IOException {
        throw new IOException("Har: setowner not allowed");
    }

    @Override
    public void setPermission(Path p, FsPermission permisssion) throws IOException {
        throw new IOException("Har: setPermission not allowed");
    }

    HarMetaData getMetadata() {
        return this.metadata;
    }

    private static class LruCache<K, V>
    extends LinkedHashMap<K, V> {
        private final int MAX_ENTRIES;

        public LruCache(int maxEntries) {
            super(maxEntries + 1, 1.0f, true);
            this.MAX_ENTRIES = maxEntries;
        }

        @Override
        protected boolean removeEldestEntry(Map.Entry<K, V> eldest) {
            return this.size() > this.MAX_ENTRIES;
        }
    }

    private class HarMetaData {
        private FileSystem fs;
        private int version;
        private Path masterIndexPath;
        private Path archiveIndexPath;
        private long masterIndexTimestamp;
        private long archiveIndexTimestamp;
        List<Store> stores = new ArrayList<Store>();
        Map<Path, HarStatus> archive = new HashMap<Path, HarStatus>();
        private Map<Path, FileStatus> partFileStatuses = new HashMap<Path, FileStatus>();

        public HarMetaData(FileSystem fs, Path masterIndexPath, Path archiveIndexPath) {
            this.fs = fs;
            this.masterIndexPath = masterIndexPath;
            this.archiveIndexPath = archiveIndexPath;
        }

        public FileStatus getPartFileStatus(Path partPath) throws IOException {
            FileStatus status = this.partFileStatuses.get(partPath);
            if (status == null) {
                status = this.fs.getFileStatus(partPath);
                this.partFileStatuses.put(partPath, status);
            }
            return status;
        }

        public long getMasterIndexTimestamp() {
            return this.masterIndexTimestamp;
        }

        public long getArchiveIndexTimestamp() {
            return this.archiveIndexTimestamp;
        }

        private int getVersion() {
            return this.version;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void parseMetaData() throws IOException {
            long read;
            Text line;
            FSDataInputStream in = null;
            LineReader lin = null;
            try {
                int b;
                in = this.fs.open(this.masterIndexPath);
                FileStatus masterStat = this.fs.getFileStatus(this.masterIndexPath);
                this.masterIndexTimestamp = masterStat.getModificationTime();
                lin = new LineReader((InputStream)in, HarFileSystem.this.getConf());
                line = new Text();
                String versionLine = line.toString();
                String[] arr = versionLine.split(" ");
                this.version = Integer.parseInt(arr[0]);
                if (this.version > 3) {
                    throw new IOException("Invalid version " + this.version + " expected " + 3);
                }
                String[] readStr = null;
                for (read = (long)lin.readLine(line); read < masterStat.getLen(); read += (long)b) {
                    b = lin.readLine(line);
                    readStr = line.toString().split(" ");
                    int startHash = Integer.parseInt(readStr[0]);
                    int endHash = Integer.parseInt(readStr[1]);
                    this.stores.add(new Store(Long.parseLong(readStr[2]), Long.parseLong(readStr[3]), startHash, endHash));
                    line.clear();
                }
            }
            catch (Throwable throwable) {
                IOUtils.cleanup(LOG, lin, in);
                throw throwable;
            }
            IOUtils.cleanup(LOG, lin, in);
            FSDataInputStream aIn = this.fs.open(this.archiveIndexPath);
            try {
                FileStatus archiveStat = this.fs.getFileStatus(this.archiveIndexPath);
                this.archiveIndexTimestamp = archiveStat.getModificationTime();
                for (Store s : this.stores) {
                    read = 0L;
                    aIn.seek(s.begin);
                    LineReader aLin = new LineReader((InputStream)aIn, HarFileSystem.this.getConf());
                    while (read + s.begin < s.end) {
                        int tmp = aLin.readLine(line);
                        read += (long)tmp;
                        String lineFeed = line.toString();
                        String[] parsed = lineFeed.split(" ");
                        parsed[0] = HarFileSystem.this.decodeFileName(parsed[0]);
                        this.archive.put(new Path(parsed[0]), new HarStatus(lineFeed));
                        line.clear();
                    }
                }
            }
            catch (Throwable throwable) {
                IOUtils.cleanup(LOG, aIn);
                throw throwable;
            }
            IOUtils.cleanup(LOG, aIn);
        }
    }

    private static class HarFSDataInputStream
    extends FSDataInputStream {
        public HarFSDataInputStream(FileSystem fs, Path p, long start, long length, int bufsize) throws IOException {
            super(new HarFsInputStream(fs, p, start, length, bufsize));
        }

        public HarFSDataInputStream(FileSystem fs, Path p, long start, long length) throws IOException {
            super(new HarFsInputStream(fs, p, start, length, 0));
        }

        private static class HarFsInputStream
        extends FSInputStream {
            private long position;
            private long start;
            private long end;
            private FSDataInputStream underLyingStream;
            private byte[] oneBytebuff = new byte[1];

            HarFsInputStream(FileSystem fs, Path path, long start, long length, int bufferSize) throws IOException {
                this.underLyingStream = fs.open(path, bufferSize);
                this.underLyingStream.seek(start);
                this.start = start;
                this.position = start;
                this.end = start + length;
            }

            @Override
            public synchronized int available() throws IOException {
                long remaining = this.end - this.underLyingStream.getPos();
                if (remaining > Integer.MAX_VALUE) {
                    return Integer.MAX_VALUE;
                }
                return (int)remaining;
            }

            @Override
            public synchronized void close() throws IOException {
                this.underLyingStream.close();
                super.close();
            }

            @Override
            public void mark(int readLimit) {
            }

            @Override
            public void reset() throws IOException {
                throw new IOException("reset not implemented.");
            }

            @Override
            public synchronized int read() throws IOException {
                int ret = this.read(this.oneBytebuff, 0, 1);
                return ret <= 0 ? -1 : this.oneBytebuff[0] & 0xFF;
            }

            @Override
            public synchronized int read(byte[] b) throws IOException {
                int ret = this.read(b, 0, b.length);
                if (ret != -1) {
                    this.position += (long)ret;
                }
                return ret;
            }

            @Override
            public synchronized int read(byte[] b, int offset, int len) throws IOException {
                int newlen = len;
                int ret = -1;
                if (this.position + (long)len > this.end) {
                    newlen = (int)(this.end - this.position);
                }
                if (newlen == 0) {
                    return ret;
                }
                ret = this.underLyingStream.read(b, offset, newlen);
                this.position += (long)ret;
                return ret;
            }

            @Override
            public synchronized long skip(long n) throws IOException {
                long tmpN = n;
                if (tmpN > 0L) {
                    if (this.position + tmpN > this.end) {
                        tmpN = this.end - this.position;
                    }
                    this.underLyingStream.seek(tmpN + this.position);
                    this.position += tmpN;
                    return tmpN;
                }
                return tmpN < 0L ? -1L : 0L;
            }

            @Override
            public synchronized long getPos() throws IOException {
                return this.position - this.start;
            }

            @Override
            public synchronized void seek(long pos) throws IOException {
                if (pos < 0L || this.start + pos > this.end) {
                    throw new IOException("Failed to seek: EOF");
                }
                this.position = this.start + pos;
                this.underLyingStream.seek(this.position);
            }

            @Override
            public boolean seekToNewSource(long targetPos) throws IOException {
                return false;
            }

            @Override
            public int read(long pos, byte[] b, int offset, int length) throws IOException {
                int nlength = length;
                if (this.start + (long)nlength + pos > this.end) {
                    nlength = (int)(this.end - (this.start + pos));
                }
                return this.underLyingStream.read(pos + this.start, b, offset, nlength);
            }

            @Override
            public void readFully(long pos, byte[] b, int offset, int length) throws IOException {
                if (this.start + (long)length + pos > this.end) {
                    throw new IOException("Not enough bytes to read.");
                }
                this.underLyingStream.readFully(pos + this.start, b, offset, length);
            }

            @Override
            public void readFully(long pos, byte[] b) throws IOException {
                this.readFully(pos, b, 0, b.length);
            }
        }
    }

    private class HarStatus {
        boolean isDir;
        String name;
        List<String> children;
        String partName;
        long startIndex;
        long length;
        long modificationTime = 0L;

        public HarStatus(String harString) throws UnsupportedEncodingException {
            String[] splits = harString.split(" ");
            this.name = HarFileSystem.this.decodeFileName(splits[0]);
            this.isDir = "dir".equals(splits[1]);
            this.partName = splits[2];
            this.startIndex = Long.parseLong(splits[3]);
            this.length = Long.parseLong(splits[4]);
            int version = HarFileSystem.this.metadata.getVersion();
            String[] propSplits = null;
            if (this.isDir) {
                if (version == 3) {
                    propSplits = HarFileSystem.decodeString(this.partName).split(" ");
                }
                this.children = new ArrayList<String>();
                for (int i = 5; i < splits.length; ++i) {
                    this.children.add(HarFileSystem.this.decodeFileName(splits[i]));
                }
            } else if (version == 3) {
                propSplits = HarFileSystem.decodeString(splits[5]).split(" ");
            }
            if (propSplits != null && propSplits.length >= 4) {
                this.modificationTime = Long.parseLong(propSplits[0]);
            }
        }

        public boolean isDir() {
            return this.isDir;
        }

        public String getName() {
            return this.name;
        }

        public String getPartName() {
            return this.partName;
        }

        public long getStartIndex() {
            return this.startIndex;
        }

        public long getLength() {
            return this.length;
        }

        public long getModificationTime() {
            return this.modificationTime;
        }
    }

    static class Store {
        public long begin;
        public long end;
        public int startHash;
        public int endHash;

        public Store() {
            this.endHash = 0;
            this.startHash = 0;
            this.begin = this.end = (long)0;
        }

        public Store(long begin, long end, int startHash, int endHash) {
            this.begin = begin;
            this.end = end;
            this.startHash = startHash;
            this.endHash = endHash;
        }
    }
}

