/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ipc;

import java.util.Collection;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import junit.framework.TestCase;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.ipc.FairCallQueue;
import org.apache.hadoop.ipc.IdentityProvider;
import org.apache.hadoop.ipc.RpcMultiplexer;
import org.apache.hadoop.ipc.RpcScheduler;
import org.apache.hadoop.ipc.Schedulable;
import org.apache.hadoop.ipc.UserIdentityProvider;
import org.apache.hadoop.security.UserGroupInformation;
import org.mockito.Matchers;
import org.mockito.Mockito;

public class TestFairCallQueue
extends TestCase {
    private FairCallQueue<Schedulable> fcq;
    private RpcScheduler alwaysZeroScheduler;

    public TestFairCallQueue() {
        RpcScheduler sched = (RpcScheduler)Mockito.mock(RpcScheduler.class);
        Mockito.when((Object)sched.getPriorityLevel((Schedulable)Matchers.any())).thenReturn((Object)0);
        this.alwaysZeroScheduler = sched;
    }

    private Schedulable mockCall(String id) {
        Schedulable mockCall = (Schedulable)Mockito.mock(Schedulable.class);
        UserGroupInformation ugi = (UserGroupInformation)Mockito.mock(UserGroupInformation.class);
        Mockito.when((Object)ugi.getUserName()).thenReturn((Object)id);
        Mockito.when((Object)mockCall.getUserGroupInformation()).thenReturn((Object)ugi);
        return mockCall;
    }

    public void setUp() {
        Configuration conf = new Configuration();
        conf.setInt("ns.faircallqueue.priority-levels", 2);
        this.fcq = new FairCallQueue(5, "ns", conf);
    }

    public void testPollReturnsNullWhenEmpty() {
        TestFairCallQueue.assertNull((Object)this.fcq.poll());
    }

    public void testPollReturnsTopCallWhenNotEmpty() {
        Schedulable call = this.mockCall("c");
        TestFairCallQueue.assertTrue((boolean)this.fcq.offer(call));
        TestFairCallQueue.assertEquals((Object)call, (Object)this.fcq.poll());
        TestFairCallQueue.assertEquals((int)0, (int)this.fcq.size());
    }

    public void testOfferSucceeds() {
        this.fcq.setScheduler(this.alwaysZeroScheduler);
        for (int i = 0; i < 5; ++i) {
            TestFairCallQueue.assertTrue((boolean)this.fcq.offer(this.mockCall("c")));
        }
        TestFairCallQueue.assertEquals((int)5, (int)this.fcq.size());
    }

    public void testOfferFailsWhenFull() {
        this.fcq.setScheduler(this.alwaysZeroScheduler);
        for (int i = 0; i < 5; ++i) {
            TestFairCallQueue.assertTrue((boolean)this.fcq.offer(this.mockCall("c")));
        }
        TestFairCallQueue.assertFalse((boolean)this.fcq.offer(this.mockCall("c")));
        TestFairCallQueue.assertEquals((int)5, (int)this.fcq.size());
    }

    public void testOfferSucceedsWhenScheduledLowPriority() {
        RpcScheduler sched = (RpcScheduler)Mockito.mock(RpcScheduler.class);
        Mockito.when((Object)sched.getPriorityLevel((Schedulable)Matchers.any())).thenReturn((Object)0, (Object[])new Integer[]{0, 0, 0, 0, 1, 0});
        this.fcq.setScheduler(sched);
        for (int i = 0; i < 5; ++i) {
            TestFairCallQueue.assertTrue((boolean)this.fcq.offer(this.mockCall("c")));
        }
        TestFairCallQueue.assertTrue((boolean)this.fcq.offer(this.mockCall("c")));
        TestFairCallQueue.assertEquals((int)6, (int)this.fcq.size());
    }

    public void testPeekNullWhenEmpty() {
        TestFairCallQueue.assertNull((Object)this.fcq.peek());
    }

    public void testPeekNonDestructive() {
        Schedulable call = this.mockCall("c");
        TestFairCallQueue.assertTrue((boolean)this.fcq.offer(call));
        TestFairCallQueue.assertEquals((Object)call, (Object)this.fcq.peek());
        TestFairCallQueue.assertEquals((Object)call, (Object)this.fcq.peek());
        TestFairCallQueue.assertEquals((int)1, (int)this.fcq.size());
    }

    public void testPeekPointsAtHead() {
        Schedulable call = this.mockCall("c");
        Schedulable next = this.mockCall("b");
        this.fcq.offer(call);
        this.fcq.offer(next);
        TestFairCallQueue.assertEquals((Object)call, (Object)this.fcq.peek());
    }

    public void testPollTimeout() throws InterruptedException {
        this.fcq.setScheduler(this.alwaysZeroScheduler);
        TestFairCallQueue.assertNull((Object)this.fcq.poll(10L, TimeUnit.MILLISECONDS));
    }

    public void testPollSuccess() throws InterruptedException {
        this.fcq.setScheduler(this.alwaysZeroScheduler);
        Schedulable call = this.mockCall("c");
        TestFairCallQueue.assertTrue((boolean)this.fcq.offer(call));
        TestFairCallQueue.assertEquals((Object)call, (Object)this.fcq.poll(10L, TimeUnit.MILLISECONDS));
        TestFairCallQueue.assertEquals((int)0, (int)this.fcq.size());
    }

    public void testOfferTimeout() throws InterruptedException {
        this.fcq.setScheduler(this.alwaysZeroScheduler);
        for (int i = 0; i < 5; ++i) {
            TestFairCallQueue.assertTrue((boolean)this.fcq.offer(this.mockCall("c"), 10L, TimeUnit.MILLISECONDS));
        }
        TestFairCallQueue.assertFalse((boolean)this.fcq.offer(this.mockCall("e"), 10L, TimeUnit.MILLISECONDS));
        TestFairCallQueue.assertEquals((int)5, (int)this.fcq.size());
    }

    public void testDrainTo() {
        Configuration conf = new Configuration();
        conf.setInt("ns.faircallqueue.priority-levels", 2);
        FairCallQueue fcq2 = new FairCallQueue(10, "ns", conf);
        this.fcq.setScheduler(this.alwaysZeroScheduler);
        fcq2.setScheduler(this.alwaysZeroScheduler);
        for (int i = 0; i < 3; ++i) {
            this.fcq.offer(this.mockCall("c"));
        }
        this.fcq.drainTo((Collection)fcq2);
        TestFairCallQueue.assertEquals((int)0, (int)this.fcq.size());
        TestFairCallQueue.assertEquals((int)3, (int)fcq2.size());
    }

    public void testDrainToWithLimit() {
        Configuration conf = new Configuration();
        conf.setInt("ns.faircallqueue.priority-levels", 2);
        FairCallQueue fcq2 = new FairCallQueue(10, "ns", conf);
        this.fcq.setScheduler(this.alwaysZeroScheduler);
        fcq2.setScheduler(this.alwaysZeroScheduler);
        for (int i = 0; i < 3; ++i) {
            this.fcq.offer(this.mockCall("c"));
        }
        this.fcq.drainTo((Collection)fcq2, 2);
        TestFairCallQueue.assertEquals((int)1, (int)this.fcq.size());
        TestFairCallQueue.assertEquals((int)2, (int)fcq2.size());
    }

    public void testInitialRemainingCapacity() {
        TestFairCallQueue.assertEquals((int)10, (int)this.fcq.remainingCapacity());
    }

    public void testFirstQueueFullRemainingCapacity() {
        this.fcq.setScheduler(this.alwaysZeroScheduler);
        while (this.fcq.offer(this.mockCall("c"))) {
        }
        TestFairCallQueue.assertEquals((int)5, (int)this.fcq.remainingCapacity());
    }

    public void testAllQueuesFullRemainingCapacity() {
        RpcScheduler sched = (RpcScheduler)Mockito.mock(RpcScheduler.class);
        Mockito.when((Object)sched.getPriorityLevel((Schedulable)Matchers.any())).thenReturn((Object)0, (Object[])new Integer[]{0, 0, 0, 0, 1, 1, 1, 1, 1});
        this.fcq.setScheduler(sched);
        while (this.fcq.offer(this.mockCall("c"))) {
        }
        TestFairCallQueue.assertEquals((int)0, (int)this.fcq.remainingCapacity());
        TestFairCallQueue.assertEquals((int)10, (int)this.fcq.size());
    }

    public void testQueuesPartialFilledRemainingCapacity() {
        RpcScheduler sched = (RpcScheduler)Mockito.mock(RpcScheduler.class);
        Mockito.when((Object)sched.getPriorityLevel((Schedulable)Matchers.any())).thenReturn((Object)0, (Object[])new Integer[]{1, 0, 1, 0});
        this.fcq.setScheduler(sched);
        for (int i = 0; i < 5; ++i) {
            this.fcq.offer(this.mockCall("c"));
        }
        TestFairCallQueue.assertEquals((int)5, (int)this.fcq.remainingCapacity());
        TestFairCallQueue.assertEquals((int)5, (int)this.fcq.size());
    }

    public void assertCanTake(BlockingQueue<Schedulable> cq, int numberOfTakes, int takeAttempts) throws InterruptedException {
        CountDownLatch latch = new CountDownLatch(numberOfTakes);
        Taker taker = new Taker(cq, takeAttempts, "default", latch);
        Thread t = new Thread(taker);
        t.start();
        latch.await();
        TestFairCallQueue.assertEquals((int)numberOfTakes, (int)taker.callsTaken);
        t.interrupt();
    }

    public void assertCanPut(BlockingQueue<Schedulable> cq, int numberOfPuts, int putAttempts) throws InterruptedException {
        CountDownLatch latch = new CountDownLatch(numberOfPuts);
        Putter putter = new Putter(cq, putAttempts, null, latch);
        Thread t = new Thread(putter);
        t.start();
        latch.await();
        TestFairCallQueue.assertEquals((int)numberOfPuts, (int)putter.callsAdded);
        t.interrupt();
    }

    public void testPutOverflows() throws InterruptedException {
        this.fcq.setScheduler(this.alwaysZeroScheduler);
        this.assertCanPut((BlockingQueue<Schedulable>)this.fcq, 8, 8);
        TestFairCallQueue.assertEquals((int)8, (int)this.fcq.size());
    }

    public void testPutBlocksWhenAllFull() throws InterruptedException {
        this.fcq.setScheduler(this.alwaysZeroScheduler);
        this.assertCanPut((BlockingQueue<Schedulable>)this.fcq, 10, 10);
        TestFairCallQueue.assertEquals((int)10, (int)this.fcq.size());
        this.assertCanPut((BlockingQueue<Schedulable>)this.fcq, 0, 1);
    }

    public void testTakeBlocksWhenEmpty() throws InterruptedException {
        this.fcq.setScheduler(this.alwaysZeroScheduler);
        this.assertCanTake((BlockingQueue<Schedulable>)this.fcq, 0, 1);
    }

    public void testTakeRemovesCall() throws InterruptedException {
        this.fcq.setScheduler(this.alwaysZeroScheduler);
        Schedulable call = this.mockCall("c");
        this.fcq.offer(call);
        TestFairCallQueue.assertEquals((Object)call, (Object)this.fcq.take());
        TestFairCallQueue.assertEquals((int)0, (int)this.fcq.size());
    }

    public void testTakeTriesNextQueue() throws InterruptedException {
        RpcScheduler q1Scheduler = (RpcScheduler)Mockito.mock(RpcScheduler.class);
        Mockito.when((Object)q1Scheduler.getPriorityLevel((Schedulable)Matchers.any())).thenReturn((Object)1);
        this.fcq.setScheduler(q1Scheduler);
        RpcMultiplexer q0mux = (RpcMultiplexer)Mockito.mock(RpcMultiplexer.class);
        Mockito.when((Object)q0mux.getAndAdvanceCurrentIndex()).thenReturn((Object)0);
        this.fcq.setMultiplexer(q0mux);
        Schedulable call = this.mockCall("c");
        this.fcq.put(call);
        TestFairCallQueue.assertEquals((Object)call, (Object)this.fcq.take());
        TestFairCallQueue.assertEquals((int)0, (int)this.fcq.size());
    }

    public class Taker
    implements Runnable {
        private final BlockingQueue<Schedulable> cq;
        public final String tag;
        public volatile int callsTaken = 0;
        public volatile Schedulable lastResult = null;
        private final int maxCalls;
        private final CountDownLatch latch;
        private IdentityProvider uip;

        public Taker(BlockingQueue<Schedulable> aCq, int maxCalls, String tag, CountDownLatch latch) {
            this.maxCalls = maxCalls;
            this.cq = aCq;
            this.tag = tag;
            this.uip = new UserIdentityProvider();
            this.latch = latch;
        }

        @Override
        public void run() {
            try {
                while (this.callsTaken < this.maxCalls || this.maxCalls < 0) {
                    Schedulable res = this.cq.take();
                    String identity = this.uip.makeIdentity(res);
                    if (this.tag != null && this.tag.equals(identity)) {
                        this.cq.put(res);
                        continue;
                    }
                    ++this.callsTaken;
                    this.latch.countDown();
                    this.lastResult = res;
                }
            }
            catch (InterruptedException e) {
                return;
            }
        }
    }

    public class Putter
    implements Runnable {
        private final BlockingQueue<Schedulable> cq;
        public final String tag;
        public volatile int callsAdded = 0;
        private final int maxCalls;
        private final CountDownLatch latch;

        public Putter(BlockingQueue<Schedulable> aCq, int maxCalls, String tag, CountDownLatch latch) {
            this.maxCalls = maxCalls;
            this.cq = aCq;
            this.tag = tag;
            this.latch = latch;
        }

        private String getTag() {
            if (this.tag != null) {
                return this.tag;
            }
            return "";
        }

        @Override
        public void run() {
            try {
                while (this.callsAdded < this.maxCalls || this.maxCalls < 0) {
                    this.cq.put(TestFairCallQueue.this.mockCall(this.getTag()));
                    ++this.callsAdded;
                    this.latch.countDown();
                }
            }
            catch (InterruptedException e) {
                return;
            }
        }
    }
}

