/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.tools;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.EnumMap;
import java.util.HashSet;
import java.util.List;
import java.util.Random;
import org.apache.hadoop.HadoopIllegalArgumentException;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.DFSUtilClient;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.protocol.SnapshotDiffReport;
import org.apache.hadoop.tools.CopyListing;
import org.apache.hadoop.tools.DiffInfo;
import org.apache.hadoop.tools.DistCp;
import org.apache.hadoop.tools.DistCpOptions;

class DistCpSync {
    private DistCpOptions inputOptions;
    private Configuration conf;
    private EnumMap<SnapshotDiffReport.DiffType, List<DiffInfo>> diffMap;
    private DiffInfo[] renameDiffs;

    DistCpSync(DistCpOptions options, Configuration conf) {
        this.inputOptions = options;
        this.conf = conf;
    }

    private boolean preSyncCheck() throws IOException {
        List<Path> sourcePaths = this.inputOptions.getSourcePaths();
        if (sourcePaths.size() != 1) {
            throw new IllegalArgumentException(sourcePaths.size() + " source paths are provided");
        }
        Path sourceDir = sourcePaths.get(0);
        Path targetDir = this.inputOptions.getTargetPath();
        FileSystem sfs = sourceDir.getFileSystem(this.conf);
        FileSystem tfs = targetDir.getFileSystem(this.conf);
        if (!(sfs instanceof DistributedFileSystem) || !(tfs instanceof DistributedFileSystem)) {
            throw new IllegalArgumentException("The FileSystems needs to be DistributedFileSystem for using snapshot-diff-based distcp");
        }
        DistributedFileSystem targetFs = (DistributedFileSystem)tfs;
        if (!this.checkNoChange(targetFs, targetDir)) {
            this.inputOptions.setSourcePaths(Arrays.asList(this.getSourceSnapshotPath(sourceDir, this.inputOptions.getToSnapshot())));
            return false;
        }
        String from = this.getSnapshotName(this.inputOptions.getFromSnapshot());
        String to = this.getSnapshotName(this.inputOptions.getToSnapshot());
        try {
            FileStatus fromSnapshotStat = sfs.getFileStatus(this.getSourceSnapshotPath(sourceDir, from));
            FileStatus toSnapshotStat = sfs.getFileStatus(this.getSourceSnapshotPath(sourceDir, to));
            if (!to.equals("") && fromSnapshotStat.getModificationTime() > toSnapshotStat.getModificationTime()) {
                throw new HadoopIllegalArgumentException("Snapshot " + to + " should be newer than " + from);
            }
        }
        catch (FileNotFoundException nfe) {
            throw new CopyListing.InvalidInputException("Input snapshot is not found", nfe);
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean sync() throws IOException {
        boolean bl;
        if (!this.preSyncCheck()) {
            return false;
        }
        if (!this.getAllDiffs()) {
            return false;
        }
        List<Path> sourcePaths = this.inputOptions.getSourcePaths();
        Path sourceDir = sourcePaths.get(0);
        Path targetDir = this.inputOptions.getTargetPath();
        FileSystem tfs = targetDir.getFileSystem(this.conf);
        DistributedFileSystem targetFs = (DistributedFileSystem)tfs;
        Path tmpDir = null;
        try {
            tmpDir = this.createTargetTmpDir(targetFs, targetDir);
            DiffInfo[] renameAndDeleteDiffs = this.getRenameAndDeleteDiffs(targetDir);
            if (renameAndDeleteDiffs.length > 0) {
                this.syncDiff(renameAndDeleteDiffs, targetFs, tmpDir);
            }
            bl = true;
            this.deleteTargetTmpDir(targetFs, tmpDir);
        }
        catch (Exception e) {
            boolean bl2;
            try {
                DistCp.LOG.warn((Object)"Failed to use snapshot diff for distcp", (Throwable)e);
                bl2 = false;
                this.deleteTargetTmpDir(targetFs, tmpDir);
            }
            catch (Throwable throwable) {
                this.deleteTargetTmpDir(targetFs, tmpDir);
                this.inputOptions.setSourcePaths(Arrays.asList(this.getSourceSnapshotPath(sourceDir, this.inputOptions.getToSnapshot())));
                throw throwable;
            }
            this.inputOptions.setSourcePaths(Arrays.asList(this.getSourceSnapshotPath(sourceDir, this.inputOptions.getToSnapshot())));
            return bl2;
        }
        this.inputOptions.setSourcePaths(Arrays.asList(this.getSourceSnapshotPath(sourceDir, this.inputOptions.getToSnapshot())));
        return bl;
    }

    private boolean getAllDiffs() throws IOException {
        List<Path> sourcePaths = this.inputOptions.getSourcePaths();
        Path sourceDir = sourcePaths.get(0);
        try {
            DistributedFileSystem fs = (DistributedFileSystem)sourceDir.getFileSystem(this.conf);
            String from = this.getSnapshotName(this.inputOptions.getFromSnapshot());
            String to = this.getSnapshotName(this.inputOptions.getToSnapshot());
            SnapshotDiffReport report = fs.getSnapshotDiffReport(sourceDir, from, to);
            this.diffMap = new EnumMap(SnapshotDiffReport.DiffType.class);
            for (SnapshotDiffReport.DiffType type : SnapshotDiffReport.DiffType.values()) {
                this.diffMap.put(type, new ArrayList());
            }
            for (SnapshotDiffReport.DiffReportEntry entry : report.getDiffList()) {
                Path source;
                if (entry.getSourcePath().length <= 0) continue;
                List<DiffInfo> list = this.diffMap.get(entry.getType());
                if (entry.getType() == SnapshotDiffReport.DiffType.MODIFY || entry.getType() == SnapshotDiffReport.DiffType.CREATE || entry.getType() == SnapshotDiffReport.DiffType.DELETE) {
                    source = new Path(DFSUtilClient.bytes2String((byte[])entry.getSourcePath()));
                    list.add(new DiffInfo(source, null, entry.getType()));
                    continue;
                }
                if (entry.getType() != SnapshotDiffReport.DiffType.RENAME) continue;
                source = new Path(DFSUtilClient.bytes2String((byte[])entry.getSourcePath()));
                Path target = new Path(DFSUtilClient.bytes2String((byte[])entry.getTargetPath()));
                list.add(new DiffInfo(source, target, entry.getType()));
            }
            return true;
        }
        catch (IOException e) {
            DistCp.LOG.warn((Object)("Failed to compute snapshot diff on " + sourceDir), (Throwable)e);
            this.diffMap = null;
            return false;
        }
    }

    private String getSnapshotName(String name) {
        return ".".equals(name) ? "" : name;
    }

    private Path getSourceSnapshotPath(Path sourceDir, String snapshotName) {
        if (".".equals(snapshotName)) {
            return sourceDir;
        }
        return new Path(sourceDir, ".snapshot/" + snapshotName);
    }

    private Path createTargetTmpDir(DistributedFileSystem targetFs, Path targetDir) throws IOException {
        Path tmp = new Path(targetDir, ".distcp.diff.tmp" + DistCp.rand.nextInt());
        if (!targetFs.mkdirs(tmp)) {
            throw new IOException("The tmp directory " + tmp + " already exists");
        }
        return tmp;
    }

    private void deleteTargetTmpDir(DistributedFileSystem targetFs, Path tmpDir) {
        try {
            if (tmpDir != null) {
                targetFs.delete(tmpDir, true);
            }
        }
        catch (IOException e) {
            DistCp.LOG.error((Object)("Unable to cleanup tmp dir: " + tmpDir), (Throwable)e);
        }
    }

    private boolean checkNoChange(DistributedFileSystem fs, Path path) {
        try {
            SnapshotDiffReport targetDiff = fs.getSnapshotDiffReport(path, this.inputOptions.getFromSnapshot(), "");
            if (!targetDiff.getDiffList().isEmpty()) {
                DistCp.LOG.warn((Object)("The target has been modified since snapshot " + this.inputOptions.getFromSnapshot()));
                return false;
            }
            return true;
        }
        catch (IOException e) {
            DistCp.LOG.warn((Object)("Failed to compute snapshot diff on " + path), (Throwable)e);
            return false;
        }
    }

    private void syncDiff(DiffInfo[] diffs, DistributedFileSystem targetFs, Path tmpDir) throws IOException {
        this.moveToTmpDir(diffs, targetFs, tmpDir);
        this.moveToTarget(diffs, targetFs);
    }

    private void moveToTmpDir(DiffInfo[] diffs, DistributedFileSystem targetFs, Path tmpDir) throws IOException {
        Arrays.sort(diffs, DiffInfo.sourceComparator);
        Random random = new Random();
        for (DiffInfo diff : diffs) {
            Path tmpTarget = new Path(tmpDir, diff.source.getName());
            while (targetFs.exists(tmpTarget)) {
                tmpTarget = new Path(tmpDir, diff.source.getName() + random.nextInt());
            }
            diff.setTmp(tmpTarget);
            targetFs.rename(diff.source, tmpTarget);
        }
    }

    private void moveToTarget(DiffInfo[] diffs, DistributedFileSystem targetFs) throws IOException {
        Arrays.sort(diffs, DiffInfo.targetComparator);
        for (DiffInfo diff : diffs) {
            if (diff.target == null) continue;
            if (!targetFs.exists(diff.target.getParent())) {
                targetFs.mkdirs(diff.target.getParent());
            }
            targetFs.rename(diff.getTmp(), diff.target);
        }
    }

    private DiffInfo[] getRenameAndDeleteDiffs(Path targetDir) {
        Path source;
        ArrayList<DiffInfo> renameAndDeleteDiff = new ArrayList<DiffInfo>();
        for (DiffInfo diff : this.diffMap.get(SnapshotDiffReport.DiffType.DELETE)) {
            source = new Path(targetDir, diff.source);
            renameAndDeleteDiff.add(new DiffInfo(source, diff.target, diff.getType()));
        }
        for (DiffInfo diff : this.diffMap.get(SnapshotDiffReport.DiffType.RENAME)) {
            source = new Path(targetDir, diff.source);
            Path target = new Path(targetDir, diff.target);
            renameAndDeleteDiff.add(new DiffInfo(source, target, diff.getType()));
        }
        return renameAndDeleteDiff.toArray(new DiffInfo[renameAndDeleteDiff.size()]);
    }

    private DiffInfo[] getCreateAndModifyDiffs() {
        List<DiffInfo> createDiff = this.diffMap.get(SnapshotDiffReport.DiffType.CREATE);
        List<DiffInfo> modifyDiff = this.diffMap.get(SnapshotDiffReport.DiffType.MODIFY);
        ArrayList<DiffInfo> diffs = new ArrayList<DiffInfo>(createDiff.size() + modifyDiff.size());
        diffs.addAll(createDiff);
        diffs.addAll(modifyDiff);
        return diffs.toArray(new DiffInfo[diffs.size()]);
    }

    private boolean isParentOf(Path parent, Path child) {
        String parentPath = parent.toString();
        String childPath = child.toString();
        if (!parentPath.endsWith("/")) {
            parentPath = parentPath + "/";
        }
        return childPath.length() > parentPath.length() && childPath.startsWith(parentPath);
    }

    private DiffInfo getRenameItem(DiffInfo diff, DiffInfo[] renameDiffArray) {
        for (DiffInfo renameItem : renameDiffArray) {
            if (!(diff.source.equals((Object)renameItem.source) ? diff.getType() == SnapshotDiffReport.DiffType.MODIFY : this.isParentOf(renameItem.source, diff.source))) continue;
            return renameItem;
        }
        return null;
    }

    private Path getTargetPath(Path sourcePath, DiffInfo renameItem) {
        if (sourcePath.equals((Object)renameItem.source)) {
            return renameItem.target;
        }
        StringBuffer sb = new StringBuffer(sourcePath.toString());
        String remain = sb.substring(renameItem.source.toString().length() + 1);
        return new Path(renameItem.target, remain);
    }

    public ArrayList<DiffInfo> prepareDiffList() {
        DiffInfo[] modifyAndCreateDiffs = this.getCreateAndModifyDiffs();
        List<DiffInfo> renameDiffsList = this.diffMap.get(SnapshotDiffReport.DiffType.RENAME);
        DiffInfo[] renameDiffArray = renameDiffsList.toArray(new DiffInfo[renameDiffsList.size()]);
        Arrays.sort(renameDiffArray, DiffInfo.sourceComparator);
        ArrayList<DiffInfo> finalListWithTarget = new ArrayList<DiffInfo>();
        for (DiffInfo diff : modifyAndCreateDiffs) {
            DiffInfo renameItem = this.getRenameItem(diff, renameDiffArray);
            diff.target = renameItem == null ? diff.source : this.getTargetPath(diff.source, renameItem);
            finalListWithTarget.add(diff);
        }
        return finalListWithTarget;
    }

    public HashSet<String> getTraverseExcludeList(Path newDir, Path prefix) {
        if (this.renameDiffs == null) {
            List<DiffInfo> renameList = this.diffMap.get(SnapshotDiffReport.DiffType.RENAME);
            this.renameDiffs = renameList.toArray(new DiffInfo[renameList.size()]);
            Arrays.sort(this.renameDiffs, DiffInfo.targetComparator);
        }
        if (this.renameDiffs.length <= 0) {
            return null;
        }
        boolean foundChild = false;
        HashSet<String> excludeList = new HashSet<String>();
        for (DiffInfo diff : this.renameDiffs) {
            if (this.isParentOf(newDir, diff.target)) {
                foundChild = true;
                excludeList.add(new Path(prefix, diff.target).toUri().getPath());
                continue;
            }
            if (foundChild) break;
        }
        return excludeList;
    }
}

