/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.security.x509.certificate.client;

import com.google.common.base.Preconditions;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.InetSocketAddress;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.security.InvalidKeyException;
import java.security.KeyPair;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Signature;
import java.security.SignatureException;
import java.security.cert.CertStore;
import java.security.cert.X509Certificate;
import java.security.spec.InvalidKeySpecException;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import javax.net.SocketFactory;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.RandomStringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.apache.commons.validator.routines.DomainValidator;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdds.HddsUtils;
import org.apache.hadoop.hdds.conf.OzoneConfiguration;
import org.apache.hadoop.hdds.protocol.SCMSecurityProtocol;
import org.apache.hadoop.hdds.protocolPB.SCMSecurityProtocolClientSideTranslatorPB;
import org.apache.hadoop.hdds.protocolPB.SCMSecurityProtocolPB;
import org.apache.hadoop.hdds.scm.protocolPB.ScmBlockLocationProtocolPB;
import org.apache.hadoop.hdds.security.x509.SecurityConfig;
import org.apache.hadoop.hdds.security.x509.certificate.client.CertificateClient;
import org.apache.hadoop.hdds.security.x509.certificate.utils.CertificateCodec;
import org.apache.hadoop.hdds.security.x509.certificates.utils.CertificateSignRequest;
import org.apache.hadoop.hdds.security.x509.exceptions.CertificateException;
import org.apache.hadoop.hdds.security.x509.keys.HDDSKeyGenerator;
import org.apache.hadoop.hdds.security.x509.keys.KeyCodec;
import org.apache.hadoop.ipc.Client;
import org.apache.hadoop.ipc.ProtobufRpcEngine;
import org.apache.hadoop.ipc.RPC;
import org.apache.hadoop.net.NetUtils;
import org.apache.hadoop.ozone.OzoneSecurityUtil;
import org.apache.hadoop.security.UserGroupInformation;
import org.bouncycastle.cert.X509CertificateHolder;
import org.slf4j.Logger;

public abstract class DefaultCertificateClient
implements CertificateClient {
    private static final String CERT_FILE_NAME_FORMAT = "%s.crt";
    private static final String CA_CERT_PREFIX = "CA-";
    private static final int CA_CERT_PREFIX_LEN = 3;
    private final Logger logger;
    private final SecurityConfig securityConfig;
    private final KeyCodec keyCodec;
    private PrivateKey privateKey;
    private PublicKey publicKey;
    private X509Certificate x509Certificate;
    private Map<String, X509Certificate> certificateMap;
    private String certSerialId;
    private String caCertId;
    private String component;

    DefaultCertificateClient(SecurityConfig securityConfig, Logger log, String certSerialId, String component) {
        Objects.requireNonNull(securityConfig);
        this.securityConfig = securityConfig;
        this.keyCodec = new KeyCodec(securityConfig, component);
        this.logger = log;
        this.certificateMap = new ConcurrentHashMap<String, X509Certificate>();
        this.certSerialId = certSerialId;
        this.component = component;
        this.loadAllCertificates();
    }

    private void loadAllCertificates() {
        Path certPath = this.securityConfig.getCertificateLocation(this.component);
        if (Files.exists(certPath, new LinkOption[0]) && Files.isDirectory(certPath, new LinkOption[0])) {
            this.getLogger().info("Loading certificate from location:{}.", (Object)certPath);
            File[] certFiles = certPath.toFile().listFiles();
            if (certFiles != null) {
                CertificateCodec certificateCodec = new CertificateCodec(this.securityConfig, this.component);
                long latestCaCertSerailId = -1L;
                for (File file : certFiles) {
                    if (!file.isFile()) continue;
                    try {
                        X509CertificateHolder x509CertificateHolder = certificateCodec.readCertificate(certPath, file.getName());
                        X509Certificate cert = CertificateCodec.getX509Certificate(x509CertificateHolder);
                        if (cert != null && cert.getSerialNumber() != null) {
                            String certFileName;
                            long tmpCaCertSerailId;
                            if (cert.getSerialNumber().toString().equals(this.certSerialId)) {
                                this.x509Certificate = cert;
                            }
                            this.certificateMap.putIfAbsent(cert.getSerialNumber().toString(), cert);
                            if (file.getName().startsWith(CA_CERT_PREFIX) && (tmpCaCertSerailId = NumberUtils.toLong((String)(certFileName = FilenameUtils.getBaseName((String)file.getName())).substring(3))) > latestCaCertSerailId) {
                                latestCaCertSerailId = tmpCaCertSerailId;
                            }
                            this.getLogger().info("Added certificate from file:{}.", (Object)file.getAbsolutePath());
                            continue;
                        }
                        this.getLogger().error("Error reading certificate from file:{}", (Object)file);
                    }
                    catch (IOException | java.security.cert.CertificateException e) {
                        this.getLogger().error("Error reading certificate from file:{}.", (Object)file.getAbsolutePath(), (Object)e);
                    }
                }
                if (latestCaCertSerailId != -1L) {
                    this.caCertId = Long.toString(latestCaCertSerailId);
                }
            }
        }
    }

    @Override
    public PrivateKey getPrivateKey() {
        if (this.privateKey != null) {
            return this.privateKey;
        }
        Path keyPath = this.securityConfig.getKeyLocation(this.component);
        if (OzoneSecurityUtil.checkIfFileExist(keyPath, this.securityConfig.getPrivateKeyFileName())) {
            try {
                this.privateKey = this.keyCodec.readPrivateKey();
            }
            catch (IOException | NoSuchAlgorithmException | InvalidKeySpecException e) {
                this.getLogger().error("Error while getting private key.", (Throwable)e);
            }
        }
        return this.privateKey;
    }

    @Override
    public PublicKey getPublicKey() {
        if (this.publicKey != null) {
            return this.publicKey;
        }
        Path keyPath = this.securityConfig.getKeyLocation(this.component);
        if (OzoneSecurityUtil.checkIfFileExist(keyPath, this.securityConfig.getPublicKeyFileName())) {
            try {
                this.publicKey = this.keyCodec.readPublicKey();
            }
            catch (IOException | NoSuchAlgorithmException | InvalidKeySpecException e) {
                this.getLogger().error("Error while getting public key.", (Throwable)e);
            }
        }
        return this.publicKey;
    }

    @Override
    public X509Certificate getCertificate() {
        if (this.x509Certificate != null) {
            return this.x509Certificate;
        }
        if (this.certSerialId == null) {
            this.getLogger().error("Default certificate serial id is not set. Can't locate the default certificate for this client.");
            return null;
        }
        this.loadAllCertificates();
        if (this.certificateMap.containsKey(this.certSerialId)) {
            this.x509Certificate = this.certificateMap.get(this.certSerialId);
        }
        return this.x509Certificate;
    }

    @Override
    public X509Certificate getCACertificate() {
        if (this.caCertId != null) {
            return this.certificateMap.get(this.caCertId);
        }
        return null;
    }

    @Override
    public X509Certificate getCertificate(String certId) throws CertificateException {
        if (this.certificateMap.containsKey(certId)) {
            return this.certificateMap.get(certId);
        }
        return this.getCertificateFromScm(certId);
    }

    private X509Certificate getCertificateFromScm(String certId) throws CertificateException {
        this.getLogger().info("Getting certificate with certSerialId:{}.", (Object)certId);
        try {
            SCMSecurityProtocol scmSecurityProtocolClient = DefaultCertificateClient.getScmSecurityClient((OzoneConfiguration)this.securityConfig.getConfiguration());
            String pemEncodedCert = scmSecurityProtocolClient.getCertificate(certId);
            this.storeCertificate(pemEncodedCert, true);
            return CertificateCodec.getX509Certificate(pemEncodedCert);
        }
        catch (Exception e) {
            this.getLogger().error("Error while getting Certificate with certSerialId:{} from scm.", (Object)certId, (Object)e);
            throw new CertificateException("Error while getting certificate for certSerialId:" + certId, e, CertificateException.ErrorCode.CERTIFICATE_ERROR);
        }
    }

    @Override
    public boolean verifyCertificate(X509Certificate certificate) {
        throw new UnsupportedOperationException("Operation not supported.");
    }

    @Override
    public byte[] signDataStream(InputStream stream) throws CertificateException {
        try {
            int len;
            Signature sign = Signature.getInstance(this.securityConfig.getSignatureAlgo(), this.securityConfig.getProvider());
            sign.initSign(this.getPrivateKey());
            byte[] buffer = new byte[4096];
            while (-1 != (len = stream.read(buffer))) {
                sign.update(buffer, 0, len);
            }
            return sign.sign();
        }
        catch (IOException | InvalidKeyException | NoSuchAlgorithmException | NoSuchProviderException | SignatureException e) {
            this.getLogger().error("Error while signing the stream", (Throwable)e);
            throw new CertificateException("Error while signing the stream", e, CertificateException.ErrorCode.CRYPTO_SIGN_ERROR);
        }
    }

    @Override
    public byte[] signData(byte[] data) throws CertificateException {
        try {
            Signature sign = Signature.getInstance(this.securityConfig.getSignatureAlgo(), this.securityConfig.getProvider());
            sign.initSign(this.getPrivateKey());
            sign.update(data);
            return sign.sign();
        }
        catch (InvalidKeyException | NoSuchAlgorithmException | NoSuchProviderException | SignatureException e) {
            this.getLogger().error("Error while signing the stream", (Throwable)e);
            throw new CertificateException("Error while signing the stream", e, CertificateException.ErrorCode.CRYPTO_SIGN_ERROR);
        }
    }

    @Override
    public boolean verifySignature(InputStream stream, byte[] signature, X509Certificate cert) throws CertificateException {
        try {
            int len;
            Signature sign = Signature.getInstance(this.securityConfig.getSignatureAlgo(), this.securityConfig.getProvider());
            sign.initVerify(cert);
            byte[] buffer = new byte[4096];
            while (-1 != (len = stream.read(buffer))) {
                sign.update(buffer, 0, len);
            }
            return sign.verify(signature);
        }
        catch (IOException | InvalidKeyException | NoSuchAlgorithmException | NoSuchProviderException | SignatureException e) {
            this.getLogger().error("Error while signing the stream", (Throwable)e);
            throw new CertificateException("Error while signing the stream", e, CertificateException.ErrorCode.CRYPTO_SIGNATURE_VERIFICATION_ERROR);
        }
    }

    @Override
    public boolean verifySignature(byte[] data, byte[] signature, X509Certificate cert) throws CertificateException {
        try {
            Signature sign = Signature.getInstance(this.securityConfig.getSignatureAlgo(), this.securityConfig.getProvider());
            sign.initVerify(cert);
            sign.update(data);
            return sign.verify(signature);
        }
        catch (InvalidKeyException | NoSuchAlgorithmException | NoSuchProviderException | SignatureException e) {
            this.getLogger().error("Error while signing the stream", (Throwable)e);
            throw new CertificateException("Error while signing the stream", e, CertificateException.ErrorCode.CRYPTO_SIGNATURE_VERIFICATION_ERROR);
        }
    }

    private boolean verifySignature(byte[] data, byte[] signature, PublicKey pubKey) throws CertificateException {
        try {
            Signature sign = Signature.getInstance(this.securityConfig.getSignatureAlgo(), this.securityConfig.getProvider());
            sign.initVerify(pubKey);
            sign.update(data);
            return sign.verify(signature);
        }
        catch (InvalidKeyException | NoSuchAlgorithmException | NoSuchProviderException | SignatureException e) {
            this.getLogger().error("Error while signing the stream", (Throwable)e);
            throw new CertificateException("Error while signing the stream", e, CertificateException.ErrorCode.CRYPTO_SIGNATURE_VERIFICATION_ERROR);
        }
    }

    @Override
    public CertificateSignRequest.Builder getCSRBuilder() throws CertificateException {
        CertificateSignRequest.Builder builder = new CertificateSignRequest.Builder().setConfiguration(this.securityConfig.getConfiguration());
        try {
            DomainValidator validator = DomainValidator.getInstance();
            OzoneSecurityUtil.getValidInetsForCurrentHost().forEach(ip -> {
                builder.addIpAddress(ip.getHostAddress());
                if (validator.isValid(ip.getCanonicalHostName())) {
                    builder.addDnsName(ip.getCanonicalHostName());
                }
            });
        }
        catch (IOException e) {
            throw new CertificateException("Error while adding ip to CSR builder", e, CertificateException.ErrorCode.CSR_ERROR);
        }
        return builder;
    }

    @Override
    public X509Certificate queryCertificate(String query) {
        throw new UnsupportedOperationException("Operation not supported");
    }

    @Override
    public void storeCertificate(String pemEncodedCert, boolean force) throws CertificateException {
        this.storeCertificate(pemEncodedCert, force, false);
    }

    @Override
    public void storeCertificate(String pemEncodedCert, boolean force, boolean caCert) throws CertificateException {
        CertificateCodec certificateCodec = new CertificateCodec(this.securityConfig, this.component);
        try {
            Path basePath = this.securityConfig.getCertificateLocation(this.component);
            X509Certificate cert = CertificateCodec.getX509Certificate(pemEncodedCert);
            String certName = String.format(CERT_FILE_NAME_FORMAT, cert.getSerialNumber().toString());
            if (caCert) {
                certName = CA_CERT_PREFIX + certName;
                this.caCertId = cert.getSerialNumber().toString();
            }
            certificateCodec.writeCertificate(basePath, certName, pemEncodedCert, force);
            this.certificateMap.putIfAbsent(cert.getSerialNumber().toString(), cert);
        }
        catch (IOException | java.security.cert.CertificateException e) {
            throw new CertificateException("Error while storing certificate.", e, CertificateException.ErrorCode.CERTIFICATE_ERROR);
        }
    }

    @Override
    public synchronized void storeTrustChain(CertStore ks) throws CertificateException {
        throw new UnsupportedOperationException("Operation not supported.");
    }

    @Override
    public synchronized void storeTrustChain(List<X509Certificate> certificates) throws CertificateException {
        throw new UnsupportedOperationException("Operation not supported.");
    }

    @Override
    public synchronized CertificateClient.InitResponse init() throws CertificateException {
        int initCase = 0;
        PrivateKey pvtKey = this.getPrivateKey();
        PublicKey pubKey = this.getPublicKey();
        X509Certificate certificate = this.getCertificate();
        if (pvtKey != null) {
            initCase |= 4;
        }
        if (pubKey != null) {
            initCase |= 2;
        }
        if (certificate != null) {
            initCase |= 1;
        }
        this.getLogger().info("Certificate client init case: {}", (Object)initCase);
        Preconditions.checkArgument((initCase < 8 ? 1 : 0) != 0, (Object)"Not a valid case.");
        InitCase init = InitCase.values()[initCase];
        return this.handleCase(init);
    }

    protected CertificateClient.InitResponse handleCase(InitCase init) throws CertificateException {
        switch (init) {
            case NONE: {
                this.getLogger().info("Creating keypair for client as keypair and certificate not found.");
                this.bootstrapClientKeys();
                return CertificateClient.InitResponse.GETCERT;
            }
            case CERT: {
                this.getLogger().error("Private key not found, while certificate is still present. Delete keypair and try again.");
                return CertificateClient.InitResponse.FAILURE;
            }
            case PUBLIC_KEY: {
                this.getLogger().error("Found public key but private key and certificate missing.");
                return CertificateClient.InitResponse.FAILURE;
            }
            case PRIVATE_KEY: {
                this.getLogger().info("Found private key but public key and certificate is missing.");
                return CertificateClient.InitResponse.FAILURE;
            }
            case PUBLICKEY_CERT: {
                this.getLogger().error("Found public key and certificate but private key is missing.");
                return CertificateClient.InitResponse.FAILURE;
            }
            case PRIVATEKEY_CERT: {
                this.getLogger().info("Found private key and certificate but public key missing.");
                if (this.recoverPublicKey()) {
                    return CertificateClient.InitResponse.SUCCESS;
                }
                this.getLogger().error("Public key recovery failed.");
                return CertificateClient.InitResponse.FAILURE;
            }
            case PUBLICKEY_PRIVATEKEY: {
                this.getLogger().info("Found private and public key but certificate is missing.");
                if (this.validateKeyPair(this.getPublicKey())) {
                    return CertificateClient.InitResponse.GETCERT;
                }
                this.getLogger().info("Keypair validation failed.");
                return CertificateClient.InitResponse.FAILURE;
            }
            case ALL: {
                this.getLogger().info("Found certificate file along with KeyPair.");
                if (this.validateKeyPairAndCertificate()) {
                    return CertificateClient.InitResponse.SUCCESS;
                }
                return CertificateClient.InitResponse.FAILURE;
            }
        }
        this.getLogger().error("Unexpected case: {} (private/public/cert)", (Object)Integer.toBinaryString(init.ordinal()));
        return CertificateClient.InitResponse.FAILURE;
    }

    protected boolean validateKeyPairAndCertificate() throws CertificateException {
        if (this.validateKeyPair(this.getPublicKey())) {
            this.getLogger().info("Keypair validated.");
            if (!this.validateKeyPair(this.getCertificate().getPublicKey())) {
                this.getLogger().error("Stored certificate is generated with different private key.");
                return false;
            }
        } else {
            this.getLogger().error("Keypair validation failed.");
            return false;
        }
        this.getLogger().info("Keypair validated with certificate.");
        return true;
    }

    protected boolean recoverPublicKey() throws CertificateException {
        PublicKey pubKey = this.getCertificate().getPublicKey();
        try {
            if (!this.validateKeyPair(pubKey)) {
                this.getLogger().error("Can't recover public key corresponding to private key.", (Object)CertificateException.ErrorCode.BOOTSTRAP_ERROR);
                return false;
            }
            this.keyCodec.writePublicKey(pubKey);
            this.publicKey = pubKey;
        }
        catch (IOException e) {
            throw new CertificateException("Error while trying to recover public key.", e, CertificateException.ErrorCode.BOOTSTRAP_ERROR);
        }
        return true;
    }

    protected boolean validateKeyPair(PublicKey pubKey) throws CertificateException {
        byte[] challenge = RandomStringUtils.random((int)1000).getBytes(StandardCharsets.UTF_8);
        byte[] sign = this.signDataStream(new ByteArrayInputStream(challenge));
        return this.verifySignature(challenge, sign, pubKey);
    }

    protected void bootstrapClientKeys() throws CertificateException {
        Path keyPath = this.securityConfig.getKeyLocation(this.component);
        if (Files.notExists(keyPath, new LinkOption[0])) {
            try {
                Files.createDirectories(keyPath, new FileAttribute[0]);
            }
            catch (IOException e) {
                throw new CertificateException("Error while creating directories for certificate storage.", CertificateException.ErrorCode.BOOTSTRAP_ERROR);
            }
        }
        KeyPair keyPair = this.createKeyPair();
        this.privateKey = keyPair.getPrivate();
        this.publicKey = keyPair.getPublic();
    }

    protected KeyPair createKeyPair() throws CertificateException {
        HDDSKeyGenerator keyGenerator = new HDDSKeyGenerator(this.securityConfig);
        KeyPair keyPair = null;
        try {
            keyPair = keyGenerator.generateKey();
            this.keyCodec.writePublicKey(keyPair.getPublic());
            this.keyCodec.writePrivateKey(keyPair.getPrivate());
        }
        catch (IOException | NoSuchAlgorithmException | NoSuchProviderException e) {
            this.getLogger().error("Error while bootstrapping certificate client.", (Throwable)e);
            throw new CertificateException("Error while bootstrapping certificate.", CertificateException.ErrorCode.BOOTSTRAP_ERROR);
        }
        return keyPair;
    }

    public Logger getLogger() {
        return this.logger;
    }

    private static SCMSecurityProtocol getScmSecurityClient(OzoneConfiguration conf) throws IOException {
        RPC.setProtocolEngine((Configuration)conf, SCMSecurityProtocolPB.class, ProtobufRpcEngine.class);
        long scmVersion = RPC.getProtocolVersion(ScmBlockLocationProtocolPB.class);
        InetSocketAddress scmSecurityProtoAdd = HddsUtils.getScmAddressForSecurityProtocol(conf);
        SCMSecurityProtocolClientSideTranslatorPB scmSecurityClient = new SCMSecurityProtocolClientSideTranslatorPB((SCMSecurityProtocolPB)RPC.getProxy(SCMSecurityProtocolPB.class, (long)scmVersion, (InetSocketAddress)scmSecurityProtoAdd, (UserGroupInformation)UserGroupInformation.getCurrentUser(), (Configuration)conf, (SocketFactory)NetUtils.getDefaultSocketFactory((Configuration)conf), (int)Client.getRpcTimeout((Configuration)conf)));
        return scmSecurityClient;
    }

    protected static enum InitCase {
        NONE,
        CERT,
        PUBLIC_KEY,
        PUBLICKEY_CERT,
        PRIVATE_KEY,
        PRIVATEKEY_CERT,
        PUBLICKEY_PRIVATEKEY,
        ALL;

    }
}

