/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.security.token;

import com.google.common.base.Strings;
import java.io.ByteArrayInputStream;
import java.io.DataInput;
import java.io.DataInputStream;
import java.io.IOException;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.X509Certificate;
import org.apache.hadoop.hdds.HddsUtils;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.security.exception.SCMSecurityException;
import org.apache.hadoop.hdds.security.token.BlockTokenException;
import org.apache.hadoop.hdds.security.token.OzoneBlockTokenIdentifier;
import org.apache.hadoop.hdds.security.token.TokenVerifier;
import org.apache.hadoop.hdds.security.x509.SecurityConfig;
import org.apache.hadoop.hdds.security.x509.certificate.client.CertificateClient;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.security.token.Token;
import org.apache.hadoop.util.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BlockTokenVerifier
implements TokenVerifier {
    private final CertificateClient caClient;
    private final SecurityConfig conf;
    private static boolean testStub = false;
    private static final Logger LOGGER = LoggerFactory.getLogger(BlockTokenVerifier.class);

    public BlockTokenVerifier(SecurityConfig conf, CertificateClient caClient) {
        this.conf = conf;
        this.caClient = caClient;
    }

    private boolean isExpired(long expiryDate) {
        return Time.now() > expiryDate;
    }

    @Override
    public void verify(String user, String tokenStr, ContainerProtos.Type cmd, String id) throws SCMSecurityException {
        if (!this.conf.isBlockTokenEnabled() || !HddsUtils.requireBlockToken((ContainerProtos.Type)cmd)) {
            return;
        }
        if (Strings.isNullOrEmpty((String)tokenStr)) {
            throw new BlockTokenException("Fail to find any token (empty or null.)");
        }
        Token token = new Token();
        OzoneBlockTokenIdentifier tokenId = new OzoneBlockTokenIdentifier();
        try {
            token.decodeFromUrlString(tokenStr);
            ByteArrayInputStream buf = new ByteArrayInputStream(token.getIdentifier());
            DataInputStream in = new DataInputStream(buf);
            tokenId.readFields((DataInput)in);
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Verifying token:{} for user:{} ", (Object)tokenId, (Object)user);
            }
        }
        catch (IOException ex) {
            throw new BlockTokenException("Failed to decode token : " + tokenStr);
        }
        if (this.caClient == null) {
            throw new SCMSecurityException("Certificate client not available to validate token");
        }
        UserGroupInformation tokenUser = tokenId.getUser();
        X509Certificate signerCert = this.caClient.getCertificate(tokenId.getOmCertSerialId());
        if (signerCert == null) {
            throw new BlockTokenException("Can't find signer certificate (OmCertSerialId: " + tokenId.getOmCertSerialId() + ") of the block token for user: " + tokenUser);
        }
        try {
            signerCert.checkValidity();
        }
        catch (CertificateExpiredException exExp) {
            throw new BlockTokenException("Block token can't be verified due to expired certificate " + tokenId.getOmCertSerialId());
        }
        catch (CertificateNotYetValidException exNyv) {
            throw new BlockTokenException("Block token can't be verified due to not yet valid certificate " + tokenId.getOmCertSerialId());
        }
        boolean validToken = this.caClient.verifySignature(tokenId.getBytes(), token.getPassword(), signerCert);
        if (!validToken) {
            throw new BlockTokenException("Invalid block token for user: " + tokenId.getUser());
        }
        if (this.isExpired(tokenId.getExpiryDate())) {
            throw new BlockTokenException("Expired block token for user: " + tokenUser);
        }
        if (!tokenId.getBlockId().equals(id)) {
            throw new BlockTokenException("Block id mismatch. Token for block ID: " + tokenId.getBlockId() + " can't be used to access block: " + id + " by user: " + tokenUser);
        }
        if (cmd == ContainerProtos.Type.ReadChunk || cmd == ContainerProtos.Type.GetBlock || cmd == ContainerProtos.Type.GetSmallFile) {
            if (!tokenId.getAccessModes().contains(HddsProtos.BlockTokenSecretProto.AccessModeProto.READ)) {
                throw new BlockTokenException("Block token with " + id + " doesn't have READ permission");
            }
        } else if (cmd == ContainerProtos.Type.WriteChunk || cmd == ContainerProtos.Type.PutBlock || cmd == ContainerProtos.Type.PutSmallFile) {
            if (!tokenId.getAccessModes().contains(HddsProtos.BlockTokenSecretProto.AccessModeProto.WRITE)) {
                throw new BlockTokenException("Block token with " + id + " doesn't have WRITE permission");
            }
        } else {
            throw new BlockTokenException("Block token does not support " + cmd);
        }
    }

    public static boolean isTestStub() {
        return testStub;
    }

    public static void setTestStub(boolean isTestStub) {
        testStub = isTestStub;
    }
}

