/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.pipeline;

import com.google.common.base.Preconditions;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.NavigableSet;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.apache.commons.lang3.builder.HashCodeBuilder;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.scm.container.ContainerID;
import org.apache.hadoop.hdds.scm.pipeline.Pipeline;
import org.apache.hadoop.hdds.scm.pipeline.PipelineID;
import org.apache.hadoop.hdds.scm.pipeline.PipelineNotFoundException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class PipelineStateMap {
    private static final Logger LOG = LoggerFactory.getLogger(PipelineStateMap.class);
    private final Map<PipelineID, Pipeline> pipelineMap = new HashMap<PipelineID, Pipeline>();
    private final Map<PipelineID, NavigableSet<ContainerID>> pipeline2container = new HashMap<PipelineID, NavigableSet<ContainerID>>();
    private final Map<PipelineQuery, List<Pipeline>> query2OpenPipelines = new HashMap<PipelineQuery, List<Pipeline>>();

    PipelineStateMap() {
        this.initializeQueryMap();
    }

    private void initializeQueryMap() {
        for (HddsProtos.ReplicationType type : HddsProtos.ReplicationType.values()) {
            for (HddsProtos.ReplicationFactor factor : HddsProtos.ReplicationFactor.values()) {
                this.query2OpenPipelines.put(new PipelineQuery(type, factor), new CopyOnWriteArrayList());
            }
        }
    }

    void addPipeline(Pipeline pipeline) throws IOException {
        Preconditions.checkNotNull((Object)pipeline, (Object)"Pipeline cannot be null");
        Preconditions.checkArgument((pipeline.getNodes().size() == pipeline.getFactor().getNumber() ? 1 : 0) != 0, (Object)String.format("Nodes size=%d, replication factor=%d do not match ", pipeline.getNodes().size(), pipeline.getFactor().getNumber()));
        if (this.pipelineMap.putIfAbsent(pipeline.getId(), pipeline) != null) {
            LOG.warn("Duplicate pipeline ID detected. {}", (Object)pipeline.getId());
            throw new IOException(String.format("Duplicate pipeline ID %s detected.", pipeline.getId()));
        }
        this.pipeline2container.put(pipeline.getId(), new TreeSet());
        if (pipeline.getPipelineState() == Pipeline.PipelineState.OPEN) {
            this.query2OpenPipelines.get(new PipelineQuery(pipeline)).add(pipeline);
        }
    }

    void addContainerToPipeline(PipelineID pipelineID, ContainerID containerID) throws IOException {
        Preconditions.checkNotNull((Object)pipelineID, (Object)"Pipeline Id cannot be null");
        Preconditions.checkNotNull((Object)containerID, (Object)"Container Id cannot be null");
        Pipeline pipeline = this.getPipeline(pipelineID);
        if (pipeline.isClosed()) {
            throw new IOException(String.format("Cannot add container to pipeline=%s in closed state", pipelineID));
        }
        this.pipeline2container.get(pipelineID).add(containerID);
    }

    Pipeline getPipeline(PipelineID pipelineID) throws PipelineNotFoundException {
        Preconditions.checkNotNull((Object)pipelineID, (Object)"Pipeline Id cannot be null");
        Pipeline pipeline = this.pipelineMap.get(pipelineID);
        if (pipeline == null) {
            throw new PipelineNotFoundException(String.format("%s not found", pipelineID));
        }
        return pipeline;
    }

    public List<Pipeline> getPipelines() {
        return new ArrayList<Pipeline>(this.pipelineMap.values());
    }

    List<Pipeline> getPipelines(HddsProtos.ReplicationType type) {
        Preconditions.checkNotNull((Object)type, (Object)"Replication type cannot be null");
        return this.pipelineMap.values().stream().filter(p -> p.getType().equals((Object)type)).collect(Collectors.toList());
    }

    List<Pipeline> getPipelines(HddsProtos.ReplicationType type, HddsProtos.ReplicationFactor factor) {
        Preconditions.checkNotNull((Object)type, (Object)"Replication type cannot be null");
        Preconditions.checkNotNull((Object)factor, (Object)"Replication factor cannot be null");
        return this.pipelineMap.values().stream().filter(pipeline -> pipeline.getType() == type && pipeline.getFactor() == factor).collect(Collectors.toList());
    }

    List<Pipeline> getPipelines(HddsProtos.ReplicationType type, Pipeline.PipelineState ... states) {
        Preconditions.checkNotNull((Object)type, (Object)"Replication type cannot be null");
        Preconditions.checkNotNull((Object)states, (Object)"Pipeline state cannot be null");
        HashSet<Pipeline.PipelineState> pipelineStates = new HashSet<Pipeline.PipelineState>();
        pipelineStates.addAll(Arrays.asList(states));
        return this.pipelineMap.values().stream().filter(pipeline -> pipeline.getType() == type && pipelineStates.contains(pipeline.getPipelineState())).collect(Collectors.toList());
    }

    List<Pipeline> getPipelines(HddsProtos.ReplicationType type, HddsProtos.ReplicationFactor factor, Pipeline.PipelineState state) {
        Preconditions.checkNotNull((Object)type, (Object)"Replication type cannot be null");
        Preconditions.checkNotNull((Object)factor, (Object)"Replication factor cannot be null");
        Preconditions.checkNotNull((Object)state, (Object)"Pipeline state cannot be null");
        if (state == Pipeline.PipelineState.OPEN) {
            return Collections.unmodifiableList(this.query2OpenPipelines.get(new PipelineQuery(type, factor)));
        }
        return this.pipelineMap.values().stream().filter(pipeline -> pipeline.getType() == type && pipeline.getPipelineState() == state && pipeline.getFactor() == factor).collect(Collectors.toList());
    }

    List<Pipeline> getPipelines(HddsProtos.ReplicationType type, HddsProtos.ReplicationFactor factor, Pipeline.PipelineState state, Collection<DatanodeDetails> excludeDns, Collection<PipelineID> excludePipelines) {
        Preconditions.checkNotNull((Object)type, (Object)"Replication type cannot be null");
        Preconditions.checkNotNull((Object)factor, (Object)"Replication factor cannot be null");
        Preconditions.checkNotNull((Object)state, (Object)"Pipeline state cannot be null");
        Preconditions.checkNotNull(excludeDns, (Object)"Datanode exclude list cannot be null");
        Preconditions.checkNotNull(excludeDns, (Object)"Pipeline exclude list cannot be null");
        return this.getPipelines(type, factor, state).stream().filter(pipeline -> !this.discardPipeline((Pipeline)pipeline, excludePipelines) && !this.discardDatanode((Pipeline)pipeline, excludeDns)).collect(Collectors.toList());
    }

    private boolean discardPipeline(Pipeline pipeline, Collection<PipelineID> excludePipelines) {
        if (excludePipelines.isEmpty()) {
            return false;
        }
        Predicate<PipelineID> predicate = p -> p.equals((Object)pipeline.getId());
        return excludePipelines.parallelStream().anyMatch(predicate);
    }

    private boolean discardDatanode(Pipeline pipeline, Collection<DatanodeDetails> excludeDns) {
        if (excludeDns.isEmpty()) {
            return false;
        }
        boolean discard = false;
        for (DatanodeDetails dn : pipeline.getNodes()) {
            Predicate<DatanodeDetails> predicate = p -> p.equals((Object)dn);
            discard = excludeDns.parallelStream().anyMatch(predicate);
            if (!discard) continue;
            break;
        }
        return discard;
    }

    NavigableSet<ContainerID> getContainers(PipelineID pipelineID) throws PipelineNotFoundException {
        Preconditions.checkNotNull((Object)pipelineID, (Object)"Pipeline Id cannot be null");
        NavigableSet<ContainerID> containerIDs = this.pipeline2container.get(pipelineID);
        if (containerIDs == null) {
            throw new PipelineNotFoundException(String.format("%s not found", pipelineID));
        }
        return new TreeSet<ContainerID>((SortedSet<ContainerID>)containerIDs);
    }

    int getNumberOfContainers(PipelineID pipelineID) throws PipelineNotFoundException {
        Preconditions.checkNotNull((Object)pipelineID, (Object)"Pipeline Id cannot be null");
        Set containerIDs = this.pipeline2container.get(pipelineID);
        if (containerIDs == null) {
            throw new PipelineNotFoundException(String.format("%s not found", pipelineID));
        }
        return containerIDs.size();
    }

    Pipeline removePipeline(PipelineID pipelineID) throws IOException {
        Preconditions.checkNotNull((Object)pipelineID, (Object)"Pipeline Id cannot be null");
        Pipeline pipeline = this.getPipeline(pipelineID);
        if (!pipeline.isClosed()) {
            throw new IOException(String.format("Pipeline with %s is not yet closed", pipelineID));
        }
        this.pipelineMap.remove(pipelineID);
        this.pipeline2container.remove(pipelineID);
        return pipeline;
    }

    void removeContainerFromPipeline(PipelineID pipelineID, ContainerID containerID) throws IOException {
        Preconditions.checkNotNull((Object)pipelineID, (Object)"Pipeline Id cannot be null");
        Preconditions.checkNotNull((Object)containerID, (Object)"container Id cannot be null");
        Set containerIDs = this.pipeline2container.get(pipelineID);
        if (containerIDs == null) {
            throw new PipelineNotFoundException(String.format("%s not found", pipelineID));
        }
        containerIDs.remove(containerID);
    }

    Pipeline updatePipelineState(PipelineID pipelineID, Pipeline.PipelineState state) throws PipelineNotFoundException {
        Preconditions.checkNotNull((Object)pipelineID, (Object)"Pipeline Id cannot be null");
        Preconditions.checkNotNull((Object)state, (Object)"Pipeline LifeCycleState cannot be null");
        Pipeline pipeline = this.getPipeline(pipelineID);
        Pipeline updatedPipeline = this.pipelineMap.compute(pipelineID, (id, p) -> Pipeline.newBuilder((Pipeline)pipeline).setState(state).build());
        PipelineQuery query = new PipelineQuery(pipeline);
        if (updatedPipeline.getPipelineState() == Pipeline.PipelineState.OPEN) {
            this.query2OpenPipelines.get(query).add(updatedPipeline);
        } else if (updatedPipeline.getPipelineState() == Pipeline.PipelineState.CLOSED) {
            this.query2OpenPipelines.get(query).remove(pipeline);
        }
        return updatedPipeline;
    }

    private static class PipelineQuery {
        private HddsProtos.ReplicationType type;
        private HddsProtos.ReplicationFactor factor;

        PipelineQuery(HddsProtos.ReplicationType type, HddsProtos.ReplicationFactor factor) {
            this.type = (HddsProtos.ReplicationType)Preconditions.checkNotNull((Object)type);
            this.factor = (HddsProtos.ReplicationFactor)Preconditions.checkNotNull((Object)factor);
        }

        PipelineQuery(Pipeline pipeline) {
            this.type = pipeline.getType();
            this.factor = pipeline.getFactor();
        }

        @SuppressFBWarnings(value={"NP_EQUALS_SHOULD_HANDLE_NULL_ARGUMENT"})
        public boolean equals(Object other) {
            if (this == other) {
                return true;
            }
            if (!this.getClass().equals(other.getClass())) {
                return false;
            }
            PipelineQuery otherQuery = (PipelineQuery)other;
            return this.type == otherQuery.type && this.factor == otherQuery.factor;
        }

        public int hashCode() {
            return new HashCodeBuilder().append((Object)this.type).append((Object)this.factor).toHashCode();
        }
    }
}

