/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.container;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.primitives.Longs;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NavigableSet;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.stream.Collectors;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.scm.container.ContainerID;
import org.apache.hadoop.hdds.scm.container.ContainerInfo;
import org.apache.hadoop.hdds.scm.container.ContainerManager;
import org.apache.hadoop.hdds.scm.container.ContainerNotFoundException;
import org.apache.hadoop.hdds.scm.container.ContainerReplica;
import org.apache.hadoop.hdds.scm.container.ContainerReplicaNotFoundException;
import org.apache.hadoop.hdds.scm.container.ContainerStateManager;
import org.apache.hadoop.hdds.scm.container.metrics.SCMContainerManagerMetrics;
import org.apache.hadoop.hdds.scm.exceptions.SCMException;
import org.apache.hadoop.hdds.scm.pipeline.Pipeline;
import org.apache.hadoop.hdds.scm.pipeline.PipelineManager;
import org.apache.hadoop.hdds.scm.pipeline.PipelineNotFoundException;
import org.apache.hadoop.hdds.server.ServerUtils;
import org.apache.hadoop.hdds.utils.BatchOperation;
import org.apache.hadoop.hdds.utils.MetadataStore;
import org.apache.hadoop.hdds.utils.MetadataStoreBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SCMContainerManager
implements ContainerManager {
    private static final Logger LOG = LoggerFactory.getLogger(SCMContainerManager.class);
    private final Lock lock;
    private final MetadataStore containerStore;
    private final PipelineManager pipelineManager;
    private final ContainerStateManager containerStateManager;
    private final int numContainerPerOwnerInPipeline;
    private final SCMContainerManagerMetrics scmContainerManagerMetrics;

    public SCMContainerManager(Configuration conf, PipelineManager pipelineManager) throws IOException {
        File containerDBPath = this.getContainerDBPath(conf);
        int cacheSize = conf.getInt("ozone.scm.db.cache.size.mb", 128);
        this.containerStore = MetadataStoreBuilder.newBuilder().setConf(conf).setDbFile(containerDBPath).setCacheSize((long)cacheSize * 0x100000L).build();
        this.lock = new ReentrantLock();
        this.pipelineManager = pipelineManager;
        this.containerStateManager = new ContainerStateManager(conf);
        this.numContainerPerOwnerInPipeline = conf.getInt("ozone.scm.pipeline.owner.container.count", 3);
        this.loadExistingContainers();
        this.scmContainerManagerMetrics = SCMContainerManagerMetrics.create();
    }

    private void loadExistingContainers() throws IOException {
        List range = this.containerStore.getSequentialRangeKVs(null, Integer.MAX_VALUE, null);
        for (Map.Entry entry : range) {
            ContainerInfo container = ContainerInfo.fromProtobuf((HddsProtos.ContainerInfoProto)((HddsProtos.ContainerInfoProto)HddsProtos.ContainerInfoProto.PARSER.parseFrom((byte[])entry.getValue())));
            Preconditions.checkNotNull((Object)container);
            this.containerStateManager.loadContainer(container);
            try {
                if (container.getState() != HddsProtos.LifeCycleState.OPEN) continue;
                this.pipelineManager.addContainerToPipeline(container.getPipelineID(), ContainerID.valueof((long)container.getContainerID()));
            }
            catch (PipelineNotFoundException ex) {
                LOG.warn("Found a Container {} which is in {} state with pipeline {} that does not exist. Closing Container.", new Object[]{container, container.getState(), container.getPipelineID()});
                this.updateContainerState(container.containerID(), HddsProtos.LifeCycleEvent.FINALIZE, true);
            }
        }
    }

    @VisibleForTesting
    public ContainerStateManager getContainerStateManager() {
        return this.containerStateManager;
    }

    @Override
    public Set<ContainerID> getContainerIDs() {
        this.lock.lock();
        try {
            Set<ContainerID> set = this.containerStateManager.getAllContainerIDs();
            return set;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public List<ContainerInfo> getContainers() {
        this.lock.lock();
        try {
            List<ContainerInfo> list = this.containerStateManager.getAllContainerIDs().stream().map(id -> {
                try {
                    return this.containerStateManager.getContainer((ContainerID)id);
                }
                catch (ContainerNotFoundException e) {
                    return null;
                }
            }).filter(Objects::nonNull).collect(Collectors.toList());
            return list;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public List<ContainerInfo> getContainers(HddsProtos.LifeCycleState state) {
        this.lock.lock();
        try {
            List<ContainerInfo> list = this.containerStateManager.getContainerIDsByState(state).stream().map(id -> {
                try {
                    return this.containerStateManager.getContainer((ContainerID)id);
                }
                catch (ContainerNotFoundException e) {
                    return null;
                }
            }).filter(Objects::nonNull).collect(Collectors.toList());
            return list;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public Integer getContainerCountByState(HddsProtos.LifeCycleState state) {
        return this.containerStateManager.getContainerCountByState(state);
    }

    @Override
    public ContainerInfo getContainer(ContainerID containerID) throws ContainerNotFoundException {
        return this.containerStateManager.getContainer(containerID);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public boolean exists(ContainerID containerID) {
        this.lock.lock();
        try {
            boolean bl = this.containerStateManager.getContainer(containerID) != null;
            return bl;
        }
        catch (ContainerNotFoundException e) {
            boolean bl = false;
            return bl;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public List<ContainerInfo> listContainer(ContainerID startContainerID, int count) {
        this.lock.lock();
        try {
            this.scmContainerManagerMetrics.incNumListContainersOps();
            long startId = startContainerID == null ? 0L : startContainerID.getId();
            ArrayList<ContainerID> containersIds = new ArrayList<ContainerID>(this.containerStateManager.getAllContainerIDs());
            Collections.sort(containersIds);
            List<ContainerInfo> list = containersIds.stream().filter(id -> id.getId() > startId).limit(count).map(id -> {
                try {
                    return this.containerStateManager.getContainer((ContainerID)id);
                }
                catch (ContainerNotFoundException ex) {
                    LOG.warn("Container Missing.", (Throwable)ex);
                    return null;
                }
            }).collect(Collectors.toList());
            return list;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public ContainerInfo allocateContainer(HddsProtos.ReplicationType type, HddsProtos.ReplicationFactor replicationFactor, String owner) throws IOException {
        try {
            this.lock.lock();
            ContainerInfo containerInfo = null;
            try {
                containerInfo = this.containerStateManager.allocateContainer(this.pipelineManager, type, replicationFactor, owner);
            }
            catch (IOException ex) {
                this.scmContainerManagerMetrics.incNumFailureCreateContainers();
                throw ex;
            }
            try {
                this.addContainerToDB(containerInfo);
            }
            catch (IOException ex) {
                this.pipelineManager.removeContainerFromPipeline(containerInfo.getPipelineID(), new ContainerID(containerInfo.getContainerID()));
                throw ex;
            }
            ContainerInfo containerInfo2 = containerInfo;
            return containerInfo2;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public void deleteContainer(ContainerID containerID) throws IOException {
        this.lock.lock();
        try {
            this.containerStateManager.removeContainer(containerID);
            byte[] dbKey = Longs.toByteArray((long)containerID.getId());
            byte[] containerBytes = this.containerStore.get(dbKey);
            if (containerBytes != null) {
                this.containerStore.delete(dbKey);
            } else {
                LOG.warn("Unable to remove the container {} from container store, it's missing!", (Object)containerID);
            }
            this.scmContainerManagerMetrics.incNumSuccessfulDeleteContainers();
        }
        catch (ContainerNotFoundException cnfe) {
            this.scmContainerManagerMetrics.incNumFailureDeleteContainers();
            throw new SCMException("Failed to delete container " + containerID + ", reason : container doesn't exist.", SCMException.ResultCodes.FAILED_TO_FIND_CONTAINER);
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public HddsProtos.LifeCycleState updateContainerState(ContainerID containerID, HddsProtos.LifeCycleEvent event) throws IOException {
        return this.updateContainerState(containerID, event, false);
    }

    private HddsProtos.LifeCycleState updateContainerState(ContainerID containerID, HddsProtos.LifeCycleEvent event, boolean skipPipelineToContainerRemove) throws IOException {
        this.lock.lock();
        try {
            ContainerInfo container = this.containerStateManager.getContainer(containerID);
            HddsProtos.LifeCycleState oldState = container.getState();
            this.containerStateManager.updateContainerState(containerID, event);
            HddsProtos.LifeCycleState newState = container.getState();
            if (!skipPipelineToContainerRemove && oldState == HddsProtos.LifeCycleState.OPEN && newState != HddsProtos.LifeCycleState.OPEN) {
                this.pipelineManager.removeContainerFromPipeline(container.getPipelineID(), containerID);
            }
            byte[] dbKey = Longs.toByteArray((long)containerID.getId());
            this.containerStore.put(dbKey, container.getProtobuf().toByteArray());
            HddsProtos.LifeCycleState lifeCycleState = newState;
            return lifeCycleState;
        }
        catch (ContainerNotFoundException cnfe) {
            throw new SCMException("Failed to update container state" + containerID + ", reason : container doesn't exist.", SCMException.ResultCodes.FAILED_TO_FIND_CONTAINER);
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void updateDeleteTransactionId(Map<Long, Long> deleteTransactionMap) throws IOException {
        if (deleteTransactionMap == null) {
            return;
        }
        this.lock.lock();
        try {
            BatchOperation batch = new BatchOperation();
            for (Map.Entry<Long, Long> entry : deleteTransactionMap.entrySet()) {
                long containerID = entry.getKey();
                byte[] dbKey = Longs.toByteArray((long)containerID);
                byte[] containerBytes = this.containerStore.get(dbKey);
                if (containerBytes == null) {
                    throw new SCMException("Failed to increment number of deleted blocks for container " + containerID + ", reason : container doesn't exist.", SCMException.ResultCodes.FAILED_TO_FIND_CONTAINER);
                }
                ContainerInfo containerInfo = ContainerInfo.fromProtobuf((HddsProtos.ContainerInfoProto)HddsProtos.ContainerInfoProto.parseFrom((byte[])containerBytes));
                containerInfo.updateDeleteTransactionId(entry.getValue().longValue());
                batch.put(dbKey, containerInfo.getProtobuf().toByteArray());
            }
            this.containerStore.writeBatch(batch);
            this.containerStateManager.updateDeleteTransactionId(deleteTransactionMap);
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public ContainerInfo getMatchingContainer(long sizeRequired, String owner, Pipeline pipeline) {
        return this.getMatchingContainer(sizeRequired, owner, pipeline, Collections.emptyList());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public ContainerInfo getMatchingContainer(long sizeRequired, String owner, Pipeline pipeline, List<ContainerID> excludedContainers) {
        try {
            NavigableSet<ContainerID> containerIDs;
            Pipeline pipeline2 = pipeline;
            synchronized (pipeline2) {
                containerIDs = this.getContainersForOwner(pipeline, owner);
                if (containerIDs.size() < this.numContainerPerOwnerInPipeline) {
                    ContainerInfo containerInfo = this.containerStateManager.allocateContainer(this.pipelineManager, owner, pipeline);
                    this.addContainerToDB(containerInfo);
                    this.containerStateManager.updateLastUsedMap(pipeline.getId(), containerInfo.containerID(), owner);
                    return containerInfo;
                }
            }
            containerIDs.removeAll(excludedContainers);
            ContainerInfo containerInfo = this.containerStateManager.getMatchingContainer(sizeRequired, owner, pipeline.getId(), containerIDs);
            if (containerInfo == null) {
                Pipeline pipeline3 = pipeline;
                synchronized (pipeline3) {
                    containerInfo = this.containerStateManager.allocateContainer(this.pipelineManager, owner, pipeline);
                    this.addContainerToDB(containerInfo);
                }
            }
            this.containerStateManager.updateLastUsedMap(pipeline.getId(), containerInfo.containerID(), owner);
            return containerInfo;
        }
        catch (Exception e) {
            LOG.warn("Container allocation failed for pipeline={} requiredSize={} {}", new Object[]{pipeline, sizeRequired, e});
            return null;
        }
    }

    protected void addContainerToDB(ContainerInfo containerInfo) throws IOException {
        try {
            byte[] containerIDBytes = Longs.toByteArray((long)containerInfo.getContainerID());
            this.containerStore.put(containerIDBytes, containerInfo.getProtobuf().toByteArray());
            this.scmContainerManagerMetrics.incNumSuccessfulCreateContainers();
        }
        catch (IOException ex) {
            this.scmContainerManagerMetrics.incNumFailureCreateContainers();
            LOG.error("Add Container to DB failed for ContainerID #{}", (Object)containerInfo.getContainerID());
            try {
                this.containerStateManager.removeContainer(containerInfo.containerID());
            }
            catch (ContainerNotFoundException containerNotFoundException) {
                // empty catch block
            }
            throw ex;
        }
    }

    private NavigableSet<ContainerID> getContainersForOwner(Pipeline pipeline, String owner) throws IOException {
        NavigableSet<ContainerID> containerIDs = this.pipelineManager.getContainersInPipeline(pipeline.getId());
        Iterator<ContainerID> containerIDIterator = containerIDs.iterator();
        while (containerIDIterator.hasNext()) {
            ContainerID cid = containerIDIterator.next();
            try {
                if (this.getContainer(cid).getOwner().equals(owner)) continue;
                containerIDIterator.remove();
            }
            catch (ContainerNotFoundException e) {
                LOG.error("Could not find container info for container id={} {}", (Object)cid, (Object)e);
                containerIDIterator.remove();
            }
        }
        return containerIDs;
    }

    @Override
    public Set<ContainerReplica> getContainerReplicas(ContainerID containerID) throws ContainerNotFoundException {
        return this.containerStateManager.getContainerReplicas(containerID);
    }

    @Override
    public void updateContainerReplica(ContainerID containerID, ContainerReplica replica) throws ContainerNotFoundException {
        this.containerStateManager.updateContainerReplica(containerID, replica);
    }

    @Override
    public void removeContainerReplica(ContainerID containerID, ContainerReplica replica) throws ContainerNotFoundException, ContainerReplicaNotFoundException {
        this.containerStateManager.removeContainerReplica(containerID, replica);
    }

    @Override
    public void close() throws IOException {
        if (this.containerStateManager != null) {
            this.containerStateManager.close();
        }
        if (this.containerStore != null) {
            this.containerStore.close();
        }
        if (this.scmContainerManagerMetrics != null) {
            this.scmContainerManagerMetrics.unRegister();
        }
    }

    @Override
    public void notifyContainerReportProcessing(boolean isFullReport, boolean success) {
        if (isFullReport) {
            if (success) {
                this.scmContainerManagerMetrics.incNumContainerReportsProcessedSuccessful();
            } else {
                this.scmContainerManagerMetrics.incNumContainerReportsProcessedFailed();
            }
        } else if (success) {
            this.scmContainerManagerMetrics.incNumICRReportsProcessedSuccessful();
        } else {
            this.scmContainerManagerMetrics.incNumICRReportsProcessedFailed();
        }
    }

    protected File getContainerDBPath(Configuration conf) {
        File metaDir = ServerUtils.getScmDbDir((Configuration)conf);
        return new File(metaDir, "scm-container.db");
    }

    protected PipelineManager getPipelineManager() {
        return this.pipelineManager;
    }

    public Lock getLock() {
        return this.lock;
    }
}

