/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.node;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.Closeable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.scm.container.ContainerID;
import org.apache.hadoop.hdds.scm.events.SCMEvents;
import org.apache.hadoop.hdds.scm.node.DatanodeInfo;
import org.apache.hadoop.hdds.scm.node.states.Node2PipelineMap;
import org.apache.hadoop.hdds.scm.node.states.NodeAlreadyExistsException;
import org.apache.hadoop.hdds.scm.node.states.NodeNotFoundException;
import org.apache.hadoop.hdds.scm.node.states.NodeStateMap;
import org.apache.hadoop.hdds.scm.pipeline.Pipeline;
import org.apache.hadoop.hdds.scm.pipeline.PipelineID;
import org.apache.hadoop.hdds.server.events.Event;
import org.apache.hadoop.hdds.server.events.EventPublisher;
import org.apache.hadoop.hdds.utils.HddsServerUtil;
import org.apache.hadoop.ozone.common.statemachine.InvalidStateTransitionException;
import org.apache.hadoop.ozone.common.statemachine.StateMachine;
import org.apache.hadoop.util.Time;
import org.apache.hadoop.util.concurrent.HadoopExecutors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NodeStateManager
implements Runnable,
Closeable {
    private static final Logger LOG = LoggerFactory.getLogger(NodeStateManager.class);
    private final StateMachine<HddsProtos.NodeState, NodeLifeCycleEvent> stateMachine;
    private final NodeStateMap nodeStateMap = new NodeStateMap();
    private final Node2PipelineMap node2PipelineMap = new Node2PipelineMap();
    private final EventPublisher eventPublisher;
    private final Map<HddsProtos.NodeState, Event<DatanodeDetails>> state2EventMap;
    private final ScheduledExecutorService executorService;
    private final long heartbeatCheckerIntervalMs;
    private final long staleNodeIntervalMs;
    private final long deadNodeIntervalMs;
    private ScheduledFuture<?> healthCheckFuture;
    private boolean checkPaused;
    private long lastHealthCheck;
    private long skippedHealthChecks;

    public NodeStateManager(Configuration conf, EventPublisher eventPublisher) {
        this.eventPublisher = eventPublisher;
        this.state2EventMap = new HashMap<HddsProtos.NodeState, Event<DatanodeDetails>>();
        this.initialiseState2EventMap();
        HashSet<HddsProtos.NodeState> finalStates = new HashSet<HddsProtos.NodeState>();
        finalStates.add(HddsProtos.NodeState.DECOMMISSIONED);
        this.stateMachine = new StateMachine((Enum)HddsProtos.NodeState.HEALTHY, finalStates);
        this.initializeStateMachine();
        this.heartbeatCheckerIntervalMs = HddsServerUtil.getScmheartbeatCheckerInterval((Configuration)conf);
        this.staleNodeIntervalMs = HddsServerUtil.getStaleNodeInterval((Configuration)conf);
        this.deadNodeIntervalMs = HddsServerUtil.getDeadNodeInterval((Configuration)conf);
        Preconditions.checkState((this.heartbeatCheckerIntervalMs > 0L ? 1 : 0) != 0, (Object)"ozone.scm.heartbeat.thread.interval should be greater than 0.");
        Preconditions.checkState((this.staleNodeIntervalMs < this.deadNodeIntervalMs ? 1 : 0) != 0, (Object)"ozone.scm.stale.node.interval should be less thanozone.scm.dead.node.interval");
        this.executorService = HadoopExecutors.newScheduledThreadPool((int)1, (ThreadFactory)new ThreadFactoryBuilder().setDaemon(true).setNameFormat("SCM Heartbeat Processing Thread - %d").build());
        this.skippedHealthChecks = 0L;
        this.checkPaused = false;
        this.scheduleNextHealthCheck();
    }

    private void initialiseState2EventMap() {
        this.state2EventMap.put(HddsProtos.NodeState.STALE, (Event<DatanodeDetails>)SCMEvents.STALE_NODE);
        this.state2EventMap.put(HddsProtos.NodeState.DEAD, (Event<DatanodeDetails>)SCMEvents.DEAD_NODE);
        this.state2EventMap.put(HddsProtos.NodeState.HEALTHY, (Event<DatanodeDetails>)SCMEvents.NON_HEALTHY_TO_HEALTHY_NODE);
    }

    private void initializeStateMachine() {
        this.stateMachine.addTransition((Enum)HddsProtos.NodeState.HEALTHY, (Enum)HddsProtos.NodeState.STALE, (Enum)NodeLifeCycleEvent.TIMEOUT);
        this.stateMachine.addTransition((Enum)HddsProtos.NodeState.STALE, (Enum)HddsProtos.NodeState.DEAD, (Enum)NodeLifeCycleEvent.TIMEOUT);
        this.stateMachine.addTransition((Enum)HddsProtos.NodeState.STALE, (Enum)HddsProtos.NodeState.HEALTHY, (Enum)NodeLifeCycleEvent.RESTORE);
        this.stateMachine.addTransition((Enum)HddsProtos.NodeState.DEAD, (Enum)HddsProtos.NodeState.HEALTHY, (Enum)NodeLifeCycleEvent.RESURRECT);
        this.stateMachine.addTransition((Enum)HddsProtos.NodeState.HEALTHY, (Enum)HddsProtos.NodeState.DECOMMISSIONING, (Enum)NodeLifeCycleEvent.DECOMMISSION);
        this.stateMachine.addTransition((Enum)HddsProtos.NodeState.STALE, (Enum)HddsProtos.NodeState.DECOMMISSIONING, (Enum)NodeLifeCycleEvent.DECOMMISSION);
        this.stateMachine.addTransition((Enum)HddsProtos.NodeState.DEAD, (Enum)HddsProtos.NodeState.DECOMMISSIONING, (Enum)NodeLifeCycleEvent.DECOMMISSION);
        this.stateMachine.addTransition((Enum)HddsProtos.NodeState.DECOMMISSIONING, (Enum)HddsProtos.NodeState.DECOMMISSIONED, (Enum)NodeLifeCycleEvent.DECOMMISSIONED);
    }

    public void addNode(DatanodeDetails datanodeDetails) throws NodeAlreadyExistsException {
        this.nodeStateMap.addNode(datanodeDetails, (HddsProtos.NodeState)this.stateMachine.getInitialState());
        this.eventPublisher.fireEvent(SCMEvents.NEW_NODE, (Object)datanodeDetails);
    }

    public void addPipeline(Pipeline pipeline) {
        this.node2PipelineMap.addPipeline(pipeline);
    }

    public int getPipelinesCount(DatanodeDetails datanodeDetails) {
        return this.node2PipelineMap.getPipelinesCount(datanodeDetails.getUuid());
    }

    public DatanodeInfo getNode(DatanodeDetails datanodeDetails) throws NodeNotFoundException {
        return this.nodeStateMap.getNodeInfo(datanodeDetails.getUuid());
    }

    public void updateLastHeartbeatTime(DatanodeDetails datanodeDetails) throws NodeNotFoundException {
        this.nodeStateMap.getNodeInfo(datanodeDetails.getUuid()).updateLastHeartbeatTime();
    }

    public HddsProtos.NodeState getNodeState(DatanodeDetails datanodeDetails) throws NodeNotFoundException {
        return this.nodeStateMap.getNodeState(datanodeDetails.getUuid());
    }

    public List<DatanodeInfo> getHealthyNodes() {
        return this.getNodes(HddsProtos.NodeState.HEALTHY);
    }

    public List<DatanodeInfo> getStaleNodes() {
        return this.getNodes(HddsProtos.NodeState.STALE);
    }

    public List<DatanodeInfo> getDeadNodes() {
        return this.getNodes(HddsProtos.NodeState.DEAD);
    }

    public List<DatanodeInfo> getNodes(HddsProtos.NodeState state) {
        ArrayList<DatanodeInfo> nodes = new ArrayList<DatanodeInfo>();
        this.nodeStateMap.getNodes(state).forEach(uuid -> {
            try {
                nodes.add(this.nodeStateMap.getNodeInfo((UUID)uuid));
            }
            catch (NodeNotFoundException e) {
                LOG.error("Inconsistent NodeStateMap! {}", (Object)this.nodeStateMap);
            }
        });
        return nodes;
    }

    public List<DatanodeInfo> getAllNodes() {
        ArrayList<DatanodeInfo> nodes = new ArrayList<DatanodeInfo>();
        this.nodeStateMap.getAllNodes().forEach(uuid -> {
            try {
                nodes.add(this.nodeStateMap.getNodeInfo((UUID)uuid));
            }
            catch (NodeNotFoundException e) {
                LOG.error("Inconsistent NodeStateMap! {}", (Object)this.nodeStateMap);
            }
        });
        return nodes;
    }

    public Set<PipelineID> getPipelineByDnID(UUID dnId) {
        return this.node2PipelineMap.getPipelines(dnId);
    }

    public int getHealthyNodeCount() {
        return this.getNodeCount(HddsProtos.NodeState.HEALTHY);
    }

    public int getStaleNodeCount() {
        return this.getNodeCount(HddsProtos.NodeState.STALE);
    }

    public int getDeadNodeCount() {
        return this.getNodeCount(HddsProtos.NodeState.DEAD);
    }

    public int getNodeCount(HddsProtos.NodeState state) {
        return this.nodeStateMap.getNodeCount(state);
    }

    public int getTotalNodeCount() {
        return this.nodeStateMap.getTotalNodeCount();
    }

    public void removePipeline(Pipeline pipeline) {
        this.node2PipelineMap.removePipeline(pipeline);
    }

    public void addContainer(UUID uuid, ContainerID containerId) throws NodeNotFoundException {
        this.nodeStateMap.addContainer(uuid, containerId);
    }

    public void setContainers(UUID uuid, Set<ContainerID> containerIds) throws NodeNotFoundException {
        this.nodeStateMap.setContainers(uuid, containerIds);
    }

    public Set<ContainerID> getContainers(UUID uuid) throws NodeNotFoundException {
        return this.nodeStateMap.getContainers(uuid);
    }

    @Override
    public void run() {
        if (this.shouldSkipCheck()) {
            ++this.skippedHealthChecks;
            LOG.info("Detected long delay in scheduling HB processing thread. Skipping heartbeat checks for one iteration.");
        } else {
            this.checkNodesHealth();
        }
        this.scheduleNextHealthCheck();
    }

    private void checkNodesHealth() {
        long processingStartTime = Time.monotonicNow();
        long healthyNodeDeadline = processingStartTime - this.staleNodeIntervalMs;
        long staleNodeDeadline = processingStartTime - this.deadNodeIntervalMs;
        Predicate<Long> healthyNodeCondition = lastHbTime -> lastHbTime >= healthyNodeDeadline;
        Predicate<Long> staleNodeCondition = lastHbTime -> lastHbTime < healthyNodeDeadline;
        Predicate<Long> deadNodeCondition = lastHbTime -> lastHbTime < staleNodeDeadline;
        try {
            for (HddsProtos.NodeState state : HddsProtos.NodeState.values()) {
                List<UUID> nodes = this.nodeStateMap.getNodes(state);
                for (UUID id : nodes) {
                    DatanodeInfo node = this.nodeStateMap.getNodeInfo(id);
                    switch (state) {
                        case HEALTHY: {
                            this.updateNodeState(node, staleNodeCondition, state, NodeLifeCycleEvent.TIMEOUT);
                            break;
                        }
                        case STALE: {
                            this.updateNodeState(node, deadNodeCondition, state, NodeLifeCycleEvent.TIMEOUT);
                            this.updateNodeState(node, healthyNodeCondition, state, NodeLifeCycleEvent.RESTORE);
                            break;
                        }
                        case DEAD: {
                            this.updateNodeState(node, healthyNodeCondition, state, NodeLifeCycleEvent.RESURRECT);
                            break;
                        }
                    }
                }
            }
        }
        catch (NodeNotFoundException e) {
            LOG.error("Inconsistent NodeStateMap! {}", (Object)this.nodeStateMap);
        }
        long processingEndTime = Time.monotonicNow();
        if (processingEndTime - processingStartTime > this.heartbeatCheckerIntervalMs) {
            LOG.error("Total time spend processing datanode HB's is greater than configured values for datanode heartbeats. Please adjust the heartbeat configs. Time Spend on HB processing: {} seconds Datanode heartbeat Interval: {} seconds.", (Object)TimeUnit.MILLISECONDS.toSeconds(processingEndTime - processingStartTime), (Object)this.heartbeatCheckerIntervalMs);
        }
    }

    private void scheduleNextHealthCheck() {
        if (!Thread.currentThread().isInterrupted() && !this.executorService.isShutdown()) {
            this.healthCheckFuture = this.executorService.schedule(this, this.heartbeatCheckerIntervalMs, TimeUnit.MILLISECONDS);
        } else {
            LOG.warn("Current Thread is interrupted, shutting down HB processing thread for Node Manager.");
        }
        this.lastHealthCheck = Time.monotonicNow();
    }

    private boolean shouldSkipCheck() {
        long minInterval;
        long currentTime = Time.monotonicNow();
        return currentTime - this.lastHealthCheck >= (minInterval = Math.min(this.staleNodeIntervalMs, this.deadNodeIntervalMs));
    }

    private void updateNodeState(DatanodeInfo node, Predicate<Long> condition, HddsProtos.NodeState state, NodeLifeCycleEvent lifeCycleEvent) throws NodeNotFoundException {
        try {
            if (condition.test(node.getLastHeartbeatTime())) {
                HddsProtos.NodeState newState = (HddsProtos.NodeState)this.stateMachine.getNextState((Enum)state, (Enum)lifeCycleEvent);
                this.nodeStateMap.updateNodeState(node.getUuid(), state, newState);
                if (this.state2EventMap.containsKey(newState)) {
                    this.eventPublisher.fireEvent(this.state2EventMap.get(newState), (Object)node);
                }
            }
        }
        catch (InvalidStateTransitionException e) {
            LOG.warn("Invalid state transition of node {}. Current state: {}, life cycle event: {}", new Object[]{node, state, lifeCycleEvent});
        }
    }

    @Override
    public void close() {
        this.executorService.shutdown();
        try {
            if (!this.executorService.awaitTermination(5L, TimeUnit.SECONDS)) {
                this.executorService.shutdownNow();
            }
            if (!this.executorService.awaitTermination(5L, TimeUnit.SECONDS)) {
                LOG.error("Unable to shutdown NodeStateManager properly.");
            }
        }
        catch (InterruptedException e) {
            this.executorService.shutdownNow();
            Thread.currentThread().interrupt();
        }
    }

    @VisibleForTesting
    long getSkippedHealthChecks() {
        return this.skippedHealthChecks;
    }

    @VisibleForTesting
    ScheduledFuture pause() {
        if (this.executorService.isShutdown() || this.checkPaused) {
            return null;
        }
        this.checkPaused = this.healthCheckFuture.cancel(false);
        return this.healthCheckFuture;
    }

    @VisibleForTesting
    ScheduledFuture unpause() {
        if (this.executorService.isShutdown()) {
            return null;
        }
        if (this.checkPaused) {
            Preconditions.checkState((this.healthCheckFuture == null || this.healthCheckFuture.isCancelled() || this.healthCheckFuture.isDone() ? 1 : 0) != 0);
            this.checkPaused = false;
            this.healthCheckFuture = this.executorService.schedule(this, this.heartbeatCheckerIntervalMs, TimeUnit.MILLISECONDS);
        }
        return this.healthCheckFuture;
    }

    private static enum NodeLifeCycleEvent {
        TIMEOUT,
        RESTORE,
        RESURRECT,
        DECOMMISSION,
        DECOMMISSIONED;

    }
}

