/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.node;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import java.io.IOException;
import java.net.InetAddress;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledFuture;
import java.util.stream.Collectors;
import javax.management.ObjectName;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdds.conf.OzoneConfiguration;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.protocol.proto.StorageContainerDatanodeProtocolProtos;
import org.apache.hadoop.hdds.scm.VersionInfo;
import org.apache.hadoop.hdds.scm.container.ContainerID;
import org.apache.hadoop.hdds.scm.container.placement.metrics.SCMNodeMetric;
import org.apache.hadoop.hdds.scm.container.placement.metrics.SCMNodeStat;
import org.apache.hadoop.hdds.scm.net.NetworkTopology;
import org.apache.hadoop.hdds.scm.net.Node;
import org.apache.hadoop.hdds.scm.node.CommandQueue;
import org.apache.hadoop.hdds.scm.node.DatanodeInfo;
import org.apache.hadoop.hdds.scm.node.NodeManager;
import org.apache.hadoop.hdds.scm.node.NodeStateManager;
import org.apache.hadoop.hdds.scm.node.SCMNodeMetrics;
import org.apache.hadoop.hdds.scm.node.states.NodeAlreadyExistsException;
import org.apache.hadoop.hdds.scm.node.states.NodeNotFoundException;
import org.apache.hadoop.hdds.scm.pipeline.Pipeline;
import org.apache.hadoop.hdds.scm.pipeline.PipelineID;
import org.apache.hadoop.hdds.scm.server.SCMStorageConfig;
import org.apache.hadoop.hdds.server.events.EventPublisher;
import org.apache.hadoop.ipc.Server;
import org.apache.hadoop.metrics2.util.MBeans;
import org.apache.hadoop.net.CachedDNSToSwitchMapping;
import org.apache.hadoop.net.DNSToSwitchMapping;
import org.apache.hadoop.net.TableMapping;
import org.apache.hadoop.ozone.protocol.VersionResponse;
import org.apache.hadoop.ozone.protocol.commands.CommandForDatanode;
import org.apache.hadoop.ozone.protocol.commands.RegisteredCommand;
import org.apache.hadoop.ozone.protocol.commands.SCMCommand;
import org.apache.hadoop.util.ReflectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SCMNodeManager
implements NodeManager {
    private static final Logger LOG = LoggerFactory.getLogger(SCMNodeManager.class);
    private final NodeStateManager nodeStateManager;
    private final VersionInfo version;
    private final CommandQueue commandQueue;
    private final SCMNodeMetrics metrics;
    private ObjectName nmInfoBean;
    private final SCMStorageConfig scmStorageConfig;
    private final NetworkTopology clusterMap;
    private final DNSToSwitchMapping dnsToSwitchMapping;
    private final boolean useHostname;
    private final ConcurrentHashMap<String, Set<String>> dnsToUuidMap = new ConcurrentHashMap();

    public SCMNodeManager(OzoneConfiguration conf, SCMStorageConfig scmStorageConfig, EventPublisher eventPublisher, NetworkTopology networkTopology) {
        this.nodeStateManager = new NodeStateManager((Configuration)conf, eventPublisher);
        this.version = VersionInfo.getLatestVersion();
        this.commandQueue = new CommandQueue();
        this.scmStorageConfig = scmStorageConfig;
        LOG.info("Entering startup safe mode.");
        this.registerMXBean();
        this.metrics = SCMNodeMetrics.create(this);
        this.clusterMap = networkTopology;
        Class dnsToSwitchMappingClass = conf.getClass("net.topology.node.switch.mapping.impl", TableMapping.class, DNSToSwitchMapping.class);
        DNSToSwitchMapping newInstance = (DNSToSwitchMapping)ReflectionUtils.newInstance((Class)dnsToSwitchMappingClass, (Configuration)conf);
        this.dnsToSwitchMapping = newInstance instanceof CachedDNSToSwitchMapping ? newInstance : new CachedDNSToSwitchMapping(newInstance);
        this.useHostname = conf.getBoolean("dfs.datanode.use.datanode.hostname", false);
    }

    private void registerMXBean() {
        this.nmInfoBean = MBeans.register((String)"SCMNodeManager", (String)"SCMNodeManagerInfo", (Object)this);
    }

    private void unregisterMXBean() {
        if (this.nmInfoBean != null) {
            MBeans.unregister((ObjectName)this.nmInfoBean);
            this.nmInfoBean = null;
        }
    }

    @Override
    public List<DatanodeDetails> getNodes(HddsProtos.NodeState nodestate) {
        return this.nodeStateManager.getNodes(nodestate).stream().map(node -> node).collect(Collectors.toList());
    }

    @Override
    public List<DatanodeDetails> getAllNodes() {
        return this.nodeStateManager.getAllNodes().stream().map(node -> node).collect(Collectors.toList());
    }

    @Override
    public int getNodeCount(HddsProtos.NodeState nodestate) {
        return this.nodeStateManager.getNodeCount(nodestate);
    }

    @Override
    public HddsProtos.NodeState getNodeState(DatanodeDetails datanodeDetails) {
        try {
            return this.nodeStateManager.getNodeState(datanodeDetails);
        }
        catch (NodeNotFoundException e) {
            return null;
        }
    }

    @Override
    public void close() throws IOException {
        this.unregisterMXBean();
        this.metrics.unRegister();
        this.nodeStateManager.close();
    }

    public VersionResponse getVersion(StorageContainerDatanodeProtocolProtos.SCMVersionRequestProto versionRequest) {
        return VersionResponse.newBuilder().setVersion(this.version.getVersion()).addValue("scmUuid", this.scmStorageConfig.getScmId()).addValue("clusterID", this.scmStorageConfig.getClusterID()).build();
    }

    public RegisteredCommand register(DatanodeDetails datanodeDetails, StorageContainerDatanodeProtocolProtos.NodeReportProto nodeReport, StorageContainerDatanodeProtocolProtos.PipelineReportsProto pipelineReportsProto) {
        block4: {
            InetAddress dnAddress = Server.getRemoteIp();
            if (dnAddress != null) {
                datanodeDetails.setHostName(dnAddress.getHostName());
                datanodeDetails.setIpAddress(dnAddress.getHostAddress());
            }
            try {
                datanodeDetails.setNetworkName(datanodeDetails.getUuidString());
                String dnsName = this.useHostname ? datanodeDetails.getHostName() : datanodeDetails.getIpAddress();
                String networkLocation = this.nodeResolve(dnsName);
                if (networkLocation != null) {
                    datanodeDetails.setNetworkLocation(networkLocation);
                }
                this.nodeStateManager.addNode(datanodeDetails);
                this.clusterMap.add((Node)datanodeDetails);
                this.addEntryTodnsToUuidMap(dnsName, datanodeDetails.getUuidString());
                this.processNodeReport(datanodeDetails, nodeReport);
                LOG.info("Registered Data node : {}", (Object)datanodeDetails);
            }
            catch (NodeAlreadyExistsException e) {
                if (!LOG.isTraceEnabled()) break block4;
                LOG.trace("Datanode is already registered. Datanode: {}", (Object)datanodeDetails.toString());
            }
        }
        return RegisteredCommand.newBuilder().setErrorCode(StorageContainerDatanodeProtocolProtos.SCMRegisteredResponseProto.ErrorCode.success).setDatanode(datanodeDetails).setClusterID(this.scmStorageConfig.getClusterID()).build();
    }

    @SuppressFBWarnings(value={"AT_OPERATION_SEQUENCE_ON_CONCURRENT_ABSTRACTION"}, justification="The method is synchronized and this is the only place dnsToUuidMap is modified")
    private synchronized void addEntryTodnsToUuidMap(String dnsName, String uuid) {
        Set<String> dnList = this.dnsToUuidMap.get(dnsName);
        if (dnList == null) {
            dnList = ConcurrentHashMap.newKeySet();
            this.dnsToUuidMap.put(dnsName, dnList);
        }
        dnList.add(uuid);
    }

    public List<SCMCommand> processHeartbeat(DatanodeDetails datanodeDetails) {
        Preconditions.checkNotNull((Object)datanodeDetails, (Object)"Heartbeat is missing DatanodeDetails.");
        try {
            this.nodeStateManager.updateLastHeartbeatTime(datanodeDetails);
            this.metrics.incNumHBProcessed();
        }
        catch (NodeNotFoundException e) {
            this.metrics.incNumHBProcessingFailed();
            LOG.error("SCM trying to process heartbeat from an unregistered node {}. Ignoring the heartbeat.", (Object)datanodeDetails);
        }
        return this.commandQueue.getCommand(datanodeDetails.getUuid());
    }

    public Boolean isNodeRegistered(DatanodeDetails datanodeDetails) {
        try {
            this.nodeStateManager.getNode(datanodeDetails);
            return true;
        }
        catch (NodeNotFoundException e) {
            return false;
        }
    }

    @Override
    public void processNodeReport(DatanodeDetails datanodeDetails, StorageContainerDatanodeProtocolProtos.NodeReportProto nodeReport) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Processing node report from [datanode={}]", (Object)datanodeDetails.getHostName());
        }
        if (LOG.isTraceEnabled()) {
            LOG.trace("HB is received from [datanode={}]: <json>{}</json>", (Object)datanodeDetails.getHostName(), (Object)nodeReport.toString().replaceAll("\n", "\\\\n"));
        }
        try {
            DatanodeInfo datanodeInfo = this.nodeStateManager.getNode(datanodeDetails);
            if (nodeReport != null) {
                datanodeInfo.updateStorageReports(nodeReport.getStorageReportList());
                this.metrics.incNumNodeReportProcessed();
            }
        }
        catch (NodeNotFoundException e) {
            this.metrics.incNumNodeReportProcessingFailed();
            LOG.warn("Got node report from unregistered datanode {}", (Object)datanodeDetails);
        }
    }

    @Override
    public SCMNodeStat getStats() {
        long capacity = 0L;
        long used = 0L;
        long remaining = 0L;
        for (SCMNodeStat stat : this.getNodeStats().values()) {
            capacity += stat.getCapacity().get().longValue();
            used += stat.getScmUsed().get().longValue();
            remaining += stat.getRemaining().get().longValue();
        }
        return new SCMNodeStat(capacity, used, remaining);
    }

    @Override
    public Map<DatanodeDetails, SCMNodeStat> getNodeStats() {
        HashMap<DatanodeDetails, SCMNodeStat> nodeStats = new HashMap<DatanodeDetails, SCMNodeStat>();
        List<DatanodeInfo> healthyNodes = this.nodeStateManager.getNodes(HddsProtos.NodeState.HEALTHY);
        List<DatanodeInfo> staleNodes = this.nodeStateManager.getNodes(HddsProtos.NodeState.STALE);
        ArrayList<DatanodeInfo> datanodes = new ArrayList<DatanodeInfo>(healthyNodes);
        datanodes.addAll(staleNodes);
        for (DatanodeInfo dnInfo : datanodes) {
            SCMNodeStat nodeStat = this.getNodeStatInternal(dnInfo);
            if (nodeStat == null) continue;
            nodeStats.put(dnInfo, nodeStat);
        }
        return nodeStats;
    }

    @Override
    public SCMNodeMetric getNodeStat(DatanodeDetails datanodeDetails) {
        SCMNodeStat nodeStat = this.getNodeStatInternal(datanodeDetails);
        return nodeStat != null ? new SCMNodeMetric(nodeStat) : null;
    }

    private SCMNodeStat getNodeStatInternal(DatanodeDetails datanodeDetails) {
        try {
            long capacity = 0L;
            long used = 0L;
            long remaining = 0L;
            DatanodeInfo datanodeInfo = this.nodeStateManager.getNode(datanodeDetails);
            List<StorageContainerDatanodeProtocolProtos.StorageReportProto> storageReportProtos = datanodeInfo.getStorageReports();
            for (StorageContainerDatanodeProtocolProtos.StorageReportProto reportProto : storageReportProtos) {
                capacity += reportProto.getCapacity();
                used += reportProto.getScmUsed();
                remaining += reportProto.getRemaining();
            }
            return new SCMNodeStat(capacity, used, remaining);
        }
        catch (NodeNotFoundException e) {
            LOG.warn("Cannot generate NodeStat, datanode {} not found.", (Object)datanodeDetails.getUuid());
            return null;
        }
    }

    @Override
    public Map<String, Integer> getNodeCount() {
        HashMap<String, Integer> nodeCountMap = new HashMap<String, Integer>();
        for (HddsProtos.NodeState state : HddsProtos.NodeState.values()) {
            nodeCountMap.put(state.toString(), this.getNodeCount(state));
        }
        return nodeCountMap;
    }

    @Override
    public Map<String, Long> getNodeInfo() {
        long diskCapacity = 0L;
        long diskUsed = 0L;
        long diskRemaning = 0L;
        long ssdCapacity = 0L;
        long ssdUsed = 0L;
        long ssdRemaining = 0L;
        List<DatanodeInfo> healthyNodes = this.nodeStateManager.getNodes(HddsProtos.NodeState.HEALTHY);
        List<DatanodeInfo> staleNodes = this.nodeStateManager.getNodes(HddsProtos.NodeState.STALE);
        ArrayList<DatanodeInfo> datanodes = new ArrayList<DatanodeInfo>(healthyNodes);
        datanodes.addAll(staleNodes);
        for (DatanodeInfo dnInfo : datanodes) {
            List<StorageContainerDatanodeProtocolProtos.StorageReportProto> storageReportProtos = dnInfo.getStorageReports();
            for (StorageContainerDatanodeProtocolProtos.StorageReportProto reportProto : storageReportProtos) {
                if (reportProto.getStorageType() == StorageContainerDatanodeProtocolProtos.StorageTypeProto.DISK) {
                    diskCapacity += reportProto.getCapacity();
                    diskRemaning += reportProto.getRemaining();
                    diskUsed += reportProto.getScmUsed();
                    continue;
                }
                if (reportProto.getStorageType() != StorageContainerDatanodeProtocolProtos.StorageTypeProto.SSD) continue;
                ssdCapacity += reportProto.getCapacity();
                ssdRemaining += reportProto.getRemaining();
                ssdUsed += reportProto.getScmUsed();
            }
        }
        HashMap<String, Long> nodeInfo = new HashMap<String, Long>();
        nodeInfo.put("DISKCapacity", diskCapacity);
        nodeInfo.put("DISKUsed", diskUsed);
        nodeInfo.put("DISKRemaining", diskRemaning);
        nodeInfo.put("SSDCapacity", ssdCapacity);
        nodeInfo.put("SSDUsed", ssdUsed);
        nodeInfo.put("SSDRemaining", ssdRemaining);
        return nodeInfo;
    }

    @Override
    public Set<PipelineID> getPipelines(DatanodeDetails datanodeDetails) {
        return this.nodeStateManager.getPipelineByDnID(datanodeDetails.getUuid());
    }

    @Override
    public int getPipelinesCount(DatanodeDetails datanodeDetails) {
        return this.nodeStateManager.getPipelinesCount(datanodeDetails);
    }

    @Override
    public void addPipeline(Pipeline pipeline) {
        this.nodeStateManager.addPipeline(pipeline);
    }

    @Override
    public void removePipeline(Pipeline pipeline) {
        this.nodeStateManager.removePipeline(pipeline);
    }

    @Override
    public void addContainer(DatanodeDetails datanodeDetails, ContainerID containerId) throws NodeNotFoundException {
        this.nodeStateManager.addContainer(datanodeDetails.getUuid(), containerId);
    }

    @Override
    public void setContainers(DatanodeDetails datanodeDetails, Set<ContainerID> containerIds) throws NodeNotFoundException {
        this.nodeStateManager.setContainers(datanodeDetails.getUuid(), containerIds);
    }

    @Override
    public Set<ContainerID> getContainers(DatanodeDetails datanodeDetails) throws NodeNotFoundException {
        return this.nodeStateManager.getContainers(datanodeDetails.getUuid());
    }

    @Override
    public void addDatanodeCommand(UUID dnId, SCMCommand command) {
        this.commandQueue.addCommand(dnId, command);
    }

    public void onMessage(CommandForDatanode commandForDatanode, EventPublisher ignored) {
        this.addDatanodeCommand(commandForDatanode.getDatanodeId(), commandForDatanode.getCommand());
    }

    @Override
    public List<SCMCommand> getCommandQueue(UUID dnID) {
        return this.commandQueue.getCommand(dnID);
    }

    @Override
    public DatanodeDetails getNodeByUuid(String uuid) {
        if (Strings.isNullOrEmpty((String)uuid)) {
            LOG.warn("uuid is null");
            return null;
        }
        DatanodeDetails temp = DatanodeDetails.newBuilder().setUuid(uuid).build();
        try {
            return this.nodeStateManager.getNode(temp);
        }
        catch (NodeNotFoundException e) {
            LOG.warn("Cannot find node for uuid {}", (Object)uuid);
            return null;
        }
    }

    @Override
    public List<DatanodeDetails> getNodesByAddress(String address) {
        LinkedList<DatanodeDetails> results = new LinkedList<DatanodeDetails>();
        if (Strings.isNullOrEmpty((String)address)) {
            LOG.warn("address is null");
            return results;
        }
        Set<String> uuids = this.dnsToUuidMap.get(address);
        if (uuids == null) {
            LOG.warn("Cannot find node for address {}", (Object)address);
            return results;
        }
        for (String uuid : uuids) {
            DatanodeDetails temp = DatanodeDetails.newBuilder().setUuid(uuid).build();
            try {
                results.add(this.nodeStateManager.getNode(temp));
            }
            catch (NodeNotFoundException e) {
                LOG.warn("Cannot find node for uuid {}", (Object)uuid);
            }
        }
        return results;
    }

    @Override
    public NetworkTopology getClusterNetworkTopologyMap() {
        return this.clusterMap;
    }

    private String nodeResolve(String hostname) {
        ArrayList<String> hosts = new ArrayList<String>(1);
        hosts.add(hostname);
        List resolvedHosts = this.dnsToSwitchMapping.resolve(hosts);
        if (resolvedHosts != null && !resolvedHosts.isEmpty()) {
            String location = (String)resolvedHosts.get(0);
            if (LOG.isDebugEnabled()) {
                LOG.debug("Resolve datanode {} return location {}", (Object)hostname, (Object)location);
            }
            return location;
        }
        LOG.error("Node {} Resolution failed. Please make sure that DNS table mapping or configured mapping is functional.", (Object)hostname);
        return null;
    }

    @VisibleForTesting
    ScheduledFuture pauseHealthCheck() {
        return this.nodeStateManager.pause();
    }

    @VisibleForTesting
    ScheduledFuture unpauseHealthCheck() {
        return this.nodeStateManager.unpause();
    }

    @VisibleForTesting
    long getSkippedHealthChecks() {
        return this.nodeStateManager.getSkippedHealthChecks();
    }
}

