/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.pipeline;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.scm.SCMCommonPlacementPolicy;
import org.apache.hadoop.hdds.scm.container.placement.metrics.SCMNodeMetric;
import org.apache.hadoop.hdds.scm.exceptions.SCMException;
import org.apache.hadoop.hdds.scm.net.NetworkTopology;
import org.apache.hadoop.hdds.scm.net.Node;
import org.apache.hadoop.hdds.scm.node.NodeManager;
import org.apache.hadoop.hdds.scm.pipeline.Pipeline;
import org.apache.hadoop.hdds.scm.pipeline.PipelineID;
import org.apache.hadoop.hdds.scm.pipeline.PipelineNotFoundException;
import org.apache.hadoop.hdds.scm.pipeline.PipelineStateManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class PipelinePlacementPolicy
extends SCMCommonPlacementPolicy {
    @VisibleForTesting
    static final Logger LOG = LoggerFactory.getLogger(PipelinePlacementPolicy.class);
    private final NodeManager nodeManager;
    private final PipelineStateManager stateManager;
    private final Configuration conf;
    private final int heavyNodeCriteria;

    public PipelinePlacementPolicy(NodeManager nodeManager, PipelineStateManager stateManager, Configuration conf) {
        super(nodeManager, conf);
        this.nodeManager = nodeManager;
        this.conf = conf;
        this.stateManager = stateManager;
        this.heavyNodeCriteria = conf.getInt("ozone.datanode.pipeline.limit", 2);
    }

    @VisibleForTesting
    boolean meetCriteria(DatanodeDetails datanodeDetails, int nodesRequired) {
        boolean meet;
        if (this.heavyNodeCriteria == 0) {
            return true;
        }
        int pipelineNumDeductable = 0;
        Set<PipelineID> pipelines = this.nodeManager.getPipelines(datanodeDetails);
        for (PipelineID pid : pipelines) {
            Pipeline pipeline;
            try {
                pipeline = this.stateManager.getPipeline(pid);
            }
            catch (PipelineNotFoundException e) {
                LOG.error("Pipeline not found in pipeline state manager during pipeline creation. PipelineID: " + pid + " exception: " + e.getMessage());
                continue;
            }
            if (pipeline == null || pipeline.getFactor().getNumber() != nodesRequired || pipeline.getType() != HddsProtos.ReplicationType.RATIS || pipeline.getPipelineState() != Pipeline.PipelineState.CLOSED) continue;
            ++pipelineNumDeductable;
        }
        boolean bl = meet = this.nodeManager.getPipelinesCount(datanodeDetails) - pipelineNumDeductable < this.heavyNodeCriteria;
        if (!meet && LOG.isDebugEnabled()) {
            LOG.debug("Pipeline Placement: can't place more pipeline on heavy datanode\uff1a " + datanodeDetails.getUuid().toString() + " Heaviness: " + this.nodeManager.getPipelinesCount(datanodeDetails) + " limit: " + this.heavyNodeCriteria);
        }
        return meet;
    }

    List<DatanodeDetails> filterViableNodes(List<DatanodeDetails> excludedNodes, int nodesRequired) throws SCMException {
        int initialHealthyNodesCount;
        List<DatanodeDetails> healthyNodes = this.nodeManager.getNodes(HddsProtos.NodeState.HEALTHY);
        if (excludedNodes != null) {
            healthyNodes.removeAll(excludedNodes);
        }
        if ((initialHealthyNodesCount = healthyNodes.size()) < nodesRequired) {
            String msg = String.format("Pipeline creation failed due to no sufficient healthy datanodes. Required %d. Found %d.", nodesRequired, initialHealthyNodesCount);
            LOG.warn(msg);
            throw new SCMException(msg, SCMException.ResultCodes.FAILED_TO_FIND_SUITABLE_NODE);
        }
        List<DatanodeDetails> healthyList = healthyNodes.stream().filter(d -> this.meetCriteria((DatanodeDetails)d, nodesRequired)).collect(Collectors.toList());
        if (healthyList.size() < nodesRequired) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Unable to find enough nodes that meet the criteria that cannot engage in more than" + this.heavyNodeCriteria + " pipelines. Nodes required: " + nodesRequired + " Found:" + healthyList.size() + " healthy nodes count in NodeManager: " + initialHealthyNodesCount);
            }
            String msg = String.format("Pipeline creation failed because nodes are engaged in other pipelines and every node can only be engaged in max %d pipelines. Required %d. Found %d", this.heavyNodeCriteria, nodesRequired, healthyList.size());
            throw new SCMException(msg, SCMException.ResultCodes.FAILED_TO_FIND_SUITABLE_NODE);
        }
        return healthyList;
    }

    @Override
    public List<DatanodeDetails> chooseDatanodes(List<DatanodeDetails> excludedNodes, List<DatanodeDetails> favoredNodes, int nodesRequired, long sizeRequired) throws SCMException {
        List<DatanodeDetails> healthyNodes = this.filterViableNodes(excludedNodes, nodesRequired);
        if (this.checkAllNodesAreEqual(this.nodeManager.getClusterNetworkTopologyMap())) {
            return super.getResultSet(nodesRequired, healthyNodes);
        }
        return this.getResultSet(nodesRequired, healthyNodes);
    }

    DatanodeDetails fallBackPickNodes(List<DatanodeDetails> nodeSet, List<DatanodeDetails> excludedNodes) throws SCMException {
        DatanodeDetails node;
        if (excludedNodes == null || excludedNodes.isEmpty()) {
            node = this.chooseNode(nodeSet);
        } else {
            List<DatanodeDetails> inputNodes = nodeSet.stream().filter(p -> !excludedNodes.contains(p)).collect(Collectors.toList());
            node = this.chooseNode(inputNodes);
        }
        if (node == null) {
            String msg = String.format("Unable to find fall back node in pipeline allocation. nodeSet size: {}", nodeSet.size());
            LOG.warn(msg);
            throw new SCMException(msg, SCMException.ResultCodes.FAILED_TO_FIND_SUITABLE_NODE);
        }
        return node;
    }

    @Override
    public List<DatanodeDetails> getResultSet(int nodesRequired, List<DatanodeDetails> healthyNodes) throws SCMException {
        DatanodeDetails nextNode;
        if (nodesRequired != HddsProtos.ReplicationFactor.THREE.getNumber()) {
            throw new SCMException("Nodes required number is not supported: " + nodesRequired, SCMException.ResultCodes.INVALID_CAPACITY);
        }
        boolean rackAwareness = false;
        ArrayList<DatanodeDetails> results = new ArrayList<DatanodeDetails>(nodesRequired);
        ArrayList<DatanodeDetails> exclude = new ArrayList<DatanodeDetails>();
        DatanodeDetails anchor = this.chooseNode(healthyNodes);
        if (anchor == null) {
            LOG.warn("Unable to find healthy node for anchor(first) node.");
            throw new SCMException("Unable to find anchor node.", SCMException.ResultCodes.FAILED_TO_FIND_SUITABLE_NODE);
        }
        results.add(anchor);
        exclude.add(anchor);
        if (LOG.isDebugEnabled()) {
            LOG.debug("First node chosen: {}", (Object)anchor);
        }
        if ((nextNode = this.chooseNodeBasedOnRackAwareness(healthyNodes, exclude, this.nodeManager.getClusterNetworkTopologyMap(), anchor)) != null) {
            rackAwareness = true;
            results.add(nextNode);
            exclude.add(nextNode);
            if (LOG.isDebugEnabled()) {
                LOG.debug("Second node chosen: {}", (Object)nextNode);
            }
        } else if (LOG.isDebugEnabled()) {
            LOG.debug("Pipeline Placement: Unable to find 2nd node on different rack based on rack awareness.");
        }
        int nodesToFind = nodesRequired - results.size();
        for (int x = 0; x < nodesToFind; ++x) {
            DatanodeDetails pick;
            DatanodeDetails datanodeDetails = pick = rackAwareness ? this.chooseNodeFromNetworkTopology(this.nodeManager.getClusterNetworkTopologyMap(), anchor, exclude) : this.fallBackPickNodes(healthyNodes, exclude);
            if (pick == null) continue;
            results.add(pick);
            exclude.add(pick);
            if (!LOG.isDebugEnabled()) continue;
            LOG.debug("Remaining node chosen: {}", (Object)pick);
        }
        if (results.size() < nodesRequired) {
            LOG.warn("Unable to find the required number of healthy nodes that  meet the criteria. Required nodes: {}, Found nodes: {}", (Object)nodesRequired, (Object)results.size());
            throw new SCMException("Unable to find required number of nodes.", SCMException.ResultCodes.FAILED_TO_FIND_SUITABLE_NODE);
        }
        return results;
    }

    @Override
    public DatanodeDetails chooseNode(List<DatanodeDetails> healthyNodes) {
        DatanodeDetails datanodeDetails;
        int secondNodeNdx;
        if (healthyNodes == null || healthyNodes.isEmpty()) {
            return null;
        }
        int firstNodeNdx = this.getRand().nextInt(healthyNodes.size());
        if (firstNodeNdx == (secondNodeNdx = this.getRand().nextInt(healthyNodes.size()))) {
            datanodeDetails = healthyNodes.get(firstNodeNdx);
        } else {
            SCMNodeMetric secondNodeMetric;
            DatanodeDetails firstNodeDetails = healthyNodes.get(firstNodeNdx);
            DatanodeDetails secondNodeDetails = healthyNodes.get(secondNodeNdx);
            SCMNodeMetric firstNodeMetric = this.nodeManager.getNodeStat(firstNodeDetails);
            datanodeDetails = firstNodeMetric.isGreater((secondNodeMetric = this.nodeManager.getNodeStat(secondNodeDetails)).get()) ? firstNodeDetails : secondNodeDetails;
        }
        healthyNodes.remove(datanodeDetails);
        return datanodeDetails;
    }

    @VisibleForTesting
    protected DatanodeDetails chooseNodeBasedOnRackAwareness(List<DatanodeDetails> healthyNodes, List<DatanodeDetails> excludedNodes, NetworkTopology networkTopology, DatanodeDetails anchor) {
        Preconditions.checkArgument((networkTopology != null ? 1 : 0) != 0);
        if (this.checkAllNodesAreEqual(networkTopology)) {
            return null;
        }
        for (DatanodeDetails node : healthyNodes) {
            if (excludedNodes.contains(node) || anchor.getNetworkLocation().equals(node.getNetworkLocation())) continue;
            return node;
        }
        return null;
    }

    private boolean checkAllNodesAreEqual(NetworkTopology topology) {
        if (topology == null) {
            return true;
        }
        return topology.getNumOfNodes(topology.getMaxLevel() - 1) == 1;
    }

    @VisibleForTesting
    protected DatanodeDetails chooseNodeFromNetworkTopology(NetworkTopology networkTopology, DatanodeDetails anchor, List<DatanodeDetails> excludedNodes) {
        Preconditions.checkArgument((networkTopology != null ? 1 : 0) != 0);
        ArrayList<DatanodeDetails> excluded = new ArrayList<DatanodeDetails>();
        if (excludedNodes != null && excludedNodes.size() != 0) {
            excluded.addAll(excludedNodes);
        }
        Node pick = networkTopology.chooseRandom(anchor.getNetworkLocation(), excluded);
        DatanodeDetails pickedNode = (DatanodeDetails)pick;
        return pickedNode;
    }
}

