/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.federation.resolver;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.hadoop.HadoopIllegalArgumentException;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.DFSUtil;
import org.apache.hadoop.hdfs.DFSUtilClient;
import org.apache.hadoop.hdfs.server.federation.resolver.FileSubclusterResolver;
import org.apache.hadoop.hdfs.server.federation.resolver.PathLocation;
import org.apache.hadoop.hdfs.server.federation.resolver.RemoteLocation;
import org.apache.hadoop.hdfs.server.federation.resolver.order.DestinationOrder;
import org.apache.hadoop.hdfs.server.federation.router.FederationUtil;
import org.apache.hadoop.hdfs.server.federation.router.Router;
import org.apache.hadoop.hdfs.server.federation.store.MountTableStore;
import org.apache.hadoop.hdfs.server.federation.store.StateStoreCache;
import org.apache.hadoop.hdfs.server.federation.store.StateStoreService;
import org.apache.hadoop.hdfs.server.federation.store.StateStoreUnavailableException;
import org.apache.hadoop.hdfs.server.federation.store.protocol.GetMountTableEntriesRequest;
import org.apache.hadoop.hdfs.server.federation.store.protocol.GetMountTableEntriesResponse;
import org.apache.hadoop.hdfs.server.federation.store.records.MountTable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MountTableResolver
implements FileSubclusterResolver,
StateStoreCache {
    private static final Logger LOG = LoggerFactory.getLogger(MountTableResolver.class);
    private final Router router;
    private final StateStoreService stateStore;
    private MountTableStore mountTableStore;
    private boolean init = false;
    private final TreeMap<String, MountTable> tree = new TreeMap();
    private final Cache<String, PathLocation> locationCache;
    private String defaultNameService = "";
    private final ReadWriteLock readWriteLock = new ReentrantReadWriteLock();
    private final Lock readLock = this.readWriteLock.readLock();
    private final Lock writeLock = this.readWriteLock.writeLock();

    @VisibleForTesting
    public MountTableResolver(Configuration conf) {
        this(conf, (StateStoreService)null);
    }

    public MountTableResolver(Configuration conf, Router routerService) {
        this(conf, routerService, null);
    }

    public MountTableResolver(Configuration conf, StateStoreService store) {
        this(conf, null, store);
    }

    public MountTableResolver(Configuration conf, Router routerService, StateStoreService store) {
        this.router = routerService;
        this.stateStore = store != null ? store : (this.router != null ? this.router.getStateStore() : null);
        int maxCacheSize = conf.getInt("dfs.federation.router.mount-table.max-cache-size", 10000);
        this.locationCache = CacheBuilder.newBuilder().maximumSize((long)maxCacheSize).build();
        this.registerCacheExternal();
        this.initDefaultNameService(conf);
    }

    private void registerCacheExternal() {
        if (this.stateStore != null) {
            this.stateStore.registerCacheExternal(this);
        }
    }

    private void initDefaultNameService(Configuration conf) {
        try {
            this.defaultNameService = conf.get("dfs.federation.router.default.nameserviceId", DFSUtil.getNamenodeNameServiceId((Configuration)conf));
        }
        catch (HadoopIllegalArgumentException e) {
            LOG.error("Cannot find default name service, setting it to the first");
            Collection nsIds = DFSUtilClient.getNameServiceIds((Configuration)conf);
            this.defaultNameService = (String)nsIds.iterator().next();
            LOG.info("Default name service: {}", (Object)this.defaultNameService);
        }
    }

    protected Router getRouter() {
        return this.router;
    }

    protected MountTableStore getMountTableStore() throws IOException {
        if (this.mountTableStore == null) {
            this.mountTableStore = this.stateStore.getRegisteredRecordStore(MountTableStore.class);
            if (this.mountTableStore == null) {
                throw new IOException("State Store does not have an interface for " + MountTableStore.class);
            }
        }
        return this.mountTableStore;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void addEntry(MountTable entry) {
        this.writeLock.lock();
        try {
            String srcPath = entry.getSourcePath();
            this.tree.put(srcPath, entry);
            this.invalidateLocationCache(srcPath);
        }
        finally {
            this.writeLock.unlock();
        }
        this.init = true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void removeEntry(String srcPath) {
        this.writeLock.lock();
        try {
            this.tree.remove(srcPath);
            this.invalidateLocationCache(srcPath);
        }
        finally {
            this.writeLock.unlock();
        }
    }

    private void invalidateLocationCache(String path) {
        LOG.debug("Invalidating {} from {}", (Object)path, this.locationCache);
        if (this.locationCache.size() == 0L) {
            return;
        }
        ConcurrentMap map = this.locationCache.asMap();
        Set entries = map.entrySet();
        Iterator it = entries.iterator();
        while (it.hasNext()) {
            Map.Entry entry = it.next();
            PathLocation loc = (PathLocation)entry.getValue();
            String src = loc.getSourcePath();
            if (src != null) {
                if (!FederationUtil.isParentEntry(src, path)) continue;
                LOG.debug("Removing {}", (Object)src);
                it.remove();
                continue;
            }
            String dest = loc.getDefaultLocation().getDest();
            if (!dest.startsWith(path)) continue;
            LOG.debug("Removing default cache {}", (Object)dest);
            it.remove();
        }
        LOG.debug("Location cache after invalidation: {}", this.locationCache);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    public void refreshEntries(Collection<MountTable> entries) {
        this.writeLock.lock();
        try {
            String srcPath;
            ConcurrentHashMap<String, MountTable> newEntries = new ConcurrentHashMap<String, MountTable>();
            for (MountTable entry : entries) {
                String srcPath2 = entry.getSourcePath();
                newEntries.put(srcPath2, entry);
            }
            TreeSet oldEntries = new TreeSet(Collections.reverseOrder());
            for (MountTable entry : this.getTreeValues("/")) {
                srcPath = entry.getSourcePath();
                oldEntries.add(srcPath);
            }
            for (String srcPath2 : oldEntries) {
                if (newEntries.containsKey(srcPath2)) continue;
                this.tree.remove(srcPath2);
                this.invalidateLocationCache(srcPath2);
                LOG.info("Removed stale mount point {} from resolver", (Object)srcPath2);
            }
            for (MountTable entry : entries) {
                srcPath = entry.getSourcePath();
                if (!oldEntries.contains(srcPath)) {
                    this.tree.put(srcPath, entry);
                    this.invalidateLocationCache(srcPath);
                    LOG.info("Added new mount point {} to resolver", (Object)srcPath);
                    continue;
                }
                MountTable existingEntry = this.tree.get(srcPath);
                if (existingEntry == null || existingEntry.equals(entry)) continue;
                LOG.info("Entry has changed from \"{}\" to \"{}\"", (Object)existingEntry, (Object)entry);
                this.tree.put(srcPath, entry);
                this.invalidateLocationCache(srcPath);
                LOG.info("Updated mount point {} in resolver", (Object)srcPath);
            }
        }
        finally {
            this.writeLock.unlock();
        }
        this.init = true;
    }

    @Override
    public boolean loadCache(boolean force) {
        try {
            MountTableStore mountTable = this.getMountTableStore();
            mountTable.loadCache(force);
            GetMountTableEntriesRequest request = GetMountTableEntriesRequest.newInstance("/");
            GetMountTableEntriesResponse response = mountTable.getMountTableEntries(request);
            List<MountTable> records = response.getEntries();
            this.refreshEntries(records);
        }
        catch (IOException e) {
            LOG.error("Cannot fetch mount table entries from State Store", (Throwable)e);
            return false;
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void clear() {
        LOG.info("Clearing all mount location caches");
        this.writeLock.lock();
        try {
            this.locationCache.invalidateAll();
            this.tree.clear();
        }
        finally {
            this.writeLock.unlock();
        }
    }

    @Override
    public PathLocation getDestinationForPath(final String path) throws IOException {
        this.verifyMountTable();
        this.readLock.lock();
        try {
            Callable<PathLocation> meh = new Callable<PathLocation>(){

                @Override
                public PathLocation call() throws Exception {
                    return MountTableResolver.this.lookupLocation(path);
                }
            };
            PathLocation pathLocation = (PathLocation)this.locationCache.get((Object)path, (Callable)meh);
            return pathLocation;
        }
        catch (ExecutionException e) {
            throw new IOException(e);
        }
        finally {
            this.readLock.unlock();
        }
    }

    public PathLocation lookupLocation(String path) {
        PathLocation ret = null;
        MountTable entry = this.findDeepest(path);
        if (entry != null) {
            ret = MountTableResolver.buildLocation(path, entry);
        } else {
            RemoteLocation remoteLocation = new RemoteLocation(this.defaultNameService, path);
            List<RemoteLocation> locations = Collections.singletonList(remoteLocation);
            ret = new PathLocation(null, locations);
        }
        return ret;
    }

    public MountTable getMountPoint(String path) throws IOException {
        this.verifyMountTable();
        return this.findDeepest(path);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public List<String> getMountPoints(String path) throws IOException {
        this.verifyMountTable();
        TreeSet<String> children = new TreeSet<String>();
        this.readLock.lock();
        try {
            LinkedList<String> linkedList;
            String from = path;
            String to = path + '\uffff';
            SortedMap<String, MountTable> subMap = this.tree.subMap(from, to);
            boolean exists = false;
            Iterator<String> i$ = subMap.keySet().iterator();
            while (i$.hasNext()) {
                String subPath;
                String child = subPath = i$.next();
                if (!path.equals("/")) {
                    int ini = path.length();
                    child = subPath.substring(ini);
                }
                if (child.isEmpty()) {
                    exists = true;
                    continue;
                }
                if (!child.startsWith("/")) continue;
                exists = true;
                int fin = (child = child.substring(1)).indexOf("/");
                if (fin > -1) {
                    child = child.substring(0, fin);
                }
                if (child.isEmpty()) continue;
                children.add(child);
            }
            if (!exists) {
                linkedList = null;
                return linkedList;
            }
            linkedList = new LinkedList<String>(children);
            return linkedList;
        }
        finally {
            this.readLock.unlock();
        }
    }

    public List<MountTable> getMounts(String path) throws IOException {
        this.verifyMountTable();
        return this.getTreeValues(path, false);
    }

    private void verifyMountTable() throws StateStoreUnavailableException {
        if (!this.init) {
            throw new StateStoreUnavailableException("Mount Table not initialized");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public String toString() {
        this.readLock.lock();
        try {
            String string = this.tree.toString();
            return string;
        }
        finally {
            this.readLock.unlock();
        }
    }

    private static PathLocation buildLocation(String path, MountTable entry) {
        String srcPath = entry.getSourcePath();
        if (!path.startsWith(srcPath)) {
            LOG.error("Cannot build location, {} not a child of {}", (Object)path, (Object)srcPath);
            return null;
        }
        String remainingPath = path.substring(srcPath.length());
        if (remainingPath.startsWith("/")) {
            remainingPath = remainingPath.substring(1);
        }
        LinkedList<RemoteLocation> locations = new LinkedList<RemoteLocation>();
        for (RemoteLocation oneDst : entry.getDestinations()) {
            String nsId = oneDst.getNameserviceId();
            String dest = oneDst.getDest();
            String newPath = dest;
            if (!newPath.endsWith("/") && !remainingPath.isEmpty()) {
                newPath = newPath + "/";
            }
            newPath = newPath + remainingPath;
            RemoteLocation remoteLocation = new RemoteLocation(nsId, newPath);
            locations.add(remoteLocation);
        }
        DestinationOrder order = entry.getDestOrder();
        return new PathLocation(srcPath, locations, order);
    }

    @Override
    public String getDefaultNamespace() {
        return this.defaultNameService;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private MountTable findDeepest(String path) {
        this.readLock.lock();
        try {
            Map.Entry<String, MountTable> entry = this.tree.floorEntry(path);
            while (entry != null && !FederationUtil.isParentEntry(path, entry.getKey())) {
                entry = this.tree.lowerEntry(entry.getKey());
            }
            if (entry == null) {
                MountTable mountTable = null;
                return mountTable;
            }
            MountTable mountTable = entry.getValue();
            return mountTable;
        }
        finally {
            this.readLock.unlock();
        }
    }

    private List<MountTable> getTreeValues(String path) {
        return this.getTreeValues(path, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private List<MountTable> getTreeValues(String path, boolean reverse) {
        LinkedList<MountTable> ret = new LinkedList<MountTable>();
        this.readLock.lock();
        try {
            String from = path;
            String to = path + '\uffff';
            SortedMap<String, MountTable> subMap = this.tree.subMap(from, to);
            for (MountTable entry : subMap.values()) {
                if (!reverse) {
                    ret.add(entry);
                    continue;
                }
                ret.addFirst(entry);
            }
        }
        finally {
            this.readLock.unlock();
        }
        return ret;
    }

    protected long getCacheSize() {
        return this.locationCache.size();
    }
}

