/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.federation.router;

import com.google.common.collect.ArrayListMultimap;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.function.Consumer;
import java.util.function.Predicate;
import org.apache.hadoop.fs.QuotaUsage;
import org.apache.hadoop.fs.StorageType;
import org.apache.hadoop.hdfs.DFSUtil;
import org.apache.hadoop.hdfs.server.federation.resolver.RemoteLocation;
import org.apache.hadoop.hdfs.server.federation.router.RemoteMethod;
import org.apache.hadoop.hdfs.server.federation.router.RemoteParam;
import org.apache.hadoop.hdfs.server.federation.router.Router;
import org.apache.hadoop.hdfs.server.federation.router.RouterQuotaManager;
import org.apache.hadoop.hdfs.server.federation.router.RouterQuotaUsage;
import org.apache.hadoop.hdfs.server.federation.router.RouterRpcClient;
import org.apache.hadoop.hdfs.server.federation.router.RouterRpcServer;
import org.apache.hadoop.hdfs.server.namenode.NameNode;
import org.apache.hadoop.security.AccessControlException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Quota {
    private static final Logger LOG = LoggerFactory.getLogger(Quota.class);
    private final RouterRpcServer rpcServer;
    private final RouterRpcClient rpcClient;
    private final Router router;

    public Quota(Router router, RouterRpcServer server) {
        this.router = router;
        this.rpcServer = server;
        this.rpcClient = server.getRPCClient();
    }

    public void setQuota(String path, long namespaceQuota, long storagespaceQuota, StorageType type, boolean checkMountEntry) throws IOException {
        if (checkMountEntry && this.isMountEntry(path)) {
            throw new AccessControlException("Permission denied: " + RouterRpcServer.getRemoteUser() + " is not allowed to change quota of " + path);
        }
        this.setQuotaInternal(path, null, namespaceQuota, storagespaceQuota, type);
    }

    void setQuotaInternal(String path, List<RemoteLocation> locations, long namespaceQuota, long storagespaceQuota, StorageType type) throws IOException {
        this.rpcServer.checkOperation(NameNode.OperationCategory.WRITE);
        if (!this.router.isQuotaEnabled()) {
            throw new IOException("The quota system is disabled in Router.");
        }
        if (locations == null) {
            locations = this.getQuotaRemoteLocations(path);
        }
        if (LOG.isDebugEnabled()) {
            for (RemoteLocation loc : locations) {
                LOG.debug("Set quota for path: nsId: {}, dest: {}.", (Object)loc.getNameserviceId(), (Object)loc.getDest());
            }
        }
        RemoteMethod method = new RemoteMethod("setQuota", new Class[]{String.class, Long.TYPE, Long.TYPE, StorageType.class}, new RemoteParam(), namespaceQuota, storagespaceQuota, type);
        this.rpcClient.invokeConcurrent(locations, method, false, false);
    }

    public QuotaUsage getQuotaUsage(String path) throws IOException {
        return this.aggregateQuota(path, this.getEachQuotaUsage(path));
    }

    Map<RemoteLocation, QuotaUsage> getEachQuotaUsage(String path) throws IOException {
        this.rpcServer.checkOperation(NameNode.OperationCategory.READ);
        if (!this.router.isQuotaEnabled()) {
            throw new IOException("The quota system is disabled in Router.");
        }
        List<RemoteLocation> quotaLocs = this.getValidQuotaLocations(path);
        RemoteMethod method = new RemoteMethod("getQuotaUsage", new Class[]{String.class}, new RemoteParam());
        Map<RemoteLocation, QuotaUsage> results = this.rpcClient.invokeConcurrent(quotaLocs, method, true, false, QuotaUsage.class);
        return results;
    }

    QuotaUsage getGlobalQuota(String path) throws IOException {
        if (!this.router.isQuotaEnabled()) {
            throw new IOException("The quota system is disabled in Router.");
        }
        long nQuota = -1L;
        long sQuota = -1L;
        long[] typeQuota = new long[StorageType.values().length];
        Quota.eachByStorageType(t -> {
            typeQuota[t.ordinal()] = -1L;
        });
        RouterQuotaManager manager = this.router.getQuotaManager();
        TreeMap<String, RouterQuotaUsage> pts = manager.getParentsContainingQuota(path);
        Map.Entry<String, RouterQuotaUsage> entry = pts.lastEntry();
        while (entry != null && (nQuota == -1L || sQuota == -1L || Quota.orByStorageType(t -> typeQuota[t.ordinal()] == -1L))) {
            String ppath = entry.getKey();
            QuotaUsage quota = entry.getValue();
            if (nQuota == -1L) {
                nQuota = quota.getQuota();
            }
            if (sQuota == -1L) {
                sQuota = quota.getSpaceQuota();
            }
            Quota.eachByStorageType(t -> {
                if (typeQuota[t.ordinal()] == -1L) {
                    typeQuota[t.ordinal()] = quota.getTypeQuota(t);
                }
            });
            entry = pts.lowerEntry(ppath);
        }
        return new QuotaUsage.Builder().quota(nQuota).spaceQuota(sQuota).typeQuota(typeQuota).build();
    }

    private boolean isMountEntry(String path) {
        return this.router.getQuotaManager().isMountEntry(path);
    }

    private List<RemoteLocation> getValidQuotaLocations(String path) throws IOException {
        List<RemoteLocation> locations = this.getQuotaRemoteLocations(path);
        ArrayListMultimap validLocations = ArrayListMultimap.create();
        for (RemoteLocation loc : locations) {
            String nsId = loc.getNameserviceId();
            List dests = validLocations.get((Object)nsId);
            boolean isChildPath = false;
            for (RemoteLocation d : dests) {
                if (!DFSUtil.isParentEntry((String)loc.getDest(), (String)d.getDest())) continue;
                isChildPath = true;
                break;
            }
            if (isChildPath) continue;
            validLocations.put((Object)nsId, (Object)loc);
        }
        return Collections.unmodifiableList(new ArrayList(validLocations.values()));
    }

    QuotaUsage aggregateQuota(String path, Map<RemoteLocation, QuotaUsage> results) throws IOException {
        long nsCount = 0L;
        long ssCount = 0L;
        long[] typeCount = new long[StorageType.values().length];
        long nsQuota = -1L;
        long ssQuota = -1L;
        long[] typeQuota = new long[StorageType.values().length];
        Quota.eachByStorageType(t -> {
            typeQuota[t.ordinal()] = -1L;
        });
        boolean hasQuotaUnset = false;
        boolean isMountEntry = this.isMountEntry(path);
        for (Map.Entry<RemoteLocation, QuotaUsage> entry : results.entrySet()) {
            RemoteLocation loc = entry.getKey();
            QuotaUsage usage = entry.getValue();
            if (isMountEntry) {
                nsCount += usage.getFileAndDirectoryCount();
                ssCount += usage.getSpaceConsumed();
                Quota.eachByStorageType(t -> {
                    int n = t.ordinal();
                    typeCount[n] = typeCount[n] + usage.getTypeConsumed(t);
                });
                continue;
            }
            if (usage == null) continue;
            if (!RouterQuotaManager.isQuotaSet(usage)) {
                hasQuotaUnset = true;
            }
            nsQuota = usage.getQuota();
            ssQuota = usage.getSpaceQuota();
            Quota.eachByStorageType(t -> {
                typeQuota[t.ordinal()] = usage.getTypeQuota(t);
            });
            nsCount += usage.getFileAndDirectoryCount();
            ssCount += usage.getSpaceConsumed();
            Quota.eachByStorageType(t -> {
                int n = t.ordinal();
                typeCount[n] = typeCount[n] + usage.getTypeConsumed(t);
            });
            LOG.debug("Get quota usage for path: nsId: {}, dest: {}, nsCount: {}, ssCount: {}, typeCount: {}.", new Object[]{loc.getNameserviceId(), loc.getDest(), usage.getFileAndDirectoryCount(), usage.getSpaceConsumed(), usage.toString(false, true, Arrays.asList(StorageType.values()))});
        }
        if (isMountEntry) {
            QuotaUsage quota = this.getGlobalQuota(path);
            nsQuota = quota.getQuota();
            ssQuota = quota.getSpaceQuota();
            Quota.eachByStorageType(t -> {
                typeQuota[t.ordinal()] = quota.getTypeQuota(t);
            });
        }
        QuotaUsage.Builder builder = new QuotaUsage.Builder().fileAndDirectoryCount(nsCount).spaceConsumed(ssCount).typeConsumed(typeCount);
        if (hasQuotaUnset) {
            builder.quota(-1L).spaceQuota(-1L);
            Quota.eachByStorageType(t -> builder.typeQuota(t, -1L));
        } else {
            builder.quota(nsQuota).spaceQuota(ssQuota);
            Quota.eachByStorageType(t -> builder.typeQuota(t, typeQuota[t.ordinal()]));
        }
        return builder.build();
    }

    public static void eachByStorageType(Consumer<StorageType> consumer) {
        for (StorageType type : StorageType.values()) {
            consumer.accept(type);
        }
    }

    public static boolean orByStorageType(Predicate<StorageType> predicate) {
        boolean res = false;
        for (StorageType type : StorageType.values()) {
            res |= predicate.test(type);
        }
        return res;
    }

    public static boolean andByStorageType(Predicate<StorageType> predicate) {
        boolean res = false;
        for (StorageType type : StorageType.values()) {
            res &= predicate.test(type);
        }
        return res;
    }

    private List<RemoteLocation> getQuotaRemoteLocations(String path) throws IOException {
        ArrayList<RemoteLocation> locations = new ArrayList<RemoteLocation>();
        RouterQuotaManager manager = this.router.getQuotaManager();
        if (manager != null) {
            Set<String> childrenPaths = manager.getPaths(path);
            for (String childPath : childrenPaths) {
                locations.addAll(this.rpcServer.getLocationsForPath(childPath, false, false));
            }
        }
        if (locations.size() >= 1) {
            return locations;
        }
        locations.addAll(this.rpcServer.getLocationsForPath(path, false, false));
        return locations;
    }
}

