/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.Closeable;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.ContentSummary;
import org.apache.hadoop.fs.FileAlreadyExistsException;
import org.apache.hadoop.fs.Options;
import org.apache.hadoop.fs.ParentNotDirectoryException;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.UnresolvedLinkException;
import org.apache.hadoop.fs.permission.FsAction;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.fs.permission.PermissionStatus;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.protocol.DirectoryListing;
import org.apache.hadoop.hdfs.protocol.FSLimitException;
import org.apache.hadoop.hdfs.protocol.HdfsFileStatus;
import org.apache.hadoop.hdfs.protocol.HdfsLocatedFileStatus;
import org.apache.hadoop.hdfs.protocol.LocatedBlocks;
import org.apache.hadoop.hdfs.protocol.QuotaExceededException;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfo;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfoUnderConstruction;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockManager;
import org.apache.hadoop.hdfs.server.blockmanagement.DatanodeDescriptor;
import org.apache.hadoop.hdfs.server.common.HdfsServerConstants;
import org.apache.hadoop.hdfs.server.common.Util;
import org.apache.hadoop.hdfs.server.namenode.FSImage;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.INode;
import org.apache.hadoop.hdfs.server.namenode.INodeDirectory;
import org.apache.hadoop.hdfs.server.namenode.INodeDirectoryWithQuota;
import org.apache.hadoop.hdfs.server.namenode.INodeFile;
import org.apache.hadoop.hdfs.server.namenode.INodeFileUnderConstruction;
import org.apache.hadoop.hdfs.server.namenode.INodeSymlink;
import org.apache.hadoop.hdfs.server.namenode.NameCache;
import org.apache.hadoop.hdfs.server.namenode.NameNode;
import org.apache.hadoop.hdfs.util.ByteArray;

public class FSDirectory
implements Closeable {
    INodeDirectoryWithQuota rootDir;
    FSImage fsImage;
    private volatile boolean ready = false;
    private static final long UNKNOWN_DISK_SPACE = -1L;
    private final int maxComponentLength;
    private final int maxDirItems;
    private final int lsLimit;
    private ReentrantReadWriteLock dirLock = new ReentrantReadWriteLock(true);
    private Condition cond = this.dirLock.writeLock().newCondition();
    private final NameCache<ByteArray> nameCache;

    void readLock() {
        this.dirLock.readLock().lock();
    }

    void readUnlock() {
        this.dirLock.readLock().unlock();
    }

    void writeLock() {
        this.dirLock.writeLock().lock();
    }

    void writeUnlock() {
        this.dirLock.writeLock().unlock();
    }

    boolean hasWriteLock() {
        return this.dirLock.isWriteLockedByCurrentThread();
    }

    boolean hasReadLock() {
        return this.dirLock.getReadHoldCount() > 0;
    }

    FSDirectory(FSNamesystem ns, Configuration conf) throws IOException {
        this(new FSImage(conf), ns, conf);
    }

    FSDirectory(FSImage fsImage, FSNamesystem ns, Configuration conf) {
        fsImage.setFSNamesystem(ns);
        this.rootDir = new INodeDirectoryWithQuota("", ns.createFsOwnerPermissions(new FsPermission(493)), Integer.MAX_VALUE, -1L);
        this.fsImage = fsImage;
        int configuredLimit = conf.getInt("dfs.ls.limit", 1000);
        this.lsLimit = configuredLimit > 0 ? configuredLimit : 1000;
        this.maxComponentLength = conf.getInt("dfs.namenode.fs-limits.max-component-length", 0);
        this.maxDirItems = conf.getInt("dfs.namenode.fs-limits.max-directory-items", 0);
        int threshold = conf.getInt("dfs.namenode.name.cache.threshold", 10);
        NameNode.LOG.info((Object)("Caching file names occuring more than " + threshold + " times "));
        this.nameCache = new NameCache(threshold);
    }

    private FSNamesystem getFSNamesystem() {
        return this.fsImage.getFSNamesystem();
    }

    private BlockManager getBlockManager() {
        return this.getFSNamesystem().getBlockManager();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void loadFSImage(HdfsServerConstants.StartupOption startOpt) throws IOException {
        if (startOpt == HdfsServerConstants.StartupOption.FORMAT) {
            this.fsImage.format(this.fsImage.getStorage().determineClusterId());
            startOpt = HdfsServerConstants.StartupOption.REGULAR;
        }
        boolean success = false;
        try {
            if (this.fsImage.recoverTransitionRead(startOpt)) {
                this.fsImage.saveNamespace();
            }
            this.fsImage.openEditLog();
            this.fsImage.setCheckpointDirectories(null, null);
            success = true;
        }
        finally {
            if (!success) {
                this.fsImage.close();
            }
        }
        this.setReady();
    }

    void imageLoadComplete() {
        Preconditions.checkState((!this.ready ? 1 : 0) != 0, (Object)"FSDirectory already loaded");
        this.setReady();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void setReady() {
        if (this.ready) {
            return;
        }
        this.writeLock();
        try {
            this.setReady(true);
            this.nameCache.initialized();
            this.cond.signalAll();
        }
        finally {
            this.writeUnlock();
        }
    }

    @VisibleForTesting
    boolean isReady() {
        return this.ready;
    }

    protected void setReady(boolean flag) {
        this.ready = flag;
    }

    private void incrDeletedFileCount(int count) {
        if (this.getFSNamesystem() != null) {
            NameNode.getNameNodeMetrics().incrFilesDeleted(count);
        }
    }

    @Override
    public void close() throws IOException {
        this.fsImage.close();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void waitForReady() {
        if (!this.ready) {
            this.writeLock();
            try {
                while (!this.ready) {
                    try {
                        this.cond.await(5000L, TimeUnit.MILLISECONDS);
                    }
                    catch (InterruptedException interruptedException) {}
                }
            }
            finally {
                this.writeUnlock();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    INodeFileUnderConstruction addFile(String path, PermissionStatus permissions, short replication, long preferredBlockSize, String clientName, String clientMachine, DatanodeDescriptor clientNode, long generationStamp) throws FileAlreadyExistsException, QuotaExceededException, UnresolvedLinkException {
        this.waitForReady();
        long modTime = Util.now();
        Path parent = new Path(path).getParent();
        if (parent == null) {
            return null;
        }
        if (!this.mkdirs(parent.toString(), permissions, true, modTime)) {
            return null;
        }
        INodeFileUnderConstruction newNode = new INodeFileUnderConstruction(permissions, replication, preferredBlockSize, modTime, clientName, clientMachine, clientNode);
        this.writeLock();
        try {
            newNode = this.addNode(path, newNode, -1L);
        }
        finally {
            this.writeUnlock();
        }
        if (newNode == null) {
            NameNode.stateChangeLog.info((Object)("DIR* FSDirectory.addFile: failed to add " + path + " to the file system"));
            return null;
        }
        if (NameNode.stateChangeLog.isDebugEnabled()) {
            NameNode.stateChangeLog.debug((Object)("DIR* FSDirectory.addFile: " + path + " is added to the file system"));
        }
        return newNode;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    INode unprotectedAddFile(String path, PermissionStatus permissions, BlockInfo[] blocks, short replication, long modificationTime, long atime, long preferredBlockSize, String clientName, String clientMachine) throws UnresolvedLinkException {
        assert (this.hasWriteLock());
        INode newNode = blocks == null ? new INodeDirectory(permissions, modificationTime) : (blocks.length == 0 || blocks[blocks.length - 1].getBlockUCState() == HdfsServerConstants.BlockUCState.UNDER_CONSTRUCTION ? new INodeFileUnderConstruction(permissions, blocks.length, replication, preferredBlockSize, modificationTime, clientName, clientMachine, null) : new INodeFile(permissions, blocks.length, replication, modificationTime, atime, preferredBlockSize));
        this.writeLock();
        try {
            try {
                newNode = this.addNode(path, newNode, -1L);
                if (newNode != null && blocks != null) {
                    int nrBlocks = blocks.length;
                    INodeFile newF = (INodeFile)newNode;
                    for (int i = 0; i < nrBlocks; ++i) {
                        newF.setBlock(i, this.getBlockManager().addINode(blocks[i], newF));
                    }
                }
            }
            catch (IOException e) {
                INode iNode = null;
                this.writeUnlock();
                return iNode;
            }
            INode iNode = newNode;
            return iNode;
        }
        finally {
            this.writeUnlock();
        }
    }

    void updateFile(INodeFile file, String path, BlockInfo[] blocks, long mtime, long atime) throws IOException {
        int i;
        file.setAccessTime(atime);
        file.setModificationTimeForce(mtime);
        BlockInfo[] oldBlocks = file.getBlocks();
        boolean isGenStampUpdate = oldBlocks.length == blocks.length;
        BlockInfo oldBlock = null;
        for (i = 0; i < oldBlocks.length && i < blocks.length; ++i) {
            boolean isLastBlock;
            oldBlock = oldBlocks[i];
            BlockInfo newBlock = blocks[i];
            boolean bl = isLastBlock = i == oldBlocks.length - 1;
            if (oldBlock.getBlockId() != newBlock.getBlockId() || oldBlock.getGenerationStamp() != newBlock.getGenerationStamp() && (!isGenStampUpdate || !isLastBlock)) {
                throw new IOException("Mismatched block IDs or generation stamps, attempting to replace block " + oldBlock + " with " + newBlock + " as block # " + i + "/" + blocks.length + " of " + path);
            }
            oldBlock.setNumBytes(newBlock.getNumBytes());
            oldBlock.setGenerationStamp(newBlock.getGenerationStamp());
        }
        if (blocks.length < oldBlocks.length) {
            if (!file.isUnderConstruction()) {
                throw new IOException("Trying to remove a block from file " + path + " which is not under construction.");
            }
            if (blocks.length != oldBlocks.length - 1) {
                throw new IOException("Trying to remove more than one block from file " + path);
            }
            this.unprotectedRemoveBlock(path, (INodeFileUnderConstruction)file, oldBlocks[oldBlocks.length - 1]);
        } else if (blocks.length > oldBlocks.length) {
            this.getBlockManager().completeBlock(file, oldBlocks.length - 1, true);
            for (i = oldBlocks.length; i < blocks.length; ++i) {
                BlockInfo newBI = blocks[i];
                this.getBlockManager().addINode(newBI, file);
                file.addBlock(newBI);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    INodeDirectory addToParent(byte[] src, INodeDirectory parentINode, INode newNode, boolean propagateModTime) throws UnresolvedLinkException {
        INodeDirectory newParent = null;
        this.writeLock();
        try {
            try {
                newParent = this.rootDir.addToParent(src, newNode, parentINode, propagateModTime);
                this.cacheName(newNode);
            }
            catch (FileNotFoundException e) {
                INodeDirectory iNodeDirectory = null;
                this.writeUnlock();
                return iNodeDirectory;
            }
            if (newParent == null) {
                INodeDirectory e = null;
                return e;
            }
            if (!newNode.isDirectory() && !newNode.isLink()) {
                INodeFile newF = (INodeFile)newNode;
                BlockInfo[] blocks = newF.getBlocks();
                for (int i = 0; i < blocks.length; ++i) {
                    newF.setBlock(i, this.getBlockManager().addINode(blocks[i], newF));
                }
            }
        }
        finally {
            this.writeUnlock();
        }
        return newParent;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    BlockInfo addBlock(String path, INode[] inodes, Block block, DatanodeDescriptor[] targets) throws QuotaExceededException {
        this.waitForReady();
        this.writeLock();
        try {
            assert (inodes[inodes.length - 1].isUnderConstruction()) : "INode should correspond to a file under construction";
            INodeFileUnderConstruction fileINode = (INodeFileUnderConstruction)inodes[inodes.length - 1];
            this.updateCount(inodes, inodes.length - 1, 0L, fileINode.getPreferredBlockSize() * (long)fileINode.getReplication(), true);
            BlockInfoUnderConstruction blockInfo = new BlockInfoUnderConstruction(block, fileINode.getReplication(), HdfsServerConstants.BlockUCState.UNDER_CONSTRUCTION, targets);
            this.getBlockManager().addINode(blockInfo, fileINode);
            fileINode.addBlock(blockInfo);
            if (NameNode.stateChangeLog.isDebugEnabled()) {
                NameNode.stateChangeLog.debug((Object)("DIR* FSDirectory.addBlock: " + path + " with " + block + " block is added to the in-memory " + "file system"));
            }
            BlockInfoUnderConstruction blockInfoUnderConstruction = blockInfo;
            return blockInfoUnderConstruction;
        }
        finally {
            this.writeUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void persistBlocks(String path, INodeFileUnderConstruction file) {
        this.waitForReady();
        this.writeLock();
        try {
            this.fsImage.getEditLog().logOpenFile(path, file);
            if (NameNode.stateChangeLog.isDebugEnabled()) {
                NameNode.stateChangeLog.debug((Object)("DIR* FSDirectory.persistBlocks: " + path + " with " + file.getBlocks().length + " blocks is persisted to the file system"));
            }
        }
        finally {
            this.writeUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void closeFile(String path, INodeFile file) {
        this.waitForReady();
        long now = Util.now();
        this.writeLock();
        try {
            file.setModificationTimeForce(now);
            this.fsImage.getEditLog().logCloseFile(path, file);
            if (NameNode.stateChangeLog.isDebugEnabled()) {
                NameNode.stateChangeLog.debug((Object)("DIR* FSDirectory.closeFile: " + path + " with " + file.getBlocks().length + " blocks is persisted to the file system"));
            }
        }
        finally {
            this.writeUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean removeBlock(String path, INodeFileUnderConstruction fileNode, Block block) throws IOException {
        this.waitForReady();
        this.writeLock();
        try {
            this.unprotectedRemoveBlock(path, fileNode, block);
            this.fsImage.getEditLog().logOpenFile(path, fileNode);
        }
        finally {
            this.writeUnlock();
        }
        return true;
    }

    void unprotectedRemoveBlock(String path, INodeFileUnderConstruction fileNode, Block block) throws IOException {
        fileNode.removeLastBlock(block);
        this.getBlockManager().removeBlockFromMap(block);
        if (NameNode.stateChangeLog.isDebugEnabled()) {
            NameNode.stateChangeLog.debug((Object)("DIR* FSDirectory.removeBlock: " + path + " with " + block + " block is removed from the file system"));
        }
        INode[] pathINodes = this.getExistingPathINodes(path);
        this.updateCount(pathINodes, pathINodes.length - 1, 0L, -fileNode.getPreferredBlockSize() * (long)fileNode.getReplication(), true);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Deprecated
    boolean renameTo(String src, String dst) throws QuotaExceededException, UnresolvedLinkException, FileAlreadyExistsException {
        if (NameNode.stateChangeLog.isDebugEnabled()) {
            NameNode.stateChangeLog.debug((Object)("DIR* FSDirectory.renameTo: " + src + " to " + dst));
        }
        this.waitForReady();
        long now = Util.now();
        this.writeLock();
        try {
            if (!this.unprotectedRenameTo(src, dst, now)) {
                boolean bl = false;
                return bl;
            }
        }
        finally {
            this.writeUnlock();
        }
        this.fsImage.getEditLog().logRename(src, dst, now);
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void renameTo(String src, String dst, Options.Rename ... options) throws FileAlreadyExistsException, FileNotFoundException, ParentNotDirectoryException, QuotaExceededException, UnresolvedLinkException, IOException {
        if (NameNode.stateChangeLog.isDebugEnabled()) {
            NameNode.stateChangeLog.debug((Object)("DIR* FSDirectory.renameTo: " + src + " to " + dst));
        }
        this.waitForReady();
        long now = Util.now();
        this.writeLock();
        try {
            if (this.unprotectedRenameTo(src, dst, now, options)) {
                this.incrDeletedFileCount(1);
            }
        }
        finally {
            this.writeUnlock();
        }
        this.fsImage.getEditLog().logRename(src, dst, now, options);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Deprecated
    boolean unprotectedRenameTo(String src, String dst, long timestamp) throws QuotaExceededException, UnresolvedLinkException, FileAlreadyExistsException {
        String srcChildName;
        INode srcChild;
        INode dstChild;
        INode[] srcInodes;
        block17: {
            boolean bl;
            block18: {
                INode[] dstInodes;
                byte[][] dstComponents;
                block15: {
                    boolean bl2;
                    block16: {
                        assert (this.hasWriteLock());
                        srcInodes = this.rootDir.getExistingPathINodes(src, false);
                        INode srcInode = srcInodes[srcInodes.length - 1];
                        if (srcInode == null) {
                            NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.unprotectedRenameTo: failed to rename " + src + " to " + dst + " because source does not exist"));
                            return false;
                        }
                        if (srcInodes.length == 1) {
                            NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.unprotectedRenameTo: failed to rename " + src + " to " + dst + " because source is the root"));
                            return false;
                        }
                        if (this.isDir(dst)) {
                            dst = dst + "/" + new Path(src).getName();
                        }
                        if (dst.equals(src)) {
                            return true;
                        }
                        if (srcInode.isLink() && dst.equals(((INodeSymlink)srcInode).getLinkValue())) {
                            throw new FileAlreadyExistsException("Cannot rename symlink " + src + " to its target " + dst);
                        }
                        if (dst.startsWith(src) && dst.charAt(src.length()) == '/') {
                            NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.unprotectedRenameTo: failed to rename " + src + " to " + dst + " because destination starts with src"));
                            return false;
                        }
                        dstComponents = INode.getPathComponents(dst);
                        dstInodes = new INode[dstComponents.length];
                        this.rootDir.getExistingPathINodes(dstComponents, dstInodes, false);
                        if (dstInodes[dstInodes.length - 1] != null) {
                            NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.unprotectedRenameTo: failed to rename " + src + " to " + dst + " because destination exists"));
                            return false;
                        }
                        if (dstInodes[dstInodes.length - 2] == null) {
                            NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.unprotectedRenameTo: failed to rename " + src + " to " + dst + " because destination's parent does not exist"));
                            return false;
                        }
                        this.verifyQuotaForRename(srcInodes, dstInodes);
                        dstChild = null;
                        srcChild = null;
                        srcChildName = null;
                        try {
                            srcChild = this.removeChild(srcInodes, srcInodes.length - 1);
                            if (srcChild != null) break block15;
                            NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.unprotectedRenameTo: failed to rename " + src + " to " + dst + " because the source can not be removed"));
                            bl2 = false;
                            if (dstChild != null || srcChild == null) break block16;
                            srcChild.setLocalName(srcChildName);
                            this.addChildNoQuotaCheck(srcInodes, srcInodes.length - 1, srcChild, -1L);
                        }
                        catch (Throwable throwable) {
                            if (dstChild == null && srcChild != null) {
                                srcChild.setLocalName(srcChildName);
                                this.addChildNoQuotaCheck(srcInodes, srcInodes.length - 1, srcChild, -1L);
                            }
                            throw throwable;
                        }
                    }
                    return bl2;
                }
                srcChildName = srcChild.getLocalName();
                srcChild.setLocalName(dstComponents[dstInodes.length - 1]);
                dstChild = this.addChildNoQuotaCheck(dstInodes, dstInodes.length - 1, srcChild, -1L);
                if (dstChild == null) break block17;
                srcChild = null;
                if (NameNode.stateChangeLog.isDebugEnabled()) {
                    NameNode.stateChangeLog.debug((Object)("DIR* FSDirectory.unprotectedRenameTo: " + src + " is renamed to " + dst));
                }
                srcInodes[srcInodes.length - 2].setModificationTime(timestamp);
                dstInodes[dstInodes.length - 2].setModificationTime(timestamp);
                this.getFSNamesystem().unprotectedChangeLease(src, dst);
                bl = true;
                if (dstChild != null || srcChild == null) break block18;
                srcChild.setLocalName(srcChildName);
                this.addChildNoQuotaCheck(srcInodes, srcInodes.length - 1, srcChild, -1L);
            }
            return bl;
        }
        if (dstChild == null && srcChild != null) {
            srcChild.setLocalName(srcChildName);
            this.addChildNoQuotaCheck(srcInodes, srcInodes.length - 1, srcChild, -1L);
        }
        NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.unprotectedRenameTo: failed to rename " + src + " to " + dst));
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean unprotectedRenameTo(String src, String dst, long timestamp, Options.Rename ... options) throws FileAlreadyExistsException, FileNotFoundException, ParentNotDirectoryException, QuotaExceededException, UnresolvedLinkException, IOException {
        INode removedDst;
        String dstChildName;
        String srcChildName;
        INode removedSrc;
        INode[] dstInodes;
        INode[] srcInodes;
        block26: {
            boolean bl;
            block27: {
                assert (this.hasWriteLock());
                boolean overwrite = false;
                if (null != options) {
                    for (Options.Rename option : options) {
                        if (option != Options.Rename.OVERWRITE) continue;
                        overwrite = true;
                    }
                }
                String error = null;
                srcInodes = this.rootDir.getExistingPathINodes(src, false);
                INode srcInode = srcInodes[srcInodes.length - 1];
                if (srcInode == null) {
                    error = "rename source " + src + " is not found.";
                    NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.unprotectedRenameTo: " + error));
                    throw new FileNotFoundException(error);
                }
                if (srcInodes.length == 1) {
                    error = "rename source cannot be the root";
                    NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.unprotectedRenameTo: " + error));
                    throw new IOException(error);
                }
                if (dst.equals(src)) {
                    throw new FileAlreadyExistsException("The source " + src + " and destination " + dst + " are the same");
                }
                if (srcInode.isLink() && dst.equals(((INodeSymlink)srcInode).getLinkValue())) {
                    throw new FileAlreadyExistsException("Cannot rename symlink " + src + " to its target " + dst);
                }
                if (dst.startsWith(src) && dst.charAt(src.length()) == '/') {
                    error = "Rename destination " + dst + " is a directory or file under source " + src;
                    NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.unprotectedRenameTo: " + error));
                    throw new IOException(error);
                }
                byte[][] dstComponents = INode.getPathComponents(dst);
                dstInodes = new INode[dstComponents.length];
                this.rootDir.getExistingPathINodes(dstComponents, dstInodes, false);
                INode dstInode = dstInodes[dstInodes.length - 1];
                if (dstInodes.length == 1) {
                    error = "rename destination cannot be the root";
                    NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.unprotectedRenameTo: " + error));
                    throw new IOException(error);
                }
                if (dstInode != null) {
                    List<INode> children;
                    if (dstInode.isDirectory() != srcInode.isDirectory()) {
                        error = "Source " + src + " and destination " + dst + " must both be directories";
                        NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.unprotectedRenameTo: " + error));
                        throw new IOException(error);
                    }
                    if (!overwrite) {
                        error = "rename destination " + dst + " already exists";
                        NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.unprotectedRenameTo: " + error));
                        throw new FileAlreadyExistsException(error);
                    }
                    List<INode> list = children = dstInode.isDirectory() ? ((INodeDirectory)dstInode).getChildrenRaw() : null;
                    if (children != null && children.size() != 0) {
                        error = "rename cannot overwrite non empty destination directory " + dst;
                        NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.unprotectedRenameTo: " + error));
                        throw new IOException(error);
                    }
                }
                if (dstInodes[dstInodes.length - 2] == null) {
                    error = "rename destination parent " + dst + " not found.";
                    NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.unprotectedRenameTo: " + error));
                    throw new FileNotFoundException(error);
                }
                if (!dstInodes[dstInodes.length - 2].isDirectory()) {
                    error = "rename destination parent " + dst + " is a file.";
                    NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.unprotectedRenameTo: " + error));
                    throw new ParentNotDirectoryException(error);
                }
                this.verifyQuotaForRename(srcInodes, dstInodes);
                removedSrc = this.removeChild(srcInodes, srcInodes.length - 1);
                if (removedSrc == null) {
                    error = "Failed to rename " + src + " to " + dst + " because the source can not be removed";
                    NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.unprotectedRenameTo: " + error));
                    throw new IOException(error);
                }
                srcChildName = removedSrc.getLocalName();
                dstChildName = null;
                removedDst = null;
                try {
                    if (dstInode != null) {
                        removedDst = this.removeChild(dstInodes, dstInodes.length - 1);
                        dstChildName = removedDst.getLocalName();
                    }
                    INode dstChild = null;
                    removedSrc.setLocalName(dstComponents[dstInodes.length - 1]);
                    dstChild = this.addChildNoQuotaCheck(dstInodes, dstInodes.length - 1, removedSrc, -1L);
                    int filesDeleted = 0;
                    if (dstChild == null) break block26;
                    removedSrc = null;
                    if (NameNode.stateChangeLog.isDebugEnabled()) {
                        NameNode.stateChangeLog.debug((Object)("DIR* FSDirectory.unprotectedRenameTo: " + src + " is renamed to " + dst));
                    }
                    srcInodes[srcInodes.length - 2].setModificationTime(timestamp);
                    dstInodes[dstInodes.length - 2].setModificationTime(timestamp);
                    this.getFSNamesystem().unprotectedChangeLease(src, dst);
                    if (removedDst != null) {
                        INode rmdst = removedDst;
                        removedDst = null;
                        ArrayList<Block> collectedBlocks = new ArrayList<Block>();
                        filesDeleted = rmdst.collectSubtreeBlocksAndClear(collectedBlocks);
                        this.getFSNamesystem().removePathAndBlocks(src, collectedBlocks);
                    }
                    boolean bl2 = bl = filesDeleted > 0;
                    if (removedSrc == null) break block27;
                    removedSrc.setLocalName(srcChildName);
                    this.addChildNoQuotaCheck(srcInodes, srcInodes.length - 1, removedSrc, -1L);
                }
                catch (Throwable throwable) {
                    if (removedSrc != null) {
                        removedSrc.setLocalName(srcChildName);
                        this.addChildNoQuotaCheck(srcInodes, srcInodes.length - 1, removedSrc, -1L);
                    }
                    if (removedDst != null) {
                        removedDst.setLocalName(dstChildName);
                        this.addChildNoQuotaCheck(dstInodes, dstInodes.length - 1, removedDst, -1L);
                    }
                    throw throwable;
                }
            }
            if (removedDst != null) {
                removedDst.setLocalName(dstChildName);
                this.addChildNoQuotaCheck(dstInodes, dstInodes.length - 1, removedDst, -1L);
            }
            return bl;
        }
        if (removedSrc != null) {
            removedSrc.setLocalName(srcChildName);
            this.addChildNoQuotaCheck(srcInodes, srcInodes.length - 1, removedSrc, -1L);
        }
        if (removedDst != null) {
            removedDst.setLocalName(dstChildName);
            this.addChildNoQuotaCheck(dstInodes, dstInodes.length - 1, removedDst, -1L);
        }
        NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.unprotectedRenameTo: failed to rename " + src + " to " + dst));
        throw new IOException("rename from " + src + " to " + dst + " failed.");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    Block[] setReplication(String src, short replication, short[] oldReplication) throws QuotaExceededException, UnresolvedLinkException {
        this.waitForReady();
        Block[] fileBlocks = null;
        this.writeLock();
        try {
            fileBlocks = this.unprotectedSetReplication(src, replication, oldReplication);
            if (fileBlocks != null) {
                this.fsImage.getEditLog().logSetReplication(src, replication);
            }
            Block[] blockArray = fileBlocks;
            return blockArray;
        }
        finally {
            this.writeUnlock();
        }
    }

    Block[] unprotectedSetReplication(String src, short replication, short[] oldReplication) throws QuotaExceededException, UnresolvedLinkException {
        assert (this.hasWriteLock());
        INode[] inodes = this.rootDir.getExistingPathINodes(src, true);
        INode inode = inodes[inodes.length - 1];
        if (inode == null) {
            return null;
        }
        assert (!inode.isLink());
        if (inode.isDirectory()) {
            return null;
        }
        INodeFile fileNode = (INodeFile)inode;
        short oldRepl = fileNode.getReplication();
        long dsDelta = (long)(replication - oldRepl) * (fileNode.diskspaceConsumed() / (long)oldRepl);
        this.updateCount(inodes, inodes.length - 1, 0L, dsDelta, true);
        fileNode.setReplication(replication);
        if (oldReplication != null) {
            oldReplication[0] = oldRepl;
        }
        return fileNode.getBlocks();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    long getPreferredBlockSize(String filename) throws UnresolvedLinkException, FileNotFoundException, IOException {
        this.readLock();
        try {
            INode inode = this.rootDir.getNode(filename, false);
            if (inode == null) {
                throw new FileNotFoundException("File does not exist: " + filename);
            }
            if (inode.isDirectory() || inode.isLink()) {
                throw new IOException("Getting block size of non-file: " + filename);
            }
            long l = ((INodeFile)inode).getPreferredBlockSize();
            return l;
        }
        finally {
            this.readUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean exists(String src) throws UnresolvedLinkException {
        src = this.normalizePath(src);
        this.readLock();
        try {
            INode inode = this.rootDir.getNode(src, false);
            if (inode == null) {
                boolean bl = false;
                return bl;
            }
            boolean bl = inode.isDirectory() || inode.isLink() ? true : ((INodeFile)inode).getBlocks() != null;
            return bl;
        }
        finally {
            this.readUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void setPermission(String src, FsPermission permission) throws FileNotFoundException, UnresolvedLinkException {
        this.writeLock();
        try {
            this.unprotectedSetPermission(src, permission);
        }
        finally {
            this.writeUnlock();
        }
        this.fsImage.getEditLog().logSetPermissions(src, permission);
    }

    void unprotectedSetPermission(String src, FsPermission permissions) throws FileNotFoundException, UnresolvedLinkException {
        assert (this.hasWriteLock());
        INode inode = this.rootDir.getNode(src, true);
        if (inode == null) {
            throw new FileNotFoundException("File does not exist: " + src);
        }
        inode.setPermission(permissions);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void setOwner(String src, String username, String groupname) throws FileNotFoundException, UnresolvedLinkException {
        this.writeLock();
        try {
            this.unprotectedSetOwner(src, username, groupname);
        }
        finally {
            this.writeUnlock();
        }
        this.fsImage.getEditLog().logSetOwner(src, username, groupname);
    }

    void unprotectedSetOwner(String src, String username, String groupname) throws FileNotFoundException, UnresolvedLinkException {
        assert (this.hasWriteLock());
        INode inode = this.rootDir.getNode(src, true);
        if (inode == null) {
            throw new FileNotFoundException("File does not exist: " + src);
        }
        if (username != null) {
            inode.setUser(username);
        }
        if (groupname != null) {
            inode.setGroup(groupname);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void concat(String target, String[] srcs) throws UnresolvedLinkException {
        this.writeLock();
        try {
            this.waitForReady();
            long timestamp = Util.now();
            this.unprotectedConcat(target, srcs, timestamp);
            this.fsImage.getEditLog().logConcat(target, srcs, timestamp);
        }
        finally {
            this.writeUnlock();
        }
    }

    public void unprotectedConcat(String target, String[] srcs, long timestamp) throws UnresolvedLinkException {
        assert (this.hasWriteLock());
        if (NameNode.stateChangeLog.isDebugEnabled()) {
            NameNode.stateChangeLog.debug((Object)("DIR* FSNamesystem.concat to " + target));
        }
        INode[] trgINodes = this.getExistingPathINodes(target);
        INodeFile trgInode = (INodeFile)trgINodes[trgINodes.length - 1];
        INodeDirectory trgParent = (INodeDirectory)trgINodes[trgINodes.length - 2];
        INodeFile[] allSrcInodes = new INodeFile[srcs.length];
        int i = 0;
        int totalBlocks = 0;
        for (String src : srcs) {
            INodeFile srcInode = this.getFileINode(src);
            allSrcInodes[i++] = srcInode;
            totalBlocks += srcInode.blocks.length;
        }
        trgInode.appendBlocks(allSrcInodes, totalBlocks);
        int count = 0;
        for (INodeFile nodeToRemove : allSrcInodes) {
            if (nodeToRemove == null) continue;
            nodeToRemove.blocks = null;
            trgParent.removeChild(nodeToRemove);
            ++count;
        }
        trgInode.setModificationTimeForce(timestamp);
        trgParent.setModificationTime(timestamp);
        this.unprotectedUpdateCount(trgINodes, trgINodes.length - 1, -count, 0L);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean delete(String src, List<Block> collectedBlocks) throws UnresolvedLinkException {
        int filesRemoved;
        if (NameNode.stateChangeLog.isDebugEnabled()) {
            NameNode.stateChangeLog.debug((Object)("DIR* FSDirectory.delete: " + src));
        }
        this.waitForReady();
        long now = Util.now();
        this.writeLock();
        try {
            filesRemoved = this.unprotectedDelete(src, collectedBlocks, now);
        }
        finally {
            this.writeUnlock();
        }
        if (filesRemoved <= 0) {
            return false;
        }
        this.incrDeletedFileCount(filesRemoved);
        this.getFSNamesystem().removePathAndBlocks(src, null);
        this.fsImage.getEditLog().logDelete(src, now);
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean isDirEmpty(String src) throws UnresolvedLinkException {
        boolean dirNotEmpty = true;
        if (!this.isDir(src)) {
            return true;
        }
        this.readLock();
        try {
            INode targetNode = this.rootDir.getNode(src, false);
            assert (targetNode != null) : "should be taken care in isDir() above";
            if (((INodeDirectory)targetNode).getChildren().size() != 0) {
                dirNotEmpty = false;
            }
        }
        finally {
            this.readUnlock();
        }
        return dirNotEmpty;
    }

    boolean isEmpty() {
        try {
            return this.isDirEmpty("/");
        }
        catch (UnresolvedLinkException e) {
            if (NameNode.stateChangeLog.isDebugEnabled()) {
                NameNode.stateChangeLog.debug((Object)"/ cannot be a symlink");
            }
            assert (false) : "/ cannot be a symlink";
            return true;
        }
    }

    void unprotectedDelete(String src, long mtime) throws UnresolvedLinkException {
        assert (this.hasWriteLock());
        ArrayList<Block> collectedBlocks = new ArrayList<Block>();
        int filesRemoved = this.unprotectedDelete(src, collectedBlocks, mtime);
        if (filesRemoved > 0) {
            this.getFSNamesystem().removePathAndBlocks(src, collectedBlocks);
        }
    }

    int unprotectedDelete(String src, List<Block> collectedBlocks, long mtime) throws UnresolvedLinkException {
        assert (this.hasWriteLock());
        INode[] inodes = this.rootDir.getExistingPathINodes(src = this.normalizePath(src), false);
        INode targetNode = inodes[inodes.length - 1];
        if (targetNode == null) {
            if (NameNode.stateChangeLog.isDebugEnabled()) {
                NameNode.stateChangeLog.debug((Object)("DIR* FSDirectory.unprotectedDelete: failed to remove " + src + " because it does not exist"));
            }
            return 0;
        }
        if (inodes.length == 1) {
            NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.unprotectedDelete: failed to remove " + src + " because the root is not allowed to be deleted"));
            return 0;
        }
        int pos = inodes.length - 1;
        targetNode = this.removeChild(inodes, pos);
        if (targetNode == null) {
            return 0;
        }
        inodes[pos - 1].setModificationTime(mtime);
        int filesRemoved = targetNode.collectSubtreeBlocksAndClear(collectedBlocks);
        if (NameNode.stateChangeLog.isDebugEnabled()) {
            NameNode.stateChangeLog.debug((Object)("DIR* FSDirectory.unprotectedDelete: " + src + " is removed"));
        }
        return filesRemoved;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void replaceNode(String path, INodeFile oldnode, INodeFile newnode) throws IOException, UnresolvedLinkException {
        this.writeLock();
        try {
            if (!oldnode.removeNode()) {
                NameNode.stateChangeLog.warn((Object)("DIR* FSDirectory.replaceNode: failed to remove " + path));
                throw new IOException("FSDirectory.replaceNode: failed to remove " + path);
            }
            this.rootDir.addNode(path, newnode);
            int index = 0;
            for (BlockInfo b : newnode.getBlocks()) {
                BlockInfo info = this.getBlockManager().addINode(b, newnode);
                newnode.setBlock(index, info);
                ++index;
            }
        }
        finally {
            this.writeUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    DirectoryListing getListing(String src, byte[] startAfter, boolean needLocation) throws UnresolvedLinkException, IOException {
        String srcs = this.normalizePath(src);
        this.readLock();
        try {
            INode targetNode = this.rootDir.getNode(srcs, true);
            if (targetNode == null) {
                DirectoryListing directoryListing = null;
                return directoryListing;
            }
            if (!targetNode.isDirectory()) {
                DirectoryListing directoryListing = new DirectoryListing(new HdfsFileStatus[]{this.createFileStatus(HdfsFileStatus.EMPTY_NAME, targetNode, needLocation)}, 0);
                return directoryListing;
            }
            INodeDirectory dirInode = (INodeDirectory)targetNode;
            List<INode> contents = dirInode.getChildren();
            int startChild = dirInode.nextChild(startAfter);
            int totalNumChildren = contents.size();
            int numOfListing = Math.min(totalNumChildren - startChild, this.lsLimit);
            HdfsFileStatus[] listing = new HdfsFileStatus[numOfListing];
            for (int i = 0; i < numOfListing; ++i) {
                INode cur = contents.get(startChild + i);
                listing[i] = this.createFileStatus(cur.name, cur, needLocation);
            }
            DirectoryListing directoryListing = new DirectoryListing(listing, totalNumChildren - startChild - numOfListing);
            return directoryListing;
        }
        finally {
            this.readUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    HdfsFileStatus getFileInfo(String src, boolean resolveLink) throws UnresolvedLinkException {
        String srcs = this.normalizePath(src);
        this.readLock();
        try {
            INode targetNode = this.rootDir.getNode(srcs, resolveLink);
            if (targetNode == null) {
                HdfsFileStatus hdfsFileStatus = null;
                return hdfsFileStatus;
            }
            HdfsFileStatus hdfsFileStatus = this.createFileStatus(HdfsFileStatus.EMPTY_NAME, targetNode);
            return hdfsFileStatus;
        }
        finally {
            this.readUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    Block[] getFileBlocks(String src) throws UnresolvedLinkException {
        this.waitForReady();
        this.readLock();
        try {
            INode targetNode = this.rootDir.getNode(src, false);
            if (targetNode == null) {
                Block[] blockArray = null;
                return blockArray;
            }
            if (targetNode.isDirectory()) {
                Block[] blockArray = null;
                return blockArray;
            }
            if (targetNode.isLink()) {
                Block[] blockArray = null;
                return blockArray;
            }
            Block[] blockArray = ((INodeFile)targetNode).getBlocks();
            return blockArray;
        }
        finally {
            this.readUnlock();
        }
    }

    INodeFile getFileINode(String src) throws UnresolvedLinkException {
        INode inode = this.getINode(src);
        if (inode == null || inode.isDirectory()) {
            return null;
        }
        assert (!inode.isLink());
        return (INodeFile)inode;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    INode getINode(String src) throws UnresolvedLinkException {
        this.readLock();
        try {
            INode iNode;
            INode iNode2 = iNode = this.rootDir.getNode(src, true);
            return iNode2;
        }
        finally {
            this.readUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    INode[] getExistingPathINodes(String path) throws UnresolvedLinkException {
        this.readLock();
        try {
            INode[] iNodeArray = this.rootDir.getExistingPathINodes(path, true);
            return iNodeArray;
        }
        finally {
            this.readUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    INodeDirectory getParent(byte[][] path) throws FileNotFoundException, UnresolvedLinkException {
        this.readLock();
        try {
            INodeDirectory iNodeDirectory = this.rootDir.getParent(path);
            return iNodeDirectory;
        }
        finally {
            this.readUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean isValidToCreate(String src) throws UnresolvedLinkException {
        String srcs = this.normalizePath(src);
        this.readLock();
        try {
            if (srcs.startsWith("/") && !srcs.endsWith("/") && this.rootDir.getNode(srcs, false) == null) {
                boolean bl = true;
                return bl;
            }
            boolean bl = false;
            return bl;
        }
        finally {
            this.readUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean isDir(String src) throws UnresolvedLinkException {
        src = this.normalizePath(src);
        this.readLock();
        try {
            INode node = this.rootDir.getNode(src, false);
            boolean bl = node != null && node.isDirectory();
            return bl;
        }
        finally {
            this.readUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void updateSpaceConsumed(String path, long nsDelta, long dsDelta) throws QuotaExceededException, FileNotFoundException, UnresolvedLinkException {
        this.writeLock();
        try {
            INode[] inodes = this.rootDir.getExistingPathINodes(path, false);
            int len = inodes.length;
            if (inodes[len - 1] == null) {
                throw new FileNotFoundException(path + " does not exist under rootDir.");
            }
            this.updateCount(inodes, len - 1, nsDelta, dsDelta, true);
        }
        finally {
            this.writeUnlock();
        }
    }

    private void updateCount(INode[] inodes, int numOfINodes, long nsDelta, long dsDelta, boolean checkQuota) throws QuotaExceededException {
        assert (this.hasWriteLock());
        if (!this.ready) {
            return;
        }
        if (numOfINodes > inodes.length) {
            numOfINodes = inodes.length;
        }
        if (checkQuota) {
            this.verifyQuota(inodes, numOfINodes, nsDelta, dsDelta, null);
        }
        for (int i = 0; i < numOfINodes; ++i) {
            if (!inodes[i].isQuotaSet()) continue;
            INodeDirectoryWithQuota node = (INodeDirectoryWithQuota)inodes[i];
            node.updateNumItemsInTree(nsDelta, dsDelta);
        }
    }

    private void updateCountNoQuotaCheck(INode[] inodes, int numOfINodes, long nsDelta, long dsDelta) {
        assert (this.hasWriteLock());
        try {
            this.updateCount(inodes, numOfINodes, nsDelta, dsDelta, false);
        }
        catch (QuotaExceededException e) {
            NameNode.LOG.warn((Object)"FSDirectory.updateCountNoQuotaCheck - unexpected ", (Throwable)e);
        }
    }

    void unprotectedUpdateCount(INode[] inodes, int numOfINodes, long nsDelta, long dsDelta) {
        assert (this.hasWriteLock());
        for (int i = 0; i < numOfINodes; ++i) {
            if (!inodes[i].isQuotaSet()) continue;
            INodeDirectoryWithQuota node = (INodeDirectoryWithQuota)inodes[i];
            node.unprotectedUpdateNumItemsInTree(nsDelta, dsDelta);
        }
    }

    private static String getFullPathName(INode[] inodes, int pos) {
        StringBuilder fullPathName = new StringBuilder();
        if (inodes[0].isRoot()) {
            if (pos == 0) {
                return "/";
            }
        } else {
            fullPathName.append(inodes[0].getLocalName());
        }
        for (int i = 1; i <= pos; ++i) {
            fullPathName.append('/').append(inodes[i].getLocalName());
        }
        return fullPathName.toString();
    }

    static String getFullPathName(INode inode) {
        int depth = 0;
        INode i = inode;
        while (i != null) {
            ++depth;
            i = i.parent;
        }
        INode[] inodes = new INode[depth];
        for (int i2 = 0; i2 < depth; ++i2) {
            inodes[depth - i2 - 1] = inode;
            inode = inode.parent;
        }
        return FSDirectory.getFullPathName(inodes, depth - 1);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean mkdirs(String src, PermissionStatus permissions, boolean inheritPermission, long now) throws FileAlreadyExistsException, QuotaExceededException, UnresolvedLinkException {
        src = this.normalizePath(src);
        String[] names = INode.getPathNames(src);
        byte[][] components = INode.getPathComponents(names);
        INode[] inodes = new INode[components.length];
        int lastInodeIndex = inodes.length - 1;
        this.writeLock();
        try {
            int i;
            this.rootDir.getExistingPathINodes(components, inodes, false);
            StringBuilder pathbuilder = new StringBuilder();
            for (i = 1; i < inodes.length && inodes[i] != null; ++i) {
                pathbuilder.append("/" + names[i]);
                if (inodes[i].isDirectory()) continue;
                throw new FileAlreadyExistsException("Parent path is not a directory: " + pathbuilder + " " + inodes[i].getLocalName());
            }
            PermissionStatus parentPermissions = permissions;
            if (inheritPermission || i < lastInodeIndex) {
                FsPermission parentFsPerm;
                FsPermission fsPermission = parentFsPerm = inheritPermission ? inodes[i - 1].getFsPermission() : permissions.getPermission();
                if (!parentFsPerm.getUserAction().implies(FsAction.WRITE_EXECUTE)) {
                    parentFsPerm = new FsPermission(parentFsPerm.getUserAction().or(FsAction.WRITE_EXECUTE), parentFsPerm.getGroupAction(), parentFsPerm.getOtherAction());
                }
                if (!parentPermissions.getPermission().equals((Object)parentFsPerm)) {
                    parentPermissions = new PermissionStatus(parentPermissions.getUserName(), parentPermissions.getGroupName(), parentFsPerm);
                    if (inheritPermission) {
                        permissions = parentPermissions;
                    }
                }
            }
            while (i < inodes.length) {
                pathbuilder.append("/" + names[i]);
                String cur = pathbuilder.toString();
                this.unprotectedMkdir(inodes, i, components[i], i < lastInodeIndex ? parentPermissions : permissions, now);
                if (inodes[i] == null) {
                    boolean bl = false;
                    return bl;
                }
                if (this.getFSNamesystem() != null) {
                    NameNode.getNameNodeMetrics().incrFilesCreated();
                }
                this.fsImage.getEditLog().logMkDir(cur, inodes[i]);
                if (NameNode.stateChangeLog.isDebugEnabled()) {
                    NameNode.stateChangeLog.debug((Object)("DIR* FSDirectory.mkdirs: created directory " + cur));
                }
                ++i;
            }
        }
        finally {
            this.writeUnlock();
        }
        return true;
    }

    INode unprotectedMkdir(String src, PermissionStatus permissions, long timestamp) throws QuotaExceededException, UnresolvedLinkException {
        assert (this.hasWriteLock());
        byte[][] components = INode.getPathComponents(src);
        INode[] inodes = new INode[components.length];
        this.rootDir.getExistingPathINodes(components, inodes, false);
        this.unprotectedMkdir(inodes, inodes.length - 1, components[inodes.length - 1], permissions, timestamp);
        return inodes[inodes.length - 1];
    }

    private void unprotectedMkdir(INode[] inodes, int pos, byte[] name, PermissionStatus permission, long timestamp) throws QuotaExceededException {
        assert (this.hasWriteLock());
        inodes[pos] = this.addChild(inodes, pos, new INodeDirectory(name, permission, timestamp), -1L);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private <T extends INode> T addNode(String src, T child, long childDiskspace) throws QuotaExceededException, UnresolvedLinkException {
        byte[][] components = INode.getPathComponents(src);
        byte[] path = components[components.length - 1];
        child.setLocalName(path);
        this.cacheName(child);
        INode[] inodes = new INode[components.length];
        this.writeLock();
        try {
            this.rootDir.getExistingPathINodes(components, inodes, false);
            T t = this.addChild(inodes, inodes.length - 1, child, childDiskspace);
            return t;
        }
        finally {
            this.writeUnlock();
        }
    }

    private void verifyQuota(INode[] inodes, int pos, long nsDelta, long dsDelta, INode commonAncestor) throws QuotaExceededException {
        int i;
        if (!this.ready) {
            return;
        }
        if (nsDelta <= 0L && dsDelta <= 0L) {
            return;
        }
        if (pos > inodes.length) {
            pos = inodes.length;
        }
        try {
            for (i = pos - 1; i >= 0; --i) {
                if (commonAncestor == inodes[i]) {
                    return;
                }
                if (!inodes[i].isQuotaSet()) continue;
                INodeDirectoryWithQuota node = (INodeDirectoryWithQuota)inodes[i];
                node.verifyQuota(nsDelta, dsDelta);
            }
        }
        catch (QuotaExceededException e) {
            e.setPathName(FSDirectory.getFullPathName(inodes, i));
            throw e;
        }
    }

    private void verifyQuotaForRename(INode[] srcInodes, INode[] dstInodes) throws QuotaExceededException {
        if (!this.ready) {
            return;
        }
        INode srcInode = srcInodes[srcInodes.length - 1];
        INode commonAncestor = null;
        int i = 0;
        while (srcInodes[i] == dstInodes[i]) {
            commonAncestor = srcInodes[i];
            ++i;
        }
        INode.DirCounts srcCounts = new INode.DirCounts();
        srcInode.spaceConsumedInTree(srcCounts);
        long nsDelta = srcCounts.getNsCount();
        long dsDelta = srcCounts.getDsCount();
        INode dstInode = dstInodes[dstInodes.length - 1];
        if (dstInode != null) {
            INode.DirCounts dstCounts = new INode.DirCounts();
            dstInode.spaceConsumedInTree(dstCounts);
            nsDelta -= dstCounts.getNsCount();
            dsDelta -= dstCounts.getDsCount();
        }
        this.verifyQuota(dstInodes, dstInodes.length - 1, nsDelta, dsDelta, commonAncestor);
    }

    protected <T extends INode> void verifyFsLimits(INode[] pathComponents, int pos, T child) throws FSLimitException {
        boolean includeChildName = false;
        try {
            INodeDirectory parent;
            int count;
            int length;
            if (this.maxComponentLength != 0 && (length = child.getLocalName().length()) > this.maxComponentLength) {
                includeChildName = true;
                throw new FSLimitException.PathComponentTooLongException(this.maxComponentLength, length);
            }
            if (this.maxDirItems != 0 && (count = (parent = (INodeDirectory)pathComponents[pos - 1]).getChildren().size()) >= this.maxDirItems) {
                throw new FSLimitException.MaxDirectoryItemsExceededException(this.maxDirItems, count);
            }
        }
        catch (FSLimitException e) {
            String badPath = FSDirectory.getFullPathName(pathComponents, pos - 1);
            if (includeChildName) {
                badPath = badPath + "/" + child.getLocalName();
            }
            e.setPathName(badPath);
            if (this.ready) {
                throw e;
            }
            NameNode.LOG.error((Object)("FSDirectory.verifyFsLimits - " + e.getLocalizedMessage()));
        }
    }

    private <T extends INode> T addChild(INode[] pathComponents, int pos, T child, long childDiskspace, boolean checkQuota) throws QuotaExceededException {
        if (checkQuota) {
            this.verifyFsLimits(pathComponents, pos, child);
        }
        INode.DirCounts counts = new INode.DirCounts();
        child.spaceConsumedInTree(counts);
        if (childDiskspace < 0L) {
            childDiskspace = counts.getDsCount();
        }
        this.updateCount(pathComponents, pos, counts.getNsCount(), childDiskspace, checkQuota);
        if (pathComponents[pos - 1] == null) {
            throw new NullPointerException("Panic: parent does not exist");
        }
        T addedNode = ((INodeDirectory)pathComponents[pos - 1]).addChild(child, true);
        if (addedNode == null) {
            this.updateCount(pathComponents, pos, -counts.getNsCount(), -childDiskspace, true);
        }
        return addedNode;
    }

    private <T extends INode> T addChild(INode[] pathComponents, int pos, T child, long childDiskspace) throws QuotaExceededException {
        return this.addChild(pathComponents, pos, child, childDiskspace, true);
    }

    private <T extends INode> T addChildNoQuotaCheck(INode[] pathComponents, int pos, T child, long childDiskspace) {
        T inode = null;
        try {
            inode = this.addChild(pathComponents, pos, child, childDiskspace, false);
        }
        catch (QuotaExceededException e) {
            NameNode.LOG.warn((Object)"FSDirectory.addChildNoQuotaCheck - unexpected", (Throwable)e);
        }
        return inode;
    }

    private INode removeChild(INode[] pathComponents, int pos) {
        INode removedNode = ((INodeDirectory)pathComponents[pos - 1]).removeChild(pathComponents[pos]);
        if (removedNode != null) {
            INode.DirCounts counts = new INode.DirCounts();
            removedNode.spaceConsumedInTree(counts);
            this.updateCountNoQuotaCheck(pathComponents, pos, -counts.getNsCount(), -counts.getDsCount());
        }
        return removedNode;
    }

    String normalizePath(String src) {
        if (src.length() > 1 && src.endsWith("/")) {
            src = src.substring(0, src.length() - 1);
        }
        return src;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    ContentSummary getContentSummary(String src) throws FileNotFoundException, UnresolvedLinkException {
        String srcs = this.normalizePath(src);
        this.readLock();
        try {
            INode targetNode = this.rootDir.getNode(srcs, false);
            if (targetNode == null) {
                throw new FileNotFoundException("File does not exist: " + srcs);
            }
            ContentSummary contentSummary = targetNode.computeContentSummary();
            return contentSummary;
        }
        finally {
            this.readUnlock();
        }
    }

    void updateCountForINodeWithQuota() {
        FSDirectory.updateCountForINodeWithQuota(this.rootDir, new INode.DirCounts(), new ArrayList<INode>(50));
    }

    private static void updateCountForINodeWithQuota(INodeDirectory dir, INode.DirCounts counts, ArrayList<INode> nodesInPath) {
        long parentNamespace = counts.nsCount;
        long parentDiskspace = counts.dsCount;
        counts.nsCount = 1L;
        counts.dsCount = 0L;
        nodesInPath.add(dir);
        for (INode child : dir.getChildren()) {
            if (child.isDirectory()) {
                FSDirectory.updateCountForINodeWithQuota((INodeDirectory)child, counts, nodesInPath);
                continue;
            }
            if (child.isLink()) {
                ++counts.nsCount;
                continue;
            }
            ++counts.nsCount;
            counts.dsCount += ((INodeFile)child).diskspaceConsumed();
        }
        if (dir.isQuotaSet()) {
            ((INodeDirectoryWithQuota)dir).setSpaceConsumed(counts.nsCount, counts.dsCount);
            if (dir.getNsQuota() >= 0L && counts.nsCount > dir.getNsQuota() || dir.getDsQuota() >= 0L && counts.dsCount > dir.getDsQuota()) {
                StringBuilder path = new StringBuilder(512);
                for (INode n : nodesInPath) {
                    path.append('/');
                    path.append(n.getLocalName());
                }
                NameNode.LOG.warn((Object)("Quota violation in image for " + path + " (Namespace quota : " + dir.getNsQuota() + " consumed : " + counts.nsCount + ")" + " (Diskspace quota : " + dir.getDsQuota() + " consumed : " + counts.dsCount + ")."));
            }
        }
        nodesInPath.remove(nodesInPath.size() - 1);
        counts.nsCount += parentNamespace;
        counts.dsCount += parentDiskspace;
    }

    INodeDirectory unprotectedSetQuota(String src, long nsQuota, long dsQuota) throws FileNotFoundException, QuotaExceededException, UnresolvedLinkException {
        assert (this.hasWriteLock());
        if (nsQuota < 0L && nsQuota != Long.MAX_VALUE && nsQuota < -1L || dsQuota < 0L && dsQuota != Long.MAX_VALUE && dsQuota < -1L) {
            throw new IllegalArgumentException("Illegal value for nsQuota or dsQuota : " + nsQuota + " and " + dsQuota);
        }
        String srcs = this.normalizePath(src);
        INode[] inodes = this.rootDir.getExistingPathINodes(src, true);
        INode targetNode = inodes[inodes.length - 1];
        if (targetNode == null) {
            throw new FileNotFoundException("Directory does not exist: " + srcs);
        }
        if (!targetNode.isDirectory()) {
            throw new FileNotFoundException("Cannot set quota on a file: " + srcs);
        }
        if (targetNode.isRoot() && nsQuota == -1L) {
            throw new IllegalArgumentException("Cannot clear namespace quota on root.");
        }
        INodeDirectory dirNode = (INodeDirectory)targetNode;
        long oldNsQuota = dirNode.getNsQuota();
        long oldDsQuota = dirNode.getDsQuota();
        if (nsQuota == Long.MAX_VALUE) {
            nsQuota = oldNsQuota;
        }
        if (dsQuota == Long.MAX_VALUE) {
            dsQuota = oldDsQuota;
        }
        if (dirNode instanceof INodeDirectoryWithQuota) {
            ((INodeDirectoryWithQuota)dirNode).setQuota(nsQuota, dsQuota);
            if (!dirNode.isQuotaSet()) {
                INodeDirectory newNode = new INodeDirectory(dirNode);
                INodeDirectory parent = (INodeDirectory)inodes[inodes.length - 2];
                dirNode = newNode;
                parent.replaceChild(newNode);
            }
        } else {
            INodeDirectoryWithQuota newNode = new INodeDirectoryWithQuota(nsQuota, dsQuota, dirNode);
            INodeDirectory parent = (INodeDirectory)inodes[inodes.length - 2];
            dirNode = newNode;
            parent.replaceChild(newNode);
        }
        return oldNsQuota != nsQuota || oldDsQuota != dsQuota ? dirNode : null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void setQuota(String src, long nsQuota, long dsQuota) throws FileNotFoundException, QuotaExceededException, UnresolvedLinkException {
        this.writeLock();
        try {
            INodeDirectory dir = this.unprotectedSetQuota(src, nsQuota, dsQuota);
            if (dir != null) {
                this.fsImage.getEditLog().logSetQuota(src, dir.getNsQuota(), dir.getDsQuota());
            }
        }
        finally {
            this.writeUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    long totalInodes() {
        this.readLock();
        try {
            long l = this.rootDir.numItemsInTree();
            return l;
        }
        finally {
            this.readUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void setTimes(String src, INode inode, long mtime, long atime, boolean force) {
        boolean status = false;
        this.writeLock();
        try {
            status = this.unprotectedSetTimes(src, inode, mtime, atime, force);
        }
        finally {
            this.writeUnlock();
        }
        if (status) {
            this.fsImage.getEditLog().logTimes(src, mtime, atime);
        }
    }

    boolean unprotectedSetTimes(String src, long mtime, long atime, boolean force) throws UnresolvedLinkException {
        assert (this.hasWriteLock());
        INode inode = this.getINode(src);
        return this.unprotectedSetTimes(src, inode, mtime, atime, force);
    }

    private boolean unprotectedSetTimes(String src, INode inode, long mtime, long atime, boolean force) {
        assert (this.hasWriteLock());
        boolean status = false;
        if (mtime != -1L) {
            inode.setModificationTimeForce(mtime);
            status = true;
        }
        if (atime != -1L) {
            long inodeTime = inode.getAccessTime();
            if (atime <= inodeTime + this.getFSNamesystem().getAccessTimePrecision() && !force) {
                status = false;
            } else {
                inode.setAccessTime(atime);
                status = true;
            }
        }
        return status;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void reset() {
        this.writeLock();
        try {
            this.setReady(false);
            this.rootDir = new INodeDirectoryWithQuota("", this.getFSNamesystem().createFsOwnerPermissions(new FsPermission(493)), Integer.MAX_VALUE, -1L);
            this.nameCache.reset();
        }
        finally {
            this.writeUnlock();
        }
    }

    private HdfsFileStatus createFileStatus(byte[] path, INode node, boolean needLocation) throws IOException {
        if (needLocation) {
            return this.createLocatedFileStatus(path, node);
        }
        return this.createFileStatus(path, node);
    }

    private HdfsFileStatus createFileStatus(byte[] path, INode node) {
        long size = 0L;
        short replication = 0;
        long blocksize = 0L;
        if (node instanceof INodeFile) {
            INodeFile fileNode = (INodeFile)node;
            size = fileNode.computeFileSize(true);
            replication = fileNode.getReplication();
            blocksize = fileNode.getPreferredBlockSize();
        }
        return new HdfsFileStatus(size, node.isDirectory(), replication, blocksize, node.getModificationTime(), node.getAccessTime(), node.getFsPermission(), node.getUserName(), node.getGroupName(), node.isLink() ? ((INodeSymlink)node).getSymlink() : null, path);
    }

    private HdfsLocatedFileStatus createLocatedFileStatus(byte[] path, INode node) throws IOException {
        assert (this.hasReadLock());
        long size = 0L;
        short replication = 0;
        long blocksize = 0L;
        LocatedBlocks loc = null;
        if (node instanceof INodeFile) {
            INodeFile fileNode = (INodeFile)node;
            size = fileNode.computeFileSize(true);
            replication = fileNode.getReplication();
            blocksize = fileNode.getPreferredBlockSize();
            loc = this.getFSNamesystem().getBlockManager().createLocatedBlocks(fileNode.getBlocks(), fileNode.computeFileSize(false), fileNode.isUnderConstruction(), 0L, size, false);
            if (loc == null) {
                loc = new LocatedBlocks();
            }
        }
        return new HdfsLocatedFileStatus(size, node.isDirectory(), replication, blocksize, node.getModificationTime(), node.getAccessTime(), node.getFsPermission(), node.getUserName(), node.getGroupName(), node.isLink() ? ((INodeSymlink)node).getSymlink() : null, path, loc);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    INodeSymlink addSymlink(String path, String target, PermissionStatus dirPerms, boolean createParent) throws UnresolvedLinkException, FileAlreadyExistsException, QuotaExceededException, IOException {
        String parent;
        this.waitForReady();
        long modTime = Util.now();
        if (createParent && !this.mkdirs(parent = new Path(path).getParent().toString(), dirPerms, true, modTime)) {
            return null;
        }
        String userName = dirPerms.getUserName();
        INodeSymlink newNode = null;
        this.writeLock();
        try {
            newNode = this.unprotectedSymlink(path, target, modTime, modTime, new PermissionStatus(userName, null, FsPermission.getDefault()));
        }
        finally {
            this.writeUnlock();
        }
        if (newNode == null) {
            NameNode.stateChangeLog.info((Object)("DIR* FSDirectory.addSymlink: failed to add " + path + " to the file system"));
            return null;
        }
        this.fsImage.getEditLog().logSymlink(path, target, modTime, modTime, newNode);
        if (NameNode.stateChangeLog.isDebugEnabled()) {
            NameNode.stateChangeLog.debug((Object)("DIR* FSDirectory.addSymlink: " + path + " is added to the file system"));
        }
        return newNode;
    }

    INodeSymlink unprotectedSymlink(String path, String target, long modTime, long atime, PermissionStatus perm) throws UnresolvedLinkException {
        assert (this.hasWriteLock());
        INodeSymlink newNode = new INodeSymlink(target, modTime, atime, perm);
        try {
            newNode = this.addNode(path, newNode, -1L);
        }
        catch (UnresolvedLinkException e) {
            throw e;
        }
        catch (IOException e) {
            return null;
        }
        return newNode;
    }

    void cacheName(INode inode) {
        if (inode.isDirectory() || inode.isLink()) {
            return;
        }
        ByteArray name = new ByteArray(inode.getLocalNameBytes());
        if ((name = this.nameCache.put(name)) != null) {
            inode.setLocalName(name.getBytes());
        }
    }
}

