/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.namenode;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CountDownLatch;

import org.apache.hadoop.hdfs.server.common.Storage.StorageDirectory;

import com.google.common.base.Preconditions;

/**
 * Context for an ongoing SaveNamespace operation. This class
 * allows cancellation, and also is responsible for accumulating
 * failed storage directories.
 */
class SaveNamespaceContext {
  private final FSNamesystem sourceNamesystem;
  private final long txid;
  private final List<StorageDirectory> errorSDs =
    Collections.synchronizedList(new ArrayList<StorageDirectory>());

  /**
   * If the operation has been canceled, set to the reason why
   * it has been canceled (eg standby moving to active)
   */
  private volatile String cancelReason = null;
  
  private CountDownLatch completionLatch = new CountDownLatch(1);
  
  SaveNamespaceContext(
      FSNamesystem sourceNamesystem,
      long txid) {
    this.sourceNamesystem = sourceNamesystem;
    this.txid = txid;
  }

  FSNamesystem getSourceNamesystem() {
    return sourceNamesystem;
  }

  long getTxId() {
    return txid;
  }

  void reportErrorOnStorageDirectory(StorageDirectory sd) {
    errorSDs.add(sd);
  }

  List<StorageDirectory> getErrorSDs() {
    return errorSDs;
  }

  /**
   * Requests that the current saveNamespace operation be
   * canceled if it is still running.
   * @param reason the reason why cancellation is requested
   * @throws InterruptedException 
   */
  void cancel(String reason) throws InterruptedException {
    this.cancelReason = reason;
    completionLatch.await();
  }
  
  void markComplete() {
    Preconditions.checkState(completionLatch.getCount() == 1,
        "Context already completed!");
    completionLatch.countDown();
  }

  void checkCancelled() throws SaveNamespaceCancelledException {
    if (cancelReason != null) {
      throw new SaveNamespaceCancelledException(
          cancelReason);
    }
  }

  boolean isCancelled() {
    return cancelReason != null;
  }
}
