/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ComparisonChain;
import com.google.common.collect.Lists;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.hdfs.server.common.Storage;
import org.apache.hadoop.hdfs.server.namenode.EditLogFileInputStream;
import org.apache.hadoop.hdfs.server.namenode.EditLogFileOutputStream;
import org.apache.hadoop.hdfs.server.namenode.EditLogInputStream;
import org.apache.hadoop.hdfs.server.namenode.EditLogOutputStream;
import org.apache.hadoop.hdfs.server.namenode.FSEditLogLoader;
import org.apache.hadoop.hdfs.server.namenode.JournalManager;
import org.apache.hadoop.hdfs.server.namenode.NNStorage;
import org.apache.hadoop.hdfs.server.namenode.NNStorageRetentionManager;
import org.apache.hadoop.hdfs.server.protocol.RemoteEditLog;

class FileJournalManager
implements JournalManager {
    private static final Log LOG = LogFactory.getLog(FileJournalManager.class);
    private final Storage.StorageDirectory sd;
    private final NNStorage storage;
    private int outputBufferCapacity = 524288;
    private static final Pattern EDITS_REGEX = Pattern.compile(NNStorage.NameNodeFile.EDITS.getName() + "_(\\d+)-(\\d+)");
    private static final Pattern EDITS_INPROGRESS_REGEX = Pattern.compile(NNStorage.NameNodeFile.EDITS_INPROGRESS.getName() + "_(\\d+)");
    private File currentInProgress = null;
    @VisibleForTesting
    NNStorageRetentionManager.StoragePurger purger = new NNStorageRetentionManager.DeletionStoragePurger();

    public FileJournalManager(Storage.StorageDirectory sd, NNStorage storage) {
        this.sd = sd;
        this.storage = storage;
    }

    @Override
    public void close() throws IOException {
    }

    @Override
    public synchronized EditLogOutputStream startLogSegment(long txid) throws IOException {
        try {
            this.currentInProgress = NNStorage.getInProgressEditsFile(this.sd, txid);
            EditLogFileOutputStream stm = new EditLogFileOutputStream(this.currentInProgress, this.outputBufferCapacity);
            ((EditLogOutputStream)stm).create();
            return stm;
        }
        catch (IOException e) {
            this.storage.reportErrorsOnDirectory(this.sd);
            throw e;
        }
    }

    @Override
    public synchronized void finalizeLogSegment(long firstTxId, long lastTxId) throws IOException {
        File inprogressFile = NNStorage.getInProgressEditsFile(this.sd, firstTxId);
        File dstFile = NNStorage.getFinalizedEditsFile(this.sd, firstTxId, lastTxId);
        LOG.info((Object)("Finalizing edits file " + inprogressFile + " -> " + dstFile));
        Preconditions.checkState((!dstFile.exists() ? 1 : 0) != 0, (Object)("Can't finalize edits file " + inprogressFile + " since finalized file " + "already exists"));
        if (!inprogressFile.renameTo(dstFile)) {
            this.storage.reportErrorsOnDirectory(this.sd);
            throw new IllegalStateException("Unable to finalize edits file " + inprogressFile);
        }
        if (inprogressFile.equals(this.currentInProgress)) {
            this.currentInProgress = null;
        }
    }

    @VisibleForTesting
    public Storage.StorageDirectory getStorageDirectory() {
        return this.sd;
    }

    @Override
    public synchronized void setOutputBufferCapacity(int size) {
        this.outputBufferCapacity = size;
    }

    @Override
    public void purgeLogsOlderThan(long minTxIdToKeep) throws IOException {
        LOG.info((Object)("Purging logs older than " + minTxIdToKeep));
        File[] files = FileUtil.listFiles((File)this.sd.getCurrentDir());
        List<EditLogFile> editLogs = FileJournalManager.matchEditLogs(files);
        for (EditLogFile log : editLogs) {
            if (log.getFirstTxId() >= minTxIdToKeep || log.getLastTxId() >= minTxIdToKeep) continue;
            this.purger.purgeLog(log);
        }
    }

    List<RemoteEditLog> getRemoteEditLogs(long firstTxId) throws IOException {
        File currentDir = this.sd.getCurrentDir();
        List<EditLogFile> allLogFiles = FileJournalManager.matchEditLogs(currentDir);
        ArrayList ret = Lists.newArrayListWithCapacity((int)allLogFiles.size());
        for (EditLogFile elf : allLogFiles) {
            if (elf.hasCorruptHeader() || elf.isInProgress()) continue;
            if (elf.getFirstTxId() >= firstTxId) {
                ret.add(new RemoteEditLog(elf.firstTxId, elf.lastTxId));
                continue;
            }
            if (firstTxId <= elf.getFirstTxId() || firstTxId > elf.getLastTxId()) continue;
            throw new IllegalStateException("Asked for firstTxId " + firstTxId + " which is in the middle of file " + elf.file);
        }
        return ret;
    }

    static List<EditLogFile> matchEditLogs(File logDir) throws IOException {
        return FileJournalManager.matchEditLogs(FileUtil.listFiles((File)logDir));
    }

    static List<EditLogFile> matchEditLogs(File[] filesInStorage) {
        ArrayList ret = Lists.newArrayList();
        for (File f : filesInStorage) {
            Matcher inProgressEditsMatch;
            String name = f.getName();
            Matcher editsMatch = EDITS_REGEX.matcher(name);
            if (editsMatch.matches()) {
                try {
                    long startTxId = Long.valueOf(editsMatch.group(1));
                    long endTxId = Long.valueOf(editsMatch.group(2));
                    ret.add(new EditLogFile(f, startTxId, endTxId));
                }
                catch (NumberFormatException nfe) {
                    LOG.error((Object)("Edits file " + f + " has improperly formatted " + "transaction ID"));
                }
            }
            if (!(inProgressEditsMatch = EDITS_INPROGRESS_REGEX.matcher(name)).matches()) continue;
            try {
                long startTxId = Long.valueOf(inProgressEditsMatch.group(1));
                ret.add(new EditLogFile(f, startTxId, startTxId, true));
            }
            catch (NumberFormatException nfe) {
                LOG.error((Object)("In-progress edits file " + f + " has improperly " + "formatted transaction ID"));
            }
        }
        return ret;
    }

    @Override
    public synchronized EditLogInputStream getInputStream(long fromTxId, boolean inProgressOk) throws IOException {
        for (EditLogFile elf : this.getLogFiles(fromTxId)) {
            if (!elf.containsTxId(fromTxId) || !inProgressOk && elf.isInProgress()) continue;
            if (elf.isInProgress()) {
                elf.validateLog();
            }
            if (LOG.isTraceEnabled()) {
                LOG.trace((Object)("Returning edit stream reading from " + elf));
            }
            EditLogFileInputStream elfis = new EditLogFileInputStream(elf.getFile(), elf.getFirstTxId(), elf.getLastTxId(), elf.isInProgress());
            long transactionsToSkip = fromTxId - elf.getFirstTxId();
            if (transactionsToSkip > 0L) {
                LOG.info((Object)String.format("Log begins at txid %d, but requested start txid is %d. Skipping %d edits.", elf.getFirstTxId(), fromTxId, transactionsToSkip));
            }
            if (!elfis.skipUntil(fromTxId)) {
                throw new IOException("failed to advance input stream to txid " + fromTxId);
            }
            return elfis;
        }
        throw new IOException("Cannot find editlog file containing " + fromTxId);
    }

    @Override
    public long getNumberOfTransactions(long fromTxId, boolean inProgressOk) throws IOException, JournalManager.CorruptionException {
        long numTxns = 0L;
        for (EditLogFile elf : this.getLogFiles(fromTxId)) {
            if (LOG.isTraceEnabled()) {
                LOG.trace((Object)("Counting " + elf));
            }
            if (elf.getFirstTxId() > fromTxId) {
                LOG.warn((Object)("Gap in transactions in " + this.sd.getRoot() + ". Gap is " + fromTxId + " - " + (elf.getFirstTxId() - 1L)));
                break;
            }
            if (!elf.containsTxId(fromTxId)) continue;
            if (!inProgressOk && elf.isInProgress()) break;
            if (elf.isInProgress()) {
                elf.validateLog();
            }
            if (elf.hasCorruptHeader()) break;
            numTxns += elf.getLastTxId() + 1L - fromTxId;
            fromTxId = elf.getLastTxId() + 1L;
            if (!elf.isInProgress()) continue;
            break;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("Journal " + this + " has " + numTxns + " txns from " + fromTxId));
        }
        long max = this.findMaxTransaction(inProgressOk);
        if (numTxns == 0L && fromTxId <= max) {
            String error = String.format("Gap in transactions, max txnid is %d, 0 txns from %d", max, fromTxId);
            LOG.error((Object)error);
            throw new JournalManager.CorruptionException(error);
        }
        return numTxns;
    }

    @Override
    public synchronized void recoverUnfinalizedSegments() throws IOException {
        File currentDir = this.sd.getCurrentDir();
        LOG.info((Object)("Recovering unfinalized segments in " + currentDir));
        List<EditLogFile> allLogFiles = FileJournalManager.matchEditLogs(currentDir);
        for (EditLogFile elf : allLogFiles) {
            if (elf.getFile().equals(this.currentInProgress) || !elf.isInProgress()) continue;
            if (elf.getFile().length() == 0L) {
                LOG.info((Object)("Deleting zero-length edit log file " + elf));
                if (elf.getFile().delete()) continue;
                throw new IOException("Unable to delete file " + elf.getFile());
            }
            elf.validateLog();
            if (elf.hasCorruptHeader()) {
                elf.moveAsideCorruptFile();
                throw new JournalManager.CorruptionException("In-progress edit log file is corrupt: " + elf);
            }
            if (elf.getNumTransactions() == 0L) {
                LOG.info((Object)("Deleting edit log file with zero transactions " + elf));
                if (elf.getFile().delete()) continue;
                throw new IOException("Unable to delete " + elf.getFile());
            }
            this.finalizeLogSegment(elf.getFirstTxId(), elf.getLastTxId());
        }
    }

    List<EditLogFile> getLogFiles(long fromTxId) throws IOException {
        File currentDir = this.sd.getCurrentDir();
        List<EditLogFile> allLogFiles = FileJournalManager.matchEditLogs(currentDir);
        ArrayList logFiles = Lists.newArrayList();
        for (EditLogFile elf : allLogFiles) {
            if (fromTxId > elf.getFirstTxId() && !elf.containsTxId(fromTxId)) continue;
            logFiles.add(elf);
        }
        Collections.sort(logFiles, EditLogFile.COMPARE_BY_START_TXID);
        return logFiles;
    }

    private long findMaxTransaction(boolean inProgressOk) throws IOException {
        boolean considerSeenTxId = true;
        long seenTxId = NNStorage.readTransactionIdFile(this.sd);
        long maxSeenTransaction = 0L;
        for (EditLogFile elf : this.getLogFiles(0L)) {
            if (elf.isInProgress() && !inProgressOk) {
                if (elf.getFirstTxId() == -12345L || elf.getFirstTxId() > seenTxId) continue;
                considerSeenTxId = false;
                continue;
            }
            if (elf.isInProgress()) {
                maxSeenTransaction = Math.max(elf.getFirstTxId(), maxSeenTransaction);
                elf.validateLog();
            }
            maxSeenTransaction = Math.max(elf.getLastTxId(), maxSeenTransaction);
        }
        if (considerSeenTxId) {
            return Math.max(maxSeenTransaction, seenTxId);
        }
        return maxSeenTransaction;
    }

    public String toString() {
        return String.format("FileJournalManager(root=%s)", this.sd.getRoot());
    }

    static class EditLogFile {
        private File file;
        private final long firstTxId;
        private long lastTxId;
        private long numTx = -1L;
        private boolean hasCorruptHeader = false;
        private final boolean isInProgress;
        static final Comparator<EditLogFile> COMPARE_BY_START_TXID = new Comparator<EditLogFile>(){

            @Override
            public int compare(EditLogFile a, EditLogFile b) {
                return ComparisonChain.start().compare(a.getFirstTxId(), b.getFirstTxId()).compare(a.getLastTxId(), b.getLastTxId()).result();
            }
        };

        EditLogFile(File file, long firstTxId, long lastTxId) {
            this(file, firstTxId, lastTxId, false);
            assert (lastTxId != -12345L && lastTxId >= firstTxId);
        }

        EditLogFile(File file, long firstTxId, long lastTxId, boolean isInProgress) {
            assert (lastTxId == -12345L && isInProgress || lastTxId != -12345L && lastTxId >= firstTxId);
            assert (firstTxId > 0L || firstTxId == -12345L);
            assert (file != null);
            this.firstTxId = firstTxId;
            this.lastTxId = lastTxId;
            this.file = file;
            this.isInProgress = isInProgress;
        }

        long getFirstTxId() {
            return this.firstTxId;
        }

        long getLastTxId() {
            return this.lastTxId;
        }

        boolean containsTxId(long txId) {
            return this.firstTxId <= txId && txId <= this.lastTxId;
        }

        void validateLog() throws IOException {
            FSEditLogLoader.EditLogValidation val = EditLogFileInputStream.validateEditLog(this.file);
            this.numTx = val.getNumTransactions();
            this.lastTxId = val.getEndTxId();
            this.hasCorruptHeader = val.hasCorruptHeader();
        }

        long getNumTransactions() {
            return this.numTx;
        }

        boolean isInProgress() {
            return this.isInProgress;
        }

        File getFile() {
            return this.file;
        }

        boolean hasCorruptHeader() {
            return this.hasCorruptHeader;
        }

        void moveAsideCorruptFile() throws IOException {
            assert (this.hasCorruptHeader);
            File src = this.file;
            File dst = new File(src.getParent(), src.getName() + ".corrupt");
            boolean success = src.renameTo(dst);
            if (!success) {
                throw new IOException("Couldn't rename corrupt log " + src + " to " + dst);
            }
            this.file = dst;
        }

        public String toString() {
            return String.format("EditLogFile(file=%s,first=%019d,last=%019d,inProgress=%b,hasCorruptHeader=%b,numTx=%d)", this.file.toString(), this.firstTxId, this.lastTxId, this.isInProgress(), this.hasCorruptHeader, this.numTx);
        }
    }
}

